#line 2 "synch.c"
#include "rs_sys_threads_manager_p.h"

static void requeue( obj thr, obj blocked_on );


void mark_thread_suspended( obj th )
{
  obj susp_count = gvec_ref( th, THREAD_SUSPEND_COUNT );
  int still_in_q = 1;

  if (FX_LT( susp_count, ZERO ))
    {
      still_in_q = 0;
      susp_count = FX_SUB( ZERO, susp_count );
    }
  else if (EQ( susp_count, ZERO ))
    {
      /* newly suspended */
      gvec_write_non_ptr( th, THREAD_STATE, int2fx( TSTATE_SUSPEND ) );
    }

  susp_count = ADD1( susp_count );

  if (still_in_q)
    gvec_write_non_ptr( th, 
			THREAD_SUSPEND_COUNT, 
			susp_count );
  else
    gvec_write_non_ptr( th, 
			THREAD_SUSPEND_COUNT, 
			FX_SUB(ZERO,susp_count) );
}

void mark_thread_resumed( obj th )
{
  obj susp_count = gvec_ref( th, THREAD_SUSPEND_COUNT );
  int still_in_q = 1;

  if (FX_LT( susp_count, ZERO ))
    {
      still_in_q = 0;
      susp_count = FX_SUB( ZERO, susp_count );
    }
  else if (EQ( susp_count, ZERO ))
    {
      /* do nothing */
      return;
    }

  susp_count = SUB1( susp_count );

  if (EQ( susp_count, ZERO ))
    {
      obj blkd_on = gvec_ref( th, THREAD_BLOCKED_ON );

      gvec_write_non_ptr( th, THREAD_SUSPEND_COUNT, ZERO );

      /* unblock it */
      if (EQ( blkd_on, ZERO ))
	{
	  /* if it IS still in the queue, then we need do nothing */

	  if (!still_in_q)
	    mark_thread_ready( th );
	}
      else if (FIXNUM_P( blkd_on ))
	{
	  /* it's an Event, so it must have been sleeping *and*
	     the timer hasn't gone off (see comments in `class.scm'
	   */
	  assert( still_in_q );
	  gvec_set( th, THREAD_STATE, int2fx( TSTATE_SLEEPING ) );
	}
      else
	{
	  /* it must be some other object, like a <mailbox> */
	  if (!still_in_q)
	    {
	      /* put it back in the queue */
	      requeue( th, blkd_on );
	    }
	}
    }
  else
    {
      /* still suspended */

      if (still_in_q)
	gvec_write_non_ptr( th, 
			    THREAD_SUSPEND_COUNT, 
			    susp_count );
      else
	gvec_write_non_ptr( th, 
			    THREAD_SUSPEND_COUNT, 
			    FX_SUB(ZERO,susp_count) );
    }
}

static void send_item_to_thread( obj mbox, obj thr, obj item )
{
  gvec_write_non_ptr( thr, THREAD_BLOCKED_ON, ZERO );
  store_resume_value( thr, item );
  mark_thread_ready( thr );
  if (dequeue_empty( mbox ))
    gvec_write_non_ptr( mbox, MAILBOX_HAS_DATA_Q, TRUE_OBJ );
}

/*
 *   requeue a thread on a blockable scheme object, 
 *   which is one of:
 *
 *      <thread>          -- see `thread-join'
 *      <semaphore>       -- see `semaphore-wait'
 *      <fd-output-port>  -- see `fd-output-port-write-string'
 *      <mailbox>         -- see `receive-message!'
 *
 *   Note that we are only called when the thread
 *   being requeued has been removed from the blocking
 *   queue.
 */


static void requeue( obj thr, obj blocked_on )
{
  if (instance_p( blocked_on, mailbox_class ))
    {
      obj mbox = blocked_on;

      /* two cases to consider
       *   1. mailbox has data now
       *   2. mailbox has no data
       */
      if (truish(gvec_ref(mbox,MAILBOX_HAS_DATA_Q)) && !dequeue_empty(mbox))
	{
	  /* mark_thread_ready() will set the state to WAITING */
	  send_item_to_thread( mbox, thr, dequeue_pop_front( mbox ) );
	}
      else
	{
	  /* put it back in the wait queue */
	  gvec_write_non_ptr( mbox, MAILBOX_HAS_DATA_Q, FALSE_OBJ );
	  dequeue_push_back( mbox, thr );
	  /* now, we're blocked again */
	  gvec_set( thr, THREAD_STATE, int2fx( TSTATE_BLOCKED ) );
	}
    }
  else
    {
      /*  it can't be a <queued-output-port>, because we are only
       *  called when the thread has been ejected from the queue,
       *  and that never happens for qout's (they work like timers)
       *  -- see comments in output.c and class.scm
       */
      assert( !instance_p( blocked_on, qout_class ) );
      assert(0); /* not implemented yet... */
    }
}

static void do_mailbox_send( obj mbox, obj item, int front_q )
{

try_again_from_top:

  if (truish(gvec_ref(mbox,MAILBOX_HAS_DATA_Q)))
    {
      /* the mbox queue contains data... */

      if (DEBUG_THREAD_MBOX)
	printf( " mailbox{%#lx}: inserting another item{%#lx}\n",
		VAL(mbox), VAL(item) );
      if (front_q)
	dequeue_push_front( mbox, item );
      else
	dequeue_push_back( mbox, item );
    }
  else
    {
      obj top;

      /* the mbox queue contains threads... */

    try_again_more:
      /*  note that we remove the longest-waiting thread 
       *  independent of whether or not this is a "prepend"
       *  data item...  I hope that's right...!
       */
      top = dequeue_pop_front(mbox);
      if (!did_remove_from_queue( top ))
	{
	  if (dequeue_empty(mbox))
	    {
	      gvec_write_non_ptr( mbox, MAILBOX_HAS_DATA_Q, TRUE_OBJ );
	      goto try_again_from_top;
	    }
	  else
	    {
	      /* more left in this queue... try another */
	      goto try_again_more;
	    }
	}

      if (DEBUG_THREAD_MBOX)
	printf( " mailbox{%#lx}: delivering item{%#lx} (%ld left waiting)\n",
		VAL(mbox), VAL(item),
		fx2int( dequeue_count(mbox) ));
      send_item_to_thread( mbox, top, item );
    }
}

void ksend_mailbox( obj mbox, obj item )
{
  do_mailbox_send( mbox, item, 0 );
}

void ksend_mailbox_pre( obj mbox, obj item )
{
  do_mailbox_send( mbox, item, 1 );
}
