#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>

#include <time.h>

#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/uio.h>
#include <fcntl.h>

#include "lsspriv.h"
#include "lssv3.h"

/* #define DEBUG_PRINT */

#define ACCESS_IN(a)   ((a)->in_buf)
#define HEADER_FOR(a)  (((struct LSSRecordHeader *)((a)->addr)) - 1)

static struct LSSMethods lss_v3;


static void lssi_close( LSS *generic_ptr );
static void lssi_write_done_z( LSS_V3 *lss, LSSAccess *a, 
			      UINT_32 z_len, UINT_32 unz_len, int z_alg );
#define lssi_write_done_noz(lss,a,len)  lssi_write_done_z(lss,a,len,len,0)

static UINT_32 lssi_write_alloc( LSS_V3 *lss, LSSAccess *a, UINT_32 space, 
				 UINT_32 type );
static void *lssi_read_it( LSS_V3 *lss, UINT_32 at, UINT_32 type );
static void lssi_update_volume_header( LSS_V3 *lss );
static int lssi_read_chk( LSS_V3 *lss, UINT_32 at, void *ptr, UINT_32 len,
			  UINT_32 type );
static int lssi_read( LSS_V3 *lss, UINT_32 at, void *ptr, UINT_32 len );


static UINT_64 lssi_time_ms( void )
{
  UINT_64 t = time(NULL);
  /* FIXME -- work with (UINT_16 * 4) implementation */
  return t*1000;
}

static UINT_64 lssi_gen_serno( int fd )
{
  struct timeval tv;
  UINT_64 serno;
  UINT_32 state;
  int i;
  struct stat sb;

  gettimeofday( &tv, NULL );

  state = (tv.tv_usec + tv.tv_sec) * 4;
  if (fstat( fd, &sb ) == 0)
    {
      state += (sb.st_dev << 10) + sb.st_ino;
    }
  state ^= 0x317febc1;
  serno = 0;

  for (i=0; i<64; i++)
    {
      serno = (serno << 1) + (state & 1);
      switch (state & 3)
	{
	case 0:
	case 3:
	  state = (state >> 1);
	  break;
	case 1:
	case 2:
	  state = (state >> 1) + 0x80000000;
	  break;
	}
    }
  return serno;
}

/**
 *   Write out a data segment (DSEG) from the contents of the
 *   output buffers.
 */

static void lssi_flush_obufs( LSS_V3 *lss )
{
  struct iovec iov[MAX_OUT_BUFS+2];
  int i, ntrailer, nbytes, rc;
  struct LSSRecordHeader gap;

  nbytes = 0;
  for (i=0; i<lss->num_outbufs; i++)
    {
      assert( lss->outbuf[i].num_accesses == 0 );

      iov[i].iov_base = (void *)lss->outbuf[i].base;
      iov[i].iov_len = lss->outbuf[i].ptr - lss->outbuf[i].base;
      nbytes += iov[i].iov_len;
#ifdef DEBUG_PRINT
      printf( "iov[%d] -- %d bytes @ %p\n", i, 
	      iov[i].iov_len, iov[i].iov_base );
#endif /* DEBUG_PRINT */
      
      lss->outbuf[i].ptr = lss->outbuf[i].base;
    }

  ntrailer = IO_BLOCK_SIZE - (nbytes & (IO_BLOCK_SIZE - 1));
  if (ntrailer > sizeof( struct LSSSegmentTrailer ))
    {
      /* add a GAP to make the DSEG appear at the end of an IO_BLOCK */

      nbytes += sizeof gap;
      ntrailer = IO_BLOCK_SIZE - (nbytes & (IO_BLOCK_SIZE - 1));

      gap.magic = GAP_MAGIC;
      gap.recnum = 0;
      /*  the `gap.space' measures the size of the gap including
       *  the gap header.  Since the gap header and the trailer
       *  are the same size, and `ntrailer' is the amount of trailer
       *  including the trailer block, the gap space = ntrailer
       */
      gap.space_word = MAKE_SPACE_WORD( ntrailer, 0 );
      gap.length = 0;

      iov[i].iov_base = (void *)&gap;
      iov[i].iov_len = sizeof gap;
      i++;
    }

  iov[i].iov_base = (void *)(lss->trailing_buf + IO_BLOCK_SIZE - ntrailer);
  iov[i].iov_len = ntrailer;
  i++;

  lss->trailer->magic = DATASEG_MAGIC;
  lss->trailer->length = nbytes + ntrailer;

  rc = lseek( lss->tip_vol->filedes, 0, SEEK_END );
  rc = ROUND_UP( rc );
#ifdef DEBUG_PRINT
  printf( "rc = %d, buf[0].offset = %lu\n", rc, lss->outbuf[0].base_offset );
#endif /* DEBUG_PRINT */

  if (rc != lss->outbuf[0].base_offset )
    {
      if (rc < 0)
	lssi_sys_error( &lss->com, "" );
      else
	lssi_signal_error( &lss->com, LSSERR_NOT_WHERE_EXPECTED,
			   "ll", (long)rc, (long)lss->outbuf[0].base_offset );
    }

#ifdef DEBUG_PRINT
  printf( "tip fd %d\n", lss->tip_vol->filedes );
#endif /* DEBUG_PRINT */
  assert( i <= (MAX_OUT_BUFS+2) );

  errno = 0;
  rc = writev( lss->tip_vol->filedes, iov, i );
  if (rc < 0)
    {
      lssi_sys_error( &lss->com, "l", (long)(nbytes + ntrailer) );
    }
  if (rc != (nbytes + ntrailer))
    {
      lssi_signal_error( &lss->com,
			 LSSERR_SHORT_WRITE,
			 "ll",
			 (long)rc,
			 (long)nbytes + (long)ntrailer );
    }

#ifdef DEBUG_PRINT
  printf( "%s: wrote %d bytes (%d is trailer)\n", 
	  lss->tip_vol->file, rc, ntrailer );
#endif /* DEBUG_PRINT */
  lss->outbuf[0].base_offset += rc;

  for (i=1; i<lss->num_outbufs; i++)
    {
      free( lss->outbuf[i].base );
    }
  lss->num_outbufs = 1;
}

/**
 *  Flush the in-memory commit record to disk,
 *  thereby committing the changes
 */

/**
 *  NOTE:  We could use the existing output buffering mechanism,
 *  if we just provide `flush_out' with a flag that says
 *  "sync before&after last IOBLOCK"
 *
 *  though we need to pad things so that the commit record goes
 *  at the end of the last io block
 */

static void lssi_write_cr( LSS_V3 *lss )
{
  int rc;
  int fd = lss->tip_vol->filedes;
  struct iovec iov[3];
  struct LSSRecordHeader gap;
  UINT_32 ntrailer, at;

  off_t lrc = lseek( fd, 0, SEEK_END );

  if (lrc < 0)
    {
      lssi_sys_error( &lss->com, "" );
    }

  at = lrc;

  rc = fsync( fd );
  if (rc < 0)
    {
      lssi_sys_error( &lss->com, "" );
    }

  lss->cr.prev_cr_at = lss->cr.self_cr_at;
  lss->cr.self_cr_at = MAKE_LOCATOR( lss->num_vols - 1, at );

  lss->cr.cr_space_w = MAKE_SPACE_WORD( CR_SPACE, 0 );

  iov[0].iov_base = (void *)&lss->cr;
  iov[0].iov_len = CR_SPACE;

  ntrailer = IO_BLOCK_SIZE - (iov[0].iov_len + sizeof gap);

  gap.magic = GAP_MAGIC;
  gap.length = 0;
  gap.recnum = 0;
  /* see above comment re `gap.space = ntrailer' */
  gap.space_word = MAKE_SPACE_WORD( ntrailer, 0 ); 

  iov[1].iov_base = (void *)&gap;
  iov[1].iov_len = sizeof gap;

  iov[2].iov_base = (void *)(lss->trailing_buf + IO_BLOCK_SIZE - ntrailer);
  iov[2].iov_len = ntrailer;

  lss->trailer->magic = EOF_MAGIC;
  lss->trailer->length = IO_BLOCK_SIZE;

  lss->cr.generation++;
  lss->cr.commit_time_ms = lssi_time_ms();

  rc = writev( fd, iov, 3 );
  if (rc != IO_BLOCK_SIZE)
    {
      if (rc < 0)
	lssi_sys_error( &lss->com, "" );
      else
	lssi_signal_error( &lss->com, LSSERR_WRITE_COM_FAILED, "" );
    }

  rc = fsync( fd );
  if (rc < 0)
    {
      lssi_sys_error( &lss->com, "" );
    }

  /*  adjust the base offset of the output buffer to refer
   *  PAST the CR we just wrote -- since this mechanism
   *  bypasses the output buffering system
   */
  lss->outbuf[0].base_offset += IO_BLOCK_SIZE;
}

static UINT_32 lssi_write_out( LSS_V3 *lss, const void *data, UINT_32 len, 
			       UINT_32 type )
{
  UINT_32 at;
  LSSAccess a;

  at = lssi_write_alloc( lss, &a, len, type );
  memcpy( a.addr, data, len );
  lssi_write_done_noz( lss, &a, len );
  return at;
}

static void lssi_init_bufs( LSS_V3 *l )
{
  l->num_outbufs = 1;
  l->outbuf[0].base = malloc( FIRST_OBUF_SIZE );
  l->outbuf[0].ptr = l->outbuf[0].base;
  l->outbuf[0].limit = l->outbuf[0].base + FIRST_OBUF_SIZE;
  l->outbuf[0].num_accesses = 0;
  l->outbuf[0].base_offset = 0;
  l->outbuf[0].last_access = NULL;

  l->trailing_buf = malloc( IO_BLOCK_SIZE );
  memset( l->trailing_buf, 0, IO_BLOCK_SIZE );
  l->trailer = (struct LSSSegmentTrailer *)
                   ((l->trailing_buf + IO_BLOCK_SIZE) - TRAILER_SIZE);
  l->trailer->trailer_len = 0;
  l->trailer->trailer_space_w = MAKE_SPACE_WORD( STORAGE_GRANULE, 0 );
}

static void lssi_write_line( LSS_V3 *lss, IndexEntry *ip )
{
  LSSAccess a;
  UINT_32 at;

  at = lssi_write_alloc( lss,
			 &a,
			 sizeof( struct LSSIndexEntries ),
			 INDEX_MAGIC );

  memcpy( a.addr, ip->mem_ptr, sizeof( struct LSSIndexEntries ) );
  HEADER_FOR(&a)->recnum = ip->m.line_key;

  lssi_write_done_noz( lss, &a, sizeof( struct LSSIndexEntries ) );
  ip->m.line_offset = at;
}
				   

static UINT_32 lssi_commit( LSS *generic_ptr )
{
  LSS_V3 *lss = (LSS_V3 *)generic_ptr;

  /* write out the index records, if needed */

  if (lss->cr.mix_offset == 0)
    {
      UINT_32 i, j, n, cap;
      struct MasterIndexEntry *mix;

      cap = lss->mindex_cap;
      n = lss->cr.mix_cnt;
      mix = ALLOCN( struct MasterIndexEntry, n );

#ifdef DEBUG_PRINT
      printf( "flushing index (%lu lines)\n", n );
#endif /* DEBUG_PRINT */

      /* collect all the lines, and write out the dirty ones */
      j = 0;
      for (i=0; i<cap; i++)
	{
	  struct IndexEntry *ip = &lss->mindex[i];

	  if (ip->mem_ptr)
	    {
	      if (ip->m.line_offset == 0)
		{
		  lssi_write_line( lss, ip );  /* updates `line_offset' */
		}
	    }
	  if (ip->m.line_offset)
	    {
	      mix[j++] = ip->m;
	    }
	}
      /*  verify that we found exactly the right
       *  number of entries
       */
      assert( j == n );

      lss->cr.mix_offset = lssi_write_out( lss, 
					   mix,
					   n*sizeof( struct MasterIndexEntry ),
					     MINDEX_MAGIC );
      free( mix ); /* that was a temporary representation */
    }

  lss->cr.vh_fuel -= COMMIT_FUEL_COST;

  /*  however, if we have not written out the volume header yet,
   *  then flush the output buffer so `update' can read it back
   *  in.
   */
  if (lss->outbuf[0].base_offset == 0)
    {
      /*  if we were really smart, we could probably just modify
       *  it in place, instead of writing it and then reading
       *  it in order to update it...  future work.
       */
      lssi_flush_obufs( lss );
    }

  /* update volume header if (necessary or) desired */
  if (lss->cr.vh_fuel <= 0)
    {
      lssi_update_volume_header( lss );
      lss->cr.vh_fuel = VOLUME_HEADER_FUEL;
    }
  /* flush the data segment */
  lssi_flush_obufs( lss );

  /*  note that we have to commit AFTER updating the volume header
   *  because new compression algorithms may have been specified,
   *  and we can't run without them!
   *
   * -- on the other hand, what if we crash before the commit;
   * will the volume header have an incorrect forward pointer to
   * the CR?  We should handle that at load time...
   */
  lssi_write_cr( lss );

#ifdef DEBUG_PRINT
  printf( "fuel left in volume header: %d\n", lss->cr.vh_fuel );
#endif /* DEBUG_PRINT */
  return lss->cr.generation;
}

static LSS_V3 *lssi_new( void )
{
  LSS_V3 *l = (LSS_V3 *)malloc( sizeof( LSS_V3 ) );
  memset( l, 0, sizeof( LSS_V3 ) );
  LSS_INIT_COM( l->com, &lss_v3 );
  l->zip_algs[0] = &lss_null_zip;
  l->zip_algs[1] = NULL;
  return l;
}

/**
 *  Return -1 on error
 */

static int lssi_read_volume_header( int fd, struct LSSVolumeHeader *vh )
{
  int rc;
  struct LSSRecordHeader rh;

  rc = lseek( fd, 0, SEEK_SET );
  if (rc != 0)
    {
      return -1;
    }

  rc = read( fd, &rh, sizeof rh );
  if (rc != sizeof rh)
    {
      errno = LSSERR_SHORT_READ;
      return -1;
    }

  if ((rh.magic != LSS_MAGIC)
      || (rh.recnum != LSSX_VERSION)
      || (rh.length != sizeof( struct LSSVolumeHeader )))
    {
      errno = LSSERR_NOT_LSS;
      return -1;
    }

  rc = read( fd, vh, sizeof( struct LSSVolumeHeader ) );
  if (rc != sizeof( struct LSSVolumeHeader ))
    {
      errno = LSSERR_SHORT_READ;
      return -1;
    }

  return 0;
}

static void lssi_update_volume_header( LSS_V3 *lss )
{
  int rc;
  int fd = lss->tip_vol->filedes;
  struct LSSVolumeHeader vh;
  int nz;

  rc = lseek( fd, sizeof( struct LSSRecordHeader ), SEEK_SET );
  if (rc != sizeof( struct LSSRecordHeader ))
    {
      lssi_signal_error( &lss->com, LSSERR_NOT_WHERE_EXPECTED,
			 "ll", 
			 (long)rc, (long)sizeof( struct LSSRecordHeader ) );
    }

  rc = read( fd, &vh, sizeof vh );
  if (rc != sizeof vh)
    {
      if (rc < 0)
	{
	  lssi_sys_error( &lss->com, "" );
	}
      else
	{
	  lssi_signal_error( &lss->com, LSSERR_SHORT_READ, 
			     "ll", (long)rc, (long)(sizeof vh) );
	}
    }

#ifdef DEBUG_PRINT
  printf( "updating volume header (generation %lu++)\n", vh.vh_generation );
#endif /* DEBUG_PRINT */

  vh.vh_generation++;

  vh.last_cr_at = lss->cr.self_cr_at;
  vh.last_cr_generation = lss->cr.generation;

  /* add any new zip algorithms */

  for (nz = 1; (nz < MAX_ZIP_ALGORITHMS) && vh.zip_alg_at[nz-1]; nz++)
    {
    }
  /*  printf( "had %d algorithms listed\n", nz ); */
  while (lss->zip_algs[nz])
    {
      const char *name = lss->zip_algs[nz]->name;
#ifdef DEBUG_PRINT
      printf( " -- adding zip algorithm[%d] = %s\n", nz, name );
#endif
      vh.zip_alg_at[nz-1] = lssi_write_out( lss, name, strlen( name ) + 1,
					    ZIPA_MAGIC );
      nz++;
    }

  rc = lseek( fd, sizeof( struct LSSRecordHeader ), SEEK_SET );
  if (rc != sizeof( struct LSSRecordHeader ))
    {
      lssi_signal_error( &lss->com, LSSERR_NOT_WHERE_EXPECTED,
			 "ll", 
			 (long)rc, (long)sizeof( struct LSSRecordHeader ) );
    }

  rc = write( fd, &vh, sizeof vh );
  if (rc != sizeof vh)
    {
      if (rc < 0)
	{
	  lssi_sys_error( &lss->com, "" );
	}
      else
	{
	  lssi_signal_error( &lss->com, LSSERR_SHORT_WRITE,
			     "ll", (long)rc, (long)(sizeof vh)  );
	}
    }
}

static void lssi_write_volume_header( LSS_V3 *lss )
{
  int fd, i;
  struct LSSVolumeHeader *vh;
  LSSAccess vha;
  UINT_32 z;

  fd = lss->tip_vol->filedes;

  memset( &vh, 0, sizeof vh );
  lss->outbuf[0].base_offset = 0;

  z = lssi_write_alloc( lss, 
			&vha,
			sizeof( struct LSSVolumeHeader ), 
			LSS_MAGIC );
  HEADER_FOR(&vha)->recnum = LSSX_VERSION;

  vh = vha.addr;

  vh->num_vols = lss->num_vols;
  vh->vol_number = lss->num_vols - 1;

  vh->vh_generation = 1;
  
  vh->vol_create_time_ms = lssi_time_ms();

  vh->last_cr_at = 0;
  vh->last_cr_generation = 0;

  /* write out the zip algorithms */
  
  for (i=1; lss->zip_algs[i]; i++)
    {
      const char *name = lss->zip_algs[i]->name;
      printf( "zip algorithm[%d] = %s\n", i, name );
      vh->zip_alg_at[i-1] = lssi_write_out( lss, name, strlen( name ) + 1,
					    ZIPA_MAGIC );
    }
  for (; i<MAX_ZIP_ALGORITHMS; i++)
    {
      vh->zip_alg_at[i-1] = 0;
    }

  /* write out the volume filenames */
  
  for (i=0; i<lss->num_vols; i++)
    {
      struct LSSVolume *v = &lss->vol[i];
      LSSAccess a;
      UINT_32 n, x;

      n = strlen( v->file );

      x = lssi_write_alloc( lss, &a, n + 1, VOLFN_MAGIC );
      strcpy( a.addr, v->file );
      lssi_write_done_noz( lss, &a, n + 1 );

      vh->vol_info[i].vol_file_at = x;
      vh->vol_info[i].cr_offset = v->cr_offset;
      vh->vol_info[i].vol_serial_num = v->serial_num;
      vh->vol_info[i].spare1 = 0;
      vh->vol_info[i].spare2 = 0;
    }
  lssi_write_done_noz( lss, &vha, sizeof( struct LSSVolumeHeader ) );
  /*lssi_flush_obufs( lss ); */
}


/* 

PrevPrime[i_] := i /; PrimeQ[i]
PrevPrime[i_] := PrevPrime[i-1]

PrevPrime /@ (2^# & /@ Range[3,20])
 */
#define NUM_CAPACITIES  (20-3+1)

/** NOTE:
 *     Because we are using a decent hash function,
 *     these numbers need not really be primes, they
 *     can even be powers of 2.
 */

static UINT_32 cap_table[NUM_CAPACITIES] =
	{ 7, 13, 31, 61, 127, 251, 509, 1021, 2039, 4093, 8191, 16381, 
	  32749, 65521, 131071, 262139, 524287, 1048573 };

LSS *lssv3_create( const char *file, int filemode )
{
  LSS_V3 *lss;

  int fd = open( file, O_RDWR | O_CREAT, filemode );
  if (fd < 0)
    return NULL;

#ifdef DEBUG_PRINT
  printf( "open (for creation) %s\n", file );
#endif

  if (ftruncate( fd, 0 ) < 0)
    {
      close( fd );
      return NULL;
    }

  lss = lssi_new();

  lss->num_vols = 1;
  lss->tip_vol = &lss->vol[0];
  lss->vol[0].filedes = fd;
  lss->vol[0].cr_offset = 0;
  lss->vol[0].serial_num = lssi_gen_serno( fd );
  lss->vol[0].file = strdup( file );

  lssi_init_bufs( lss );

  /* write the volume descriptor */

  lssi_write_volume_header( lss );

  /* initialize the in-memory commit record image */

  lss->cr.magic = COMMIT_MAGIC;
  lss->cr.generation = 0; /* first commit will be generation 1 */
  lss->cr.cr_space_w = MAKE_SPACE_WORD( IO_BLOCK_SIZE, 0 );
  lss->cr.cr_length = 0;
  lss->cr.commit_time_ms = 0;
  lss->cr.mix_cnt = 0;
  lss->cr.mix_offset = 0;  /* causes write at next commit */
  lss->cr.num_diffs = 0;
  lss->cr.vh_fuel = VOLUME_HEADER_FUEL;
  lss->cr.minor_version = LSSV3_MINOR_VERSION; /* latest minor version */

  {
    unsigned i;

    for (i=0; i<NUM_BOOKMARKS; i++)
      lss->cr.bookmarks[i] = 0;
  }

  {
    unsigned i, n;

    n = lss->mindex_cap = cap_table[0];
    lss->mindex = ALLOCN( struct IndexEntry, n );
    for (i=0; i<n; i++)
      {
	lss->mindex[i].mem_ptr = NULL;
	lss->mindex[i].m.line_key = 0;
	lss->mindex[i].m.line_offset = 0;
      }
  }

  return &lss->com;
}

static UINT_32 next_higher_capacity( UINT_32 old_cap )
{
unsigned i;

    /* find the next higher capacity; Note that if
       the old capacity is taken from a different table,
       this may not increase the capacity by much! (but at least 1)
    */
    for (i=0; i<NUM_CAPACITIES; i++)
    {
	if (cap_table[i] > old_cap)
	{
	    return cap_table[i];
	}
    }
    
    /* make it 1.75 times bigger, and fairly odd */

    return ((old_cap * 3) / 2) | 15;
}

static UINT_32 cap_to_use( UINT_32 cnt )
{
  next_higher_capacity( (cnt * 400) / 256 ); 
}


static IndexEntry *lssi_get_line( LSS_V3 *lss, unsigned h )
{
  IndexEntry *ip = &lss->mindex[h];

  if (ip->mem_ptr)
    {
      return ip;
    }
  else if (ip->m.line_offset)
    {
      struct LSSRecordHeader h;
      struct LSSIndexLine *line;
      int fd, rc;
      UINT_32 at = ip->m.line_offset;

      fd = lssi_read( lss, at, &h, sizeof( struct LSSRecordHeader ) );
      if (h.magic != INDEX_MAGIC)
	{
	  lssi_signal_error( &lss->com, LSSERR_BAD_TYPE,
			     "ll", (long)h.magic, (long)INDEX_MAGIC );
	}
      
      ip->mem_ptr = ALLOC( struct LSSIndexEntries );
      rc = read( fd, ip->mem_ptr, sizeof( struct LSSIndexEntries ) );

      if (rc != sizeof( struct LSSIndexEntries ))
	{
	  lssi_signal_error( &lss->com, LSSERR_SHORT_READ,
			     "ll",
			     (long)rc,
			     (long)(sizeof( struct LSSIndexLine)) );
	}
      if (ip->m.line_key != h.recnum)
	{
	  lssi_signal_error( &lss->com, LSSERR_BAD_TYPE, "" );
	}
      return ip;
    }
  else
    {
      return NULL;
    }
}

static void lssi_grow_index( LSS_V3 *lss )
{
  UINT_32 i;
  UINT_32 old_cap = lss->mindex_cap;
  UINT_32 new_cap = next_higher_capacity( old_cap );
  IndexEntry *old_index, *new_index;

  old_index = lss->mindex;
  new_index = ALLOCN( IndexEntry, new_cap );

  for (i=0; i<new_cap; i++)
    {
      new_index[i].mem_ptr = NULL;
      new_index[i].m.line_key = 0;
      new_index[i].m.line_offset = 0;
    }

#ifdef DEBUG_PRINT
  printf( "--- at fill %lu, expanded index capacity from %lu to %lu\n", 
	  lss->index_cnt, old_cap, new_cap );
#endif /* DEBUG_PRINT */
  for (i=0; i<old_cap; i++)
    {
      /* TODO -- fix this to not read everything in */
      IndexEntry *ip = &old_index[i];

      if (ip->m.line_offset || ip->mem_ptr)
	{
	  unsigned h = lssi_hash( ip->m.line_key ) % new_cap;

	  while (new_index[h].m.line_offset || new_index[h].mem_ptr)
		{
	      h = (h + 1) % new_cap;
	    }
	  new_index[h] = *ip;
	}
    }
  lss->mindex = new_index;
  lss->mindex_cap = new_cap;

  lss->cr.mix_offset = 0;   /* force a write */
  lss->cr.num_diffs = 0;    /* roll in diffs, too */
}


/**
 *   Store a key/value pair into the index.
 *   This may involve writing an entry into the diff list,
 *   and/or allocating a new index line for the index.
 */

/**
 *   Store an entry into the diff list
 */

static void lssi_diff_store( LSS_V3 *lss, UINT_32 key, UINT_32 value )
{
  unsigned i;

  /* store it in the diff list */

  /* don't bother if we are going to
   * write out the whole index anyway
   */

  if (lss->cr.mix_offset == 0)
    return;

  /* if the diff list is full, mark
   * the index as flushable
   */

  if (lss->cr.num_diffs >= MAX_DIFFS)
    {
      /* force a flush of the index to disk */
      lss->cr.mix_offset = 0;
      lss->cr.num_diffs = 0;  /* diffs are rolled in */
#ifdef DEBUG_PRINT
      printf( "      flushing diff list\n" );
#endif /* DEBUG_PRINT */
      return;
    }

  /* if the key is already in the list, just update the value */

  for (i=0; i<lss->cr.num_diffs; i++)
    {
      if (lss->cr.diffs[i].diff_recnum == key)
	{
	  lss->cr.diffs[i].diff_offset = value;
#ifdef DEBUG_PRINT
	  printf( "      already in entry diffs[%u]\n", i );
#endif /* DEBUG_PRINT */
	  return;
	}
    }

  /* store it in the diff list */

  i = lss->cr.num_diffs++;

  lss->cr.diffs[i].diff_recnum = key;
  lss->cr.diffs[i].diff_offset = value;
#ifdef DEBUG_PRINT
  printf( "      entry in diffs[%u]\n", i );
#endif /* DEBUG_PRINT */
}

/**
 *   Store an entry into the main index
 *   (use lssi_diff_store separately to store diff-list entries,
 *   so this procedure can be used when elaborating the diff list
 *   itself at load time)
 */


static void lssi_index_store( LSS_V3 *lss, UINT_32 key, UINT_32 value )
{
  UINT_32 hash, my_key, in_line;

  /* compute the hash value */

  my_key = key / INDEX_LINE_SIZE;
  in_line = key % INDEX_LINE_SIZE;

restart_store:

  hash = lssi_hash( my_key ) % lss->mindex_cap;

#ifdef DEBUG_PRINT
  printf( "STORE key %lu: hash %lu, entry %lu\n", key, hash, in_line );
#endif /* DEBUG_PRINT */


  /* find the index line */
  while (1)
    {
      IndexEntry *ip = lssi_get_line( lss, hash );

      if (!ip)
	{
	  /* allocate a new line */
	  int i;
	      
#ifdef DEBUG_PRINT
	  printf( "hash %lu not in line index...\n", hash );
#endif /* DEBUG_PRINT */
	  if (lss->cr.mix_cnt >= MAX_FILL(lss->mindex_cap))
	    {
	      lssi_grow_index( lss );
	      goto restart_store;
	    }
	      
	  ip = &lss->mindex[hash];
	  ip->mem_ptr = ALLOC( struct LSSIndexEntries );
	  ip->m.line_key = my_key;
	  ip->m.line_offset = 0;
	      
	  for (i=0; i<INDEX_LINE_SIZE; i++)
	    ip->mem_ptr->entry[i] = 0;

	  ip->mem_ptr->entry[in_line] = value;
	  lss->cr.mix_cnt++;
	  lss->cr.mix_offset = 0; /* force a MIDX flush */
	  lss->cr.num_diffs = 0; /* roll in the diffs */

#ifdef DEBUG_PRINT
	  printf( "hash %lu makes %lu lines indexed\n", 
		  hash, lss->cr.mix_cnt );
#endif /* DEBUG_PRINT */
	      
	  return;
	}
      else if (ip->m.line_key == my_key)
	{
	  ip->mem_ptr->entry[ in_line ] = value;
	  ip->m.line_offset = 0;	  /* mark the line for output */
	  return;
	}
      hash = (hash + 1) % lss->mindex_cap;
    }
}

static UINT_32 lssi_index_load( LSS_V3 *lss, UINT_32 key )
{
  UINT_32 hash, my_key, in_line;

  /* compute the hash value */

  my_key = key / INDEX_LINE_SIZE;
  in_line = key % INDEX_LINE_SIZE;
  hash = lssi_hash( my_key ) % lss->mindex_cap;

  /* find the index line */
  while (1)
    {
      IndexEntry *ip = lssi_get_line( lss, hash );

      if (!ip)
	{
#ifdef DEBUG_PRINT
	  printf( "LOAD key %lu: hash %lu, entry %lu: NO LINE\n", 
		  key, hash, in_line );
#endif /* DEBUG_PRINT */
	  return 0;
	}
      else if (ip->m.line_key == my_key)
	{
#ifdef DEBUG_PRINT
	  printf( "LOAD key %lu: hash %lu, entry %lu: %#lx\n", 
		  key, hash, in_line, line->entries.entry[ in_line ] );
#endif /* DEBUG_PRINT */
	  return ip->mem_ptr->entry[ in_line ];
	}
      hash = (hash + 1) % lss->mindex_cap;
    }
}

static int lssi_read( LSS_V3 *lss, UINT_32 at, void *ptr, UINT_32 len )
{
  int rc;
  UINT_32 vol, off;
  int fd;

  vol = EXTRACT_VOLUME( at );
  off = EXTRACT_OFFSET( at );
  /* printf( "%#lx ==> v.%lu offset %#lx\n", at, vol, off ); */
  fd = lss->vol[vol].filedes;
  
  rc = lseek( fd, off, SEEK_SET );
  if (rc != off)
    {
      if (rc < 0)
	lssi_sys_error( &lss->com, "" );
      else
	lssi_signal_error( &lss->com, LSSERR_NOT_WHERE_EXPECTED,
			   "ll", (long)rc, (long)off );
    }

  rc = read( fd, ptr, len );
  if (rc != len)
    {
      if (rc < 0)
	lssi_sys_error( &lss->com, "" );
      else
	lssi_signal_error( &lss->com, LSSERR_SHORT_READ,
			   "ll", (long)rc, (long)len );
    }
  return fd;
}

static int lssi_read_chk( LSS_V3 *lss, UINT_32 at, void *ptr, UINT_32 len,
			  UINT_32 type )
{
  int fd = lssi_read( lss, at, ptr, len );
  assert( ((struct LSSRecordHeader *)ptr)->magic == type );
  return fd;
}

static void *lssi_read_ith( LSS_V3 *lss, UINT_32 at, UINT_32 type,
			    struct LSSRecordHeader *h )
{
  void *ptr;
  int rc;
  int fd = lssi_read( lss, at, h, sizeof( struct LSSRecordHeader ) );
  int n;

  assert( h->magic == type );

  /* `h->length' is the uncompressed length -- we want to read
   * the compressed len, which we can only approximate
   */

  n = GET_SPACE_BYTES( h->space_word );

  ptr = malloc( n );
  rc = read( fd, ptr, n );
  if (rc != n)
    {
      if (rc < 0)
	lssi_sys_error( &lss->com, "" );
      else
	lssi_signal_error( &lss->com, LSSERR_SHORT_READ,
			   "ll", (long)rc, (long)n );
    }
  return ptr;
}

static void *lssi_read_it( LSS_V3 *lss, UINT_32 at, UINT_32 type )
{
  struct LSSRecordHeader h;
  return lssi_read_ith( lss, at, type, &h );
}

static int lssi_read_data( LSS_V3 *lss, UINT_32 at, void *ptr, UINT_32 len,
			   UINT_32 type )
{
  struct LSSRecordHeader h;
  int rc;
  int fd = lssi_read( lss, at, &h, sizeof h );

  assert( h.magic == type );
  assert( h.length = len );

  rc = read( fd, ptr, len );
  if (rc != len)
    {
      lssi_signal_error( &lss->com, LSSERR_SHORT_READ,
			 "ll", (long)rc, (long)len );
    }
  return fd;
}


static UINT_32 lssi_find_last_cr( LSS_V3 *lss, UINT_32 at )
{
  int rc, fd;
  struct LSSRecordHeader h;
  UINT_32 i;

  fd = lss->tip_vol->filedes;
 
  /* find the last commit record */

  /* check the EOF; if there is an '*EOF' segment there, then that's it */

  rc = lseek( fd, -IO_BLOCK_SIZE, SEEK_END );
  if ((rc & (IO_BLOCK_SIZE-1)) == 0)
    {
      char ioblock[IO_BLOCK_SIZE];
      struct LSSSegmentTrailer *t;
      int nr;

      t = (void *)&ioblock[IO_BLOCK_SIZE - sizeof( struct LSSSegmentTrailer )];

      nr = read( fd, ioblock, IO_BLOCK_SIZE );
      if (nr == IO_BLOCK_SIZE)
	{
	  if (t->magic == EOF_MAGIC)
	    {
	      at = MAKE_LOCATOR( lss->num_vols - 1, rc );

#ifdef DEBUG_PRINT
	      printf( "lssi_find_cr: found COMM block at %#lx\n", at );
#endif /* DEBUG_PRINT */
	      goto found_at;
	    }
	  else
	    {
#ifdef DEBUG_PRINT
	      printf( "lssi_find_cr: magic is %08lx, not EOF_MAGIC\n", 
		      t->magic );
#endif /* DEBUG_PRINT */
	    }
	}
      else
	{
#ifdef DEBUG_PRINT
	  printf( "lssi_find_cr: read %d at end -- not an IOBlock\n", nr );
#endif /* DEBUG_PRINT */
	}
    }
  
  /* start at the last known commit record */

#ifdef DEBUG_PRINT
  printf( "lssi_find_cr( %#lx ), %u vols\n", at, lss->num_vols );
#endif /* DEBUG_PRINT */

  assert( EXTRACT_VOLUME( at ) == (lss->num_vols - 1) );
  i = EXTRACT_OFFSET( at );

  while (1)
    {
#ifdef DEBUG_PRINT
      printf( "  checking %#lx\n", i );
#endif /* DEBUG_PRINT */
      rc = lseek( fd, i, SEEK_SET );
      if (rc != i)
	break;
      
      rc = read( lss->tip_vol->filedes, &h, sizeof h );
      if (rc != sizeof h)
	break;
      if (h.magic == COMMIT_MAGIC)
	{
	  at = MAKE_LOCATOR( lss->num_vols - 1, i );
	}
      i += GET_SPACE_BYTES( h.space_word );
    }

found_at:

  lssi_read_chk( lss, at, &lss->cr, sizeof( struct LSSCommitRecord ),
		 COMMIT_MAGIC );
  return at;
}

static UINT_32 lssi_find_cr( LSS_V3 *lss, int gen, UINT_32 at )
{
  at = lssi_find_last_cr( lss, at );

  if (gen != 0)
    {
      while (gen != lss->cr.generation)
	{
	  at = lss->cr.prev_cr_at;
#ifdef DEBUG_PRINT
	  printf( "previous at: %#lx... ", at );
#endif /* DEBUG_PRINT */
	  lssi_read_chk( lss, at, &lss->cr, sizeof( struct LSSCommitRecord ),
			 COMMIT_MAGIC );
#ifdef DEBUG_PRINT
	  printf( "gen %lu\n", lss->cr.generation );
#endif /* DEBUG_PRINT */
	}
      /* look backwards for it */
    }
  return at;
}

LSS *lssv3_open( const char *file, int fd, int writable, int gen )
{
  LSS_V3 *lss;
  struct LSSVolumeHeader vh;

  if (lssi_read_volume_header( fd, &vh ) < 0)
    {
      close( fd );
      return NULL;
    }

  lss = lssi_new();

  lssi_init_bufs( lss );
  lss->outbuf[0].base_offset = ROUND_UP( lseek( fd, 0, SEEK_END ) );

  lss->num_vols = vh.num_vols;
  lss->tip_vol = &lss->vol[ lss->num_vols - 1 ];

  lss->vh_last_cr_at = vh.last_cr_at;
  lss->vh_last_cr_generation = vh.last_cr_generation;

  {
    struct LSSVolume *v = lss->tip_vol;
    unsigned i;
    
    v->filedes = fd;
    v->cr_offset = 0;
    v->file = strdup( file );  /* ignore stored value */
    v->serial_num = vh.vol_info[vh.num_vols-1].vol_serial_num;

    /* open older volumes */

    for (i=0; i<vh.num_vols-1; i++)
      {
	int fd;
	char *file = lssi_read_it( lss, 
				   vh.vol_info[i].vol_file_at, 
				   VOLFN_MAGIC );

	/*  make sure we remember the ptr in case we bail,
	 *  so we don't leak
	 */
	lss->vol[i].file = file;

#ifdef DEBUG_PRINT
	printf( "opening volume %d: %s\n", i, file );
#endif /* DEBUG_PRINT */
	fd = open( file, O_RDONLY );
	if (fd < 0)
	  {
	    lssi_sys_error( &lss->com, "s", file );
	  }

	lss->vol[i].filedes = fd;
	lss->vol[i].cr_offset = vh.vol_info[i].cr_offset;
	lss->vol[i].serial_num = vh.vol_info[i].vol_serial_num;

	/*  read the volume header to make sure the
	 *  serial numbers match
	 */
	{
	  struct LSSVolumeHeader backing_vh;

	  if (lssi_read_volume_header( fd, &backing_vh ) < 0)
	    {
	      lssi_close( &lss->com );
	      errno = LSSERR_BAD_VOLHDR;
	      return NULL;
	    }
	  /*
	  printf( "%s[%d]: expect: %Lx  got: %Lx\n",
		  file,
		  i,
		  vh.vol_info[i].vol_serial_num,
		  backing_vh.vol_info[i].vol_serial_num );
		  */
	  if (backing_vh.vol_info[i].vol_serial_num 
	      != vh.vol_info[i].vol_serial_num)
	    {
	      lssi_close( &lss->com );
	      errno = LSSERR_VOLSER_MISMATCH;
	      return NULL;
	    }
	}
      }
  }

  {
    /* link to compression algorithms used */
    unsigned i;

    for (i=1; i<MAX_ZIP_ALGORITHMS; i++)
      {
	zip_algorithm *use = NULL;
	if (vh.zip_alg_at[i-1])
	  {
	    char *name = lssi_read_it( lss, vh.zip_alg_at[i-1], ZIPA_MAGIC );
	    use = lss_find_zip_algorithm( name );
	    if (!use)
	      {
		lssi_close( &lss->com );
		errno = LSSERR_ZIP_ALGORITHM_NOT_DEF;
		return NULL;
	      }
	    free( name );
	  }
	lss->zip_algs[i] = use;
      }
  }

  /* find and read the commit record */

  {
    UINT_32 at = lssi_find_cr( lss, gen, vh.last_cr_at );
#ifdef DEBUG_PRINT
    printf( "%#lx: commit record (gen %lu)\n", at, lss->cr.generation );
#endif /* DEBUG_PRINT */

    if (lss->cr.minor_version > (LSSV3_MINOR_VERSION + LSSV3_MINOR_WIGGLE))
      {
	/* refuse to read minor versions (too much) newer than we are */

	lssi_close( &lss->com );
	errno = LSSERR_BAD_VER;
	return NULL;
      }
    /*  if there were minor versions we cared to read, we could
     *  detect and deal with them here...
     */
  }

  /* read in the master index */

  /* (note... this could be improved, but a basic cost of
     packing it on disk is that it costs more to load the master 
     index) */
  
  {
    UINT_32 i, num, cap;
    struct MasterIndexEntry *mix;
    
    num = lss->cr.mix_cnt;
    cap = cap_to_use( num );
    mix = ALLOCN( struct MasterIndexEntry, num );

    lssi_read_data( lss, lss->cr.mix_offset, 
		    mix, sizeof( struct MasterIndexEntry ) * num,
		    MINDEX_MAGIC );

    lss->mindex = ALLOCN( IndexEntry, cap );
    lss->mindex_cap = cap;

    for (i=0; i<cap; i++)
      {
	lss->mindex[i].m.line_key = 0;
	lss->mindex[i].m.line_offset = 0;
	lss->mindex[i].mem_ptr = NULL;
      }

    for (i=0; i<num; i++)
      {
	/* insert into mindex */
	unsigned h = lssi_hash( mix[i].line_key ) % cap;

	assert( mix[i].line_offset != 0 );

	while (lss->mindex[h].m.line_offset != 0)
	  {
	    h = (h + 1) % cap;
	  }
	lss->mindex[h].m = mix[i];
      }
  }
  /* apply the diffs */

  {
    unsigned i, n = lss->cr.num_diffs;

    for (i=0; i<n; i++)
      {
	lssi_index_store( lss, 
			  lss->cr.diffs[i].diff_recnum,
			  lss->cr.diffs[i].diff_offset );
      }
  }
  return &lss->com;
}

LSS *lssv3_extend( const char *file, int filemode,
		   const char *from, int gen )
{
  LSS_V3 *lss;
  int back_fd;
  int fd = open( file, O_RDWR | O_CREAT, filemode );

  if (fd < 0)
    return NULL;

  ftruncate( fd, 0 );

  back_fd = open( from, O_RDONLY );
  if (back_fd < 0)
    {
      close( fd );
      unlink( file );
      return NULL;
    }

  lss = (LSS_V3 *)lssv3_open( from, back_fd, 0, gen );
  if (!lss)
    {
      int e = errno;

      close( fd );
      close( back_fd );
      unlink( file );
      errno = e;

      return NULL;
    }

  /* push a new file onto the stack of volumes */

  lss->vol[ lss->num_vols ].filedes = fd;
  lss->vol[ lss->num_vols ].cr_offset = 0;
  lss->vol[ lss->num_vols ].serial_num = lssi_gen_serno( fd );
  lss->vol[ lss->num_vols ].file = strdup( file );
  
  lss->tip_vol = &lss->vol[ lss->num_vols ];
  lss->num_vols++;

  lssi_write_volume_header( lss );

  return &lss->com;
}


static void lssi_close( LSS *generic_ptr )
{
  int i;
  LSS_V3 *lss = (LSS_V3 *)generic_ptr;

  for (i=0; i<lss->num_vols; i++)
    {
      if (lss->vol[i].file)
	free( lss->vol[i].file );

      close( lss->vol[i].filedes );
    }
  
  for (i=0; i<lss->num_outbufs; i++)
    {
      free( lss->outbuf[i].base );
    }
  free( lss->trailing_buf );

  for (i=0; i<lss->mindex_cap; i++)
    {
      if (lss->mindex[i].mem_ptr)
	free( lss->mindex[i].mem_ptr );
    }
  free( lss->mindex );
  free( lss );
}

static void lssi_write_done_z( LSS_V3 *lss,
			       LSSAccess *a, 
			       UINT_32 z_length, UINT_32 unz_len,
			       int z_alg )
{
  struct LSSRecordHeader *h = (void *)((UINT_8 *)a->addr - HEADER_SIZE);
  struct IOBuf *in = ACCESS_IN(a);
  UINT_32 used_space = ROUND_UP( z_length + HEADER_SIZE );

  assert( in );

  if (in->last_access == a)
    {
      in->ptr = (UINT_8 *)a->addr - HEADER_SIZE + used_space;
      in->last_access = NULL;
    }

#ifdef DEBUG_PRINT
  printf( "%s:%#lx: done with write: %lu bytes (%lu unz) (space=%lu) @ %p\n",
	  lss->tip_vol->file, 
	  in->base_offset + ((UINT_8 *)a->addr - in->base),
	  z_length, unz_len, used_space,
	  a->addr );
#endif /* DEBUG_PRINT */

  lss->cr.vh_fuel -= BLOCK_FUEL_COST;
  h->length = unz_len;
  h->space_word = MAKE_SPACE_WORD( used_space, z_alg );
  in->num_accesses--;
  ACCESS_IN(a) = NULL;
}

static UINT_32 lssi_write_alloc( LSS_V3 *lss, LSSAccess *a, UINT_32 space, 
				 UINT_32 type )
{
  struct IOBuf *obuf = &lss->outbuf[ lss->num_outbufs - 1 ];
  UINT_32 real_space = ROUND_UP( space + HEADER_SIZE );
  UINT_8 *nxt_ptr = obuf->ptr + real_space;

  if (nxt_ptr > obuf->limit)
    {
      if (lss->num_outbufs < MAX_OUT_BUFS)
	{
	  struct IOBuf *no = &lss->outbuf[ lss->num_outbufs++ ];
	  UINT_32 w;

	  if (real_space < MIN_OBUF_SIZE)
	    w = MIN_OBUF_SIZE;
	  else
	    w = real_space;
	  
#ifdef DEBUG_PRINT
	  printf( "adding new obuf -- %lu bytes\n", w );
#endif /* DEBUG_PRINT */
	  no->base = malloc( w );
	  no->ptr = no->base;
	  no->limit = no->base + w;
	  no->num_accesses = 0;
	  no->base_offset = obuf->base_offset + (obuf->ptr - obuf->base);
	  no->last_access = NULL;
	}
      else
	{
	  lssi_flush_obufs( lss );
	}
      return lssi_write_alloc( lss, a, space, type );
    }
  else
    {
      struct LSSRecordHeader *h = (void *)obuf->ptr;
      UINT_32 off;

      /* more for debugging than anything, 
	 clear out the space being reserved */
      
      memset( obuf->ptr, 0, real_space );
      
      h->magic = type;
      h->space_word = MAKE_SPACE_WORD( real_space, 0 ); /*uncompressed so far*/
      h->recnum = 0;
      h->length = 0;

      off = obuf->base_offset + (obuf->ptr - obuf->base);
      ACCESS_IN(a) = obuf;
      a->addr = obuf->ptr + HEADER_SIZE;
      a->bytes = real_space - HEADER_SIZE;
      obuf->ptr = nxt_ptr;
      obuf->num_accesses++;
      obuf->last_access = a;
#ifdef DEBUG_PRINT
      printf( "write alloc %lu bytes: @ %p\n", real_space, a->addr );
#endif /* DEBUG_PRINT */
      return MAKE_LOCATOR( (lss->num_vols - 1), off );
    }
}

static LSSAccess *lssi_read_access( LSS *generic_ptr, UINT_32 recnum )
{
  LSS_V3 *lss = (LSS_V3 *)generic_ptr;
   UINT_32 at;

  at = lssi_index_load( lss, recnum );
#ifdef DEBUG_PRINT
  printf( "rec %lu ==> at: %#lx\n", recnum, at );
#endif /* DEBUG_PRINT */
  if (at)
    {
      LSSAccess *a = ALLOC( LSSAccess ); /* okay; expensive for now! */
      struct LSSRecordHeader h;
      int alg;

      /*  TODO -- please support reading back a record before 
       *  it has been flushed to disk!
       */
      a->addr = lssi_read_ith( lss, at, DATAREC_MAGIC, &h );
      a->bytes = h.length;
      a->space = GET_SPACE_BYTES( h.space_word );

      /*
      printf( "space word = %#08x, %lu unz\n", h.space_word, h.length );
      print_hex_data( a->addr, GET_SPACE_BYTES( h.space_word ) );
      printf( "\n" );
      */

      alg = GET_WHICH_ZIP( h.space_word );
      a->uses = lss->zip_algs[ alg ];
#ifdef DEBUG_PRINT
      printf( "using zip[%d] = %s\n", alg, a->uses->name );
#endif

      ACCESS_IN(a) = NULL;
      return a;
    }
  else
    {
      return NULL;
    }
}

static void lssi_read_release( LSS *generic_ptr, LSSAccess *a )
{
  /* LSS_V3 *lss = (LSS_V3 *)generic_ptr; */

  if (a)
    {
      if (a->bytes)
	{
	  assert( a->addr );
	  free( a->addr );
	}
      a->addr = NULL;
      free( a );
    }
}

static size_t lssi_copy_record( LSS *generic_ptr,
			        LSS *dest,
			        UINT_32 recnum )
{
  LSSAccess *a = lssi_read_access( generic_ptr, recnum );

  if (a)
    {
      size_t n = a->space;

      dest->fn->raw_write_meth( dest, recnum, a->addr, a->bytes, a->uses, n );
      lssi_read_release( generic_ptr, a );

      return n;
    }
  else
   {
     return 0;
   }
}

static unsigned count_used( IndexEntry *ip )
{
  unsigned i, n = 0;

  if (ip)
    {
      for (i=0; i<INDEX_LINE_SIZE; i++)
	{
	  if (ip->mem_ptr->entry[i])
	    n++;
	}
    }
  return n;
}

static UINT_32 *lssi_get_index( LSS *gp, UINT_32 *cnt )
{
  LSS_V3 *lss = (LSS_V3 *)gp;

  UINT_32 *ix;
  UINT_32 i, j, k, cap, n = 0;

  cap = lss->mindex_cap;

  for (i=0; i<cap; i++)
    {
      n += count_used( lssi_get_line( lss, i ) );
    }

  ix = ALLOCN( UINT_32, n );
  j = 0;

#ifdef DEBUG_PRINT
  printf( "get-index: %lu entries\n", n );
#endif
  for (i=0; i<cap; i++)
    {
      IndexEntry *ip = lssi_get_line( lss, i );
      if (ip)
	{
	  /* first record number in index line */
	  UINT_32 frn = ip->m.line_key * INDEX_LINE_SIZE;

	  for (k=0; k<INDEX_LINE_SIZE; k++)
	    {
	      if (ip->mem_ptr->entry[k])
		{
		  ix[j++] = k + frn;
		}
	    }
	}
    }
  assert( j == n );
  *cnt = n;
  return ix;
}

static void lssi_get_info( LSS *gp, UINT_32 rec, struct LSSRecordInfo *info )
{
  LSS_V3 *lss = (LSS_V3 *)gp;
  UINT_32 at = lssi_index_load( lss, rec );

  info->record_num = rec;
  info->volume = EXTRACT_VOLUME(at);
  info->offset = EXTRACT_OFFSET(at);
}


static void lssi_readv( LSS *generic_ptr,
			zipbuf *dest,
			LSSAccess *a )
{
  /* LSS_V3 *lss = (LSS_V3 *)generic_ptr; */
  void *end_ptr;

  if (!a)
    return; /* nothing to do */

#ifdef DEBUG_PRINT
  printf( " readv using %s\n", a->uses->name );
#endif

  end_ptr = a->uses->unzip_meth( a->uses, dest, a->addr );
  /*  should check something about the end_ptr, to make
   *  we ate exactly the input (as near as we can tell,
   *  because the compressed size was rounded off to the
   *  granularity)
   */
#ifdef DEBUG_PRINT
  printf( " --> ate %u bytes\n", 
	  (unsigned)((char *)end_ptr - (char *)a->addr) );
#endif
}

static int get_zip_id( LSS *generic_ptr, zip_algorithm *use )
{
  LSS_V3 *lss = (LSS_V3 *)generic_ptr;
  if (use)
    {
      int zip_id;

      for (zip_id=0; lss->zip_algs[zip_id]; zip_id++)
	{
	  if (lss->zip_algs[zip_id] == use)
	    return zip_id;
	}
      if (zip_id >= MAX_ZIP_ALGORITHMS)
	{
	  lssi_signal_error( (LSS *)lss, 
			     LSSERR_TOO_MANY_ALGORITHMS, 
			     "s", use->name );
	}
      lss->zip_algs[zip_id] = use;
      lss->zip_algs[zip_id+1] = NULL; /* reinstall NULL end-marker */
      /* force a flush of the volume header */
      lss->cr.vh_fuel = -1;
      return zip_id;
    }
  else
    {
      use = &lss_null_zip;
      return 0;
    }
}

static void lssi_writev( LSS *generic_ptr, 
			 UINT_32 recnum,
			 zipbuf *datav,
			 zip_algorithm *use )
{
  LSS_V3 *lss = (LSS_V3 *)generic_ptr;
  int zip_id = get_zip_id( generic_ptr, use );
  struct LSSRecordHeader *h;
  UINT_32 at;
  size_t unz_len;
  LSSAccess a;
  void *end_ptr;

  /* allocate the record space */

  unz_len = zipbuf_len( datav );
  at = lssi_write_alloc( lss, &a, MAX_ZLEN(unz_len), DATAREC_MAGIC );
  h = HEADER_FOR(&a);
  h->recnum = recnum;
  lssi_index_store( lss, recnum, at );
  lssi_diff_store( lss, recnum, at );

  /* compress into the buffer */

  if (use)
    end_ptr = use->zip_meth( use, a.addr, datav );
  else
    end_ptr = lss_null_zip.zip_meth( &lss_null_zip, a.addr, datav );

#ifdef DEBUG_PRINT
  printf( " --> wrote %u bytes\n", 
	  (unsigned)((char *)end_ptr - (char *)a.addr) );
#endif
  /* close it off */

  lssi_write_done_z( lss, &a, 
		     (char *)end_ptr - (char *)a.addr, 
		     unz_len,
		     zip_id );
}

static void lssi_raw_write( LSS *generic_ptr,
			    UINT_32 recnum,
			    void *data,
			    size_t unz_len,
			    zip_algorithm *used,
			    size_t z_len )
{
  LSS_V3 *lss = (LSS_V3 *)generic_ptr;
  int zip_id = get_zip_id( generic_ptr, used );
  struct LSSRecordHeader *h;
  UINT_32 at;
  LSSAccess a;
  void *end_ptr;

  /* allocate the record space */

  at = lssi_write_alloc( lss, &a, z_len, DATAREC_MAGIC );
  h = HEADER_FOR(&a);
  h->recnum = recnum;
  lssi_index_store( lss, recnum, at );
  lssi_diff_store( lss, recnum, at );

  /* compress into the buffer */

  memcpy( a.addr, data, z_len );

#ifdef DEBUG_PRINT
  printf( " --> wrote %u bytes\n", 
	  (unsigned)((char *)end_ptr - (char *)a.addr) );
#endif
  /* close it off */

  lssi_write_done_z( lss, &a, z_len, unz_len, zip_id );
}

/* (vol_num == -1) => latest volume */

static const char *lssi_filename( LSS *generic_ptr, int vol_num )
{
  LSS_V3 *lss = (LSS_V3 *)generic_ptr;

  if (vol_num < 0)
    /* -1 => last, -2 => next to last, etc */
    vol_num = lss->num_vols + vol_num;

  if ((vol_num < 0) || (vol_num >= lss->num_vols))
    return NULL;
  else
    return lss->vol[vol_num].file;
}

static struct LSSMethods lss_v3 = {
  "V3",
  lssi_close,
  lssi_commit,
  lssi_read_access,
  lssi_readv,
  lssi_read_release,
  lssi_writev,
  lssi_get_index,
  lssi_get_info,
  lssi_copy_record,
  lssi_raw_write,
  lssi_filename
};
