/*
 * GSPR_QD.C - PGS QuickDraw primitive routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"

void
 _PG_set_text_font_mac(PG_device *dev, int index),
 _PG_set_text_size_mac(PG_device *dev, int size_index, int flag);
 
 static void _PG_qd_draw_polyline(PG_device *dev, REAL *x, REAL *y, int n);

/*--------------------------------------------------------------------------*/

/*                         STATE QUERY ROUTINES                             */

/*--------------------------------------------------------------------------*/

/* _PG_QD_GET_TEXT_EXT_NDC - return the text extent in NDC of the given string */

void _PG_qd_get_text_ext_NDC(dev, s, px, py)
   PG_device *dev;
   char *s;
   REAL *px, *py;
   {REAL x, y;
    int ix, iy;
    FontInfo info;

    SetPort(dev->window);

    STRING_WIDTH(ix, s);

    GetFontInfo(&info);
    iy = info.ascent + info.descent;

    x = ((REAL) ix)/dev->window_width;
    y = ((REAL) iy)/dev->window_height;

    *px = x;
    *py = 0.7*y;

    return;}

/*--------------------------------------------------------------------------*/

/*                          STATE CHANGE ROUTINES                           */

/*--------------------------------------------------------------------------*/
 
/* _PG_QD_SET_LOGICAL_OP - set the logical operation */
 
void _PG_qd_set_logical_op(dev, lop)
   PG_device *dev;
   int lop;
   {

    dev->logical_op = lop;
    SetPort(dev->window);

    switch (lop)
       {case GS_XOR :
/* GOTCHA: Put something appropriate here */
             break; 

        default      :
        case GS_COPY :
/* GOTCHA: Put something appropriate here */
             break;};

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_SET_LINE_STYLE - set the line style */
 
void _PG_qd_set_line_style(dev, style)
   PG_device *dev;
   int style;
   {Pattern patt;

    dev->line_style = style;
    SetPort(dev->window);

    switch (style)
       {case LINE_DASHED    :
             StuffHex(&patt, "\p0607333303CCFC78");
             break; 
        case LINE_DOTTED    :
             StuffHex(&patt, "\p44BA455545BA4439");
             break;
        case LINE_DOTDASHED :
             StuffHex(&patt, "\p02397D7D7D39827C");
             break;
        case LINE_SOLID     :
        default             :
             StuffHex(&patt, "\pFFFFFFFFFFFFFFFF");
             break;};

    PenPat(&patt);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_SET_LINE_WIDTH - set the line width */
 
void _PG_qd_set_line_width(dev, width)
   PG_device *dev;
   double width;
   {int lwd;

    dev->line_width = width;
    SetPort(dev->window);

    lwd = max(1, width);
    PenSize(lwd, lwd);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_SET_LINE_COLOR - set the line color */
 
void _PG_qd_set_line_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {dev->line_color = color;
    SetPort(dev->window);

    if (mapped)
       color = _PG_trans_color(dev, color);

#ifdef MAC_COLOR
    if (dev->ncolor > 2)
       RGBForeColor(&dev->current_palette->true_colormap[color]);
#endif
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_SET_TEXT_COLOR - set the color of the text */
 
void _PG_qd_set_text_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {dev->text_color = color;
    SetPort(dev->window);

    if (mapped)
       color = _PG_trans_color(dev, color);

#ifdef MAC_COLOR
    if (dev->ncolor > 2)
       RGBForeColor(&dev->current_palette->true_colormap[color]);
#endif
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_SET_FILL_COLOR - sets current fill color */

void _PG_qd_set_fill_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {dev->fill_color = color;
    SetPort(dev->window);

#ifdef MAC_COLOR

    if (dev->ncolor > 2)
       {if (mapped)
           RGBForeColor(&dev->current_palette->true_colormap[color]);
        else
           {RGBColor pixcolor;

            Index2Color((long) color, &pixcolor);
            RGBForeColor(&pixcolor);};};
#endif

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_SET_FONT - set the character font */

int _PG_qd_set_font(dev, face, style, size)
   PG_device *dev;
   char *face, *style;
   int size;
   {int nfont, nstyle;
    char *font_name;
    short f;
    FontInfo font_data;

    if (!PG_setup_font(dev, face, style, size, &font_name, &nfont, &nstyle))
       return(FALSE);

    SetPort(dev->window);

/* set the type face */
    GET_FNUM(font_name, &f);
    _PG_set_text_font_mac(dev, (int) f);

/* set the type style */
    switch (nstyle)
       {case 0  :
             f = normal;             /* medium -> normal */
             break;
        case 1  :
             f = italic;             /* italic -> italic */
             break;
        case 2  :
             f = bold;               /* bold -> bold */
             break;
        case 3  :
             f = bold & italic;      /* bold-italic -> bold&italic */
             break;
        default :
             f = normal;
             break;};

    TextFace(f);
    TextSize((short) size);
    GetFontInfo(&font_data);
    
    dev->char_width_s  = font_data.widMax/dev->window_width;
    dev->char_height_s = dev->txt_ratio*(font_data.ascent + font_data.descent)/
                         dev->window_height;
    
    PG_release_current_device(dev);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_SET_CHAR_SIZE_NDC - set the character size in NCD */

void _PG_qd_set_char_size_NDC(dev, x, y)
   PG_device *dev;
   double x, y;
   {int size;

    dev->char_height_s = (REAL) y;
    dev->char_width_s  = (REAL) x;

    size = dev->char_width_s*dev->window_width;
    if (size > 4)
       _PG_set_text_size_mac(dev, size, FALSE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_SET_CHAR_PATH - set the direction along which text will be written
 *                      - defaults to (1, 0)
 */

void _PG_qd_set_char_path(dev, x, y)
   PG_device *dev;
   double x, y;
   {dev->char_path_x = x;
    dev->char_path_y = y;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_SET_CHAR_PRECISION - set the character precision
 *                           - fast and fixed size or
 *                           - slow and flexible
 */

void _PG_qd_set_char_precision(dev, p)
   PG_device *dev;
   int p;
   {dev->char_precision = p;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_SET_CHAR_SPACE - set the space between characters */

void _PG_qd_set_char_space(dev, s)
   PG_device *dev;
   double s;
   {dev->char_space = (REAL) s;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_SET_CHAR_UP - set the direction which is up for individual
 *                    - characters
 *                    - defaults to (0, 1)
 */

void _PG_qd_set_char_up(dev, x, y)
   PG_device *dev;
   double x, y;
   {dev->char_up_x = (REAL) x;
    dev->char_up_y = (REAL) y;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_SET_CHAR_LINE - set the number characters per line */
 
void _PG_qd_set_char_line(dev, n)
   PG_device *dev;
   int n;
   {return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_SET_CLIPPING - set clipping
 *                     - flag = FALSE  -->  clipping off
 *                     - flag = TRUE   -->  clipping on
 */

void _PG_qd_set_clipping(dev, flag)
   PG_device *dev;
   int flag;
   {int ix0, iy0, ix1, iy1;
    REAL xmin, xmax, ymin, ymax;
    Rect shape;

    PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);

    _PG_find_clip_region(dev, xmin, xmax, ymin, ymax,
                         &ix0, &iy0, &ix1, &iy1, flag, FALSE);

/* add a thick line worth of pixels on the bottom and right */
    ix1 += 4;
    iy1 += 4;

    SetPort(dev->window);
    SETRECT(&shape, ix0, iy0, ix1, iy1);
    ClipRect(&shape);
 
    dev->clipping = flag;

    return;}
 
/*--------------------------------------------------------------------------*/

/*                          MOVE AND DRAW ROUTINES                          */

/*--------------------------------------------------------------------------*/
 
/* _PG_QD_MOVE_GR_ABS - move the current graphics cursor position to the
 *                    - given absolute coordinates in WC
 */
 
void _PG_qd_move_gr_abs(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;
 
/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->gcurx = x;
    dev->gcury = y;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    SetPort(dev->window);
    MoveTo(x1, y1);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_MOVE_TX_ABS - move the current text cursor position to the
 *                    - given coordinates in WC
 */
 
void _PG_qd_move_tx_abs(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;
 
/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->tcurx = x;
    dev->tcury = y;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    SetPort(dev->window);
    MoveTo(x1, y1);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_MOVE_TX_REL - move the current text cursor position to the
 *                    - given relative coordinates in WC
 */
 
void _PG_qd_move_tx_rel(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;

/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->tcurx += x;
    dev->tcury += y;

    x = dev->tcurx;
    y = dev->tcury;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    SetPort(dev->window);
    Move(x1, y1);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_DRAW_TO_ABS - draw a line from current position to
 *                    - absolute position (x, y)
 *                    - in WC
 */
 
void _PG_qd_draw_to_abs(dev, x, y)
   PG_device *dev;
   double x, y;
   {REAL x1, y1;

/* if log axis options are on, take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);

    dev->gcurx = x;
    dev->gcury = y;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    SetPort(dev->window);
    LineTo(x1, y1);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_DRAW_TO_REL - draw a line from current position to
 *                    - relative position (x, y)
 *                    - in WC
 */
 
void _PG_qd_draw_to_rel(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;

/* if log axis options are on, take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->gcurx += x;
    dev->gcury += y;

/* resuse x and y to rescale to pixel coordinates */
    x = dev->gcurx;
    y = dev->gcury;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    SetPort(dev->window);
    LineTo(x1, y1);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_DRAW_POLYLINE - plot line segments between points specified
 *                      - by values from an array of x and an array of
 *                      - y values make our own calls to lineabs so that
 *                      - log plots will always work
 */
 
void _PG_qd_draw_polyline(dev, x, y, n)
   PG_device *dev;
   REAL *x, *y;
   int n;
   {int i;

    PG_move_gr_abs(dev, x[0], y[0]);

    for (i = 1; i < n; i++)
        PG_draw_to_abs(dev, x[i], y[i]);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_DRAW_CURVE - draw a PG_curve */
 
void _PG_qd_draw_curve(dev, crv, clip)
   PG_device *dev;
   PG_curve *crv;
   int clip;
   {int i, n, xo, yo, ix, iy;
    int *x, *y;

    n  = crv->n;
    x  = crv->x;
    y  = crv->y;
    xo = crv->x_origin;
    yo = crv->y_origin;

    SetPort(dev->window);

    ix = x[0] + xo;
    iy = y[0] + yo;
    MoveTo(ix, iy);

    for (i = 1; i < n; i++)
        {ix = x[i] + xo;
         iy = y[i] + yo;
         LineTo(ix, iy);};

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_DRAW_DISJOINT_POLYLINE_2 - draws disjoint two dimensional
 *                                 - line segments specified in WC
 */

void _PG_qd_draw_disjoint_polyline_2(dev, x, y, n, flag, coord)
   PG_device *dev;
   REAL *x, *y;
   long n;
   int flag, coord;
   {REAL *px, *py;
    int i;

/* if auto ranging or domaining is on the data will control the WC system */
    if (flag && (dev->autorange || dev->autodomain))
       PG_set_limits(dev, x, y, 2*n, CARTESIAN);

    px = x;
    py = y;

    PG_move_gr_abs(dev, x[0], y[0]);

    if (coord)
       {for (i = 0; i < n; i++)
            PG_draw_line(dev, *px++, *py++, *px++, *py++);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_SHADE_POLY - polygon shading routine */

void _PG_qd_shade_poly(dev, x, y, n)
   PG_device *dev;
   REAL *x, *y;
   int n;
   {int i;
    PolyHandle poly;
    Pattern patt;

    SetPort(dev->window);

    poly = OpenPoly();
    PG_move_gr_abs(dev, x[0], y[0]);
    for (i = 1; i < n; i++)
        PG_draw_to_abs(dev, x[i], y[i]);

    ClosePoly();

    PaintPoly(poly);
    KillPoly(poly);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_QD_FILL_CURVE - fill a closed PG_curve */

void _PG_qd_fill_curve(dev, crv)
   PG_device *dev;
   PG_curve *crv;
   {int i, n, xo, yo, ix, iy;
    int *x, *y;
    PolyHandle poly;
    Pattern patt;

    n  = crv->n;
    x  = crv->x;
    y  = crv->y;
    xo = crv->x_origin;
    yo = crv->y_origin;

    SetPort(dev->window);
    poly = OpenPoly();
    ix = x[0] + xo;
    iy = y[0] + yo;
    MoveTo(ix, iy);
    for (i = 1; i < n; i++)
        {ix = x[i] + xo;
         iy = y[i] + yo;
         LineTo(ix, iy);};

    ClosePoly();

    PaintPoly(poly);
    KillPoly(poly);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_PUT_IMAGE - put the image on the screen
 *                  - the image buffer may be overwritten by the pseudo
 *                  - color mapping if it is needed!!
 */

void _PG_qd_put_image(dev, bf, ix, iy, nx, ny)
   PG_device *dev;
   unsigned char *bf;
   int ix, iy, nx, ny;
   {Rect srect, drect;
    WindowPtr win;
    REAL rv, scale;
    int i, k, l, n_pal_colors, n_dev_colors;
    PG_palette *pal;
    RGB_color_map *pseudo_cm, *true_cm;

    SetPort(dev->window);

    pal          = dev->current_palette;
    n_dev_colors = dev->absolute_n_color;
    n_pal_colors = pal->n_pal_colors;
    pseudo_cm    = pal->pseudo_colormap;
    true_cm      = pal->true_colormap;

    PG_invert_image_data(bf, nx, ny, 1);

    if ((n_dev_colors < n_pal_colors) && (pseudo_cm != NULL))
       {unsigned char *pbf;

        pbf = bf;
        PM_random(-1);
        for (l = 0; l < ny; l++)
            for (k = 0; k < nx; k++)
                {i      = *pbf;
                 rv     = 3.5*PM_random(1) - 0.83;
                 *pbf++ = (rv < pseudo_cm[i].red) ?
                          pseudo_cm[i].green : pseudo_cm[i].blue;};};

    win = dev->window;
    SelectWindow(win);

    k = nx*ny;

    SETRECT(&srect, 0, 0, nx, ny);
    SETRECT(&drect, ix, iy, ix+nx, iy+ny);

    if (n_pal_colors > 2)
       {PixMap **shbits, *spm, *dpm;
        CTabHandle sct, dct;
        ColorSpec *dcs;
        PG_palette *stdpal;
        RGBColor pixcolor, *bfRGB;
        long val;
        static RGBColor white = {0xFFFF, 0xFFFF, 0xFFFF};
        static RGBColor black = {0, 0, 0};

        for (i = 0; i < k; i++)
            bf[i] += 16;

        dpm = *(((CWindowPtr) win)->portPixMap);
        dct = dpm->pmTable;
        
        if (dpm->pixelSize <= 8)
           {shbits = NewPixMap();
            spm           = *shbits;
            sct           = spm->pmTable;
            spm->rowBytes = nx | 0x8000;
            spm->bounds   = srect;
            spm->pmTable  = dct;
            dcs = (*dct)->ctTable;
            spm->baseAddr = (char *) bf;}
        else
           {dcs   = FMAKE_N(ColorSpec, n_pal_colors + 18, "_PG_qd_put_image:dcs");};

/* NOTE: the following loop maxes out at n_pal_colors + 2 because
 * n_pal_colors does not include black and white!
 */

        for (i = 0, l = 16; i < n_pal_colors + 2; i++, l++)
            {dcs[l].value = l;
             dcs[l].rgb   = true_cm[i];};

        stdpal  = PG_set_palette(dev, "standard");
        true_cm = stdpal->true_colormap;
        for (i = 0; i < 16; i++)
            {dcs[i].value = i;
             dcs[i].rgb   = true_cm[i];};
        dev->current_palette = pal;

/* NOTE: cruft abounds on the wonderful Macintosh!!!
 * the problem seems to be that the mapping from index->RGB through
 * a PixMap is unknown
 * hence PGS is forced to filter the indices through
 * an index->PGS-RGB->MAC-RGB->index process
 * the MAC-RGB->index process is limited by the inverse color table
 * which is essentially undocumented
 *
 * GOTCHA: the THINK C debugger does something funny with the
 * color table(s) or palettes or something so that one has to 
 * reboot between debugging runs or the correlation between this
 * coding and the on screen behavior is a hard 0!!!!!!!
 *
 * Don't know if this is an issue anymore.
 */
        if (dpm->pixelSize <= 8)
           {for (i = 0; i < k; i++)
                {Index2Color((long) bf[i], &pixcolor);
                 val      = Color2Index(&pixcolor);
                 bf[i]    = val;};
            RGBForeColor(&black);
            RGBBackColor(&white);

            CopyBits((BitMap *) spm, (BitMap *) dpm,
                     &srect, &drect, srcCopy, NULL);}
        else
           {short j, k;
            for (i = 0, j = iy; j < iy + ny; j++)
                {for (k = ix; k < ix + nx; k++, i++)
                    SetCPixel(k, j, &dcs[bf[i]].rgb);};}        

        if (dpm->pixelSize > 8)
           {SFREE(dcs);}
        else
           {spm->pmTable = sct;
            DisposePixMap(shbits);};}

    else
       {int bx;
        BitMap *spm, *dpm;
        unsigned char *bm;

        bx = _PG_byte_bit_map(bf, nx, ny, FALSE);

        spm = FMAKE(BitMap, "_PG_QD_PUT_IMAGE:spm");
        spm->baseAddr = (char *) bf;
        spm->rowBytes = bx;
        spm->bounds   = srect;

        dpm = &(win->portBits);

        CopyBits(spm, dpm, &srect, &drect, srcCopy, NULL);

        SFREE(spm);};

    return;}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
#if 0
/* _PG_QD_PUT_IMAGE - put the image on the screen
 *                  - the image buffer may be overwritten by the pseudo
 *                  - color mapping if it is needed!!
 */

void _PG_qd_put_image(dev, bf, ix, iy, nx, ny)
   PG_device *dev;
   unsigned char *bf;
   int ix, iy, nx, ny;
   {Rect srect, drect;
    WindowPtr win;
    REAL rv, scale;
    int i, k, l, n_pal_colors, n_dev_colors;
    PG_palette *pal;
    RGB_color_map *pseudo_cm, *true_cm;

    SetPort(dev->window);

    pal          = dev->current_palette;
    n_dev_colors = dev->absolute_n_color;
    n_pal_colors = pal->n_pal_colors;
    pseudo_cm    = pal->pseudo_colormap;
    true_cm      = pal->true_colormap;

    PG_invert_image_data(bf, nx, ny, 1);

    if ((n_dev_colors < n_pal_colors) && (pseudo_cm != NULL))
       {unsigned char *pbf;

        pbf = bf;
        PM_random(-1);
        for (l = 0; l < ny; l++)
            for (k = 0; k < nx; k++)
                {i      = *pbf;
                 rv     = 3.5*PM_random(1) - 0.83;
                 *pbf++ = (rv < pseudo_cm[i].red) ?
                          pseudo_cm[i].green : pseudo_cm[i].blue;};};

    win = dev->window;
    SelectWindow(win);

    k = nx*ny;

    SETRECT(&srect, 0, 0, nx, ny);
    SETRECT(&drect, ix, iy, ix+nx, iy+ny);

    if (n_pal_colors > 2)
       {PixMap **shbits, *spm, *dpm;
        CTabHandle sct, dct;
        ColorSpec *dcs;
        PG_palette *stdpal;
        RGBColor pixcolor, *bfRGB;
        long val;
        static RGBColor white = {0xFFFF, 0xFFFF, 0xFFFF};
        static RGBColor black = {0, 0, 0};

        for (i = 0; i < k; i++)
            bf[i] += 16;

        dpm = *(((CWindowPtr) win)->portPixMap);
        dct = dpm->pmTable;
        
        shbits = NewPixMap();
        spm           = *shbits;
        sct           = spm->pmTable;
        spm->rowBytes = nx | 0x8000;
        spm->bounds   = srect;
        spm->pmTable  = dct;
        if (dpm->pixelSize <= 8)
           {dcs = (*dct)->ctTable;
            spm->baseAddr = (char *) bf;}
        else
           {dcs   = FMAKE_N(ColorSpec, n_pal_colors + 18, "_PG_qd_put_image:dcs");
            bfRGB = FMAKE_N(RGBColor, k, "_PG_qd_put_image:bfRGB");
            spm->baseAddr = (char *) bfRGB;};

/* NOTE: the following loop maxes out at n_pal_colors + 2 because
 * n_pal_colors does not include black and white!
 */

        for (i = 0, l = 16; i < n_pal_colors + 2; i++, l++)
            {dcs[l].value = l;
             dcs[l].rgb   = true_cm[i];};

        stdpal  = PG_set_palette(dev, "standard");
        true_cm = stdpal->true_colormap;
        for (i = 0; i < 16; i++)
            {dcs[i].value = i;
             dcs[i].rgb   = true_cm[i];};
        dev->current_palette = pal;

/* NOTE: cruft abounds on the wonderful Macintosh!!!
 * the problem seems to be that the mapping from index->RGB through
 * a PixMap is unknown
 * hence PGS is forced to filter the indices through
 * an index->PGS-RGB->MAC-RGB->index process
 * the MAC-RGB->index process is limited by the inverse color table
 * which is essentially undocumented
 *
 * GOTCHA: the THINK C debugger does something funny with the
 * color table(s) or palettes or something so that one has to 
 * reboot between debugging runs or the correlation between this
 * coding and the on screen behavior is a hard 0!!!!!!!
 *
 * Don't know if this is an issue anymore.
 */
        if (dpm->pixelSize <= 8)
           {for (i = 0; i < k; i++)
                {Index2Color((long) bf[i], &pixcolor);
                 val      = Color2Index(&pixcolor);
                 bf[i]    = val;};}
        else
           {(*sct)->ctFlags = 0x4000;
            for (i = 0; i < k; i++)
                bfRGB[i] = dcs[bf[i]].rgb;};

        RGBForeColor(&black);
        RGBBackColor(&white);

        CopyBits((BitMap *) spm, (BitMap *) dpm,
                 &srect, &drect, srcCopy, NULL);

        spm->pmTable = sct;

        if (dpm->pixelSize > 8)
           {SFREE(dcs);
            SFREE(bfRGB);};

        DisposePixMap(shbits);}

    else
       {int bx;
        BitMap *spm, *dpm;
        unsigned char *bm;

        bx = _PG_byte_bit_map(bf, nx, ny, FALSE);

        spm = FMAKE(BitMap, "_PG_QD_PUT_IMAGE:spm");
        spm->baseAddr = (char *) bf;
        spm->rowBytes = bx;
        spm->bounds   = srect;

        dpm = &(win->portBits);

        CopyBits(spm, dpm, &srect, &drect, srcCopy, NULL);

        SFREE(spm);};

    return;}

#endif
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_QD_GET_IMAGE - get the image on the screen into
 *                  - the image buffer
 */

void _PG_qd_get_image(dev, bf, ix, iy, nx, ny)
   PG_device *dev;
   unsigned char *bf;
   int ix, iy, nx, ny;
   {return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

