/*
 * GSPR_PS.C - PGS PostScript primitive routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"

/*--------------------------------------------------------------------------*/

/*                         STATE QUERY ROUTINES                             */

/*--------------------------------------------------------------------------*/

/* _PG_PS_GET_TEXT_EXT_NDC - return the text extent in NDC
 *                         - of the given string
 */

void _PG_PS_get_text_ext_NDC(dev, s, px, py)
   PG_device *dev;
   char *s;
   REAL *px, *py;
   {REAL tx, ty;

    tx = strlen(s)*(dev->char_width_s + dev->char_space_s) -
         dev->char_space_s;
    ty = dev->char_height_s;

/* probably wrong factor for fixed width font */
    tx *= 0.55*_PG_ps_dots_inch/600.0;
    ty *= 1.1*_PG_ps_dots_inch/600.0;

    *px = tx;
    *py = ty;

    return;}

/*--------------------------------------------------------------------------*/

/*                          STATE CHANGE ROUTINES                           */

/*--------------------------------------------------------------------------*/

/* _PG_PS_SET_FONT - set the character font */

int _PG_PS_set_font(dev, face, style, size)
   PG_device *dev;
   char *face, *style;
   int size;
   {int nfont, nstyle, dx, dy, nc;
    char *font_name;

    if (!PG_setup_font(dev, face, style, size, &font_name, &nfont, &nstyle))
       return(FALSE);

    PG_query_screen(dev, &dx, &dy, &nc);
    dev->char_width_s  = size/((dev->window_width/dx)
                                *612*(_PG_ps_dots_inch/600.0));
    dev->char_height_s = size/((dev->window_height/dy)
                                *792*(_PG_ps_dots_inch/600.0));
    nc = (int) (size*_PG_ps_dots_inch/72.0);
    io_printf(dev->file, "%d /%s SF\n",
	      nc, font_name);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_SET_LOGICAL_OP - set the logical operation */
 
void _PG_PS_set_logical_op(dev, lop)
   PG_device *dev;
   int lop;
   {

    dev->logical_op = lop;

    switch (lop)
       {case GS_XOR :
/* GOTCHA: Put something appropriate here */
             break;

        default      :
        case GS_COPY :
/* GOTCHA: Put something appropriate here */
             break;};
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_SET_LINE_STYLE - set the line style */
 
void _PG_PS_set_line_style(dev, style)
   PG_device *dev;
   int style;
   {int lv, sv;

    dev->line_style = style;

    sv = (int) (_PG_ps_dots_inch/24.0);
    lv = sv << 1;
    switch (style)
       {case LINE_DASHED    :
             io_printf(dev->file, "[%d %d] 0 d\n",
		       lv, sv);
             break;
        case LINE_DOTTED    :
             io_printf(dev->file, "[%d %d] 0 d\n",
		       sv, sv);
             break;
        case LINE_DOTDASHED :
             io_printf(dev->file, "[%d %d %d %d] 0 d\n",
		       lv, sv, sv, sv);
             break;
        case LINE_SOLID     :
        default             :
             io_printf(dev->file, "[] 0 d\n");
             break;};
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_SET_LINE_WIDTH - set the line width */
 
void _PG_PS_set_line_width(dev, width)
   PG_device *dev;
   double width;
   {
 
    dev->line_width = width;
    io_printf(dev->file, "%7.3f w\n", _PG_ps_dots_inch*width/24.0);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_SET_LINE_COLOR - set the line color */
 
void _PG_PS_set_line_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {dev->line_color     = color;
    dev->map_line_color = mapped;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_SET_TEXT_COLOR - set the color of the text */
 
void _PG_PS_set_text_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {dev->text_color     = color;
    dev->map_text_color = mapped;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_SET_FILL_COLOR - sets current fill color */

void _PG_PS_set_fill_color(dev, color, mapped)
   PG_device *dev;
   int color, mapped;
   {dev->fill_color     = color;
    dev->map_fill_color = mapped;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_SET_CHAR_SIZE_NDC - set the character size in NCD */

void _PG_PS_set_char_size_NDC(dev, x, y)
   PG_device *dev;
   double x, y;
   {dev->char_height_s = (REAL) y;
    dev->char_width_s  = (REAL) x;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_SET_CHAR_PATH - set the direction along which text will be written
 *                      - defaults to (1, 0)
 */

void _PG_PS_set_char_path(dev, x, y)
   PG_device *dev;
   double x, y;
   {dev->char_path_x = x;
    dev->char_path_y = y;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_SET_CHAR_PRECISION - set the character precision
 *                           - fast and fixed size or
 *                           - slow and flexible
 */

void _PG_PS_set_char_precision(dev, p)
   PG_device *dev;
   int p;
   {dev->char_precision = p;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_SET_CHAR_SPACE - set the space between characters */

void _PG_PS_set_char_space(dev, s)
   PG_device *dev;
   double s;
   {dev->char_space = (REAL) s;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_SET_CHAR_UP - set the direction which is up for individual
 *                    - characters
 *                    - defaults to (0, 1)
 */

void _PG_PS_set_char_up(dev, x, y)
   PG_device *dev;
   double x, y;
   {dev->char_up_x = (REAL) x;
    dev->char_up_y = (REAL) y;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_SET_CHAR_LINE - set the number characters per line */
 
void _PG_PS_set_char_line(dev, n)
   PG_device *dev;
   int n;
   {return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_SET_CLIPPING - set clipping
 *                     - flag = FALSE  -->  clipping off
 *                     - flag = TRUE   -->  clipping on
 */

void _PG_PS_set_clipping(dev, flag)
   PG_device *dev;
   int flag;
   {int ix0, iy0, ix1, iy1;
    REAL xmin, xmax, ymin, ymax;

    PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);

    if (flag)
       {if (dev->clipping)
           {io_printf(dev->file, "\n%% Clipping OFF\n");
            io_printf(dev->file, "Cf\n");};

        _PG_find_clip_region(dev, xmin, xmax, ymin, ymax,
                             &ix0, &iy0, &ix1, &iy1, flag, TRUE);

        io_printf(dev->file, "\n%% Clipping ON\n");
        io_printf(dev->file, "%d %d %d %d Cn\n",
		  (ix1 - ix0), (iy0 - iy1), ix0, iy1);}
    
    else if (dev->clipping)
       {io_printf(dev->file, "\n%% Clipping OFF\n");
        io_printf(dev->file, "Cf\n");};
 
    dev->clipping = flag;

    return;}
 
/*--------------------------------------------------------------------------*/

/*                          MOVE AND DRAW ROUTINES                          */

/*--------------------------------------------------------------------------*/
 
/* _PG_PS_MOVE_GR_ABS - move the current graphics cursor position to the
 *                    - given absolute coordinates in WC
 */
 
void _PG_PS_move_gr_abs(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;
 
/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->gcurx = x;
    dev->gcury = y;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    io_printf(dev->file, "%d %d m\n", x1, y1);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_MOVE_TX_ABS - move the current text cursor position to the
 *                    - given coordinates in WC
 */
 
void _PG_PS_move_tx_abs(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;

/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->tcurx = x;
    dev->tcury = y;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    io_printf(dev->file, "1 0 0 1 %d %d 0 Tp\n", x1, y1);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_MOVE_TX_REL - move the current text cursor position to the
 *                    - given relative coordinates in WC
 */
 
void _PG_PS_move_tx_rel(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;

/* if log axis options have been used take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->tcurx += x;
    dev->tcury += y;

    x = dev->tcurx;
    y = dev->tcury;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);
 
    io_printf(dev->file, "1 0 0 1 %d %d 0 Tp\n", x1, y1);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_DRAW_TO_ABS - draw a line from current position to
 *                    - absolute position (x, y)
 *                    - in WC
 */
 
void _PG_PS_draw_to_abs(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;

    _PG_PS_set_dev_color(dev, PG_LINE);

/* if log axis options are on, take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);

    dev->gcurx = x;
    dev->gcury = y;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);

    io_printf(dev->file, "%d %d L S\n", x1, y1);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_DRAW_TO_REL - draw a line from current position to
 *                    - relative position (x, y)
 *                    - in WC
 */
 
void _PG_PS_draw_to_rel(dev, x, y)
   PG_device *dev;
   double x, y;
   {int x1, y1;

    _PG_PS_set_dev_color(dev, PG_LINE);

/* if log axis options are on, take logs */
    if (dev->ifxlog)
       x = log10(ABS(x) + SMALL);
    if (dev->ifylog)
       y = log10(ABS(y) + SMALL);
 
    dev->gcurx += x;
    dev->gcury += y;

/* resuse x and y to rescale to pixel coordinates */
    x = dev->gcurx;
    y = dev->gcury;

    WtoS(dev, x, y);
    StoP(dev, x, y, x1, y1);

    io_printf(dev->file, "%d %d rlineto S\n", x1, y1);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_DRAW_CURVE - draw a PG_curve */
 
void _PG_PS_draw_curve(dev, crv, clip)
   PG_device *dev;
   PG_curve *crv;
   int clip;
   {int i, flag, n, xo, yo, ix, ixo, iy, iyo, nl;
    int cxx, cxn, cyx, cyn;
    int *x, *y;
    REAL xmin, xmax, ymin, ymax;
    
    flag = TRUE;

    _PG_PS_set_dev_color(dev, PG_LINE);

    PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);
    _PG_find_clip_region(dev, xmin, xmax, ymin, ymax,
                         &cxn, &cyn, &cxx, &cyx, dev->clipping, TRUE);
    
    nl = 0;
    n  = crv->n;
    x  = crv->x;
    y  = crv->y;
    xo = crv->x_origin;
    yo = crv->y_origin;

    if (cyn > cyx)
       {i   = cyn;
        cyn = cyx;
        cyx = i;};
    
    for (i = 0; i < n; i++)
        {ixo = ix;
         iyo = iy;
         ix  = x[i] + xo;
         iy  = y[i] + yo;
       
         if (i == 0)
            continue;
    
         if ((ix == ixo) && (iy == iyo))
            continue;

         if (clip && dev->clipping)
	    {if (((ix < cxn) && (ixo < cxn)) ||
		 ((ix > cxx) && (ixo > cxx)))
	       {flag = TRUE;
		continue;}

	     if (((iy < cyn) && (iyo < cyn)) ||
		 ((iy > cyx) && (iyo > cyx)))
	       {flag = TRUE;
		continue;};}

	 if (flag)
	    {io_printf(dev->file, "%d %d m\n", ixo, iyo);
	     flag = FALSE;}

         io_printf(dev->file, "%d %d L\n", ix, iy);

         if (++nl >= 256)
            {nl = 0;
	     io_printf(dev->file, "S\n");
             io_printf(dev->file, "%d %d m\n", ix, iy);};};

    io_printf(dev->file, "S\n");

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_DRAW_DISJOINT_POLYLINE_2 - draws disjoint two dimensional
 *                                 - line segments specified in WC
 */

void _PG_PS_draw_disjoint_polyline_2(dev, x, y, n, flag, coord)
   PG_device *dev;
   REAL *x, *y;
   long n;
   int flag, coord;
   {int i;
    int ix1, ix2, iy1, iy2;
    int cxx, cxn, cyx, cyn;
    REAL *px, *py;
    REAL rx1, ry1, rx2, ry2;
    REAL xmin, xmax, ymin, ymax;

/* if auto ranging or domaining is on the data will control the WC system */
    if (flag && (dev->autorange || dev->autodomain))
       PG_set_limits(dev, x, y, 2*n, CARTESIAN);

    px = x;
    py = y;

    _PG_PS_set_dev_color(dev, PG_LINE);

    PG_move_gr_abs(dev, x[0], y[0]);

    PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);
    _PG_find_clip_region(dev, xmin, xmax, ymin, ymax,
                         &cxn, &cyx, &cxx, &cyn, dev->clipping, TRUE);

    for (i = 0; i < n; i++)
        {rx1 = *px++;
         ry1 = *py++;
         rx2 = *px++;
         ry2 = *py++;
         
         if (coord)
            {if (dev->ifxlog)
                {rx1 = log10(ABS(rx1) + SMALL);
                 rx2 = log10(ABS(rx2) + SMALL);};
             if (dev->ifylog)
                {ry1 = log10(ABS(ry1) + SMALL);
                 ry2 = log10(ABS(ry2) + SMALL);};
 
             WtoS(dev, rx1, ry1);
             StoP(dev, rx1, ry1, ix1, iy1);
             WtoS(dev, rx2, ry2);
             StoP(dev, rx2, ry2, ix2, iy2);}
         else
            {StoP(dev, rx1, ry1, ix1, iy1);
             StoP(dev, rx2, ry2, ix2, iy2);};

         if ((ix1 == ix2) && (iy1 == iy2))
            continue;

/* do simple clipping */
         if (dev->clipping &&
             (((ix1 < cxn) && (ix2 < cxn)) ||
              ((ix1 > cxx) && (ix2 > cxx)) ||
              ((iy1 < cyn) && (iy2 < cyn)) ||
              ((iy1 > cyx) && (iy2 > cyx))))
            continue;

         io_printf(dev->file, "%d %d m %d %d L S\n", ix1, iy1, ix2, iy2);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_SHADE_POLY - polygon shading routine */

void _PG_PS_shade_poly(dev, x, y, n)
   PG_device *dev;
   REAL *x, *y;
   int n;
   {REAL rx, ry;
    int i, ix, iy;

    _PG_PS_set_dev_color(dev, PG_FILL);

    rx = x[0];
    ry = y[0];
    WtoS(dev, rx, ry);
    StoP(dev, rx, ry, ix, iy);

    io_printf(dev->file, "newpath\n");
    io_printf(dev->file, "%d %d m\n", ix, iy);

    for (i = 1; i < n; i++)
      {rx = x[i];
       ry = y[i];
       WtoS(dev, rx, ry);
       StoP(dev, rx, ry, ix, iy);
       io_printf(dev->file, "%d %d L\n", ix, iy);};

    io_printf(dev->file, "closepath\n");

    io_printf(dev->file, "fill\n");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_PS_FILL_CURVE - fill a closed PG_curve */

void _PG_PS_fill_curve(dev, crv)
   PG_device *dev;
   PG_curve *crv;
   {int i, n, xo, yo;
    int *x, *y;

    _PG_PS_set_dev_color(dev, PG_FILL);

    n  = crv->n;
    x  = crv->x;
    y  = crv->y;
    xo = crv->x_origin;
    yo = crv->y_origin;

    io_printf(dev->file, "newpath\n");
    io_printf(dev->file, "%d %d m\n", x[0]+xo, y[0]+yo);

    for (i = 1; i < n; i++)
        io_printf(dev->file, "%d %d L\n", x[i]+xo, y[i]+yo);

    io_printf(dev->file, "closepath\n");
/*
    int lcolor;

    lcolor = dev->fill_color;
    if (COLOR_POSTSCRIPT_DEVICE(dev))
       {RGB_color_map *true_cm;
        int red_fl, green_fl, blue_fl;

        true_cm  = dev->current_palette->true_colormap;
        red_fl   = true_cm[lcolor].red/MAXPIX;
        green_fl = true_cm[lcolor].green/MAXPIX;
        blue_fl  = true_cm[lcolor].blue/MAXPIX;

        io_printf(dev->file, "%d %d %d Srgb\n",
		  red_fl, green_fl, blue_fl);}
    else
       {REAL val;

        val = ((REAL) (lcolor)) / ((REAL) (dev->ncolor - 1));
        io_printf(dev->file, "%7.3f setgray\n", val);};
*/
    io_printf(dev->file, "fill\n");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_PUT_IMAGE - put the image on the screen
 *                  - the image buffer may be overwritten by the pseudo
 *                  - color mapping if it is needed!!
 */

void _PG_PS_put_image(dev, bf, ix, iy, nx, ny)
   PG_device *dev;
   unsigned char *bf;
   int ix, iy, nx, ny;
   {FILE *fp;
    REAL sf, rv, scale;
    int bpp, l, k, i;
    unsigned char *pbf;
    int n_pal_colors, n_dev_colors;
    PG_palette *pal;
    RGB_color_map *pseudo_cm, *true_cm;

    pal          = dev->current_palette;
    n_dev_colors = dev->absolute_n_color;
    n_pal_colors = pal->n_pal_colors;
    pseudo_cm    = pal->pseudo_colormap;
    true_cm      = pal->true_colormap;

    if ((n_dev_colors < n_pal_colors) && (pseudo_cm != NULL))
       {unsigned char *pbf;

        pbf = bf;
        PM_random(-1);
        for (l = 0; l < ny; l++)
            for (k = 0; k < nx; k++)
                {i      = *pbf;
                 rv     = 3.5*PM_random(1) - 0.83;
                 *pbf++ = (rv < pseudo_cm[i].red) ?
                          pseudo_cm[i].green : pseudo_cm[i].blue;};};

    pbf = bf;
    bpp = log((double) (dev->absolute_n_color))/log(2.0) + 0.5;
    fp  = dev->file;
    if (bpp == 1)
       nx = _PG_byte_bit_map(bf, nx, ny, TRUE);

    sf = dev->resolution_scale_factor;

    io_printf(fp, "\n%% Image\n");
    io_printf(fp, "%d %d %d %d %d %f Di\n",
	      nx, ny, bpp, ix, iy, sf);
    
    if (COLOR_POSTSCRIPT_DEVICE(dev))
       {int color, red_fl, green_fl, blue_fl;

        scale = 255.0/((double) MAXPIX);

        for (l = 0; l < ny; l++)
            {pbf = bf + l*nx;
             for (k = 0; k < nx; k++)
                 {color    = *pbf++;
                  red_fl   = scale*true_cm[color].red;
                  green_fl = scale*true_cm[color].green;
                  blue_fl  = scale*true_cm[color].blue;
                  if ((red_fl < 0) || (255 < red_fl))
                     {printf("Bad Red Value for index %d: %d\n", color, red_fl);
                      exit(2);};
                  if ((green_fl < 0) || (255 < green_fl))
                     {printf("Bad Green Value for index %d: %d\n", color, green_fl);
                      exit(2);};
                  if ((blue_fl < 0) || (255 < blue_fl))
                     {printf("Bad Blue Value for index %d: %d\n", color, blue_fl);
                      exit(2);};
                  io_printf(fp, "%02x%02x%02x", red_fl, green_fl, blue_fl);};

             if (l < ny-1)
                io_printf(fp, "\n");};}

    else
       {scale = 255.0/((double) (n_pal_colors - 1));

        for (l = 0; l < ny; l++)
            {pbf = bf + l*nx;
             for (k = 0; k < nx; k++)
                 io_printf(fp, "%02x", (int) (scale*(*pbf++ - 2)));

             if (l < ny-1)
                io_printf(fp, "\n");};};

    io_printf(fp, "\n%% End of Image\n");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PS_set_dev_color                       */

void _PG_PS_set_dev_color(dev, prim)
   PG_device *dev;
   int prim;
   {PG_palette *pal;
    int color;
    static int last_color = -1;
    static int last_prim  = -1;
    static PG_device *last_dev = NULL;

    pal = dev->current_palette;

    switch(prim)
       {case PG_TEXT :
             color = (!dev->map_text_color) ? dev->text_color : 
                           _PG_trans_color(dev, dev->text_color);
             break;

        case PG_LINE :
             color = (!dev->map_line_color) ? dev->line_color : 
                           _PG_trans_color(dev, dev->line_color);
             break;

        case PG_FILL :
             color = (!dev->map_fill_color) ? dev->fill_color : 
                           _PG_trans_color(dev, dev->fill_color);
             break;};

    if ((dev == last_dev) && (prim == last_prim) 
                          && (color == last_color))
       return;
    else
       {last_dev   = dev;
        last_prim  = prim;
        last_color = color;}
 
    if (COLOR_POSTSCRIPT_DEVICE(dev))
       {RGB_color_map *true_cm;
        double red_fl, green_fl, blue_fl, scale;

        scale = 1.0/((double) MAXPIX);

        true_cm  = pal->true_colormap;
        red_fl   = ABS(scale*true_cm[color].red);
        green_fl = ABS(scale*true_cm[color].green);
        blue_fl  = ABS(scale*true_cm[color].blue);

        io_printf(dev->file, "%.2f %.2f %.2f Srgb\n",
		  red_fl, green_fl, blue_fl);}

    else
       {REAL val;

        val = ((REAL) (color)) / ((REAL) (pal->n_pal_colors - 1));

        io_printf(dev->file, "%7.3f setgray\n", val);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/





