/*
 * GSPR.C - PGS host independent primitive routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"

static PG_marker
 *_PG_marker_list = NULL;

int
 _PG_marker_index = 0;

REAL
 _PG_line_width = 0.1;

static int
 _PG_marker_max = 0;

/*--------------------------------------------------------------------------*/

/*                         STATE QUERY ROUTINES                             */

/*--------------------------------------------------------------------------*/

/* _PG_FIND_CLIP_REGION - convert a rectangle to pixel coordinates
 *                      - depending on the value of flag return the
 *                      - whole window (FALSE) or the given rectangle (TRUE)
 */

void _PG_find_clip_region(dev, xmn, xmx, ymn, ymx,
                          px0, py0, px1, py1, flag, swflag)
   PG_device *dev;
   double xmn, xmx, ymn, ymx;
   int *px0, *py0, *px1, *py1;
   int flag, swflag;
   {int ix0, iy0, ix1, iy1;
    double xma, yma;

    if (dev->ifxlog)
       {xmx = log10(ABS(xmx) + SMALL);
        xmn = log10(ABS(xmn) + SMALL);
        xma = xmx - _PG_axis_n_decades;

/* limit xmn to be within PG_axis_n_decades of the max value */
        xmn = max(xmn, xma);}
 
    if (dev->ifylog)
       {ymx = log10(ABS(ymx) + SMALL);
        ymn = log10(ABS(ymn) + SMALL);
        yma = ymx - _PG_axis_n_decades;

/* limit ymn to be within PG_axis_n_decades of the max value */
        ymn = max(ymn, yma);};
 
/* if we have a single window device such as a PostScript device do this */
    if (swflag)
       {if (flag)
           {WtoS(dev, xmn, ymn);
            WtoS(dev, xmx, ymx);
            StoP(dev, xmn, ymx, ix0, iy0);
            StoP(dev, xmx, ymn, ix1, iy1);}
        else
           {ix0 = dev->window_x;
            iy0 = dev->window_y;
            ix1 = ix0 + dev->window_width;
            iy1 = iy0 + dev->window_height;};}

/* otherwise for a multi-window device do this */
    else
       {if (flag)
           {WtoS(dev, xmn, ymn);
            WtoS(dev, xmx, ymx);}
        else
           {xmn = 0.0;
            xmx = 1.0;
            ymn = 0.0;
            ymx = 1.0;};

        StoP(dev, xmn, ymx, ix0, iy0);
        StoP(dev, xmx, ymn, ix1, iy1);};

/* return the pixel coordinates of the rectangle */
    *px0 = ix0;
    *py0 = iy0;
    *px1 = ix1;
    *py1 = iy1;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_TRANS_COLOR - shift the color to some more appropriate value
 *                 - the background is logically BLACK
 *                 - regardless of the screen color chosen
 */
 
int _PG_trans_color(dev, color)
   PG_device *dev;
   int color;
   {int c, nc;
    PG_palette *pal;

    if (dev == NULL)
       return(color);

    if (color == dev->BLACK)
       return(dev->WHITE);

    else
       {pal = dev->current_palette;
	if (pal != NULL)
           {nc = min(pal->n_pal_colors, dev->ncolor);
	    c = (color - 1) % (nc + 1) + 1;}
	else
	   c = (color - 1) % (dev->ncolor + 1) + 1;
        if ((c == dev->BLACK) || (c < 0) || (dev->ncolor <= c))
           return(dev->WHITE);
        else
           return(c);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_FILL_COLOR - inquire about the fill color */

void PG_get_fill_color(dev, pcl)
   PG_device *dev;
   int *pcl;
   {

    *pcl = dev->fill_color;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_LINE_COLOR - inquire about the current line color */

void PG_get_line_color(dev, pcl)
   PG_device *dev;
   int *pcl;
   {

    *pcl = dev->line_color;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_TEXT_COLOR - inquire about the text color */

void PG_get_text_color(dev, pcl)
   PG_device *dev;
   int *pcl;
   {

    *pcl = dev->text_color;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_CHAR_SPACE - inquire about the space
 *                   - to be used between characters
 */

void PG_get_char_space(dev, pcsp)
   PG_device *dev;
   REAL *pcsp;
   {

    *pcsp = dev->char_space;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_CHAR_PRECISION - inquire about the character precision
 *                       - fast and fixed size or
 *                       - slow and flexible
 */

void PG_get_char_precision(dev, pcp)
   PG_device *dev;
   int *pcp;
   {

    *pcp = dev->char_precision;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_CURVE_EXTENT - compute the rectangle in which the curve
 *                     - in inscribed (in PC)
 */

void PG_get_curve_extent(dev, crv, mode, pxmn, pxmx, pymn, pymx)
   PG_device *dev;
   PG_curve *crv;
   int mode;
   REAL *pxmn, *pxmx, *pymn, *pymx;
   {int i, n, ixmn, ixmx, iymn, iymx;
    int *x, *y;
    REAL xmn, xmx, ymn, ymx;

    n = crv->n;
    x = crv->x;
    y = crv->y;

    ixmn = INT_MAX;
    ixmx = INT_MIN;
    iymn = INT_MAX;
    iymx = INT_MIN;
    for (i = 0; i < n; i++)
        {ixmn = min(ixmn, x[i]);
         ixmx = max(ixmx, x[i]);
         iymn = min(iymn, y[i]);
         iymx = max(iymx, y[i]);};
/*
    crv->x_origin = ixmn;
    crv->y_origin = iymn;
*/
    switch (mode)
       {case WORLDC :
	     PtoS(dev, ixmn, iymn, xmn, ymn);
	     StoW(dev, xmn, ymn);
	     if (pxmn != NULL)
	        *pxmn = xmn;
	     if (pymn != NULL)
	        *pymn = ymn;
	     PtoS(dev, ixmx, iymx, xmx, ymx);
	     StoW(dev, xmx, ymx);
	     if (pxmx != NULL)
                *pxmx = xmx;
	     if (pymx != NULL)
	        *pymx = ymx;
	     break;

        case NORMC :
	     PtoS(dev, ixmn, iymn, xmn, ymn);
	     if (pxmn != NULL)
	        *pxmn = xmn;
	     if (pymn != NULL)
	        *pymn = ymn;
	     PtoS(dev, ixmx, iymx, xmx, ymx);
	     if (pxmx != NULL)
	        *pxmx = xmx;
             *pymx = ymx;
	     break;

        case PIXELC :
	     if (pxmn != NULL)
	        *pxmn = ixmn;
	     if (pymn != NULL)
	        *pymn = iymn;
	     if (pxmx != NULL)
	        *pxmx = ixmx;
	     if (pymx != NULL)
	        *pymx = iymx;
	     break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_FRAME - return the frame limits */

void PG_get_frame(dev, xmn, xmx, ymn, ymx)
   PG_device *dev;
   REAL *xmn, *xmx, *ymn, *ymx;
   {

    *xmn = dev->fxmin;
    *xmx = dev->fxmax;
    *ymn = dev->fymin;
    *ymx = dev->fymax;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_GET_VIEWPORT_PC - return the PC limits of the viewport */
 
void PG_get_viewport_PC(dev, ixmn, ixmx, iymn, iymx)
   PG_device *dev;
   int *ixmn, *ixmx, *iymn, *iymx;
   {int an, bn, ax, bx;
    REAL xmn, xmx, ymn, ymx;

    PG_get_viewport_NDC(dev, &xmn, &xmx, &ymn, &ymx);

    StoP(dev, xmn, ymn, an, bn)
    StoP(dev, xmx, ymx, ax, bx)

    *ixmn = an;
    *ixmx = ax;

    if (dev->quadrant == QUAD_FOUR)
       {*iymn = bx;
	*iymx = bn;}
    else
       {*iymn = bn;
	*iymx = bx;}
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_GET_VIEWPORT_NDC - return the NDC limits of the viewport */
 
void PG_get_viewport_NDC(dev, xmn, xmx, ymn, ymx)
   PG_device *dev;
   REAL *xmn, *xmx, *ymn, *ymx;
   {

    PG_get_viewport_WC(dev, xmn, xmx, ymn, ymx);

    WtoS(dev, *xmn, *ymn)
    WtoS(dev, *xmx, *ymx)

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_VIEWPORT_WC - return the WC limits of the viewport */

void PG_get_viewport_WC(dev, xmn, xmx, ymn, ymx)
   PG_device *dev;
   REAL *xmn, *xmx, *ymn, *ymx;
   {

    *xmn = dev->gxmin;
    *xmx = dev->gxmax;
    *ymn = dev->gymin;
    *ymx = dev->gymax;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_VIEWPORT - return the 2d view port limits */

void PG_get_viewport(dev, xmn, xmx, ymn, ymx)
   PG_device *dev;
   REAL *xmn, *xmx, *ymn, *ymx;
   {REAL fxn, dfx, fyn, dfy;

    fxn = dev->fxmin;
    dfx = 1.0/(dev->fxmax - fxn + SMALL);
    fyn = dev->fymin;
    dfy = 1.0/(dev->fymax - fyn + SMALL);

    *xmn = (dev->sxmin - fxn)*dfx;
    *xmx = (dev->sxmax - fxn)*dfx;
    *ymn = (dev->symin - fyn)*dfy;
    *ymx = (dev->symax - fyn)*dfy;
/*
    *xmn = dev->sxmin;
    *xmx = dev->sxmax;
    *ymn = dev->symin;
    *ymx = dev->symax;
*/
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_GET_BOUND_PC - return the PC limits of the viewport boundary */
 
void PG_get_bound_PC(dev, ixmn, ixmx, iymn, iymx)
   PG_device *dev;
   int *ixmn, *ixmx, *iymn, *iymx;
   {int an, bn, ax, bx, ny;
    REAL vxmn, vxmx, vymn, vymx;

    PG_get_bound_NDC(dev, &vxmn, &vxmx, &vymn, &vymx);

    StoP(dev, vxmn, vymn, an, bn)
    StoP(dev, vxmx, vymx, ax, bx)

    *ixmn = an;
    *ixmx = ax;

    if (dev->quadrant == QUAD_FOUR)
       {ny = dev->window_height;

	*iymn = bx;
	*iymx = bn;}
    else
       {*iymn = bn;
	*iymx = bx;}
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_GET_BOUND_NDC - return the NDC limits of the viewport boundary */
 
void PG_get_bound_NDC(dev, xmn, xmx, ymn, ymx)
   PG_device *dev;
   REAL *xmn, *xmx, *ymn, *ymx;
   {REAL vxmn, vxmx, vymn, vymx;

    PG_get_viewport(dev, &vxmn, &vxmx, &vymn, &vymx);

    *xmn = vxmn;
    *xmx = vxmx;
    *ymn = vymn;
    *ymx = vymx;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_BOUND_WC - return the WC limits of the viewport boundary */

void PG_get_bound_WC(dev, xmn, xmx, ymn, ymx)
   PG_device *dev;
   REAL *xmn, *xmx, *ymn, *ymx;
   {REAL ax, ay, bx, by;

    ax = dev->xmin;
    bx = dev->xmax;
    ay = dev->ymin;
    by = dev->ymax;

/* since this was not given to PG_set_window as logs
 * don't return it that way
 */
    if (dev->ifxlog)
       {ax = POW(10.0, ax);
        bx = POW(10.0, bx);};
    if (dev->ifylog)
       {ay = POW(10.0, ay);
        by = POW(10.0, by);};

    *xmn = ax;
    *xmx = bx;
    *ymn = ay;
    *ymx = by;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_TEXT_EXT - return the text extent in WC of the given string */

void PG_get_text_ext(dev, s, px, py)
   PG_device *dev;
   char *s;
   REAL *px, *py;
   {REAL tx, ty;

    PG_get_text_ext_NDC(dev, s, &tx, &ty);

    tx *= dev->bxs_w;
    ty *= dev->bys_w;

/* since this was not given to PG_set_window as logs
 * don't return it that way
 */
    if (dev->ifxlog)
       tx = POW(10.0, tx);
    if (dev->ifylog)
       ty = POW(10.0, ty);

    *px = tx;
    *py = ty;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_LINE_WIDTH - return the width of a line */

void PG_get_line_width(dev, plw)
   PG_device *dev;
   REAL *plw;
   {

    *plw = dev->line_width;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_CHAR_UP - inquire about the direction which constitutes up for
 *                - the characters
 *                - defaults to (0, 1)
 */

void PG_get_char_up(dev, px, py)
   PG_device *dev;
   REAL *px, *py;
   {

    *px = dev->char_up_x;
    *py = dev->char_up_y;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_GET_CLIPPING - return the state of the clipping */

void PG_get_clipping(dev, flag)
   PG_device *dev;
   int *flag;
   {

    *flag = dev->clipping;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_FONT - return the current font info */

void PG_get_font(dev, pf, pst, psz)
   PG_device *dev;
   char **pf, **pst;
   int *psz;
   {

    *pf  = SC_strsavef(dev->type_face, "char*:PG_GET_FONT:face");
    *pst = SC_strsavef(dev->type_style, "char*:PG_GET_FONT:style");
    *psz = dev->type_size;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_CHAR_SIZE_NDC - return the character size in NDC */

void PG_get_char_size_NDC(dev, pw, ph)
   PG_device *dev;
   REAL *pw, *ph;
   {

    *ph = dev->char_height_s;
    *pw = dev->char_width_s;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_CHAR_SIZE - return the character size in WC */

void PG_get_char_size(dev, pw, ph)
   PG_device *dev;
   REAL *pw, *ph;
   {

    *ph = (dev->char_height_s)*(dev->bys_w);
    *pw = (dev->char_width_s)*(dev->bxs_w);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_LOGICAL_OP - return the logical operation */

void PG_get_logical_op(dev, plop)
   PG_device *dev;
   int *plop;
   {

    *plop = dev->logical_op;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_LINE_STYLE - return the line style */

void PG_get_line_style(dev, pl)
   PG_device *dev;
   int *pl;
   {


    *pl = dev->line_style;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_CHAR_PATH - return the direction in which text is to be written
 *                  - defaults to (1, 0)
 */

void PG_get_char_path(dev, px, py)
   PG_device *dev;
   REAL *px, *py;
   {

    *px = dev->char_path_x;
    *py = dev->char_path_y;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_ATTRIBUTES - return the graphical attributes of the device in a
 *                   - structure
 *                   - the attributes returned are:
 *                   -
 *                   - struct s_PG_dev_attributes
 *                   -    {int clipping;
 *                   -     int char_font;
 *                   -     REAL char_frac;
 *                   -     int char_precision;
 *                   -     REAL char_space;
 *                   -     char charsz[2];
 *                   -     REAL char_up_x;
 *                   -     REAL char_up_y;
 *                   -     int fill_color;
 *                   -     int line_color;
 *                   -     int line_style;
 *                   -     REAL line_width;
 *                   -     int logical_op;
 *                   -     int text_color;
 *                   -     PG_palette *palette;};
 *                   -
 *                   - typedef struct s_PG_dev_attributes PG_dev_attributes
 *                   -
 */

PG_dev_attributes *PG_get_attributes(dev)
   PG_device *dev;
   {PG_dev_attributes *attr;

    attr = FMAKE(PG_dev_attributes, "PG_GET_ATTRIBUTES:attr");

    attr->clipping       = dev->clipping;
    attr->char_frac      = dev->char_frac;
    attr->char_precision = dev->char_precision;
    attr->char_space     = dev->char_space;
    attr->char_up_x      = dev->char_up_x;
    attr->char_up_y      = dev->char_up_y;
    attr->fill_color     = dev->fill_color;
    attr->line_color     = dev->line_color;
    attr->line_style     = dev->line_style;
    attr->line_width     = dev->line_width;
    attr->logical_op     = dev->logical_op;
    attr->text_color     = dev->text_color;
    attr->palette        = dev->current_palette;

    return(attr);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_FONT - map the given font name to the host font name and
 *               - perform other generic font setup
 *               - return TRUE iff successful
 */

int PG_setup_font(dev, face, style, size, pfn, pnf, pns)
   PG_device *dev;
   char *face, *style;
   int size;
   char **pfn;
   int *pnf, *pns;
   {int l, fn, nfont;
    PG_font_family *ff;
    char **fs;

    if (dev->type_face == NULL)
       dev->type_face = SC_strsavef(face, "char*:PG_SETUP_FONT:face");

    else if (strcmp(dev->type_face, face) != 0)
       {SFREE(dev->type_face);
	dev->type_face = SC_strsavef(face, "char*:PG_SETUP_FONT:face");};

    if (dev->type_style == NULL)
       dev->type_style = SC_strsavef(style, "char*:PG_SETUP_FONT:style");

    else if (strcmp(dev->type_style, style) != 0)
       {SFREE(dev->type_style);
	dev->type_style = SC_strsavef(style, "char*:PG_SETUP_FONT:style");};

    dev->type_size  = size;

    nfont = 0;
    for (ff = dev->font_family; ff != NULL; ff = ff->next)
        {if (strcmp(ff->type_face, face) == 0)
           break;
         nfont += ff->n_styles;};

    if (ff == NULL)
       return(FALSE);

    fs = ff->type_styles;
    fn = ff->n_styles;
    l  = 0;
    if (strcmp(style, "medium") == 0)
       l = 0;
    else if (strcmp(style, "italic") == 0)
       l = 1;
    else if (strcmp(style, "bold") == 0)
       l = 2;
    else if (strcmp(style, "bold-italic") == 0)
       l = 3;

    if (l >= fn)
       return(FALSE);

    *pfn = fs[l];

    nfont += l;
    *pnf   = nfont;
    *pns   = l;

    return(TRUE);}

/*--------------------------------------------------------------------------*/

/*                          STATE CHANGE ROUTINES                           */

/*--------------------------------------------------------------------------*/

/* PG_SET_EXPOSE_EVENT_HANDLER - set the handler to the new FNC and return
 *                             - the old one
 */

PFByte PG_set_expose_event_handler(dev, fnc)
   PG_device *dev;
   PFByte fnc;
   {PFByte oldfnc;

    if (dev != NULL)
       {oldfnc = (PFByte) dev->expose_event_handler.fnc;
	dev->expose_event_handler.fnc = (PFVoid) fnc;}
    else
       oldfnc = NULL;

    return(oldfnc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_UPDATE_EVENT_HANDLER - set the handler to the new FNC and return
 *                             - the old one
 */

PFByte PG_set_update_event_handler(dev, fnc)
   PG_device *dev;
   PFByte fnc;
   {PFByte oldfnc;

    if (dev != NULL)
       {oldfnc = (PFByte) dev->update_event_handler.fnc;
	dev->update_event_handler.fnc = (PFVoid) fnc;}
    else
       oldfnc = NULL;

    return(oldfnc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_MOUSE_DOWN_EVENT_HANDLER - set the handler to the new FNC
 *                                 - and return the old one
 */

PFByte PG_set_mouse_down_event_handler(dev, fnc)
   PG_device *dev;
   PFByte fnc;
   {PFByte oldfnc;

    if (dev != NULL)
       {oldfnc = (PFByte) dev->mouse_down_event_handler.fnc;
	dev->mouse_down_event_handler.fnc = (PFVoid) fnc;}
    else
       oldfnc = NULL;

    return(oldfnc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_MOUSE_UP_EVENT_HANDLER - set the handler to the new FNC
 *                               - and return the old one
 */

PFByte PG_set_mouse_up_event_handler(dev, fnc)
   PG_device *dev;
   PFByte fnc;
   {PFByte oldfnc;

    if (dev != NULL)
       {oldfnc = (PFByte) dev->mouse_up_event_handler.fnc;
	dev->mouse_up_event_handler.fnc = (PFVoid) fnc;}
    else
       oldfnc = NULL;

    return(oldfnc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_MOTION_EVENT_HANDLER - set the handler to the new FNC
 *                             - and return the old one
 */

PFByte PG_set_motion_event_handler(dev, fnc)
   PG_device *dev;
   PFByte fnc;
   {PFByte oldfnc;

    if (dev != NULL)
       {oldfnc = (PFByte) dev->motion_event_handler.fnc;
	dev->motion_event_handler.fnc = (PFVoid) fnc;}
    else
       oldfnc = NULL;

    return(oldfnc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_KEY_DOWN_EVENT_HANDLER - set the handler to the new FNC
 *                               - and return the old one
 */

PFByte PG_set_key_down_event_handler(dev, fnc)
   PG_device *dev;
   PFByte fnc;
   {PFByte oldfnc;

    if (dev != NULL)
       {oldfnc = (PFByte) dev->key_down_event_handler.fnc;
	dev->key_down_event_handler.fnc = (PFVoid) fnc;}
    else
       oldfnc = NULL;

    return(oldfnc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_KEY_UP_EVENT_HANDLER - set the handler to the new FNC
 *                             - and return the old one
 */

PFByte PG_set_key_up_event_handler(dev, fnc)
   PG_device *dev;
   PFByte fnc;
   {PFByte oldfnc;

    if (dev != NULL)
       {oldfnc = (PFByte) dev->key_up_event_handler.fnc;
	dev->key_up_event_handler.fnc = (PFVoid) fnc;}
    else
       oldfnc = NULL;

    return(oldfnc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_DEFAULT_EVENT_HANDLER - set the handler to the new FNC
 *                              - and return the old one
 */

PFByte PG_set_default_event_handler(dev, fnc)
   PG_device *dev;
   PFByte fnc;
   {PFByte oldfnc;

    if (dev != NULL)
       {oldfnc = (PFByte) dev->default_event_handler.fnc;
	dev->default_event_handler.fnc = (PFVoid) fnc;}
    else
       oldfnc = NULL;

    return(oldfnc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_ATTRIBUTES - set the graphical attributes of the device to 
 *                   - those stored in the PG_dev_attribute structure
 *                   - the attributes which were saved are:
 *                   -
 *                   - struct s_PG_dev_attributes
 *                   -    {int clipping;
 *                   -     REAL char_frac;
 *                   -     int char_precision;
 *                   -     REAL char_space;
 *                   -     char charsz[2];
 *                   -     REAL char_up_x;
 *                   -     REAL char_up_y;
 *                   -     int fill_color;
 *                   -     int line_color;
 *                   -     int line_style;
 *                   -     REAL line_width;
 *                   -     int logical_op;
 *                   -     int text_color;
 *                   -     PG_palette *palette;};
 *                   -
 *                   - typedef struct s_PG_dev_attributes PG_dev_attributes
 *                   -
 */

void PG_set_attributes(dev, attr)
   PG_device *dev;
   PG_dev_attributes *attr;
   {

    PG_set_clipping(dev, attr->clipping);

    dev->char_frac       = attr->char_frac;
    dev->char_precision  = attr->char_precision;
    dev->char_space      = attr->char_space;
    dev->char_up_x       = attr->char_up_x;
    dev->char_up_y       = attr->char_up_y;
    dev->fill_color      = attr->fill_color;
    dev->line_color      = attr->line_color;
    dev->line_style      = attr->line_style;
    dev->logical_op      = attr->logical_op;
    dev->text_color      = attr->text_color;
    dev->current_palette = attr->palette;

/* because of the line width mapping this cannot be done as the other are */
    PG_set_line_width(dev, attr->line_width);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FIX_WTOS - compute the WC to NDC transformation coefficients */

static void _PG_fix_wtos(dev, wh)
   PG_device *dev;
   int wh;
   {REAL xmn, xmx, ymn, ymx;

/* use the WC limits of the viewport */
    if (wh)
       {xmn = dev->gxmin;
	xmx = dev->gxmax;
	ymn = dev->gymin;
	ymx = dev->gymax;}

/* use the WC limits of the boundary */
    else
       {xmn = dev->xmin;
	xmx = dev->xmax;
	ymn = dev->ymin;
	ymx = dev->ymax;};

    if (xmn == xmx)
       xmx = xmn + 1.0;

    if (ymn == ymx)
       ymx = ymn + 1.0;

/* WC to NDC transformations */
    dev->bxw_s = (dev->sxmax - dev->sxmin)/(xmx - xmn);
    dev->axw_s = dev->sxmin - xmn*dev->bxw_s;
    dev->byw_s = (dev->symax - dev->symin)/(ymx - ymn);
    dev->ayw_s = dev->symin - ymn*dev->byw_s;
 
/* NDC to WC transformations */
    dev->bxs_w = (xmx - xmn)/(dev->sxmax - dev->sxmin);
    dev->axs_w = xmn - dev->sxmin*dev->bxs_w;
    dev->bys_w = (ymx - ymn)/(dev->symax - dev->symin);
    dev->ays_w = ymn - dev->symin*dev->bys_w;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_SET_BOUND - set the boundary limits for the viewport in WC
 *               - the axes are drawn on the boundary
 *               - and almost nobody else should use this routine
 */
 
void _PG_set_bound(dev, xmn, xmx, ymn, ymx)
   PG_device *dev;
   double xmn, xmx, ymn, ymx;
   {

    dev->xmin = xmn;
    dev->ymin = ymn;
    dev->xmax = xmx;
    dev->ymax = ymx;
 
/* set transformations */
    _PG_fix_wtos(dev, FALSE);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_SET_VIEWPORT - set the viewport for the device
 *                 - the viewport is that part of the window or screen which
 *                 - can be drawn on when clipping is on
 *                 - that is the viewport and clipping window are synonomous
 *                 - view port is specified in NDC
 */
 
void PG_set_viewport(dev, x1, x2, y1, y2)
   PG_device *dev;
   double x1, y1, x2, y2;
   {REAL fxn, dfx, fyn, dfy;

    fxn = dev->fxmin;
    dfx = dev->fxmax - fxn;
    fyn = dev->fymin;
    dfy = dev->fymax - fyn;

    x1 = fxn + x1*dfx;
    x2 = fxn + x2*dfx;
    y1 = fyn + y1*dfy;
    y2 = fyn + y2*dfy;

    dev->sxmin = x1;
    dev->symin = y1;
    dev->sxmax = x2;
    dev->symax = y2;

/* set transformations */
    _PG_fix_wtos(dev, TRUE);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_FRAME_VIEWPORT - shift a viewport from window relative to frame
 *                   - relative values
 */
 
void PG_frame_viewport(dev, x1, x2, y1, y2)
   PG_device *dev;
   REAL *x1, *y1, *x2, *y2;
   {REAL fxn, dfx, fyn, dfy;

    fxn = dev->fxmin;
    dfx = dev->fxmax - fxn;
    fyn = dev->fymin;
    dfy = dev->fymax - fyn;

    *x1 = fxn + (*x1)*dfx;
    *x2 = fxn + (*x2)*dfx;
    *y1 = fyn + (*y1)*dfy;
    *y2 = fyn + (*y2)*dfy;

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_SET_FRAME - set the frame for the device */
 
void PG_set_frame(dev, x1, x2, y1, y2)
   PG_device *dev;
   double x1, y1, x2, y2;
   {REAL sx1, sx2, sy1, sy2;

    PG_get_viewport(dev, &sx1, &sx2, &sy1, &sy2);

    dev->fxmin = x1;
    dev->fymin = y1;
    dev->fxmax = x2;
    dev->fymax = y2;

    PG_set_viewport(dev, sx1, sx2, sy1, sy2);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_AXIS_LOG_SCALE - return the settings of the log axis flags */

void PG_get_axis_log_scale(dev, pxls, pyls)
   PG_device *dev;
   int *pxls, *pyls;
   {

    *pxls = dev->ifxlog;
    *pyls = dev->ifylog;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SET_AXIS_LOG_SCALE - make sure that a log scale is handled correctly */

void PG_set_axis_log_scale(dev, xls, yls)
   PG_device *dev;
   int xls, yls;
   {REAL xmx, xmn, ymx, ymn;

    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);

    dev->ifxlog = xls;
    dev->ifylog = yls;

    PG_set_window(dev, xmn, xmx, ymn, ymx);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_SET_WINDOW - set up the window in WC:
 *               - this is the WC version of the viewport
 *               - when clipping is on it is possible to draw to those
 *               - portions of the screen or window for which
 *               -
 *               -   xmn <= x <= xmx
 *               -   ymn <= y <= ymx
 */
 
void PG_set_window(dev, xmn, xmx, ymn, ymx)
   PG_device *dev;
   double xmn, xmx, ymn, ymx;
   {REAL xrange, yrange;
    double xma, yma;

    dev->gxmin = xmn;
    dev->gymin = ymn;
    dev->gxmax = xmx;
    dev->gymax = ymx;
 
/* if log axis options have been used
 * take logs of min and max
 */
    if (dev->ifxlog)
       {xmx = log10(ABS(xmx) + SMALL);
        xmn = log10(ABS(xmn) + SMALL);
        xma = xmx - _PG_axis_n_decades;

/* limit xmn to be within PG_axis_n_decades of the max value */
        xmn = max(xmn, xma);}

    if (dev->ifylog)
       {ymx = log10(ABS(ymx) + SMALL);
	ymn = log10(ABS(ymn) + SMALL);
        yma = ymx - _PG_axis_n_decades;

/* limit ymn to be within PG_axis_n_decades of the max value */
        ymn = max(ymn, yma);};

    xrange = xmx - xmn;
    if (xrange < 0.0)
       xrange = min(xrange, -SMALL);
    else
       xrange = max(xrange, SMALL);

    yrange = ymx - ymn;
    if (yrange < 0.0)
       yrange = min(yrange, -SMALL);
    else
       yrange = max(yrange, SMALL);

/* leave a little space around the curves */
    PG_set_bound(dev, xmn - dev->leftspace*xrange,
                      xmx + dev->rightspace*xrange,
                      ymn - dev->botspace*yrange,
                      ymx + dev->topspace*yrange);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_MARKERS - set up a set of common markers */

void PG_setup_markers()
   {REAL x1[10], y1[10], x2[10], y2[10];

    if (_PG_marker_list != NULL)
       return;

/* define '+' marker */
    x1[0] = -1.0;
    y1[0] =  0.0;

    x2[0] =  1.0;
    y2[0] =  0.0;

    x1[1] =  0.0;
    y1[1] = -1.0;

    x2[1] =  0.0;
    y2[1] =  1.0;

    PG_def_marker(2, x1, y1, x2, y2);

/* define '*' marker */
    x1[2] = -0.707;
    y1[2] = -0.707;

    x2[2] =  0.707;
    y2[2] =  0.707;

    x1[3] = -0.707;
    y1[3] =  0.707;

    x2[3] =  0.707;
    y2[3] = -0.707;

    PG_def_marker(4, x1, y1, x2, y2);

/* define triangle marker */
    x1[0] = -1.0;
    y1[0] = -1.0;

    x2[0] =  1.0;
    y2[0] = -1.0;

    x1[1] =  1.0;
    y1[1] = -1.0;

    x2[1] =  0.0;
    y2[1] =  1.0;

    x1[2] =  0.0;
    y1[2] =  1.0;

    x2[2] = -1.0;
    y2[2] = -1.0;

    PG_def_marker(3, x1, y1, x2, y2);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RL_MARKERS - release memory associated with markers */

void _PG_rl_markers()
   {int i;
    PG_marker *mrk;

    for (i = 0; i < _PG_marker_index; i++)
        {mrk = &_PG_marker_list[i];
	 SFREE(mrk->x1);
	 SFREE(mrk->y1);
	 SFREE(mrk->x2);
	 SFREE(mrk->y2);};

    SFREE(_PG_marker_list);

    _PG_marker_index = 0;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_DEF_MARKER  - define a marker character and return an index into
 *                - an internal list of markers
 *                - a marker is defined in terms of a list of line segments
 *                - in a box (-1, -1) to (1, 1)
 *                - a scale factor may be separately specified
 *                - the marker made contains copies of the input arrays
 */

int PG_def_marker(n, x1, y1, x2, y2)
   int n;
   REAL *x1, *y1, *x2, *y2;
   {int ne;
    PG_marker *mrk;
    REAL *x1a, *y1a, *x2a, *y2a;

    if (_PG_marker_list == NULL)
       {_PG_marker_max = 10;
        _PG_marker_list = FMAKE_N(PG_marker, _PG_marker_max,
                                  "PG_DEF_MARKER:marker_list");};

    x1a = FMAKE_N(REAL, n, "PG_DEF_MARKER:x1a");    
    y1a = FMAKE_N(REAL, n, "PG_DEF_MARKER:y1a");    
    x2a = FMAKE_N(REAL, n, "PG_DEF_MARKER:x2a");    
    y2a = FMAKE_N(REAL, n, "PG_DEF_MARKER:y2a");    

    ne = n*sizeof(REAL);
    memcpy(x1a, x1, ne);
    memcpy(y1a, y1, ne);
    memcpy(x2a, x2, ne);
    memcpy(y2a, y2, ne);

    mrk        = &_PG_marker_list[_PG_marker_index++];
    mrk->x1    = x1a;
    mrk->y1    = y1a;
    mrk->x2    = x2a;
    mrk->y2    = y2a;
    mrk->n_seg = n;

    if (_PG_marker_index >= _PG_marker_max)
       {_PG_marker_max += 10;
        REMAKE_N(_PG_marker_list, PG_marker, _PG_marker_max);};

    return(_PG_marker_index - 1);}

/*--------------------------------------------------------------------------*/

/*                          MOVE AND DRAW ROUTINES                          */

/*--------------------------------------------------------------------------*/
 
/* PG_DRAW_MARKERS - plot out marker at world coordinates points */
 
void PG_draw_markers(dev, n, x, y, marker)
   PG_device *dev;
   int n;
   REAL *x, *y;
   int marker;
   {int i, j, nm;
    PG_marker *mrk;
    REAL *xa, *xb, *ya, *yb, *xd, *yd, *pxd, *pyd;
    REAL x0, y0, xw, yw, sc;

    if (n <= 0)
       return;

    mrk = &_PG_marker_list[marker];
    sc  = dev->marker_scale;
    xa  = mrk->x1;
    xb  = mrk->x2;
    ya  = mrk->y1;
    yb  = mrk->y2;
    nm  = mrk->n_seg;

    xd = FMAKE_N(REAL, 2*nm*n, "PG_DRAW_MARKERS:xd");
    yd = FMAKE_N(REAL, 2*nm*n, "PG_DRAW_MARKERS:yd");

    pxd = xd;
    pyd = yd;
    for (i = 0; i < n; i++)
        {x0 = x[i];
         y0 = y[i];
         if (dev->ifxlog)
            x0 = log10(ABS(x0) + SMALL);
         if (dev->ifylog)
            y0 = log10(ABS(y0) + SMALL);

         WtoS(dev, x0, y0);

         for (j = 0; j < nm; j++)
             {xw = x0 + sc*xa[j];
              yw = y0 + sc*ya[j];

              *pxd++ = xw;
              *pyd++ = yw;

              xw = x0 + sc*xb[j];
              yw = y0 + sc*yb[j];

              *pxd++ = xw;
              *pyd++ = yw;};};

    PG_draw_disjoint_polyline_2(dev, xd, yd, n*nm, FALSE, FALSE);

    SFREE(xd);
    SFREE(yd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_DRAW_BOX - draw a box around the viewport (WC) */
 
void PG_draw_box(dev, xmin, xmax, ymin, ymax)
   PG_device *dev;
   double xmin, xmax, ymin, ymax;
   {REAL x[5], y[5];

    x[0] = xmin;
    y[0] = ymin;
    x[1] = xmin;
    y[1] = ymax;
    x[2] = xmax;
    y[2] = ymax;
    x[3] = xmax;
    y[3] = ymin;
    x[4] = xmin;
    y[4] = ymin;

    PG_draw_polyline(dev, x, y, 5, TRUE);

    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_DRAW_LINE - draw a line seqment between the specified points
 *              - in WC
 *              - with log plots this is the cleanest way to handle it
 */
 
void PG_draw_line(dev, x1, y1, x2, y2)
   PG_device *dev;
   double x1, y1, x2, y2;
   {

    PG_move_gr_abs(dev, x1, y1);
    PG_draw_to_abs(dev, x2, y2);
 
    return;}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

