/*
 * Copyright (c) 1996, 1998, 1999 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Memory object. Mostly an interface to the dlmalloc routines.
 * This implementation overrides the equivalent implementation in
 * the clientos library.
 */
#include <oskit/com.h>
#include <oskit/com/mem.h>
#include <oskit/com/services.h>
#include <stdlib.h>
#include <malloc.h>			/* mem_lock - BOGUS! */

/* the COM object */
static struct oskit_mem_ops mem_ops;
static oskit_mem_t	oskit_mem  = { &mem_ops };

/*
 * This is a common symbol. The memory interface needs to be defined
 * here so that it is available in the registry immediately.
 */
oskit_mem_t		*oskit_memory_interface = &oskit_mem;

static OSKIT_COMDECL 
mem_query(oskit_mem_t *m, const oskit_iid_t *iid, void **out_ihandle)
{ 
        if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
            memcmp(iid, &oskit_mem_iid, sizeof(*iid)) == 0) {
                *out_ihandle = m;
                return 0; 
        }
  
        *out_ihandle = 0;
        return OSKIT_E_NOINTERFACE;
};

static OSKIT_COMDECL_U
mem_addref(oskit_mem_t *m)
{
	/* No reference counting; only one of them */
	return 1;
}

static OSKIT_COMDECL_U
mem_release(oskit_mem_t *m)
{
	/* No reference counting; only one of them */
	return 1;
}

static void * OSKIT_COMCALL
mem_alloc(oskit_mem_t *m, oskit_u32_t size, oskit_u32_t flags)
{
	void *dl_malloc(size_t size);
	void *ptr;
	
	mem_lock();
	ptr = dl_malloc(size);
	mem_unlock();
	return ptr;
}

static void * OSKIT_COMCALL
mem_realloc(oskit_mem_t *m, void *ptr,
	    oskit_u32_t oldsize, oskit_u32_t newsize, oskit_u32_t flags)
{
	void *dl_realloc(void *buf, size_t new_size);
	void *newptr;
	
	mem_lock();
	newptr = dl_realloc(ptr, newsize);
	mem_unlock();
	return newptr;
}

static void *OSKIT_COMCALL
mem_alloc_aligned(oskit_mem_t *m,
		  oskit_u32_t size, oskit_u32_t flags, oskit_u32_t align)
{
	void *dl_memalign(size_t align, size_t size);
	void *ptr;
	
	mem_lock();
	ptr = dl_memalign(align, size);
	mem_unlock();
	return ptr;
}

static void OSKIT_COMCALL
mem_free(oskit_mem_t *m, void *ptr, oskit_u32_t size, oskit_u32_t flags)
{
	void dl_free(void *buf);

	mem_lock();
	dl_free(ptr);
	mem_unlock();
}

static oskit_u32_t OSKIT_COMCALL
mem_getsize(oskit_mem_t *m, void *ptr)
{
	panic("mem_getsize: Makes no sense in this implementation");
	return 0;
}

static void *OSKIT_COMCALL
mem_alloc_gen(oskit_mem_t *m,
	      oskit_u32_t size, oskit_u32_t lmm_flags,
	      oskit_u32_t align_bits, oskit_u32_t align_ofs)
{
	void *dl_malloc(size_t size);
	void *ptr;
	
	mem_lock();
	ptr = dl_malloc(size);
	mem_unlock();
	return ptr;
}

static oskit_size_t OSKIT_COMCALL
mem_avail(oskit_mem_t *m, oskit_u32_t flags)
{
	return 0;
}

static OSKIT_COMDECL_V
mem_dump(oskit_mem_t *m)
{
}

static struct oskit_mem_ops mem_ops = {
	mem_query,
	mem_addref,
	mem_release,
	mem_alloc,
	mem_realloc,
	mem_alloc_aligned,
	mem_free,
	mem_getsize,
	mem_alloc_gen,
	mem_avail,
	mem_dump,
};

/*
 * Nothing to be done in this impl.
 */
oskit_mem_t *
oskit_mem_init(void)
{
	/* Return a reference to the one-and-only memory object */
	return &oskit_mem;
}
