/*
 * Copyright (c) 1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/* FLASK */

#include "policydb.h"
#include "services.h"

#ifdef __KERNEL__
#define yyin   oskit_security_yyin
#define yyparse   oskit_security_yyparse
#endif

extern FILE    *yyin;
extern int      yyparse(void);

#ifndef __KERNEL__
char           *policyfile = POLICYDB;
char           *progname;
extern char    *optarg;
extern int      optind;
#else
char            progname[] = "ss";
#endif


static int 
load_initial_sid(hashtab_key_t key,
		 hashtab_datum_t datum, void *data)
{
	initial_sid_datum_t *datump;
	sidtab_t       *sidtabp;
	sid_datum_t    siddatum;
	unsigned int    rc;


	sidtabp = (sidtab_t *) data;
	datump = (initial_sid_datum_t *) datum;

	if (!datump->isvalid) {
		fprintf(stderr, "A context for SID %s was never defined.\n", (char *) key);
		return -1;
	}
	if (!ss_root_context_valid(&datump->context, &policydb, &rc)) {
		fprintf(stderr, "Initial SID %s has an invalid context.\n", (char *) key);
		return -1;
	}

	context_init(&siddatum.context, TRUE);
	if (!root_context_cpy(&RCONTEXT(&siddatum.context), 
			      &datump->context)) {
		fprintf(stderr, "out of memory");
		return -1;
	}
	siddatum.children.table = 0;
	siddatum.children.nprim = 0;
	siddatum.child_val_to_name = 0;

	return sidtab_insert(sidtabp, datump->sid, &siddatum);
}


#ifndef __KERNEL__
static int 
print_sid(security_id_t key,
	  sid_datum_t *datum, void *data)
{
	security_context_t scontext;
	security_econtext_t econtext;
	unsigned int    scontext_len, econtext_len;

	security_sid_to_econtext(key, &scontext, &scontext_len,
				&econtext, &econtext_len);
	if (econtext[0])
		printf("sid %ld -> scontext %s econtext %s\n", key, 
		       scontext, econtext);
	else
		printf("sid %ld -> scontext %s\n", key, scontext);
	FREE(scontext, scontext_len);
	FREE(econtext, econtext_len);	
	return 0;
}

static int 
find_perm(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	unsigned int   *valuep;
	perm_datum_t   *perdatum;

	valuep = (unsigned int *) p;
	perdatum = (perm_datum_t *) datum;

	if (*valuep == perdatum->value)
		return (int) key;

	return 0;
}
#endif

#ifdef __KERNEL__
#include <oskit/dev/osenv.h>
#include <oskit/dev/osenv_mem.h>
#include <oskit/dev/osenv_log.h>
#include <oskit/flask/avc_ss.h>
#include <oskit/flask/security.h>
#include <oskit/c/stdarg.h>

typedef struct avc_node {
	oskit_avc_ss_t *avc;
	ebitmap_t classes;
	struct avc_node *next;
} avc_node_t;

typedef struct {
	oskit_security_t securityi;
	unsigned count;
	oskit_osenv_mem_t *mem;
	oskit_osenv_log_t *log;
	avc_node_t *avc_list;
} security_server_t;

static security_server_t ss;

static struct oskit_security_ops ss_ops;


int oskit_security_avc_ss_reset(oskit_u32_t seqno)
{
	avc_node_t *cur;
	
	for (cur = ss.avc_list; cur; cur = cur->next) 
		oskit_avc_ss_reset(cur->avc, seqno);
	return 0;
}


void *oskit_security_malloc(unsigned size) 
{
	return oskit_osenv_mem_alloc(ss.mem, size, OSENV_AUTO_SIZE, 0);
}

#undef free
void oskit_security_free(void *addr) 
{
	return oskit_osenv_mem_free(ss.mem, addr, OSENV_AUTO_SIZE, 0);
}


void *oskit_security_realloc(void *addr, unsigned size) 
{
        if (addr == NULL)
		return oskit_osenv_mem_alloc(ss.mem, size, OSENV_AUTO_SIZE, 0);
	oskit_security_panic("realloc");
	return NULL;
}


int oskit_security_printf(const char * fmt, ...)
{
	va_list ap;
	
	va_start(ap, fmt);
	oskit_osenv_log_vlog(ss.log, OSENV_LOG_INFO, fmt, ap);
	va_end(ap);
	return 0;
}


void oskit_security_panic(const char * fmt, ...)
{
	va_list ap;
	
	va_start(ap, fmt);
	oskit_osenv_log_vpanic(ss.log, fmt, ap);
	va_end(ap);
}



#endif

#ifndef __KERNEL__
int 
main(int argc, char **argv)
#else
oskit_error_t
oskit_security_init(oskit_services_t *osenv,
		    struct oskit_openfile *f,
		    oskit_security_t **out_security)
#endif
{
#ifndef __KERNEL__
	avdef_datum_t  *avddatum;
	class_datum_t  *cladatum;
	security_class_t tclass;
	security_id_t   ssid, tsid;
	access_vector_t allowed, decided, auditallow, auditdeny, notify;
	unsigned int    seqno;
	security_context_t scontext;
	security_econtext_t econtext;
	unsigned int    scontext_len, econtext_len, pathlen;
	char            ans[80 + 1], *path, *perm, trans;
	int             i, ret;
	FILE		*fp;
#endif


#ifndef __KERNEL__
	progname = argv[0];
	if (argc > 1) {
		policyfile = argv[1];
	}
#endif

#ifdef __KERNEL__
	oskit_services_lookup_first(osenv, &oskit_osenv_log_iid, 
				    (void **) &ss.log);
	if (!ss.log)
		return OSKIT_EINVAL;
	oskit_osenv_log_addref(ss.log);

	oskit_services_lookup_first(osenv, &oskit_osenv_mem_iid, 
				    (void **) &ss.mem);
	if (!ss.mem)
		return OSKIT_EINVAL;
	oskit_osenv_mem_addref(ss.mem);

	ss.securityi.ops = &ss_ops;
	ss.count = 1;
	ss.avc_list = NULL;

	*out_security = &ss.securityi;

	printf("\nSecurity Policy Server starting up (compiled " __DATE__ ")\n");
#endif

	printf("%s:  loading policy configuration\n", progname);

#ifdef __KERNEL__
	yyin = f;
#else
	yyin = fopen(policyfile, "r");
	if (!yyin) {
		fprintf(stderr, "%s:  unable to open %s\n", progname, policyfile);
		exit(1);
	}
#endif
	if (policydb_init(&policydb))
		exit(1);

	id_queue = queue_create();
	if (!id_queue) {
		fprintf(stderr, "%s:  out of memory\n", progname);
		exit(1);
	}
	policydbp = &policydb;
	policydb_errors = 0;
	if (yyparse() || policydb_errors) {
		fprintf(stderr, "%s:  error(s) encountered while parsing configuration\n", progname);
		exit(1);
	}
	queue_destroy(id_queue);

	if (policydb_transform(&policydb)) {
		fprintf(stderr, "%s:  unable to transform policydb\n", progname);
		exit(1);
	}
	printf("%s:  %d users, %d roles, %d types, %d levels, %d transitions\n",
	       progname, policydb.users->nprim, policydb.roles->nprim,
	       policydb.types->nprim, policydb.nlevels,
	       policydb.transitions->nel);

	printf("%s:  %d access vectors (%d constrained), %d access rules (%d constrained)\n",
	       progname, policydb.avdefs->nel, policydb.avdefs->ncons,
	       policydb.rules->nel, policydb.rules->ncons);

	avtab_hash_eval(policydb.rules, progname, "rules");

	if (sidtab_init(&sidtab)) {
		fprintf(stderr, "%s:  unable to initialize SID table\n", progname);
		exit(1);
	}
	if (hashtab_map(policydb.sids->table, load_initial_sid, &sidtab)) {
		fprintf(stderr, "%s:  unable to load initial SIDs into SID table\n", progname);
		exit(1);
	}
	if (extavtab_init(&extavtab)) {
		fprintf(stderr, "%s:  unable to initialize extension policy table\n", progname);
		exit(1);
	}
	if (exttrtab_init(&exttrtab)) {
		fprintf(stderr, "%s:  unable to initialize extension transition table\n", progname);
		exit(1);
	}
	ss_initialized = 1;

#ifndef __KERNEL__
	printf("%s:  policy configuration loaded\n", progname);
	while (1) {
		printf("\nSelect an option:\n");
		printf("0)  Call compute_access_vector\n");
		printf("1)  Call sid_to_context\n");
		printf("2)  Call context_to_sid\n");
		printf("3)  Call sid_to_econtext\n");
		printf("4)  Call econtext_to_sid\n");
		printf("5)  Call transition_sid\n");
		printf("6)  Call member_sid\n");
		printf("7)  Call list_sids\n");
		printf("8)  Call load_policy\n");
		printf("9)  Call load_extension\n");
		printf("q)  Exit\n");
		printf("\nChoose:  ");
		fgets(ans, sizeof(ans), stdin);
		trans = 0;
		switch (ans[0]) {
		case '0':
			printf("source sid?  ");
			fgets(ans, sizeof(ans), stdin);
			ssid = atoi(ans);

			printf("target sid?  ");
			fgets(ans, sizeof(ans), stdin);
			tsid = atoi(ans);

			printf("target class?  ");
			fgets(ans, sizeof(ans), stdin);
			if (isdigit(ans[0])) {
				tclass = atoi(ans);
				if (!tclass || tclass > policydb.classes->nprim) {
					printf("\nNo such class.\n");
					break;
				}
			} else {
				ans[strlen(ans) - 1] = 0;
				cladatum = (class_datum_t *) hashtab_search(policydb.classes->table,
								       ans);
				if (!cladatum) {
					printf("\nNo such class\n");
					break;
				}
				tclass = cladatum->value;
			}

			avddatum = avdeftab_search(policydb.avdefs, tclass);
			if (!avddatum) {
				printf("\nNo access vector definition for that class\n");
				break;
			}
			ret = security_compute_av(ssid, tsid, tclass, 0,
						  &allowed, &decided,
#ifdef CONFIG_FLASK_AUDIT
						  &auditallow, &auditdeny,
#endif
#ifdef CONFIG_FLASK_NOTIFY
						  &notify,
#endif
						  &seqno);
			switch (ret) {
			case 0:
				printf("\nallowed {");
				for (i = 1; i <= sizeof(allowed) * 8; i++) {
					if (allowed & (1 << (i - 1))) {
						perm = (char *) hashtab_map(avddatum->private.table,
							     find_perm, &i);

						if (!perm && avddatum->common) {
							perm = (char *) hashtab_map(avddatum->common->table,
							     find_perm, &i);
						}
						if (perm)
							printf(" %s", perm);
					}
				}
				printf(" }\n");
				break;
			case -EINVAL:
				printf("\ninvalid sid\n");
				break;
			default:
				printf("return code 0x%x\n", ret);
			}
			break;
		case '1':
			printf("sid?  ");
			fgets(ans, sizeof(ans), stdin);
			ssid = atoi(ans);
			ret = security_sid_to_context(ssid,
						  &scontext, &scontext_len);
			switch (ret) {
			case 0:
				printf("\nscontext %s\n", scontext);
				FREE(scontext, scontext_len);
				break;
			case -EINVAL:
				printf("\ninvalid sid\n");
				break;
			case -ENOMEM:
				printf("\nout of memory\n");
				break;
			default:
				printf("return code 0x%x\n", ret);
			}
			break;
		case '2':
			printf("scontext?  ");
			fgets(ans, sizeof(ans), stdin);
			scontext = (char *) strdup(ans);
			scontext_len = strlen(ans);
			scontext[scontext_len - 1] = 0;
			ret = security_context_to_sid(scontext, scontext_len,
						      &ssid);
			switch (ret) {
			case 0:
				printf("\nsid %ld\n", ssid);
				break;
			case -EINVAL:
				printf("\ninvalid context\n");
				break;
			case -ENOMEM:
				printf("\nout of memory\n");
				break;
			default:
				printf("return code 0x%x\n", ret);
			}

			FREE(scontext, scontext_len);
			break;
		case '3':
			printf("sid?  ");
			fgets(ans, sizeof(ans), stdin);
			ssid = atoi(ans);
			ret = security_sid_to_econtext(ssid,
						       &scontext, 
						       &scontext_len,
						       &econtext,
						       &econtext_len);
			switch (ret) {
			case 0:
				printf("\nscontext %s econtext %s\n", 
				       scontext, econtext);
				FREE(scontext, scontext_len);
				FREE(econtext, econtext_len);
				break;
			case -EINVAL:
				printf("\ninvalid sid\n");
				break;
			case -ENOMEM:
				printf("\nout of memory\n");
				break;
			default:
				printf("return code 0x%x\n", ret);
			}
			break;
		case '4':
			printf("scontext?  ");
			fgets(ans, sizeof(ans), stdin);
			scontext = (char *) strdup(ans);
			scontext_len = strlen(ans);
			scontext[scontext_len - 1] = 0;
			printf("econtext?  ");
			fgets(ans, sizeof(ans), stdin);
			econtext = (char *) strdup(ans);
			econtext_len = strlen(ans);
			econtext[econtext_len - 1] = 0;
			ret = security_econtext_to_sid(scontext, scontext_len,
						       econtext, econtext_len,
						       &ssid);
			switch (ret) {
			case 0:
				printf("\nsid %ld\n", ssid);
				break;
			case -EINVAL:
				printf("\ninvalid context\n");
				break;
			case -ENOMEM:
				printf("\nout of memory\n");
				break;
			default:
				printf("return code 0x%x\n", ret);
			}

			FREE(scontext, scontext_len);
			FREE(econtext, econtext_len);
			break;
		case '5':
			trans = 1;
		case '6':
			printf("source sid?  ");
			fgets(ans, sizeof(ans), stdin);
			ssid = atoi(ans);
			printf("target sid?  ");
			fgets(ans, sizeof(ans), stdin);
			tsid = atoi(ans);

			printf("object class?  ");
			fgets(ans, sizeof(ans), stdin);
			if (isdigit(ans[0])) {
				tclass = atoi(ans);
				if (!tclass || tclass > policydb.classes->nprim) {
					printf("\nNo such class.\n");
					break;
				}
			} else {
				ans[strlen(ans) - 1] = 0;
				cladatum = (class_datum_t *) hashtab_search(policydb.classes->table,
								       ans);
				if (!cladatum) {
					printf("\nNo such class\n");
					break;
				}
				tclass = cladatum->value;
			}

			if (trans)
				ret = security_transition_sid(ssid, tsid, tclass, &ssid);
			else
				ret = security_member_sid(ssid, tsid, tclass, &ssid);
			switch (ret) {
			case 0:
				printf("\nsid %ld\n", ssid);
				break;
			case -EINVAL:
				printf("\ninvalid sid\n");
				break;
			case -ENOMEM:
				printf("\nout of memory\n");
				break;
			default:
				printf("return code 0x%x\n", ret);
			}
			break;
		case '7':
			sidtab_map(&sidtab, print_sid, 0);
			break;
		case '8':
			printf("pathname?  ");
			fgets(ans, sizeof(ans), stdin);
			path = (char *) strdup(ans);
			pathlen = strlen(ans);
			path[pathlen - 1] = 0;
			fp = fopen(path, "r");
			if (!fp) {
				perror(path);
				free(path);
				break;
			}
			ret = security_load_policy(fp);
			switch (ret) {
			case 0:
				printf("\nsuccess\n");
				break;
			case -EINVAL:
				printf("\ninvalid path or policy\n");
				break;
			case -ENOMEM:
				printf("\nout of memory\n");
				break;
			default:
				printf("return code 0x%x\n", ret);
			}
			FREE(path, pathlen);
			break;
		case '9':
			printf("parent sid?  ");
			fgets(ans, sizeof(ans), stdin);
			ssid = atoi(ans);
			printf("pathname?  ");
			fgets(ans, sizeof(ans), stdin);
			path = (char *) strdup(ans);
			pathlen = strlen(ans);
			path[pathlen - 1] = 0;
			fp = fopen(path, "r");
			if (!fp) {
				perror(path);
				free(path);
				break;
			}
			ret = security_load_extension(ssid, fp);
			switch (ret) {
			case 0:
				printf("\nsuccess\n");
				break;
			case -EINVAL:
				printf("\ninvalid path or policy\n");
				break;
			case -ENOMEM:
				printf("\nout of memory\n");
				break;
			default:
				printf("return code 0x%x\n", ret);
			}
			FREE(path, pathlen);
			break;
		case 'q':
			exit(0);
			break;
		default:
			printf("\nUnknown option %s.\n", ans);
		}
	}
#endif
	return 0;
}

#ifdef __KERNEL__
static struct oskit_security_ops ss_ops;

static OSKIT_COMDECL ss_query(oskit_security_t *s,
			      const struct oskit_guid *iid,
			      void **out_ihandle)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;
	
	if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
	    memcmp(iid, &oskit_security_iid, sizeof(*iid)) == 0) {
		*out_ihandle = &ss.securityi;
		++ss.count;
		return 0;
	}

	*out_ihandle = NULL;
	return OSKIT_E_NOINTERFACE;    
}


static OSKIT_COMDECL_U ss_addref(oskit_security_t *s)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return ++ss.count;
}


static OSKIT_COMDECL_U ss_release(oskit_security_t *s)
{
	oskit_osenv_mem_t *mem;	
	avc_node_t *node, *tmp;
	unsigned newcount;

	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	newcount = --ss.count;
	if (newcount == 0) {
		/* 
		 * This is not complete.  Still need to add
		 * support for cleaning up the sidtab and
		 * the extension policy tables (extavtab, exttrtab).
		 */
		policydb_destroy(&policydb);
		oskit_osenv_log_release(ss.log);
		mem = ss.mem;
		node = ss.avc_list;
		while (node) {
			tmp = node;
			node = node->next;
			oskit_avc_ss_release(tmp->avc);
			ebitmap_destroy(&tmp->classes);
			oskit_osenv_mem_free(mem, tmp, 0, 
					     sizeof(struct avc_node));
		}
		oskit_osenv_mem_release(mem);
	}
	
	return newcount;
}


static OSKIT_COMDECL ss_compute_av(oskit_security_t *s,
				   oskit_security_id_t ssid,
				   oskit_security_id_t tsid,
				   oskit_security_class_t tclass,
				   oskit_access_vector_t requested,
				   oskit_access_vector_t *allowed,
				   oskit_access_vector_t *decided,
				   oskit_access_vector_t *auditallow,
				   oskit_access_vector_t *auditdeny,
				   oskit_u32_t *seqno)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_compute_av(ssid,tsid,tclass,requested,
				   allowed,decided,auditallow,auditdeny,
				   seqno);
}


static OSKIT_COMDECL ss_transition_sid(oskit_security_t *s,
				       oskit_security_id_t ssid,
				       oskit_security_id_t tsid,
				       oskit_security_class_t tclass,
				       oskit_security_id_t *out_sid)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_transition_sid(ssid,tsid,tclass,out_sid);
}


static OSKIT_COMDECL ss_member_sid(oskit_security_t *s,
				       oskit_security_id_t ssid,
				       oskit_security_id_t tsid,
				       oskit_security_class_t tclass,
				       oskit_security_id_t *out_sid)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_member_sid(ssid,tsid,tclass,out_sid);
}


static OSKIT_COMDECL ss_sid_to_context(oskit_security_t *s,
				       oskit_security_id_t sid,
				       oskit_security_context_t *scontext,
				       oskit_u32_t *scontext_len)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_sid_to_context(sid,scontext,scontext_len);
}


static OSKIT_COMDECL ss_context_to_sid(oskit_security_t *s,
				       oskit_security_context_t scontext,
				       oskit_u32_t scontext_len,
				       oskit_security_id_t *out_sid)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_context_to_sid(scontext,scontext_len,out_sid);
}


static OSKIT_COMDECL ss_sid_to_econtext(oskit_security_t *s,
					oskit_security_id_t sid,
					oskit_security_context_t *scontext,
					oskit_u32_t *scontext_len,
					oskit_security_econtext_t *econtext,
					oskit_u32_t *econtext_len)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_sid_to_econtext(sid,scontext,scontext_len,
					econtext, econtext_len);
}


static OSKIT_COMDECL ss_econtext_to_sid(oskit_security_t *s,
					oskit_security_context_t scontext,
					oskit_u32_t scontext_len,
					oskit_security_econtext_t econtext,
					oskit_u32_t econtext_len,
					oskit_security_id_t *out_sid)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_econtext_to_sid(scontext,scontext_len,
					econtext,econtext_len,out_sid);
}


static OSKIT_COMDECL ss_register_avc(oskit_security_t *s,
				     oskit_security_class_t *classes,
				     oskit_u32_t nclasses,
				     struct oskit_avc_ss *avc)
{
	struct avc_node *node;
	int i;

	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	node = oskit_osenv_mem_alloc(ss.mem,sizeof(struct avc_node),0,0);
	if  (!node) 
		return OSKIT_ENOMEM;
	node->avc = avc;
	oskit_avc_ss_addref(avc);
	ebitmap_init(&node->classes);
	for (i = 0; i < nclasses; i++) {
		ebitmap_set_bit(&node->classes, classes[i], TRUE);
	}
	
	node->next = ss.avc_list;
	ss.avc_list = node;
	return 0;
}


static OSKIT_COMDECL ss_unregister_avc(oskit_security_t *s,
				       struct oskit_avc_ss *avc)
{
	struct avc_node *prev, *cur;

	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	prev = NULL;
	cur = ss.avc_list;
	while (cur) {
		if (cur->avc == avc)
			break;
		prev = cur;
		cur = cur->next;
	}
	
	if (!cur)
		return OSKIT_ENOENT;
	
	if (prev)
		prev->next = cur->next;
	else
		ss.avc_list = cur->next;

	oskit_avc_ss_release(cur->avc);	
	ebitmap_destroy(&cur->classes);
	oskit_osenv_mem_free(ss.mem, cur, 0, sizeof(struct avc_node));
	return 0;
}


static OSKIT_COMDECL ss_load_policy(oskit_security_t *s,
				    struct oskit_openfile *f)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_load_policy(f);
}


static OSKIT_COMDECL ss_load_extension(oskit_security_t *s,
				       security_id_t parent,
				       struct oskit_openfile *f)
{
	if (s != ((oskit_security_t *) &ss))
		return OSKIT_EINVAL;

	return security_load_extension(parent,f);
}


static struct oskit_security_ops ss_ops = {
	ss_query,
	ss_addref,
	ss_release,
	ss_compute_av,
	ss_transition_sid,
	ss_member_sid,	
	ss_sid_to_context,
	ss_context_to_sid,
	ss_sid_to_econtext,
	ss_econtext_to_sid,
	ss_register_avc,
	ss_unregister_avc,
	ss_load_policy,
	ss_load_extension
};

#endif

/* FLASK */
