package com.jclark.xsl.sax;

import org.xml.sax.*;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.File;
import java.io.FileDescriptor;
import java.net.URL;

public class Driver {
  public static void main(String[] args) {
    if (args.length != 2 && args.length != 3) {
      System.err.println("usage: java com.jclark.xsl.sax.Driver source stylesheet [result]");
      System.exit(1);
    }
    Parser parser = createParser();
    if (parser == null)
      System.exit(1);
    XSLProcessor xsl = new XSLProcessorImpl(parser);
    ResultTypeHandlerImpl resultTypeHandler = new ResultTypeHandlerImpl(xsl);
    xsl.setResultTypeHandler(resultTypeHandler);
    boolean succeeded = true;
    File in = new File(args[0]);
    File stylesheet = new File(args[1]);
    File out = args.length == 3 ? new File(args[2]) : null;
    if (!in.isDirectory())
      succeeded = transformFile(xsl, resultTypeHandler, in, stylesheet, out);
    else {
      String[] inFiles = in.list();
      for (int i = 0; i < inFiles.length; i++) {
	File inFile = new File(in, inFiles[i]);
	if (!inFile.isDirectory()) {
	  if (!stylesheet.isDirectory()) {
	    // FIXME optimize this case by loading the stylesheet only once
	    if (!transformFile(xsl, resultTypeHandler,
			       inFile, stylesheet, new File(out, inFiles[i])))
	      succeeded = false;
	  }
	  else {
	    int ext = inFiles[i].lastIndexOf('.');
	    File stylesheetFile
	      = new File(stylesheet,
			 ext < 0
			 ? inFiles[i]
			 : inFiles[i].substring(0, ext) + ".xsl");
	    if (stylesheetFile.exists()
		&& !transformFile(xsl,
				  resultTypeHandler,
				  inFile,
				  stylesheetFile,
				  new File(out, inFiles[i])))
	      succeeded = false;
	  }
	}
      }
    }
    System.exit(succeeded ? 0 : 1);
  }

  static boolean transformFile(XSLProcessor xsl,
			       ResultTypeHandlerImpl resultTypeHandler,
			       File inputFile,
			       File stylesheetFile,
			       File outputFile) {
    FileOutputStream out = null;
    if (outputFile == null)
      out = new FileOutputStream(FileDescriptor.out);
    else {
      try {
	out = new FileOutputStream(outputFile);
      }
      catch (IOException e) {
	System.err.println(e.toString());
	System.exit(1);
      }
    }
    resultTypeHandler.setOutputStream(out);
    xsl.setStylesheet(fileInputSource(stylesheetFile));
    return transform(xsl, fileInputSource(inputFile));
  }

  static boolean transform(XSLProcessor xsl, InputSource source) {
    try {
      xsl.parse(source);
      return true;
    }
    catch (SAXParseException e) {
      String systemId = e.getSystemId();
      int lineNumber = e.getLineNumber();
      if (systemId != null)
	System.err.print(systemId + ":");
      if (lineNumber >= 0)
	System.err.print(lineNumber + ":");
      if (systemId != null || lineNumber >= 0)
	System.err.print(" ");
      System.err.println(e.getMessage());
    }
    catch (SAXException e) {
      System.err.println(e.getMessage());
    }
    catch (IOException e) {
      System.err.println(e.toString());
    }
    return false;
  }

  static Parser createParser() {
    String parserClass = System.getProperty("com.jclark.xsl.sax.parser");
    if (parserClass == null)
      parserClass = System.getProperty("org.xml.sax.parser");
    if (parserClass == null)
      parserClass = "com.jclark.xml.sax.CommentDriver";
    try {
      return (Parser)Class.forName(parserClass).newInstance();
    }
    catch (ClassNotFoundException e) {
      System.err.println(e.toString());
    }
    catch (InstantiationException e) {
      System.err.println(e.toString());
    }
    catch (IllegalAccessException e) {
      System.err.println(e.toString());
    }
    catch (ClassCastException e) {
      System.err.println(parserClass + " is not a SAX driver");
    }
    return null;
  }

  /**
   * Generates an <code>InputSource</code> from a file name.
   */
  static public InputSource fileInputSource(String str) {
    return fileInputSource(new File(str));
  }

  static public InputSource fileInputSource(File file) {
    String path = file.getAbsolutePath();
    String fSep = System.getProperty("file.separator");
    if (fSep != null && fSep.length() == 1)
      path = path.replace(fSep.charAt(0), '/');
    if (path.length() > 0 && path.charAt(0) != '/')
      path = '/' + path;
    try {
      return new InputSource(new URL("file", "", path).toString());
    }
    catch (java.net.MalformedURLException e) {
      /* According to the spec this could only happen if the file
	 protocol were not recognized. */
      throw new Error("unexpected MalformedURLException");
    }
  }

}
