/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <sys/stat.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>
#include <ctype.h>
#include <locale.h>
#include <string.h>

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "gbwidget.h"
#include "source.h"

#define GB_SOURCE_BUFFER_INCREMENT      8192
#define GB_SOURCE_BUFFER_ENSURE_SPACE   4096

#define MAX_IDENTIFIER_LEN              1024
#define MAX_LINE_LEN                    1024

#define TEXT_BUFFER_INCREMENT           1024

static GbStatusCode real_source_write (gchar * project_name,
                                       gchar * directory);
static void destroy_standard_widget (gchar * key,
                                     GtkWidget * widget,
                                     gpointer data);

static void source_write_configure_in (GbWidgetWriteSourceData * data);
static void source_write_makefile_am (GbWidgetWriteSourceData * data);
static void source_write_makefile (GbWidgetWriteSourceData * data);

static void source_write_main_c_preamble (GbWidgetWriteSourceData * data);
static void source_write_main_h_preamble (GbWidgetWriteSourceData * data);
static void source_write_signal_c_preamble (GbWidgetWriteSourceData * data);
static void source_write_signal_h_preamble (GbWidgetWriteSourceData * data);
static void source_write_preamble (GbWidgetWriteSourceData * data,
                                   FILE * fp);

static void source_write_component_create (GtkWidget * component,
                                           GbWidgetWriteSourceData * data);
static void source_write_component (GtkWidget * component,
                                    GbWidgetWriteSourceData * data);
static gboolean file_exists(gchar *filename);
static GbStatusCode backup_file_if_exists (gchar * filename);

static void add_to_buffer (gchar ** buffer,
                           gint * buffer_pos,
                           gint * buffer_space,
                           gchar * text);
static void add_char_to_buffer (gchar ** buffer,
                                gint * buffer_pos,
                                gint * buffer_space,
                                gchar ch);

/* We need this so that numbers are written in C syntax rather than the
   current locale, which may use ',' instead of '.' and then the code
   will not compile. This code is from glibc info docs. */
GbStatusCode
source_write (gchar * project_name, gchar * directory)
{
  gchar *old_locale, *saved_locale;
  GbStatusCode status;

  old_locale = setlocale (LC_NUMERIC, NULL);
  saved_locale = g_strdup (old_locale);
  setlocale (LC_NUMERIC, "C");
  status = real_source_write (project_name, directory);
  setlocale (LC_NUMERIC, saved_locale);
  g_free (saved_locale);
  return status;
}


static GbStatusCode
real_source_write (gchar * project_name, gchar * directory)
{
  GbWidgetWriteSourceData data;
  FILE *fp;

  data.status = GB_OK;
  data.project_name = project_name;
  data.main_c_filename = "gladesrc.inc";
  data.main_h_filename = "gladesrch.inc";
  data.signal_c_filename = "gladesig.inc";
  data.signal_h_filename = "gladesigh.inc";
  data.set_widget_names = FALSE;
  data.use_widget_hash = TRUE;
  data.standard_widgets = g_hash_table_new (g_str_hash, g_str_equal);

  if (chdir (directory) == -1)
    return GB_CHDIR_ERROR;

  source_write_configure_in (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_makefile_am (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_makefile (&data);
  if (data.status != GB_OK)
    return data.status;

  /* Create all 4 files. Note that we will not overwrite signal files in the
     final version. */
  backup_file_if_exists (data.main_c_filename);
  backup_file_if_exists (data.main_h_filename);
  backup_file_if_exists (data.signal_c_filename);
  backup_file_if_exists (data.signal_h_filename);

  data.cfp = data.hfp = data.sigcfp = data.sighfp = NULL;
  data.cfp = fopen (data.main_c_filename, "w");
  if (data.cfp == NULL)
    return GB_FILE_OPEN_ERROR;
  data.hfp = fopen (data.main_h_filename, "w");
  if (data.hfp == NULL)
    return GB_FILE_OPEN_ERROR;
  data.sigcfp = fopen (data.signal_c_filename, "w");
  if (data.sigcfp == NULL)
    return GB_FILE_OPEN_ERROR;
  data.sighfp = fopen (data.signal_h_filename, "w");
  if (data.sighfp == NULL)
    return GB_FILE_OPEN_ERROR;

  source_write_main_c_preamble (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_main_h_preamble (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_signal_c_preamble (&data);
  if (data.status != GB_OK)
    return data.status;
  source_write_signal_h_preamble (&data);
  if (data.status != GB_OK)
    return data.status;

  data.buffer_space = GB_SOURCE_BUFFER_INCREMENT;
  data.buffer = g_new (gchar, data.buffer_space);
  data.buffer[0] = '\0';
  data.buffer_pos = 0;

  data.decl_buffer_space = GB_SOURCE_BUFFER_INCREMENT;
  data.decl_buffer = g_new (gchar, data.decl_buffer_space);
  data.decl_buffer[0] = '\0';
  data.decl_buffer_pos = 0;

  /* This outputs code in main() to create one of each component, just so that
     the user sees something after first building the project. */
  project_foreach_component ((GtkCallback) source_write_component_create,
                             &data);

  /* write gladetest.pp */
  fp = fopen ("gladetest.pp", "w");
  if (fp == NULL)
    return GB_FILE_OPEN_ERROR;
  fprintf (fp,
           "program gladetest;\n"
           "uses\n"
           "  glib,gdk,gtk;\n"
           "\n"
           "{$I gladesig.inc}\n"
           "{$I gladesrc.inc}\n"
           "\n"
           "var\n"
           "%s"
           "begin\n"
           "  gtk_set_locale ();\n"
           "  gtk_init (@argc, @argv);\n"
           "\n"
           "  {\n"
           "    The following code was added by Glade to create one of each component\n"
           "    (except popup menus), just so that you see something after building\n"
           "    the project. Delete any components that you don't want shown initially.\n"
           "  }\n"
           "%s"
           "\n"
           "  gtk_main ();\n"
           "end.\n",data.decl_buffer,data.buffer);
  fclose (fp);

  /* This outputs the code to create the components and the signal handler
     prototypes. */
  project_foreach_component ((GtkCallback) source_write_component, &data);

  g_free (data.buffer);
  g_free (data.decl_buffer);
  g_hash_table_foreach (data.standard_widgets,
                        (GHFunc) destroy_standard_widget, NULL);
  g_hash_table_destroy (data.standard_widgets);


  if (data.cfp)
    fclose (data.cfp);
  if (data.hfp)
    fclose (data.hfp);
  if (data.sigcfp)
    fclose (data.sigcfp);
  if (data.sighfp)
    fclose (data.sighfp);
  return data.status;
}


static void
destroy_standard_widget (gchar * key, GtkWidget * widget, gpointer data)
{
  gtk_widget_destroy (widget);
}


static void
source_write_configure_in (GbWidgetWriteSourceData * data)
{
}


static void
source_write_makefile_am (GbWidgetWriteSourceData * data)
{
}


static void
source_write_makefile (GbWidgetWriteSourceData * data)
{
}


static void
source_write_main_c_preamble (GbWidgetWriteSourceData * data)
{
  source_write_preamble (data, data->cfp);

  /* Write a function to get a widget from the component's hash. */
  if (data->use_widget_hash)
    {
      fprintf (data->cfp,
               "function get_widget(widget:PGtkWidget;widget_name:pchar):PGtkWidget;\n"
               "var\n"
               "  found_widget : PGtkWidget;\n"
               "begin\n"
               "  if assigned(widget^.parent) then\n"
               "    widget := gtk_widget_get_toplevel (widget);\n"
               "  found_widget := gtk_object_get_data (PGtkObject(widget),widget_name);\n"
               "  {if not assigned(found_widget) then\n"
               "    g_warning (\"Widget not found: %%s\", widget_name);}\n"
               "  get_widget := found_widget;\n"
               "end;\n"
               "\n");
    }

  /* Write a real kludgey function to set the tab of an already-created
     notebook page. */
  fprintf (data->cfp,
           "{ This is an internally used function to set notebook tab widgets. }\n"
           "procedure set_notebook_tab(notebook:PGtkWidget;page_num:gint;widget:PGtkWidget);\n"
           "var\n"
           "  page : PGtkNotebookPage;\n"
           "  notebook_page : PGtkWidget;\n"
           "begin\n"
           "  page := g_list_nth (PGtkNoteBook(notebook)^.children, page_num)^.data;\n"
           "  notebook_page := page^.child;\n"
           "  gtk_widget_ref (notebook_page);\n"
           "  gtk_notebook_remove_page (PGtkNoteBook(notebook), page_num);\n"
           "  gtk_notebook_insert_page (PGtkNoteBook(notebook), notebook_page, widget, page_num);\n"
           "  gtk_widget_unref (notebook_page);\n"
           "end;\n"
           "\n");
}


static void
source_write_main_h_preamble (GbWidgetWriteSourceData * data)
{
  source_write_preamble (data, data->hfp);

/*  if (data->use_widget_hash)
 *   {
 *     fprintf (data->hfp,
 *             "");
 *   } */
}


static void
source_write_signal_c_preamble (GbWidgetWriteSourceData * data)
{
  source_write_preamble (data, data->sigcfp);
}


static void
source_write_signal_h_preamble (GbWidgetWriteSourceData * data)
{
  source_write_preamble (data, data->sighfp);
}


static void
source_write_preamble (GbWidgetWriteSourceData * data, FILE * fp)
{
}


static void
source_write_component_create (GtkWidget * component,
                               GbWidgetWriteSourceData * data)
{
  gchar *component_name = gtk_widget_get_name (component);

  /* Don't show popup menus. */
  if (GTK_IS_MENU (component))
    return;

  component_name = source_create_valid_identifier (component_name);
  source_add_decl (data, "  GtkWidget *%s;\n", component_name);
  source_add (data, "  %s = create_%s ();\n  gtk_widget_show (%s);\n",
              component_name, component_name, component_name);
  g_free (component_name);
}


static void
source_write_component (GtkWidget * component, GbWidgetWriteSourceData * data)
{
  data->component = component;
  data->component_name = source_create_valid_identifier (gtk_widget_get_name (component));
  data->need_tooltips = FALSE;
  data->need_accel_group = FALSE;
  data->buffer_pos = 0;
  data->decl_buffer_pos = 0;
  data->parent = NULL;
  data->create_widget = TRUE;
  data->write_children = TRUE;

  gb_widget_write_source (component, data);

  fprintf (data->cfp,
           "Function create_%s:PGtkWidget;\n"
           "var\n",data->component_name);

  if (data->need_accel_group)
    {
#ifdef GLD_HAVE_GTK_1_1
      fprintf (data->cfp, "  accel_group : PGtkAccelGroup;\n");
#else
      fprintf (data->cfp, "  accelerator_table : PGtkAcceleratorTable;\n");
#endif
    }
  if (data->need_tooltips)
    fprintf (data->cfp,"  tooltips : PGtkTooltips;\n");

  fprintf (data->cfp, "%s", data->decl_buffer);

  fprintf (data->cfp, "begin\n");

  if (data->need_tooltips)
    fprintf (data->cfp,"  tooltips:=gtk_tooltips_new();\n");

  fprintf (data->cfp, "%s", data->buffer);

  fprintf (data->cfp, "  exit(%s);\n"
                      "end;\n"
                      "\n", data->component_name);
  g_free (data->component_name);
}


gchar*
type_c_to_pas (gchar *ccode, gchar *pascode)
{
  /* convert C -> pascal
   * Example: "GtkWidget *window1;" -> "window1 : PGtkWidget" */
  gchar *q,*bufp,*typep,*varp,*endp;
  gint star=0;

  bufp=pascode;
  typep=ccode;
  /* remove leading spaces */
  while (*typep==' ')
    typep++;

  varp=typep;
  while (*varp!=' ')
    varp++;

  q=varp;
  while (*q == ' ')
    q++;

  if (*q == '*')
    {
      star=1;
      q++;
    }
  while ((*q != ';') && (*q != ' ') && (*q != ')') && (*q != ','))
    {
      *bufp=*q;
      bufp++;
      q++;
    }
  endp=q;

  *bufp=' ';
  bufp++;
  *bufp=':';
  bufp++;
  *bufp=' ';
  bufp++;
  if (star==1)
    {
      *bufp='P';
      bufp++;
    }
  q=typep;
  while (q != varp)
    {
      *bufp=*q;
      bufp++;
      q++;
    }
  *bufp='\0';

  return endp;
}


void
c_to_pas (gchar *ccode, gchar *pascode)
{
  gchar *pc,*pp;

  pc=ccode;
  pp=pascode;
  while (*pc != '\0')
    {
      switch (*pc)
        {
        case '=':
          if (*(pc+1)==' ')
            {
              *pp=':';
              pp++;
            }
          *pp=*pc;
          pp++;
          pc++;
          break;
        case '-':
          if (*(pc+1)=='>')
            {
              *pp='^';
              pp++;
              *pp='.';
              pp++;
              pc+=2;
              break;
            }
          *pp=*pc;
          pp++;
          pc++;
          break;
        case '|':
          if (*(pc+1)=='|')
            pc++;
          pc++;
          *pp='o';
          pp++;
          *pp='r';
          pp++;
          break;
        case '&':
          if (*(pc+1)=='&')
            pc++;
          pc++;
          *pp='a';
          pp++;
          *pp='n';
          pp++;
          *pp='d';
          pp++;
          break;
        case 'N':
          if (strncmp(pc,"NULL",4)==0)
            {
              strncpy(pp,"nil",3);
              pp+=3;
              pc+=4;
              break;
            }
          *pp=*pc;
          pp++;
          pc++;
          break;
        case '{':
          strncpy(pp,"begin",5);
          pp+=5;
          pc++;
          break;
        case '}':
          strncpy(pp,"end;",4);
          pp+=4;
          pc++;
          break;
        case '\"':
          *pp='\'';
          pp++;
          pc++;
          break;
        default:
          *pp=*pc;
          pp++;
          pc++;
        }
    }
  *pp='\0';
}


void
signal_c_to_pas (gchar *ccode, gchar *pascode)
{
  gchar *pc,*pp;

  pc=ccode;
  pp=pascode;
  while (*pc != '\0')
    {
      switch (*pc)
        {
        case 'v':
          if (strncmp(pc,"void\n",5)==0)
            {
              strncpy(pp,"procedure ",10);
              pp+=10;
              pc+=5;
              break;
            }
          *pp=*pc;
          pp++;
          pc++;
          break;
        case ' ':
          while (*pc == ' ')
            pc++;
          *pp=' ';
          pp++;
          break;
        case '(':
          *pp='(';
          pp++;
          while (*pc != ')')
            {
              pc++; /* skip ( or , or ) */
              if (*pc=='\n')
                pc++;
              pc=type_c_to_pas(pc,pp);
              pp+=strlen(pp);
              while (*pc == ' ')
               pc++;
              if (*pc == ',')
                {
                  *pp=';';
                  pp++;
                }
            }
          pc++;
          if (*pc==';')
            pc++;
          strncpy(pp,");cdecl;",8);
          pp+=6;
          break;
        case '{':
          strncpy(pp,"begin",5);
          pp+=5;
          pc++;
          break;
        case '}':
          strncpy(pp,"end;",4);
          pp+=4;
          pc++;
          break;
        default:
          *pp=*pc;
          pp++;
          pc++;
        }
    }
  *pp='\0';
}


void
source_add (GbWidgetWriteSourceData * data, gchar * fmt,...)
{
  va_list args;
  gchar cbuffer[MAX_LINE_LEN],pbuffer[MAX_LINE_LEN];

  if (data->buffer_space - data->buffer_pos < GB_SOURCE_BUFFER_ENSURE_SPACE)
    {
      data->buffer_space += GB_SOURCE_BUFFER_INCREMENT;
      data->buffer = g_realloc (data->buffer, data->buffer_space);
    }

  va_start (args, fmt);
  vsprintf (&cbuffer[0], fmt, args);
  va_end (args);

  c_to_pas(&cbuffer[0],&pbuffer[0]);

  sprintf(data->buffer + data->buffer_pos,"%s",&pbuffer[0]);
  data->buffer_pos += strlen (data->buffer + data->buffer_pos);
}


void
source_add_decl (GbWidgetWriteSourceData * data, gchar * fmt,...)
{
  va_list args;
  gchar *bufp,buffer[MAX_LINE_LEN];

  if (data->decl_buffer_space - data->decl_buffer_pos
      < GB_SOURCE_BUFFER_ENSURE_SPACE)
    {
      data->decl_buffer_space += GB_SOURCE_BUFFER_INCREMENT;
      data->decl_buffer = g_realloc (data->decl_buffer,
                                     data->decl_buffer_space);
    }

  bufp=&buffer[0];
  *bufp=' ';
  bufp++;
  *bufp=' ';
  bufp++;
  type_c_to_pas(fmt,bufp);
  bufp+=strlen(bufp);
  *bufp=';';
  bufp++;
  *bufp='\n';
  bufp++;
  *bufp='\0';
  fmt=&buffer[0];

  va_start (args,fmt);
  vsprintf (data->decl_buffer + data->decl_buffer_pos, fmt, args);
  data->decl_buffer_pos += strlen (data->decl_buffer + data->decl_buffer_pos);
  va_end (args);
}


void
signal_source_add (GbWidgetWriteSourceData * data, gchar * fmt,...)
{
  va_list args;
  gchar cbuffer[MAX_LINE_LEN],pbuffer[MAX_LINE_LEN];

  va_start (args, fmt);
  vsprintf (&cbuffer[0], fmt, args);
  va_end (args);

  signal_c_to_pas(&cbuffer[0],&pbuffer[0]);

  fprintf(data->sigcfp,"%s",&pbuffer[0]);
}


void
signal_header_add (GbWidgetWriteSourceData * data, gchar * fmt,...)
{
  va_list args;
  gchar cbuffer[MAX_LINE_LEN],pbuffer[MAX_LINE_LEN];

  va_start (args, fmt);
  vsprintf (&cbuffer[0], fmt, args);
  va_end (args);

  signal_c_to_pas(&cbuffer[0],&pbuffer[0]);

  fprintf(data->sighfp,"%s",&pbuffer[0]);
}


static gboolean
file_exists(gchar *filename)
{
  int status;
  struct stat filestat;

  status = stat(filename, &filestat);
  if (status == -1 && errno == ENOENT)
    return FALSE;
  return TRUE;
}


static GbStatusCode
backup_file_if_exists (gchar * filename)
{
  int status;
  gchar buffer[1024];

  /* FIXME: better error handling. */
  if (file_exists (filename))
    {
      sprintf (buffer, "%s.bak", filename);
      status = rename (filename, buffer);
      if (status == 0)
        return GB_OK;
    }
  return GB_FILE_OPEN_ERROR;
}


gchar *
source_create_valid_identifier (gchar * name)
{
  gchar buffer[MAX_IDENTIFIER_LEN];
  gint i;

  if ((name[0] >= 'a' && name[0] <= 'z')
      || (name[0] >= 'A' && name[0] <= 'Z')
      || name[0] == '_')
    buffer[0] = name[0];
  else
    buffer[0] = '_';

  for (i = 1; i < strlen (name); i++)
    {
      if ((name[i] >= 'a' && name[i] <= 'z')
          || (name[i] >= 'A' && name[i] <= 'Z')
          || (name[i] >= '0' && name[i] <= '9')
          || name[i] == '_')
        buffer[i] = name[i];
      else
        buffer[i] = '_';
    }
  buffer[i] = '\0';

  return g_strdup (buffer);
}


/* This converts a string so that it can be output as part of the C source
 * code. It converts non-printable characters to escape codes.
 * Note that it uses one dynamically allocated buffer, so the result is only
 * valid until the next call to the function.
 */
gchar *
source_make_string (gchar * text)
{
  static gchar *buffer = NULL;
  static gint buffer_space = 0;
  static gint buffer_pos;

  gchar escape_buffer[16];
  gchar *p;
  int quote;

  g_return_val_if_fail (text != NULL, "");

  buffer_pos = 0;
  quote = 0;
  for (p = text; *p; p++)
    {
      if (isprint (*p) && (*p != '\''))
        add_char_to_buffer (&buffer, &buffer_pos, &buffer_space, *p);
      else
        {
          sprintf (escape_buffer, "'#%d'", (guchar) *p);
          add_to_buffer (&buffer, &buffer_pos, &buffer_space,escape_buffer);
        }
    }
  add_char_to_buffer (&buffer, &buffer_pos, &buffer_space, '\0');

  return buffer;
}


static void
add_char_to_buffer (gchar ** buffer, gint * buffer_pos, gint * buffer_space,
                    gchar ch)
{
  if (*buffer_pos == *buffer_space)
    {
      *buffer_space += TEXT_BUFFER_INCREMENT;
      *buffer = g_realloc (*buffer, *buffer_space);
    }
  *(*buffer + *buffer_pos) = ch;
  *buffer_pos = *buffer_pos + 1;
}


static void
add_to_buffer (gchar ** buffer, gint * buffer_pos, gint * buffer_space,
               gchar * text)
{
  gchar *pos;

  for (pos = text; *pos; pos++)
    add_char_to_buffer (buffer, buffer_pos, buffer_space, *pos);
}
