/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "project.h"
#include "property.h"
#include "gbwidget.h"
#include "gbwidgetarray.h"
#include "save.h"
#include "editor.h"
#include "load.h"
#include "source.h"
#include "utils.h"
#include "tree.h"

/*
 * The hash table associates a Gtk widget class name with a GbWidget struct,
 * which contains a table of functions used by the builder, e.g. creating a
 * new widget, saving the widget etc.
 */
GHashTable *gb_widget_table = NULL;

/* The id hash stores the last id used for each widget class. */
GHashTable *gb_widget_id_hash = NULL;

/* The style data hash stores named GbStyles with the name as the key.
   Note that the GbStyle name field is also used as the key so be careful when
   freeing or updating it. */
GHashTable *gb_style_hash = NULL;

/* The default style of created widgets. We need to use a separate default
   style for the widgets created so that the Glade UI doesn't change. */
GbStyle *gb_widget_default_gb_style;

/* These are used to return what is in a table cell */
#define GB_CELL_EMPTY           1
#define GB_CELL_WIDGET          2
#define GB_CELL_PLACEHOLDER     3

/* Tooltips for the created widgets */
static GtkTooltips *gb_widget_tooltips;

gchar *GbStateNames[] =
{"NORMAL", "ACTIVE", "PRELIGHT",
 "SELECTED", "INSENSITIVE"};
gchar *GbColorNames[] =
{"fg", "bg", "text", "base"};
/* FIXME: Delete?
   gchar*  GbColorNames[] =     { "fg", "bg", "light", "dark",
   "middle", "text", "base" };
 */
gchar *GbBgPixmapName = "bg_pixmap";


/* These are used for outputting signal handler prototypes. */
#define GB_PARAM_INDENT         40
#define GB_PARAM_TYPE_WIDTH     16

static void on_widget_destroy (GtkWidget * widget,
                               gpointer data);
static void gb_widget_free_accelerators (GtkWidget * widget);
static void gb_widget_free_signals (GtkWidget * widget);

static void show_color_properties (GdkColor colors[],
                                   gchar * name);
static void show_accelerators (GtkWidget * widget,
                               GbWidgetGetArgData * data);
static void show_signals (GtkWidget * widget,
                          GbWidgetGetArgData * data);

static void set_standard_properties (GtkWidget * widget,
                                     GbWidgetSetArgData * data);
static void set_position_properties (GtkWidget * widget,
                                     GbWidgetSetArgData * data);
static void set_special_child_properties (GtkWidget * widget,
                                          GbWidgetSetArgData * data);
static void apply_style (GtkWidget * widget,
                         GbWidgetSetArgData * data);
static gboolean apply_colors (GtkWidget * widget,
                              GbWidgetSetArgData * data,
                              GdkColor colors[],
                              GdkColor new_colors[],
                              gchar * name);
static void apply_accelerators (GtkWidget * widget,
                                GbWidgetSetArgData * data);
static void apply_signals (GtkWidget * widget,
                           GbWidgetSetArgData * data);

static void add_standard_top_menu_items (GtkWidget * widget,
                                         GbWidgetCreateMenuData * data);
static void add_standard_bottom_menu_items (GtkWidget * widget,
                                            GbWidgetCreateMenuData * data);

static void gb_widget_load_properties (GbWidgetSetArgData * data);
static void load_special_child_properties (GbWidgetSetArgData * data);
static void load_accelerator (GbWidgetSetArgData * data);
static void load_signal (GbWidgetSetArgData * data);
static gboolean load_colors (GbWidgetSetArgData * data,
                             GdkColor colors[],
                             gchar * name,
                             gchar * element,
                             gchar * cdata);

static void get_standard_properties (GtkWidget * widget,
                                     GbWidgetGetArgData * data);

static void get_position_properties (GtkWidget * widget,
                                     GbWidgetGetArgData * data);
static void get_special_child_properties (GtkWidget * widget,
                                          GbWidgetGetArgData * data);

static void save_style (GtkWidget * widget,
                        GbWidgetGetArgData * data);
static void save_colors (GbWidgetGetArgData * data,
                         gboolean save_all,
                         GdkColor colors[],
                         GdkColor default_colors[],
                         gchar * name);
static void save_accelerators (GtkWidget * widget,
                               GbWidgetGetArgData * data);
static gchar* create_modifiers_string (guint8 modifiers);
static void save_signals (GtkWidget * widget,
                          GbWidgetGetArgData * data);


static void gb_widget_write_signals_source (GtkWidget * widget,
                                            GbWidgetWriteSourceData * data);
static gchar *get_type_name (GtkType type,
                             gboolean * is_pointer);
static gchar *get_gdk_event (gchar * signal_name);
static gchar **lookup_signal_arg_names (gchar * type,
                                        gchar * signal_name);
static void gb_widget_write_accelerators_source (GtkWidget * widget,
                                            GbWidgetWriteSourceData * data);

static void gb_widget_add_alignment (GtkWidget * menuitem,
                                     GtkWidget * widget);
static void gb_widget_remove_alignment (GtkWidget * menuitem,
                                        GtkWidget * widget);
static void gb_widget_add_event_box (GtkWidget * menuitem,
                                     GtkWidget * widget);
static void gb_widget_remove_event_box (GtkWidget * menuitem,
                                        GtkWidget * widget);


static void split_table_placeholder (GtkWidget * table,
                                     GtkWidget * placeholder,
                                     gint left,
                                     gint right,
                                     gint top,
                                     gint bottom,
                                     gint skip_row,
                                     gint skip_col);
static gint is_cell_occupied (GtkWidget * table,
                              gint row,
                              gint col);
static void remove_table_placeholders (GtkWidget * table,
                                       gint row,
                                       gint col);


static void table_foreach (GtkTable * table,
                           GtkCallback callback,
                           gpointer callback_data);
static void notebook_tabs_foreach (GtkNotebook *notebook,
                                   GtkCallback callback,
                                   gpointer callback_data);

static GtkNotebookPage *find_notebook_page (GtkWidget * widget,
                                            GtkWidget * current_child,
                                            gint * pos);
static void update_box_size_callback (GtkWidget * widget,
                                      gint data[2]);

static void update_ids (gchar * name);
void gb_widget_add_callback (GtkWidget * widget,
                             gpointer dummy);

struct GbUpdateStyleData
  {
    GbStyle *old_gbstyle;
    GbStyle *new_gbstyle;
  };

static void update_style (GtkWidget * component,
                          struct GbUpdateStyleData *data);


/*************************************************************************
 * Initialization functions
 *************************************************************************/

void
gb_widgets_init ()
{
  gint index;
  GbWidget *(*init_func) ();
  GbWidget *gbwidget;

  gb_widget_table = g_hash_table_new (g_str_hash, g_str_equal);
  gb_widget_id_hash = g_hash_table_new (g_str_hash, g_str_equal);

  gb_widget_reset_gb_styles ();

  /* Create tooltips */
  gb_widget_tooltips = gtk_tooltips_new ();

  /* Add all known gbwidgets to hash.
     Skip the first row, the selector, which isn't really a gbwidget */
  index = 2;
  for (;;)
    {
      if (widgets[index] == NULL)
        {
          if (widgets[index + 1] == NULL)
            break;
        }
      else
        {
          /* FIXME: Not ANSI compliant. */
          init_func = widgets[index + 1];
          gbwidget = (*init_func) ();
          g_hash_table_insert (gb_widget_table, widgets[index], gbwidget);
        }
      index += 2;
    }
}


void
gb_widget_init_struct (GbWidget * gbwidget)
{
  gbwidget->properties_page_number = GB_PROPERTIES_NOT_CREATED;
  gbwidget->pixmap_struct = NULL;
  gbwidget->tooltip = NULL;

  gbwidget->gb_widget_new = NULL;
  gbwidget->gb_widget_create_properties = NULL;
  gbwidget->gb_widget_get_properties = NULL;
  gbwidget->gb_widget_set_properties = NULL;
  gbwidget->gb_widget_create_popup_menu = NULL;
  gbwidget->gb_widget_write_source = NULL;
  gbwidget->gb_widget_get_child = NULL;
}


/*************************************************************************
 * Functions for creating & destroying GbWidgets
 *************************************************************************/

static void
gb_widget_add_callback (GtkWidget * widget, gpointer dummy)
{
  tree_add_widget (widget);
  /* for dialogs, etc.
     FIXME: probably want to move these here after they've been added
     with gb_widget_create_from(...)
     not before gtktree gets fixed
  */
  if (GTK_IS_CONTAINER (widget))
    {
      gtk_container_foreach (GTK_CONTAINER (widget),
                             (GtkCallback) gb_widget_add_callback, NULL);
    }
}


GtkWidget *
gb_widget_new (gchar * class_name, GtkWidget * parent)
{
  return gb_widget_new_full (class_name, parent, NULL, 0, 0,
                             (GbWidgetNewCallback) gb_widget_add_callback,
                             GB_CREATING, NULL);
}


GtkWidget *
gb_widget_new_full (gchar * class_name, GtkWidget * parent,
                    GtkWidget * current_child, gint x, gint y,
                    GbWidgetNewCallback callback, GbWidgetAction action,
                    GbWidgetSetArgData * loading_data)
{
  GbWidgetNewData *data;
  GtkWidget *new_widget;
  GbWidget *gbwidget;
  gint type;

  gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
  g_return_val_if_fail (gbwidget != NULL, NULL);

  type = gtk_type_from_name (class_name);
  g_return_val_if_fail (type != 0, NULL);

  data = g_new (GbWidgetNewData, 1);
  data->name = gb_widget_new_name (class_name);
  data->type = type;
  data->callback = callback;
  data->parent = parent;
  if (parent)
    gtk_widget_ref (parent);
  data->current_child = current_child;
  if (current_child)
    gtk_widget_ref (current_child);
  data->x = x;
  data->y = y;
  data->widget_data = gb_widget_new_widget_data ();
  data->action = action;
  data->loading_data = loading_data;

  if (gbwidget->gb_widget_new)
    new_widget = (gbwidget->gb_widget_new) (data);
  else
    new_widget = gtk_type_new (data->type);

  /* If the widget has been created immediately, then we can finish it off now,
     and free the GbWidgetNewData struct, otherwise we leave that to the
     dialog. */
  if (new_widget)
    {
      gb_widget_initialize (new_widget, data);
      if (data->callback)
        (*data->callback) (new_widget, data);
      gb_widget_free_new_data (data);
    }

  return new_widget;
}


/* This turns a normal widget into a GbWidget, adding a GbWidgetData struct
   and the necessary signals. The widget must have its parent set already. The
   name will have a unique ID added on to it, e.g. "ok_button1". */
void
gb_widget_create_from (GtkWidget * widget,
                       gchar * name)
{
  GbWidgetData *wdata;

  MSG ("In create_from");
  gtk_widget_set_name (widget, gb_widget_new_name (name));
  wdata = gb_widget_new_widget_data ();
  gb_widget_real_initialize (widget, wdata);
}


/* This returns the last ID used for the given name. */
gint
gb_widget_get_last_id (gchar *name)
{
  return GPOINTER_TO_INT (g_hash_table_lookup (gb_widget_id_hash, name));
}

/* This sets the last ID used for the given name. It checks if the key
   already exists, in which case we don't have to g_dtrdup it. */
void
gb_widget_set_last_id (gchar *name, gint last_id)
{
  if (g_hash_table_lookup (gb_widget_id_hash, name))
    g_hash_table_insert (gb_widget_id_hash, name,
                         GINT_TO_POINTER (last_id));
  else
    g_hash_table_insert (gb_widget_id_hash, g_strdup (name),
                         GINT_TO_POINTER (last_id));
}

gchar *
gb_widget_new_name (gchar * class_name)
{
  char new_widget_name[128];
  gint widget_id, i;

  /* Check we won't overflow the buffer. */
  g_return_val_if_fail (strlen (class_name) < 100, class_name);

  /* Skip 'Gtk' at the start of all class names */
  if (!strncmp (class_name, "Gtk", 3))
    class_name += 3;
  strcpy (new_widget_name, class_name);
  /* convert name to lower case (only normal ASCII chars) */
  for (i = 0; i < strlen (new_widget_name); i++)
    {
      if ((new_widget_name[i] >= 'A') && (new_widget_name[i] <= 'Z'))
        new_widget_name[i] += 'a' - 'A';
    }

  widget_id = GPOINTER_TO_INT (g_hash_table_lookup (gb_widget_id_hash,
                                                    new_widget_name));
  if (widget_id == 0)
    {
      widget_id = 1;
      g_hash_table_insert (gb_widget_id_hash, g_strdup (new_widget_name),
                           GINT_TO_POINTER (widget_id));
    }
  else
    {
      widget_id++;
      /* We don't need to g_strdup new_widget_name since it is already in the
         hash. */
      g_hash_table_insert (gb_widget_id_hash, new_widget_name,
                           GINT_TO_POINTER (widget_id));
    }

  /* Add the ID onto the end of the name. */
  sprintf (new_widget_name + strlen (new_widget_name), "%i", widget_id);

  return g_strdup (new_widget_name);
}


GbWidgetData *
gb_widget_new_widget_data ()
{
  GbWidgetData *widget_data = g_new (GbWidgetData, 1);
  widget_data->flags = GB_VISIBLE | GB_SENSITIVE | GB_STYLE_IS_UNNAMED
    | GB_STYLE_PROPAGATE | GB_INITIAL_EXPOSE;
  widget_data->x = -1;
  widget_data->y = -1;
  widget_data->width = 0;
  widget_data->height = 0;
  widget_data->events = 0;
  widget_data->tooltip = NULL;
  widget_data->signals = NULL;
  widget_data->accelerators = NULL;
  widget_data->gbstyle = gb_widget_default_gb_style;
  gb_widget_ref_gb_style (gb_widget_default_gb_style);
  return widget_data;
}


void
gb_widget_initialize (GtkWidget * widget, GbWidgetNewData * data)
{
  gtk_widget_set_name (widget, data->name);
  gb_widget_real_initialize (widget, data->widget_data);

  /* Now we set the widget's real style */
  /* FIXME: check if style should be propagated down from an ancestor? */
  if (widget->style != data->widget_data->gbstyle->style)
    {
      gtk_widget_set_style (widget, data->widget_data->gbstyle->style);
    }

  /* FIXME: GTK workarounds to make sure that some widgets have reasonable
     sizes initially. Quite a few widgets default to a width and height of 0,
     which means that if there is little space available they will disappear,
     so we may need to do more here. */
  if (GTK_IS_ARROW (widget))
    gtk_widget_set_usize (widget, 16, 16);

  /* Set this to NULL so we don't try to free it later. */
  data->widget_data = NULL;
}



void
gb_widget_real_initialize (GtkWidget * widget, GbWidgetData * wdata)
{
  gtk_object_set_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY, wdata);
  gtk_signal_connect (GTK_OBJECT (widget), "destroy",
                      GTK_SIGNAL_FUNC (on_widget_destroy), &widget);
  /* We don't add signals to menu items. */
  if (!GTK_IS_MENU_ITEM (widget))
    {
      editor_add_mouse_signals (widget);
      editor_add_draw_signals (widget);
      if (GTK_IS_WINDOW (widget))
        editor_add_key_signals (widget);
    }
  if (!GTK_IS_WINDOW (widget) && !GTK_IS_MENU (widget))
    gtk_widget_show (widget);
}


/* This returns TRUE if it is OK to complete the new() procedure, i.e. that
   the widget to replace or the parent widget still exist. It is used after
   the OK button is pressed in the dialog boxes for creating new tables/boxes.
   FIXME: I'm not too sure what we should do here. */
gboolean
gb_widget_can_finish_new (GbWidgetNewData * data)
{
  if (data->current_child)
    {
      if (!GTK_OBJECT_DESTROYED (data->current_child)
          && data->current_child->parent)
        return TRUE;
    }
  else if (data->parent)
    {
      if (!GTK_OBJECT_DESTROYED (data->parent)
          && (GTK_IS_WINDOW (data->parent) || data->parent->parent))
        return TRUE;
    }
  return FALSE;
}


void
gb_widget_free_new_data (GbWidgetNewData * data)
{
  g_free (data->name);
  g_free (data->widget_data);
  if (data->parent)
    gtk_widget_unref (data->parent);
  if (data->current_child)
    gtk_widget_unref (data->current_child);
  g_free (data);
}


static void
on_widget_destroy (GtkWidget * widget, gpointer data)
{
  GbWidgetData *widget_data;

  MSG1 ("IN on_widget_destroy widget:%s", gtk_widget_get_name (widget));
  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_free (widget_data->tooltip);
  gb_widget_free_accelerators (widget);
  gb_widget_free_signals (widget);
  g_free (widget_data);
  g_free (gtk_object_get_data (GTK_OBJECT (widget), GB_CHILD_NAME_KEY));
  MSG1 ("OUT on_widget_destroy widget:%s", gtk_widget_get_name (widget));
}


static void
gb_widget_free_accelerators (GtkWidget * widget)
{
  GbWidgetData *widget_data;
  GList *item;
  GbAccelerator *accelerator;

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  if (widget_data->accelerators)
    {
      item = widget_data->accelerators;
      while (item)
        {
          accelerator = (GbAccelerator *) item->data;
          g_free (accelerator->key);
          g_free (accelerator->signal);
          item = item->next;
        }
      g_list_free (widget_data->accelerators);
      widget_data->accelerators = NULL;
    }
}


static void
gb_widget_free_signals (GtkWidget * widget)
{
  GbWidgetData *widget_data;
  GList *item;
  GbSignal *signal;

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  if (widget_data->signals)
    {
      item = widget_data->signals;
      while (item)
        {
          signal = (GbSignal *) item->data;
          g_free (signal->name);
          g_free (signal->handler);
          g_free (signal->object);
          g_free (signal->data);
          item = item->next;
        }
      g_list_free (widget_data->signals);
      widget_data->signals = NULL;
    }
}


/* Re-initialise the widget ID hash. */
static void
free_id (gchar * key, gchar * value, gpointer data)
{
  g_free (key);
}


void
gb_widget_reset_ids ()
{
  g_hash_table_foreach (gb_widget_id_hash, (GHFunc) free_id, NULL);
  g_hash_table_destroy (gb_widget_id_hash);
  gb_widget_id_hash = g_hash_table_new (g_str_hash, g_str_equal);
}


/* Check if the name has a trailing ID number, and if so compare it to the
   current maximum in the ID hash and update if necessary. */
static void
update_ids (gchar * name)
{
  gchar buffer[128];
  gchar *id_start = name + strlen (name) - 1;
  gint id = 0, current_id;

  while (*id_start >= '0' && *id_start <= '9')
    id_start--;
  id_start++;

  /* If there is no trailing number or the name is too long just return. */
  if (*id_start == '\0' || id_start - name >= 128)
    return;

  id = atoi (id_start);
  if (id == 0)
    return;

  strncpy (buffer, name, id_start - name);
  buffer[id_start - name] = '\0';

  MSG2 ("Name: %s ID: %i", buffer, id);
  current_id = GPOINTER_TO_INT (g_hash_table_lookup (gb_widget_id_hash, buffer));
  if (current_id == 0)
    {
      MSG ("Adding to hash");
      g_hash_table_insert (gb_widget_id_hash, g_strdup (buffer),
                           GINT_TO_POINTER (id));
    }
  else if (id > current_id)
    {
      MSG ("Updating hash");
      /* We don't need to g_strdup buffer since it is already in the hash. */
      g_hash_table_insert (gb_widget_id_hash, buffer, GINT_TO_POINTER (id));
    }
}



/*************************************************************************
 * Functions for creating the page of properties specific to this widget
 *************************************************************************/

/* Returns the page number of the new page in the notebook which contains the
   widget's specific properties, or GB_PROPERTIES_NOT_NEEDED if it has none. */
gint
gb_widget_create_properties (GtkWidget * widget)
{
  GtkWidget *page;
  gint page_number;
  GbWidgetCreateArgData data;
  GbWidget *gbwidget;
  gchar *class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));
  gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);

  g_return_val_if_fail (gbwidget != NULL, GB_PROPERTIES_NOT_NEEDED);
  if (gbwidget->gb_widget_create_properties || GTK_IS_CONTAINER (widget))
    {
      /* Create skeleton of properties page, so gbwidget just has to add
         properties */
      page = gtk_table_new (10, 3, FALSE);
      gtk_widget_show (page);
      page_number = property_add_gbwidget_page (page);
      property_set_table_position (page, 0);

      /* If widget is a container add a border width property */
      if (GTK_IS_CONTAINER (widget))
        {
          gchar buf[128];
          sprintf (buf, "%s::border_width", class_name);
          property_add_int_range (buf, _("Border Width:"),
                             _("The width of the border around the container"),
                                  0, 1000, 1, 10, 1);
        }

      if (gbwidget->gb_widget_create_properties)
        (gbwidget->gb_widget_create_properties) (widget, &data);
      return page_number;
    }
  else
    {
      return GB_PROPERTIES_NOT_NEEDED;
    }
}


/*************************************************************************
 * Functions for showing the widget's properties
 *************************************************************************/

void
gb_widget_show_properties (GtkWidget * widget)
{
  GbWidgetGetArgData data;
  gchar *class_name;
  GbWidget *gbwidget;
  GbWidgetData *widget_data;
  gint page;

  /* If properties of widget are already shown, just return */
  if (property_get_widget () == widget)
    return;

  /* If widget is a placeholder reset the properties window and return */
  if (GB_IS_PLACEHOLDER (widget))
    {
      property_set_widget (NULL);
      return;
    }

  class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));
  gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
  g_return_if_fail (gbwidget != NULL);

  /* Turn off auto-apply so we can set properties without the 'changed'
     callbacks calling gb_widget_apply_properties (). */
  property_set_auto_apply (FALSE);

  /* Need this here to make sure properties notebook is sensitive */
  property_set_widget (widget);

  page = gbwidget->properties_page_number;
  /* If widget's properties page hasn't been created, create it now */
  if (page == GB_PROPERTIES_NOT_CREATED)
    {
      page = gb_widget_create_properties (widget);
      gbwidget->properties_page_number = page;
    }

  /* Show the widget's own properties page if it has one.
     Need to show the page before setting properties because of the
     Text widget - it must be realized before setting the text :-( */
  if (page == GB_PROPERTIES_NOT_NEEDED)
    property_hide_gbwidget_page ();
  else
    property_show_gbwidget_page (page);

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (widget_data != NULL);
  data.action = GB_SHOWING;
  data.widget_data = widget_data;
  get_standard_properties (widget, &data);

  if (gbwidget->gb_widget_get_properties)
    (gbwidget->gb_widget_get_properties) (widget, &data);

  /* Turn auto-apply back on again */
  property_set_auto_apply (TRUE);
}


/* This is called when the widget's size or position has changed, so that we
   should update the values shown in the properties editor. */
void
gb_widget_show_position_properties (GtkWidget * widget)
{
  GbWidgetGetArgData data;
  GbWidgetData *widget_data;

  /* Make sure this is the widget shown in the properties editor. */
  if (property_get_widget () != widget)
    return;

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (widget_data != NULL);
  data.action = GB_SHOWING;
  data.widget_data = widget_data;

  property_set_auto_apply (FALSE);
  get_position_properties (widget, &data);
  property_set_auto_apply (TRUE);
}



void
gb_widget_show_style (GtkWidget * widget)
{
  GbWidgetData *wdata;
  GbStyle *gbstyle;
  GtkStyle *style = widget->style;
  gchar buffer[128];
  gint i;

  wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (wdata != NULL);
  gbstyle = wdata->gbstyle;

  property_set_bool (GbStylePropagate, wdata->flags & GB_STYLE_PROPAGATE);

  property_set_dialog (GbStyleName, wdata->flags & GB_STYLE_IS_UNNAMED ?
                       "" : gbstyle->name, NULL);
  property_set_font (GbStyleFont, style->font, gbstyle->xlfd_fontname);

  /* Colors */
  show_color_properties (style->fg, "fg");
  show_color_properties (style->bg, "bg");
  show_color_properties (style->text, "text");
  show_color_properties (style->base, "base");
  /* FIXME: Delete?
     show_color_properties (style->light, "light");
     show_color_properties (style->dark,  "dark");
     show_color_properties (style->mid,   "middle");
   */

  /* Background pixmaps */
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      sprintf (buffer, "GtkStyle::%s[%s]", GbBgPixmapName, GbStateNames[i]);
      property_set_bgpixmap (buffer, style->bg_pixmap[i],
                             gbstyle->bg_pixmap_filenames[i]);
    }
}


static void
show_color_properties (GdkColor colors[], gchar * name)
{
  GdkColor gdkcolor;
  guchar color[4];
  gint state;
  gchar buf[128];

  for (state = 0; state < GB_NUM_STYLE_STATES; state++)
    {
      sprintf (buf, "GtkStyle::%s[%s]", name, GbStateNames[state]);
      gdkcolor = colors[state];
      color[0] = gdkcolor.red >> 8;
      color[1] = gdkcolor.green >> 8;
      color[2] = gdkcolor.blue >> 8;
      property_set_color (buf, color);
    }
}


static void
show_accelerators (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GList *element = data->widget_data->accelerators;
  property_clear_accelerators ();
  while (element)
    {
      property_add_accelerator ((GbAccelerator *) element->data);
      element = element->next;
    }
}


static void
show_signals (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GList *element = data->widget_data->signals;
  property_clear_signals ();
  MSG1 ("Num signals: %i", g_list_length (element));
  while (element)
    {
      property_add_signal ((GbSignal *) element->data);
      element = element->next;
    }
}



/*************************************************************************
 * Functions for applying properties to a widget
 *************************************************************************/

void
gb_widget_apply_properties (GtkWidget * widget, GtkWidget * property)
{
  GbWidgetSetArgData data;
  GbWidget *gbwidget;
  GbWidgetData *widget_data;
  gchar *class_name;

  MSG1 ("Applying properties: %s", gtk_widget_get_name (widget));

  class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));
  gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
  g_return_if_fail (gbwidget != NULL);

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  g_return_if_fail (widget_data != NULL);
  data.action = GB_APPLYING;
  data.widget_data = widget_data;
  data.property_to_apply = property;
  set_standard_properties (widget, &data);

  MSG ("Calling widget's own apply_properties");
  if (gbwidget->gb_widget_set_properties)
    (gbwidget->gb_widget_set_properties) (widget, &data);
  MSG ("Called widget's own apply_properties");
}


static void
set_standard_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gchar *name, *tooltip, *events_str, *ext_events;
  gboolean visible, sensitive, can_default, has_default, can_focus, has_focus;
  gint events, i;

  /* Properties on widget page */
  name = gb_widget_input_string (data, GbName);
  if (data->apply)
    {
      tree_rename_widget (widget, name);
      gtk_widget_set_name (widget, name);
      /* If widget is a toplevel window/dialog set the component's name in the
         project window */
      if (widget->parent == NULL)
        project_rename_component (widget, name);

      /* If we are loading, we have to check if the name as a trailing ID and
         if so update the id hash. */
      if (data->action == GB_LOADING)
        update_ids (name);
    }

  /* If widget is a container, show the border width */
  if (GTK_IS_CONTAINER (widget))
    {
      gchar buf[128];
      gint border_width;
      gchar *class_name;

      class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));
      sprintf (buf, "%s::border_width", class_name);
      border_width = gb_widget_input_int (data, buf);
      if (data->apply && GTK_CONTAINER (widget)->border_width != border_width)
        {
          gtk_container_border_width (GTK_CONTAINER (widget), border_width);
          /*if (data->action == GB_APPLYING) */
          editor_refresh_widget (widget);
        }
    }

  /* Special child properties page */
  set_special_child_properties (widget, data);

  /* Properties on standard page */
  set_position_properties (widget, data);

  visible = gb_widget_input_bool (data, GbVisible);
  if (data->apply)
    {
      if (visible)
        wdata->flags |= GB_VISIBLE;
      else
        wdata->flags &= ~GB_VISIBLE;
    }

  sensitive = gb_widget_input_bool (data, GbSensitive);
  if (data->apply)
    {
      if (sensitive)
        wdata->flags |= GB_SENSITIVE;
      else
        wdata->flags &= ~GB_SENSITIVE;
    }

  tooltip = gb_widget_input_string (data, GbTooltip);
  if (data->apply)
    {
      g_free (wdata->tooltip);
      if (tooltip && strlen (tooltip) > 0)
        {
          gtk_tooltips_set_tip (gb_widget_tooltips, widget, tooltip, NULL);
          wdata->tooltip = g_strdup (tooltip);
        }
      else
        {
          gtk_tooltips_set_tip (gb_widget_tooltips, widget, NULL, NULL);
          wdata->tooltip = NULL;
        }
    }
  can_default = gb_widget_input_bool (data, GbCanDefault);
  if (data->apply)
    {
      if (can_default)
        GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_DEFAULT);
      else
        GTK_WIDGET_UNSET_FLAGS (widget, GTK_CAN_DEFAULT);
      if (data->action == GB_APPLYING)
        editor_refresh_widget (widget);
    }
  has_default = gb_widget_input_bool (data, GbHasDefault);
  if (data->apply)
    {
      if (has_default)
        wdata->flags |= GB_GRAB_DEFAULT;
      else
        wdata->flags &= ~GB_GRAB_DEFAULT;
    }
  can_focus = gb_widget_input_bool (data, GbCanFocus);
  if (data->apply)
    {
      if (can_focus)
        GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_FOCUS);
      else
        GTK_WIDGET_UNSET_FLAGS (widget, GTK_CAN_FOCUS);
    }
  has_focus = gb_widget_input_bool (data, GbHasFocus);
  if (data->apply)
    {
      if (has_focus)
        wdata->flags |= GB_GRAB_FOCUS;
      else
        wdata->flags &= ~GB_GRAB_FOCUS;
    }

  /* Events & ext events. */
  if (!GTK_WIDGET_NO_WINDOW (widget))
    {
      if (data->action == GB_APPLYING)
        {
          events = property_events_string_to_value (gb_widget_input_string (data,
                                                                 GbEvents));
          if (data->apply)
            wdata->events = events;
        }
      else
        {
          events_str = gb_widget_input_string (data, GbEvents);
          if (data->apply)
            {
              for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
                {
                  if (gb_strstr (events_str, GbEventMaskSymbols[i]))
                    wdata->events |= GbEventMaskValues[i];
                }
            }
        }

      ext_events = gb_widget_input_choice (data, GbExtEvents);
      if (data->apply)
        {
          for (i = 0; GbExtensionModeChoices[i]; i++)
            {
              if (!strcmp (ext_events, GbExtensionModeChoices[i])
                  || !strcmp (ext_events, GbExtensionModeSymbols[i]))
                {
                  gtk_widget_set_extension_events (widget, GbExtensionModeValues
                                                   [i]);
                  break;
                }
            }
        }
    }

  if (data->action == GB_APPLYING)
    {
      apply_style (widget, data);
      apply_accelerators (widget, data);
      apply_signals (widget, data);
    }
  else
    {

      data->widget_data->signals = data->signals;
      data->widget_data->accelerators = data->accelerators;
    }
}


static void
set_special_child_properties (GtkWidget * widget,
                              GbWidgetSetArgData * data)
{
  GtkWidget *parent = widget->parent;

  if (!parent)
    return;

  /* Tell the load functions to use the child properties array. */
  data->loading_type = GB_CHILD_PROPERTIES;

  if (GTK_IS_TABLE (parent))
    {
      GtkTableChild *tchild = gb_table_find_child (GTK_TABLE (parent), widget);
      gint xpad, ypad, left_attach, right_attach, top_attach, bottom_attach;
      gint old_left_attach, old_right_attach, old_top_attach, old_bottom_attach;
      gint xexpand, yexpand, xshrink, yshrink, xfill, yfill;
      gboolean myApply;

      g_return_if_fail (tchild != NULL);
      old_left_attach = tchild->left_attach;
      old_right_attach = tchild->right_attach;
      old_top_attach = tchild->top_attach;
      old_bottom_attach = tchild->bottom_attach;

      xpad = gb_widget_input_int (data, GbXPad);
      myApply = data->apply;
      ypad = gb_widget_input_int (data, GbYPad);
      myApply |= data->apply;
      xexpand = (gb_widget_input_bool (data, GbXExpand)) ? GTK_EXPAND : 0;
      myApply |= data->apply;
      yexpand = (gb_widget_input_bool (data, GbYExpand)) ? GTK_EXPAND : 0;
      myApply |= data->apply;
      xshrink = (gb_widget_input_bool (data, GbXShrink)) ? GTK_SHRINK : 0;
      myApply |= data->apply;
      yshrink = (gb_widget_input_bool (data, GbYShrink)) ? GTK_SHRINK : 0;
      myApply |= data->apply;
      xfill = (gb_widget_input_bool (data, GbXFill)) ? GTK_FILL : 0;
      myApply |= data->apply;
      yfill = (gb_widget_input_bool (data, GbYFill)) ? GTK_FILL : 0;
      myApply |= data->apply;

      if (data->action == GB_APPLYING)
        {
          left_attach = gb_widget_input_int (data, GbCellX);
          myApply |= data->apply;
          top_attach = gb_widget_input_int (data, GbCellY);
          myApply |= data->apply;
          right_attach = gb_widget_input_int (data, GbColSpan) + left_attach;
          myApply |= data->apply;
          bottom_attach = gb_widget_input_int (data, GbRowSpan) + top_attach;
          myApply |= data->apply;
        }
      else
        {
          left_attach = gb_widget_input_int (data, "GtkWidget::left_attach");
          myApply |= data->apply;
          top_attach = gb_widget_input_int (data, "GtkWidget::top_attach");
          myApply |= data->apply;
          right_attach = gb_widget_input_int (data, "GtkWidget::right_attach");
          myApply |= data->apply;
          bottom_attach = gb_widget_input_int (data, "GtkWidget::bottom_attach");
          myApply |= data->apply;
          if (right_attach <= left_attach)
            right_attach = left_attach + 1;
          if (bottom_attach <= top_attach)
            bottom_attach = top_attach + 1;
        }

      if (myApply)
        {
          if (xpad != tchild->xpadding || ypad != tchild->ypadding
           || (xexpand && !tchild->xexpand) || (!xexpand && tchild->xexpand)
           || (yexpand && !tchild->yexpand) || (!yexpand && tchild->yexpand)
           || (xshrink && !tchild->xshrink) || (!xshrink && tchild->xshrink)
           || (yshrink && !tchild->yshrink) || (!yshrink && tchild->yshrink)
              || (xfill && !tchild->xfill) || (!xfill && tchild->xfill)
              || (yfill && !tchild->yfill) || (!yfill && tchild->yfill)
              || left_attach != tchild->left_attach
              || right_attach != tchild->right_attach
              || top_attach != tchild->top_attach
              || bottom_attach != tchild->bottom_attach)
            {

              gtk_container_block_resize (GTK_CONTAINER (parent));
              gtk_widget_ref (widget);
              gtk_container_remove (GTK_CONTAINER (parent), widget);
              gtk_table_attach (GTK_TABLE (parent), widget,
                       left_attach, right_attach, top_attach, bottom_attach,
                       xexpand | xshrink | xfill, yexpand | yshrink | yfill,
                                xpad, ypad);
              gtk_widget_unref (widget);
              gb_widget_update_table_placeholders (parent, -1, -1);
              gtk_container_unblock_resize (GTK_CONTAINER (parent));
            }
        }
    }
  else if (GTK_IS_BOX (parent) && !GTK_IS_BUTTON_BOX (parent))
    {
      gint padding;
      gboolean expand, fill, pack, myApply;

      padding = gb_widget_input_int (data, GbPadding);
      myApply = data->apply;
      expand = gb_widget_input_bool (data, GbExpand);
      myApply |= data->apply;
      fill = gb_widget_input_bool (data, GbFill);
      myApply |= data->apply;

      if (data->action == GB_APPLYING)
        {
          pack = gb_widget_input_bool (data, GbPack);
        }
      else
        {
          gchar *pack_symbol = gb_widget_input_string (data, GbPack);
          pack = (!data->apply || (!strcmp (pack_symbol, "GTK_PACK_START")));
        }
      myApply |= data->apply;
      if (myApply)
        gtk_box_set_child_packing (GTK_BOX (parent), widget, expand, fill, padding,
                                   pack ? GTK_PACK_START : GTK_PACK_END);
    }
#ifdef GLD_HAVE_GTK_1_1
  else if (GTK_IS_PACKER (parent))
    {
      gchar *side_str, *anchor_str;
      GtkSideType side = GTK_SIDE_TOP;
      GtkAnchorType anchor = GTK_ANCHOR_CENTER;
      gboolean expand, fillx, filly, myApply;
      gint border, padx, pady, ipadx, ipady, i;

      side_str = gb_widget_input_choice (data, GbPackerSide);
      myApply = data->apply;
      anchor_str = gb_widget_input_choice (data, GbPackerAnchor);
      myApply |= data->apply;
      expand = (gb_widget_input_bool (data, GbPackerExpand)) ? GTK_PACK_EXPAND : 0;
      myApply |= data->apply;
      fillx = (gb_widget_input_bool (data, GbPackerFillX)) ? GTK_FILL_X : 0;
      myApply |= data->apply;
      filly = (gb_widget_input_bool (data, GbPackerFillY)) ? GTK_FILL_Y : 0;
      myApply |= data->apply;
      border = gb_widget_input_int (data, GbPackerBorder);
      myApply |= data->apply;
      padx = gb_widget_input_int (data, GbPackerPadX);
      myApply |= data->apply;
      pady = gb_widget_input_int (data, GbPackerPadY);
      myApply |= data->apply;
      ipadx = gb_widget_input_int (data, GbPackerIPadX);
      myApply |= data->apply;
      ipady = gb_widget_input_int (data, GbPackerIPadY);
      myApply |= data->apply;

      if (myApply)
        {
          for (i = 0; GbPackerSideChoices[i]; i++)
            {
              if (!strcmp (side_str, GbPackerSideChoices[i])
                  || !strcmp (side_str, GbPackerSideSymbols[i]))
                {
                  side = GbPackerSideValues[i];
                  break;
                }
            }

          for (i = 0; GbPackerAnchorChoices[i]; i++)
            {
              if (!strcmp (anchor_str, GbPackerAnchorChoices[i])
                  || !strcmp (anchor_str, GbPackerAnchorSymbols[i]))
                {
                  anchor = GbPackerAnchorValues[i];
                  break;
                }
            }

          /* FIXME: only set if changed. */
          gtk_packer_configure (GTK_PACKER (parent), widget, side, anchor,
                                expand | fillx | filly,
                                border, padx, pady, ipadx, ipady);
        }
    }
#endif

  data->loading_type = GB_STANDARD_PROPERTIES;
}


/* This ensures that placeholders are placed in every unoccupied cell.
   If rows and cols are not -1, then they are the new dimensions of the table.
 */
void
gb_widget_update_table_placeholders (GtkWidget * table, gint rows, gint cols)
{
  gint row, col, cell_contents;

  if (rows == -1)
    rows = GTK_TABLE (table)->nrows;
  if (cols == -1)
    cols = GTK_TABLE (table)->ncols;

  for (row = 0; row < rows; row++)
    {
      for (col = 0; col < cols; col++)
        {
          /* Find out what is in the cell */
          cell_contents = is_cell_occupied (table, row, col);
          if (cell_contents == GB_CELL_WIDGET)
            {
              /* If cell is occupied, delete any placeholders there. If a placeholder
                 occupies the cell but spans multiple rows/cols split it into single
                 cells but without a placeholder in this cell */
              remove_table_placeholders (table, row, col);

            }
          else if (cell_contents == GB_CELL_EMPTY)
            {
              /* If the cell is empty, put a placeholder in it */
              gtk_table_attach_defaults (GTK_TABLE (table),
                                         editor_new_placeholder (),
                                         col, col + 1, row, row + 1);
            }
        }
    }
}


/* Finds out if cell is occupied by a real widget. If not, returns whether
   the cell is empty or whether a placeholder is in it */
static gint
is_cell_occupied (GtkWidget * table, gint row, gint col)
{
  GList *children;
  GtkTableChild *child;
  gint return_val = GB_CELL_EMPTY;

  children = GTK_TABLE (table)->children;
  while (children)
    {
      child = children->data;
      if (child->top_attach <= row && child->bottom_attach > row
          && child->left_attach <= col && child->right_attach > col)
        {
          if (GB_IS_PLACEHOLDER (child->widget))
            return_val = GB_CELL_PLACEHOLDER;
          else
            return GB_CELL_WIDGET;
        }
      children = children->next;
    }
  return return_val;
}


static void
remove_table_placeholders (GtkWidget * table, gint row, gint col)
{
  GList *children, *next;
  GtkTableChild *child;
  gint left, right, top, bottom;

  children = GTK_TABLE (table)->children;
  while (children)
    {
      next = children->next;
      child = children->data;
      left = child->left_attach;
      right = child->right_attach;
      top = child->top_attach;
      bottom = child->bottom_attach;

      if (top <= row && bottom > row && left <= col && right > col)
        {
          /* If the widget is a placeholder remove it */
          if (GB_IS_PLACEHOLDER (child->widget))
            {
              if (bottom - top > 1 || right - left > 1)
                {
                  split_table_placeholder (table, child->widget,
                                           left, right, top, bottom,
                                           row, col);
                }
              else
                {
                  gtk_container_remove (GTK_CONTAINER (table), child->widget);
                }
            }
        }
      children = next;
    }
}


static void
split_table_placeholder (GtkWidget * table, GtkWidget * placeholder,
                         gint left, gint right, gint top, gint bottom,
                         gint skip_row, gint skip_col)
{
  gint row, col;

  gtk_container_remove (GTK_CONTAINER (table), placeholder);
  for (row = top; row < bottom; row++)
    {
      for (col = left; col < right; col++)
        {
          if (!(row == skip_row && col == skip_col))
            {
              gtk_table_attach_defaults (GTK_TABLE (table),
                                         editor_new_placeholder (),
                                         col, col + 1, row, row + 1);
            }
        }
    }
}


static void
set_position_properties (GtkWidget * widget,
                         GbWidgetSetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gint x, y, w, h;
  gboolean applyX, applyY, applyWidth, applyHeight, set_usize = FALSE;

  x = gb_widget_input_int (data, GbX);
  applyX = data->apply;
  y = gb_widget_input_int (data, GbY);
  applyY = data->apply;
  w = gb_widget_input_int (data, GbWidth);
  applyWidth = data->apply;
  h = gb_widget_input_int (data, GbHeight);
  applyHeight = data->apply;

#if 0
  g_print ("In set_position_properties X:%i Y:%i W:%i H:%i\n", x, y, w, h);
#endif
  if (GTK_IS_WINDOW (widget))
    {
      if (applyX)
        wdata->x = x;
      if (applyY)
        wdata->y = y;

      if (applyWidth && wdata->width != w)
        {
          wdata->width = w;
          if (w != 0)
            wdata->flags |= GB_WIDTH_SET;
          else
            wdata->flags &= ~GB_WIDTH_SET;
          set_usize = TRUE;
        }
      if (applyHeight && wdata->height != h)
        {
          wdata->height = h;
          if (h != 0)
            wdata->flags |= GB_HEIGHT_SET;
          else
            wdata->flags &= ~GB_HEIGHT_SET;
          set_usize = TRUE;
        }

      if (set_usize)
        {
          gint w = wdata->flags & GB_WIDTH_SET ? wdata->width : -1;
          gint h = wdata->flags & GB_HEIGHT_SET ? wdata->height : -1;
          gb_widget_set_usize (widget, w, h);
        }
    }
  else if (widget->parent && GTK_IS_FIXED (widget->parent))
    {
      if (applyWidth || applyHeight)
        {
          if (w != widget->allocation.width || h != widget->allocation.height)
            {
              gb_widget_set_usize (widget, w, h);
            }
        }
      if (applyX || applyY)
        {
          if (x != widget->allocation.x || y != widget->allocation.y)
            {
              /* FIXME: Do we need both? */
              gtk_fixed_move (GTK_FIXED (widget->parent), widget, x, y);
              gtk_widget_set_uposition (widget, x, y);
            }
        }
    }
  else
    {
      if (applyWidth && wdata->width != w)
        {
          wdata->width = w;
          if (w != 0)
            wdata->flags |= GB_WIDTH_SET;
          else
            wdata->flags &= ~GB_WIDTH_SET;
          set_usize = TRUE;
        }
      if (applyHeight && wdata->height != h)
        {
          wdata->height = h;
          if (h != 0)
            wdata->flags |= GB_HEIGHT_SET;
          else
            wdata->flags &= ~GB_HEIGHT_SET;
          set_usize = TRUE;
        }

      if (set_usize)
        {
          gint w = wdata->flags & GB_WIDTH_SET ? wdata->width : 0;
          gint h = wdata->flags & GB_HEIGHT_SET ? wdata->height : 0;
          gb_widget_set_usize (widget, w, h);
        }
      MSG2 ("*** Width set:%i Height set:%i", wdata->flags & GB_WIDTH_SET,
            wdata->flags & GB_HEIGHT_SET);
    }
}


static void
apply_style (GtkWidget * widget,
             GbWidgetSetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  GtkStyle *style = widget->style, *old_style = NULL;
  GbStyle *gbstyle = wdata->gbstyle, *new_gbstyle;
  GdkFont *font = NULL;
  gchar *style_name, *xlfd_fontname;
  GdkColor fg[GB_NUM_STYLE_STATES];
  GdkColor bg[GB_NUM_STYLE_STATES];
  GdkColor text[GB_NUM_STYLE_STATES];
  GdkColor base[GB_NUM_STYLE_STATES];
  /* FIXME: Delete?
     GdkColor light[GB_NUM_STYLE_STATES];
     GdkColor dark[GB_NUM_STYLE_STATES];
     GdkColor mid[GB_NUM_STYLE_STATES];
   */
  GdkPixmap *bg_pixmap[GB_NUM_STYLE_STATES];
  gchar *bg_pixmap_filenames[GB_NUM_STYLE_STATES];
  gint recreate = FALSE, redraw = FALSE, i;
  gchar buf[128], *filename;
  gboolean named_style;

  style_name = gb_widget_input_dialog (data, GbStyleName);
  named_style = (strlen (style_name) == 0) ? FALSE : TRUE;
  if (data->apply)
    {
      if (named_style)
        {
          new_gbstyle = (GbStyle *) g_hash_table_lookup (gb_style_hash, style_name);
          g_return_if_fail (new_gbstyle != NULL);
          if (new_gbstyle != gbstyle)
            {
              gbstyle = new_gbstyle;
              gb_widget_set_gb_style (widget, gbstyle);
              wdata->flags &= ~GB_STYLE_IS_UNNAMED;
              redraw = TRUE;
            }
        }
      else
        {
          wdata->flags |= GB_STYLE_IS_UNNAMED;
        }
    }

  font = gb_widget_input_font (data, GbStyleFont, &xlfd_fontname);
  if (data->apply)
    {
      if (font != style->font)
        recreate = TRUE;
    }

  recreate |= apply_colors (widget, data, style->fg, fg, "fg");
  recreate |= apply_colors (widget, data, style->bg, bg, "bg");
  recreate |= apply_colors (widget, data, style->text, text, "text");
  recreate |= apply_colors (widget, data, style->base, base, "base");
  /* FIXME: Delete?
     recreate |= apply_colors(widget, data, style->light,light, "light");
     recreate |= apply_colors(widget, data, style->dark,        dark,   "dark");
     recreate |= apply_colors(widget, data, style->mid, mid,    "middle");
   */

  /* Background pixmaps */
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      sprintf (buf, "GtkStyle::%s[%s]", GbBgPixmapName, GbStateNames[i]);
      bg_pixmap[i] = gb_widget_input_bgpixmap (data, buf, &filename);
      bg_pixmap_filenames[i] = filename;
      if (data->apply)
        {
          if (bg_pixmap[i] != style->bg_pixmap[i])
            recreate = TRUE;
        }
    }

  if (recreate)
    {
      old_style = style;

      /* If the widget is supposedly using an unnamed GbStyle, but currently is
         actually using a named GbStyle (for convenience), then we need to
         create a copy of the GbStyle and place our new style in it. */
      if ((wdata->flags & GB_STYLE_IS_UNNAMED) && gbstyle->name)
        {
          gbstyle = gb_widget_copy_gb_style (gbstyle);
          g_free (gbstyle->name);
          gbstyle->name = NULL;
        }

      style = gtk_style_new ();
      for (i = 0; i < GB_NUM_STYLE_STATES; i++)
        {
          style->fg[i] = fg[i];
          style->bg[i] = bg[i];
          style->text[i] = text[i];
          style->base[i] = base[i];
          /* FIXME: Delete?
             style->light[i]     = light[i];
             style->dark[i]      = dark[i];
             style->mid[i]       = mid[i];
           */
          style->bg_pixmap[i] = bg_pixmap[i];
          if (bg_pixmap[i])
            gdk_pixmap_ref (bg_pixmap[i]);

          if (gbstyle->bg_pixmap_filenames[i] != bg_pixmap_filenames[i])
            {
              g_free (gbstyle->bg_pixmap_filenames[i]);
              gbstyle->bg_pixmap_filenames[i] = g_strdup (bg_pixmap_filenames
                                                          [i]);
            }
        }
      if (font)
        {
          gdk_font_unref (style->font);
          style->font = font;
          gdk_font_ref (style->font);
        }
      if (strcmp (gbstyle->xlfd_fontname, xlfd_fontname))
        {
          g_free (gbstyle->xlfd_fontname);
          gbstyle->xlfd_fontname = g_strdup (xlfd_fontname);
        }

      gbstyle->style = style;
      gtk_style_ref (style);
      gb_widget_set_gb_style (widget, gbstyle);
    }


  /* If a named style has been changed/recreated we have to update all
     widget's that use it. */
  if (recreate || redraw)
    {
      if (named_style)
        {
          gb_widget_update_gb_styles (gbstyle, gbstyle);
        }
      else
        {
          editor_refresh_widget (widget);
        }
    }

  if (old_style)
    gtk_style_unref (old_style);
}


/* This makes sure a widget's gbstyle & its style are up to date, and
   if the propagate flag is set it also updates any children.
   But it won't change a descendant's style if it has been set explicitly.
   FIXME: only propagates one level at present, and always sets child's style,
   even if it has a different GbStyle! */
void
gb_widget_set_gb_style (GtkWidget * widget,
                        GbStyle * gbstyle)
{
  GbWidgetData *wdata;

  if (!GB_IS_PLACEHOLDER (widget))
    {
      if (widget->style != gbstyle->style)
        gtk_widget_set_style (widget, gbstyle->style);
    }

  wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  if (wdata)
    {
      if (wdata->gbstyle != gbstyle)
        {
          gb_widget_unref_gb_style (wdata->gbstyle);
          wdata->gbstyle = gbstyle;
          gb_widget_ref_gb_style (gbstyle);
        }
      /* If propagate style flags is set, propagate style to children */
      if (wdata->flags & GB_STYLE_PROPAGATE)
        {
          if (GTK_IS_CONTAINER (widget))
            {
              gtk_container_foreach (GTK_CONTAINER (widget),
                             (GtkCallback) gb_widget_set_gb_style, gbstyle);
            }
        }
    }
}


/* This steps through all widgets in all components and updates the GbStyle
   and GtkStyles as appropriate. */
void
gb_widget_update_gb_styles (GbStyle * old_gbstyle, GbStyle * new_gbstyle)
{
  struct GbUpdateStyleData data;
  data.old_gbstyle = old_gbstyle;
  data.new_gbstyle = new_gbstyle;
  project_foreach_component ((GtkCallback) update_style, &data);
}


static void
update_style (GtkWidget * widget, struct GbUpdateStyleData *data)
{
  GbWidgetData *wdata;

  wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);

  if (wdata && wdata->gbstyle == data->old_gbstyle)
    {
      wdata->gbstyle = data->new_gbstyle;
      if (widget->style != wdata->gbstyle->style)
        {
          gtk_widget_set_style (widget, wdata->gbstyle->style);
          gtk_widget_queue_resize (widget);
          gtk_widget_queue_draw (widget);
        }
    }
  if (GTK_IS_CONTAINER (widget))
    {
      gtk_container_foreach (GTK_CONTAINER (widget), (GtkCallback) update_style,
                             data);
    }
}


static gboolean
apply_colors (GtkWidget * widget, GbWidgetSetArgData * data, GdkColor colors[],
              GdkColor new_colors[], gchar * name)
{
  gint state;
  gchar buf[128];
  guchar *color;
  GdkColor *old_gdkcolor, *new_gdkcolor;
  gboolean need_redraw = FALSE;

  for (state = 0; state < GB_NUM_STYLE_STATES; state++)
    {
      sprintf (buf, "GtkStyle::%s[%s]", name, GbStateNames[state]);
      old_gdkcolor = &colors[state];
      new_gdkcolor = &new_colors[state];
      new_gdkcolor->red = old_gdkcolor->red;
      new_gdkcolor->green = old_gdkcolor->green;
      new_gdkcolor->blue = old_gdkcolor->blue;
      new_gdkcolor->pixel = old_gdkcolor->pixel;

      color = gb_widget_input_color (data, buf);
      if (data->apply)
        {
          new_gdkcolor->red = color[0] << 8;
          new_gdkcolor->green = color[1] << 8;
          new_gdkcolor->blue = color[2] << 8;
          if (!gdk_color_equal (new_gdkcolor, old_gdkcolor))
            need_redraw = TRUE;
        }
    }
  return need_redraw;
}


/* Currently this frees all the GbAccelerators and creates them from scratch */
static void
apply_accelerators (GtkWidget * widget, GbWidgetSetArgData * data)
{
  if (data->property_to_apply == NULL
      || property_is_accel_clist (data->property_to_apply))
    {
      gb_widget_free_accelerators (widget);
      data->widget_data->accelerators = property_get_accelerators ();
    }
}


/* Currently this frees all the GbSignals and creates them from scratch. */
static void
apply_signals (GtkWidget * widget, GbWidgetSetArgData * data)
{
  if (data->property_to_apply == NULL
      || property_is_signal_clist (data->property_to_apply))
    {
      gb_widget_free_signals (widget);
      data->widget_data->signals = property_get_signals ();
    }
}



/*************************************************************************
 * Functions for showing the popup context-sensitive menu of a widget
 *************************************************************************/

void
gb_widget_show_popup_menu (GtkWidget * widget,
                           GdkEventButton * event)
{
  static GtkWidget *menu = NULL;

  GbWidget *gbwidget;
  gchar *class_name, *name;
  GtkWidget *menuitem, *ancestor, *submenu, *child;
  GbWidgetCreateMenuData data;

  class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));
  gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
  g_return_if_fail (gbwidget != NULL);

  name = gtk_widget_get_name (widget);
  if (GB_IS_PLACEHOLDER (widget))
    name = "Placeholder";

  /* bug workaround - delete last popup menu */
  if (menu)
    gtk_widget_destroy (menu);

  menu = gtk_menu_new ();
  /* doesn't call menuitem's 'activate' handler if we do this
     gtk_signal_connect_object (GTK_OBJECT (menu), "deactivate",
     GTK_SIGNAL_FUNC(gtk_widget_destroy),
     GTK_OBJECT(menu));
   */
  menuitem = gtk_menu_item_new_with_label (name);
  gtk_widget_show (menuitem);
  gtk_widget_set_sensitive (menuitem, FALSE);
  gtk_menu_append (GTK_MENU (menu), menuitem);

  data.menu = menu;
  data.child = NULL;
  add_standard_top_menu_items (widget, &data);

  if (gbwidget->gb_widget_create_popup_menu)
    (gbwidget->gb_widget_create_popup_menu) (widget, &data);

  add_standard_bottom_menu_items (widget, &data);

  child = widget;
  ancestor = widget->parent;
  while (ancestor)
    {
      name = gtk_widget_get_name (ancestor);
      if (GB_IS_PLACEHOLDER (ancestor))
        name = "Placeholder";

      /* Skip widgets which aren't GbWidgets */
      if (GB_IS_GB_WIDGET (ancestor))
        {
          /* Add a separator */
          menuitem = gtk_menu_item_new ();
          gtk_menu_append (GTK_MENU (menu), menuitem);
          gtk_widget_show (menuitem);

          menuitem = gtk_menu_item_new_with_label (name);
          gtk_widget_show (menuitem);
          gtk_menu_append (GTK_MENU (menu), menuitem);

          /* Create submenu */
          submenu = gtk_menu_new ();
          gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), submenu);

          data.menu = submenu;
          data.child = child;
          add_standard_top_menu_items (ancestor, &data);

          /* Call ancestors function to add any menu items */
          class_name = gtk_type_name (GTK_OBJECT_TYPE (ancestor));
          gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
          if (gbwidget != NULL && gbwidget->gb_widget_create_popup_menu)
            (gbwidget->gb_widget_create_popup_menu) (ancestor, &data);

          add_standard_bottom_menu_items (ancestor, &data);
        }
      child = ancestor;
      ancestor = ancestor->parent;
    }

  MSG ("showing popup menu");
  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
                  event->button, event->time);
}


static void
add_standard_top_menu_items (GtkWidget * widget, GbWidgetCreateMenuData * data)
{
  GtkWidget *menuitem;

  menuitem = gtk_menu_item_new_with_label (_("Select"));
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                      GTK_SIGNAL_FUNC (editor_on_select_activate), widget);
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
}


static void
add_standard_bottom_menu_items (GtkWidget * widget, GbWidgetCreateMenuData * data)
{
  GtkWidget *menuitem;

  /* FIXME: Removed this for now, until I've checked it.
     I think I'll probably use a form of drag-and-drop instead. */
  /*
     if (data->child && (GTK_IS_BOX(widget) || GTK_IS_TABLE(widget))) {
     menuitem = gtk_menu_item_new_with_label ("Move");
     gtk_widget_show (menuitem);
     gtk_menu_append (GTK_MENU (data->menu), menuitem);
     gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
     GTK_SIGNAL_FUNC(gb_widget_move_activate), data->child);
     }
   */

  if (!GTK_IS_WINDOW (widget)
      && !GTK_IS_MISC (widget)
      && !GTK_IS_ALIGNMENT (widget)
      && !GB_IS_PLACEHOLDER (widget))
    {
      if (GTK_IS_ALIGNMENT (widget->parent))
        {
          menuitem = gtk_menu_item_new_with_label (_("Remove Alignment"));
          gtk_widget_show (menuitem);
          gtk_menu_append (GTK_MENU (data->menu), menuitem);
          gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                      GTK_SIGNAL_FUNC (gb_widget_remove_alignment), widget);
        }
      else
        {
          menuitem = gtk_menu_item_new_with_label (_("Add Alignment"));
          gtk_widget_show (menuitem);
          gtk_menu_append (GTK_MENU (data->menu), menuitem);
          gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                         GTK_SIGNAL_FUNC (gb_widget_add_alignment), widget);
        }
    }

  if (GTK_WIDGET_NO_WINDOW (widget)
      && !GTK_IS_EVENT_BOX (widget)
      && !GB_IS_PLACEHOLDER (widget))
    {
      if (GTK_IS_EVENT_BOX (widget->parent))
        {
          menuitem = gtk_menu_item_new_with_label (_("Remove Event Box"));
          gtk_widget_show (menuitem);
          gtk_menu_append (GTK_MENU (data->menu), menuitem);
          gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                      GTK_SIGNAL_FUNC (gb_widget_remove_event_box), widget);
        }
      else
        {
          menuitem = gtk_menu_item_new_with_label (_("Add Event Box"));
          gtk_widget_show (menuitem);
          gtk_menu_append (GTK_MENU (data->menu), menuitem);
          gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                         GTK_SIGNAL_FUNC (gb_widget_add_event_box), widget);
        }
    }

  menuitem = gtk_menu_item_new_with_label (_("Delete"));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (data->menu), menuitem);
  gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                      GTK_SIGNAL_FUNC (editor_on_delete_activate), widget);
}



/*************************************************************************
 * Functions for loading project files
 *************************************************************************/

/* This reads in a widget, and is recursively called to read it's children.
   It assumes a '<widget>' has just been read, and reads up to and including
   '</widget>'. Note that if any gbwidgets have their own gb_widget_load
   function they are responsible for calling functions to set the standard
   properties, add the widget to its parent, and load any children. */
void
gb_widget_load (GtkWidget * widget, GbWidgetSetArgData * data, GtkWidget * parent)
{
  gchar *class_name, *child_name, *ancestor_class_name;
  GbWidget *gbwidget, *ancestor_gbwidget;
  GtkWidget *ancestor;
  GbWidgetData *wdata;

  data->action = GB_LOADING;
  data->loading_type = GB_STANDARD_PROPERTIES;
  gb_widget_load_properties (data);

  if (data->status != GB_OK)
    {
      MSG ("Load error");
      return;
    }

#ifdef GLD_DEBUG
  {
    gint i;
    for (i = 0; i < data->nproperties; i++)
      {
        MSG3 ("Line number: %i Property: %s Value: %s",
              data->properties[i].line_number,
              data->buffer + data->properties[i].tag_index,
              data->buffer + data->properties[i].cdata_index);

      }
  }
#endif
  class_name = load_get_value (data, "class");
  MSG1 ("Reading Class: %s", class_name);
  if (!class_name)
    {
      MSG ("Load error");
      data->status = GB_CLASS_NAME_MISSING;
      return;
    }

  if (!g_strcasecmp (class_name, "placeholder"))
    {
      MSG ("found placeholder");
      gb_widget_add_child (parent, data, editor_new_placeholder ());
      return;
    }

  gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
  if (gbwidget == NULL)
    {
      MSG ("Load error");
      data->status = GB_CLASS_UNKNOWN;
      return;
    }

  /* If this is a special child of a widget, step through the ancestors of
     the widget and try to find it. Note that some special child widgets
     have to be created normally, and then added by the parent container,
     e.g. notbook tabs and clist titles - see gb_widget_add_child(). */
  child_name = load_get_value (data, "child_name");
  if (child_name)
    {
      MSG1 ("Child name: %s", child_name);
      ancestor = parent;
      while (ancestor)
        {
          ancestor_class_name = gtk_type_name (GTK_OBJECT_TYPE (ancestor));
          MSG1 ("Ancestor type: %s", ancestor_class_name);
          ancestor_gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table,
                                                       ancestor_class_name);
          if (ancestor_gbwidget && ancestor_gbwidget->gb_widget_get_child)
            {
              widget = (ancestor_gbwidget->gb_widget_get_child) (ancestor,
                                                                 child_name);
              if (widget)
                break;
            }

          ancestor = ancestor->parent;
        }
      /* FIXME: take out */
#if 0
      if (!widget)
        g_print ("Child widget %s not found - may be a problem\n", child_name);
#endif
    }

  /* If this is a standard widget, we need to create it and add it to its
     parent. If the widget has already been created by its parent, we can
     just set the properties. */
  if (widget == NULL)
    {
      widget = gb_widget_new_full (class_name, parent, NULL,
                                   0, 0, NULL, GB_LOADING, data);
      wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
      data->widget_data = wdata;
      gb_widget_add_child (parent, data, widget);
      /* This must come after add_child so we can set special child properties,
         and also we may need to realize the widget. */
      tree_add_widget (widget);
      set_standard_properties (widget, data);
    }
  else
    {
      wdata = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
      if (wdata == NULL)
        g_warning ("Null widget_data\n");
      data->widget_data = wdata;
      tree_add_widget (widget);
      set_standard_properties (widget, data);
    }

  if (child_name)
    gtk_object_set_data (GTK_OBJECT (widget), GB_CHILD_NAME_KEY,
                         g_strdup (child_name));

  if (data->status != GB_OK)
    {
      MSG ("Load error");
      return;
    }

  MSG ("Calling widgets set_properties()");
  if (gbwidget->gb_widget_set_properties)
    (gbwidget->gb_widget_set_properties) (widget, data);
  MSG ("Called widgets set_properties()");

  if (data->status != GB_OK)
    {
      MSG ("Load error");
      return;
    }

  /* All the necessary properties should have been copied from the buffer
     by now, se we can clear it. Note: actually we use it below but that
     should be OK?? */
  load_buffer_release (data);

  MSG ("Loading children");

  /* load children. */
  while (data->token_type == GB_TOKEN_START_TAG)
    {
      if (!strcmp (data->buffer + data->token, "widget"))
        {
          /* We have to reset the widget_data since this may have been changed
             while loading the last child (and its children) */
          data->widget_data = wdata;
          gb_widget_load (NULL, data, widget);
          if (data->status != GB_OK)
            {
              MSG ("Load error");
              return;
            }
        }
      else
        {
          break;
        }
      load_token_skip_whitespace (data);
      if (data->status != GB_OK)
        {
          MSG ("Load error");
          return;
        }
    }

  MSG ("Loaded children");

  if (data->token_type != GB_TOKEN_END_TAG
      || strcmp (data->buffer + data->token, "widget"))
    {
      data->status = GB_END_TAG_EXPECTED;
      MSG ("Load error");
      return;
    }

  /* If the widget is a table, we make sure all empty cells have placeholders
     in them. */
  if (GTK_IS_TABLE (widget))
    {
      gb_widget_update_table_placeholders (widget, -1, -1);
    }
}


/* Adds a widget to a parent, using code for each type of container.
   If the parent is NULL, then this is a toplevel window/dialog etc. and so
   we add it to the component list. */
void
gb_widget_add_child (GtkWidget * parent,
                     GbWidgetSetArgData * data,
                     GtkWidget * child)
{
  if (parent == NULL)
    {
      gchar *name = gtk_widget_get_name (child);

      MSG1 ("Component name: %s", name);

      if (!project_add_component (child, name))
        {
          data->status = GB_INVALID_COMPONENT;
          MSG ("Load error");
        }
      return;
    }

  if (GTK_IS_MENU_ITEM (parent) && GTK_IS_MENU (child))
    {
      MSG ("Trying to add a menu to a menu item");
      gtk_menu_item_set_submenu (GTK_MENU_ITEM (parent), child);
    }
  else if (GTK_IS_BIN (parent))
    {
      if (GTK_BIN (parent)->child)
        gtk_container_remove (GTK_CONTAINER (parent), GTK_BIN (parent)->child);
      gtk_container_add (GTK_CONTAINER (parent), child);

    }
  else if (GTK_IS_BOX (parent))
    {
      gtk_container_add (GTK_CONTAINER (parent), child);

    }
  else if (GTK_IS_LIST (parent))
    {
      gtk_container_add (GTK_CONTAINER (parent), child);

    }
  else if (GTK_IS_NOTEBOOK (parent))
    {
      /* See if this is a tab widget. */
      gchar *child_name = load_get_value (data, "child_name");
      /* FIXME: This should use NotebookTab from gbnotebook.c */
      if (child_name && !strcmp (child_name, "Notebook:tab"))
        {
          /* We store the last tab read in 'last_child' */
          GtkNotebookPage *page;
          GtkWidget *notebook_page;
          gint pos = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (parent),
                                                           "last_child"));
          MSG1 ("Adding notebook tab: %i", pos);
          /* SPECIAL CODE to replace the notebooks default tab label with the
             loaded widget. We remove the page and add it with the new tab,
             just like in gb_widget_replace_child(). */
          page = (GtkNotebookPage*) g_list_nth (GTK_NOTEBOOK (parent)->children, pos)->data;
          notebook_page = page->child;
          gtk_widget_ref (notebook_page);
          gtk_notebook_remove_page (GTK_NOTEBOOK (parent), pos);
          gtk_notebook_insert_page (GTK_NOTEBOOK (parent), notebook_page,
                                    child, pos);
          gtk_widget_unref (notebook_page);
          gtk_object_set_data (GTK_OBJECT (parent), "last_child",
                               GINT_TO_POINTER (pos + 1));
        }
      else
        {
          gtk_container_add (GTK_CONTAINER (parent), child);
        }
    }
  else if (GTK_IS_PANED (parent))
    {
      gtk_container_add (GTK_CONTAINER (parent), child);

    }
  else if (GTK_IS_TABLE (parent))
    {
      gtk_container_add (GTK_CONTAINER (parent), child);

    }
  else if (GTK_IS_TREE (parent))
    {
      gtk_container_add (GTK_CONTAINER (parent), child);

    }
  else if (GTK_IS_CLIST (parent))
    {
      /* See if this is a title widget. */
      gchar *child_name = load_get_value (data, "child_name");
      /* FIXME: This should use CListTitle from gbclist.c */
      if (child_name && !strcmp (child_name, "CList:title"))
        {
          /* We store the last column title read in 'last_child' */
          gint col = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (parent),
                                                           "last_child"));
          gtk_clist_set_column_widget (GTK_CLIST (parent), col, child);
          gtk_object_set_data (GTK_OBJECT (parent), "last_child",
                               GINT_TO_POINTER (col + 1));

          /* We need to add signals to the clist button, just in case the
             title widget has no window and so doesn't get signals itself.
             Since Clist always creates 1 button initially, the signals would
             be added to this button in gb_widget_new, so we could skip it,
             but it doesn't hurt. */
          editor_add_mouse_signals_to_existing (GTK_CLIST (parent)->column[col].button);
        }
      else
        {
          gtk_container_add (GTK_CONTAINER (parent), child);
        }

    }
  else if (GTK_IS_BUTTON (parent))
    {
      if (GTK_BUTTON (parent)->child)
        gtk_container_remove (GTK_CONTAINER (parent),
                              GTK_BUTTON (parent)->child);
      gtk_container_add (GTK_CONTAINER (parent), child);

    }
  else if (GTK_IS_CONTAINER (parent))
    {
      gtk_container_add (GTK_CONTAINER (parent), child);
    }

}



/* This will load all the properties/accelerators/signals of a widget.
   It assumes a <widget> has just been read and reads up to a <widget>,
   </widget>. */
static void
gb_widget_load_properties (GbWidgetSetArgData * data)
{
  gint property;

  /* Initialise the widget's properties. */
  data->nproperties = 0;
  data->nchild_properties = 0;

  data->accelerators = NULL;
  data->signals = NULL;
  data->gbstyle = NULL;

  for (;;)
    {
      load_token_skip_whitespace (data);
      if (data->status != GB_OK)
        {
          MSG ("Load error");
          return;
        }

      if (!strcmp (data->buffer + data->token, "widget"))
        return;

      if (data->token_type != GB_TOKEN_START_TAG)
        {
          data->status = GB_START_TAG_EXPECTED;
          MSG ("Load error");
          return;
        }

      if (!strcmp (data->buffer + data->token, "style"))
        gb_widget_load_style (data);
      else if (!strcmp (data->buffer + data->token, "accelerator"))
        load_accelerator (data);
      else if (!strcmp (data->buffer + data->token, "signal"))
        load_signal (data);
      else if (!strcmp (data->buffer + data->token, "child"))
        load_special_child_properties (data);
      else
        {
          property = data->token;
          load_token (data);
          if (data->status != GB_OK)
            {
              MSG ("Load error");
              return;
            }

          /* Remember the property could be empty, so we may get the end tag. */
          if (data->token_type == GB_TOKEN_START_TAG)
            {
              data->status = GB_DATA_EXPECTED;
              MSG ("Load error");
              return;
            }

          if (data->token_type == GB_TOKEN_DATA)
            {
              if (data->nproperties == data->properties_space)
                {
                  data->properties_space += GB_LOADED_PROPERTIES_INC;
                  data->properties = g_realloc (data->properties,
                                                sizeof (GbLoadedProperty)
                                                * data->properties_space);
                }
              data->properties[data->nproperties].tag_index = property;
              data->properties[data->nproperties].cdata_index = data->token;
              data->properties[data->nproperties].line_number
                = data->line_number;
              data->nproperties++;

              load_token (data);
              if (data->status != GB_OK)
                {
                  MSG ("Load error");
                  return;
                }
              if (data->token_type != GB_TOKEN_END_TAG
                  || strcmp (data->buffer + data->token,
                             data->buffer + property))
                {
                  data->status = GB_END_TAG_EXPECTED;
                  MSG ("Load error");
                  return;
                }
            }
          else
            /* An empty tag may have been read. Check the start and end tags
               match. */
            {
              if (strcmp (data->buffer + data->token, data->buffer + property))
                {
                  data->status = GB_END_TAG_EXPECTED;
                  MSG ("Load error");
                  return;
                }
              if (data->nproperties == data->properties_space)
                {
                  data->properties_space += GB_LOADED_PROPERTIES_INC;
                  data->properties = g_realloc (data->properties,
                                                sizeof (GbLoadedProperty)
                                                * data->properties_space);
                }
              data->properties[data->nproperties].tag_index = property;
              data->properties[data->nproperties].cdata_index = 0;
              data->properties[data->nproperties].line_number
                = data->line_number;
              data->nproperties++;
            }
        }
    }
}


/* This will load all the accelerators of a widget. It assumes a <accelerator>
   has just been read and reads up to to </accelerator>. */
static void
load_special_child_properties (GbWidgetSetArgData * data)
{
  gint element_index, cdata_index;

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);

      if (data->status != GB_OK)
        {
          if (data->status == GB_START_TAG_EXPECTED
              && data->token_type == GB_TOKEN_END_TAG
              && (!strcmp (data->buffer + data->token, "child")))
            {
              data->status = GB_OK;
            }
          return;
        }

      if (data->nchild_properties == data->child_properties_space)
        {
          data->child_properties_space += GB_LOADED_PROPERTIES_INC;
          data->child_properties = g_realloc (data->child_properties,
                                              sizeof (GbLoadedProperty)
                                            * data->child_properties_space);
        }
      data->child_properties[data->nchild_properties].tag_index
        = element_index;
      data->child_properties[data->nchild_properties].cdata_index
        = cdata_index;
      data->child_properties[data->nchild_properties].line_number
        = data->line_number;
      data->nchild_properties++;
    }
}


/* This will load all the accelerators of a widget. It assumes a <accelerator>
   has just been read and reads up to to </accelerator>. */
static void
load_accelerator (GbWidgetSetArgData * data)
{
  GbAccelerator *accel;
  gint element_index, cdata_index;
  gchar *element, *cdata;

  accel = g_new (GbAccelerator, 1);
  accel->modifiers = 0;
  accel->key = NULL;
  accel->signal = NULL;

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);
      element = data->buffer + element_index;
      cdata = data->buffer + cdata_index;

      if (data->status != GB_OK)
        {
          if (data->status == GB_START_TAG_EXPECTED
              && data->token_type == GB_TOKEN_END_TAG
              && (!strcmp (data->buffer + data->token, "accelerator")))
            {
              data->accelerators = g_list_append (data->accelerators, accel);
              data->status = GB_OK;
            }
          else
            {
              g_free (accel->key);
              g_free (accel->signal);
              g_free (accel);
            }
          return;
        }

      if (!strcmp (element, "modifiers"))
        {
          /* We simple scan for the flags in the cdata, although they should
             be separated by '|'. */
          if (gb_strstr (cdata, "GDK_CONTROL_MASK"))
            accel->modifiers |= GDK_CONTROL_MASK;
          if (gb_strstr (cdata, "GDK_SHIFT_MASK"))
            accel->modifiers |= GDK_SHIFT_MASK;
          if (gb_strstr (cdata, "GDK_MOD1_MASK"))
            accel->modifiers |= GDK_MOD1_MASK;
        }
      else if (!strcmp (element, "key"))
        {
          if ((strlen (cdata) < 5) || strncmp (cdata, "GDK_", 4))
            {
              /* FIXME: add to error messages */
              g_warning ("Invalid accelerator key: %s\n", cdata);
            }
          else
            {
              g_free (accel->key);
              /* The initial 'GDK_' is removed. */
              accel->key = g_strdup (cdata + 4);
            }
        }
      else if (!strcmp (element, "signal"))
        {
          g_free (accel->signal);
          accel->signal = g_strdup (cdata);
        }
      else
        /* FIXME: add to error messages */
        g_warning ("Invalid element in accelerator: %s\n", element);
    }
}


/* This will load all the accelerators of a widget. It assumes a <signal>
   has just been read and reads up to to </signal>. */
static void
load_signal (GbWidgetSetArgData * data)
{
  GbSignal *signal;
  gint element_index, cdata_index;
  gchar *element, *cdata;

  signal = g_new (GbSignal, 1);
  signal->name = NULL;
  signal->handler = NULL;
  signal->object = NULL;
  signal->after = FALSE;
  signal->data = NULL;

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);
      element = data->buffer + element_index;
      cdata = data->buffer + cdata_index;

      if (data->status != GB_OK)
        {
          if (data->status == GB_START_TAG_EXPECTED
              && data->token_type == GB_TOKEN_END_TAG
              && (!strcmp (data->buffer + data->token, "signal")))
            {
              data->signals = g_list_append (data->signals, signal);
              data->status = GB_OK;
            }
          else
            {
              g_free (signal->name);
              g_free (signal->handler);
              g_free (signal->object);
              g_free (signal->data);
              g_free (signal);
            }
          return;
        }

      MSG2 ("Signal element:%s cdata:%s", element, cdata);

      if (!strcmp (element, "name"))
        {
          g_free (signal->name);
          signal->name = g_strdup (cdata);
        }
      else if (!strcmp (element, "handler"))
        {
          g_free (signal->handler);
          signal->handler = g_strdup (cdata);
        }
      else if (!strcmp (element, "object"))
        {
          g_free (signal->object);
          signal->object = g_strdup (cdata);
        }
      else if (!strcmp (element, "after"))
        {
          signal->after = load_parse_bool (data, cdata);
        }
      else if (!strcmp (element, "data"))
        {
          g_free (signal->data);
          signal->data = g_strdup (cdata);
        }
      else
        /* FIXME: add to error messages */
        g_warning ("Invalid element in signal: %s\n", element);
    }
}


/* This will load a style. It assumes a <style> has just been read and reads
   up to to </style>. If the style is named it is added to the GbStyle hash.
   The created GbStyle is returned in data->gbstyle. */
void
gb_widget_load_style (GbWidgetSetArgData * data)
{
  GbStyle *gbstyle;
  GtkStyle *style;
  gchar *element, *cdata;
  gint element_index, cdata_index;
  gboolean loading_default_style = FALSE;

  gbstyle = gb_widget_new_gb_style ();
  /* We copy the default style, since only changed values will appear in the
     input file. */
  style = gbstyle->style = gtk_style_copy (gb_widget_default_gb_style->style);
  gtk_style_ref (style);

  for (;;)
    {
      load_element (data, &element_index, &cdata_index);
      element = data->buffer + element_index;
      cdata = data->buffer + cdata_index;

      if (data->status != GB_OK)
        {
          if (data->status == GB_START_TAG_EXPECTED
              && data->token_type == GB_TOKEN_END_TAG
              && (!strcmp (data->buffer + data->token, "style")))
            {
              /* Make sure the new GbStyle has a font name set. */
              if (!gbstyle->xlfd_fontname)
                {
                  gbstyle->xlfd_fontname
                    = g_strdup (gb_widget_default_gb_style->xlfd_fontname);
                }

              /* If we are loading the Default style, replace current one. */
              if (loading_default_style)
                {
                  gb_widget_destroy_gb_style (gb_widget_default_gb_style, TRUE);
                  gb_widget_default_gb_style = gbstyle;
                }

              /* If the style is named, add it to the hash. */
              if (gbstyle->name)
                g_hash_table_insert (gb_style_hash, gbstyle->name, gbstyle);

              data->status = GB_OK;
              return;
            }
          MSG ("Load error");
          return;
        }

      if (!strcmp (element, "style_name"))
        {
          g_free (gbstyle->name);
          gbstyle->name = g_strdup (cdata);
          if (!strcmp (gbstyle->name, GB_STYLE_DEFAULT))
            loading_default_style = TRUE;
          MSG1 ("Loading named style:%s", gbstyle->name);
        }
      else if (!strcmp (element, "style_font"))
        {
          GdkFont *font;
          g_free (gbstyle->xlfd_fontname);
          gbstyle->xlfd_fontname = g_strdup (cdata);
          font = gdk_font_load (cdata);
          /* FIXME: add to error messages */
          if (font)
            {
              gdk_font_unref (style->font);
              style->font = font;
              gdk_font_ref (font);
            }
          else
            {
              g_warning ("Couldn't load font");
            }
        }
      else
        {
          gboolean found;

          found = load_colors (data, style->fg, "fg", element, cdata);
          if (!found)
            found = load_colors (data, style->bg, "bg", element, cdata);
          if (!found)
            found = load_colors (data, style->text, "text", element, cdata);
          if (!found)
            found = load_colors (data, style->base, "base", element, cdata);

          /* Check if element is a color or bgpixmap of one of the states. */
          if (!found)
            {
              gint state;
              gchar buf[128];
              GdkPixmap *gdkpixmap;

              for (state = 0; state < GB_NUM_STYLE_STATES; state++)
                {
                  /* Remember the tag will be converted to lowercase. */
                  sprintf (buf, "GtkStyle::%s:%s", GbBgPixmapName,
                           GbStateNames[state]);
                  g_strdown (buf);
                  if (!strcmp (element, find_start_of_tag_name (buf))
                      && strlen (cdata) > 0)
                    {
                      g_free (gbstyle->bg_pixmap_filenames[state]);
                      gbstyle->bg_pixmap_filenames[state] = g_strdup (cdata);

                      gdkpixmap = gdk_pixmap_create_from_xpm (NULL,
                                                              NULL,
                                               &style->bg[GTK_STATE_NORMAL],
                                                              cdata);
                      if (gdkpixmap)
                        {
                          style->bg_pixmap[state] = gdkpixmap;
                        }
                      else
                        {
                          /* FIXME: add to error messages */
                          g_warning ("Couldn't load pixmap: %s\n", cdata);
                        }
                      found = TRUE;
                    }
                }
            }

          /* FIXME: add to error messages */
          if (!found)
            g_warning ("Invalid element in style: %s\n", element);
        }
    }
}


static gboolean
load_colors (GbWidgetSetArgData * data, GdkColor colors[], gchar * name,
             gchar * element, gchar * cdata)
{
  gint state;
  gchar buf[128];
  guchar *color;

  for (state = 0; state < GB_NUM_STYLE_STATES; state++)
    {
      /* Remember the tag will be converted to lowercase. */
      sprintf (buf, "GtkStyle::%s:%s", name, GbStateNames[state]);
      g_strdown (buf);
      if (!strcmp (element, find_start_of_tag_name (buf)))
        {
          color = load_parse_color (data, cdata);
          if (color)
            {
              MSG3 ("RGB: %i,%i,%i", color[0], color[1], color[2]);
              colors[state].red = color[0] << 8;
              colors[state].green = color[1] << 8;
              colors[state].blue = color[2] << 8;
              return TRUE;
            }
          else
            {
              /* FIXME: add to error messages */
              g_warning ("Invalid color\n");
              return TRUE;
            }
        }
    }
  return FALSE;
}


/*************************************************************************
 * Functions for saving project files
 *************************************************************************/

void
gb_widget_save (GtkWidget * widget,
                GbWidgetGetArgData * data)
{
  GbWidget *gbwidget;
  GbWidgetData *widget_data;
  gchar *class_name, *child_name;

  /* Don't save a placeholder if its parent is a table, since we don't really
     want placeholders in the XML when the interface is finished, but it
     is quite possible that some table cells will be left blank. */
  if (GB_IS_PLACEHOLDER (widget))
    {
      if (!GTK_IS_TABLE (widget->parent))
        {
          save_newline (data);
          save_tag (data, "widget");
          data->indent++;
          save_string (data, "class", "Placeholder");
          data->indent--;
          save_end_tag (data, "widget");
        }
      return;
    }

  class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));

  /* Write out widget data and any child widgets */
  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  /* If this isn't a gbwidget, skip it. */
  if (widget_data)
    {
      data->action = GB_SAVING;
      data->widget_data = widget_data;

      save_newline (data);
      save_tag (data, "widget");
      data->indent++;
      save_string (data, "class", class_name);
      child_name = gtk_object_get_data (GTK_OBJECT (widget), GB_CHILD_NAME_KEY);
      if (child_name)
        save_string (data, "child_name", child_name);
      get_standard_properties (widget, data);

      /* Call gbwidgets save function for any extra properties */
      gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
      g_return_if_fail (gbwidget != NULL);
      if (gbwidget->gb_widget_get_properties)
        (gbwidget->gb_widget_get_properties) (widget, data);
    }

  /* If widget is a container, recursively save children.
     But don't save children of file/color selection dialogs or status bar */
  if (GTK_IS_CONTAINER (widget))
    {
      /* SPECIAL CODE: for table, so we output in the reverse order. */
      if (GTK_IS_TABLE (widget))
        table_foreach (GTK_TABLE (widget), (GtkCallback) gb_widget_save, data);
      else
        gtk_container_foreach (GTK_CONTAINER (widget),
                               (GtkCallback) gb_widget_save, data);

      /* SPECIAL CODE: output notebook tabs. */
      if (GTK_IS_NOTEBOOK (widget))
        notebook_tabs_foreach (GTK_NOTEBOOK (widget),
                               (GtkCallback) gb_widget_save, data);

      /* SPECIAL CODE: output menuitem submenus. */
      if (GTK_IS_MENU_ITEM (widget)
          && GTK_MENU_ITEM (widget)->submenu)
        gb_widget_save (GTK_MENU_ITEM (widget)->submenu, data);
    }

  if (widget_data)
    {
      data->indent--;
      save_end_tag (data, "widget");
    }
}


/* This outputs all the standard widget properties, for showing or saving.
   Note that when saving we try not to save default values. */
static void
get_standard_properties (GtkWidget * widget,
                         GbWidgetGetArgData * data)
{
  gboolean can_default, has_default, can_focus, has_focus, visible, sensitive;
  GbWidgetData *wdata = data->widget_data;
  gchar *name, *class_name;
  GbWidgetAction action = data->action;

  /* Save widget's name if it has been set */
  name = gtk_widget_get_name (widget);
  class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));
  if (action != GB_SAVING || strcmp (name, class_name))
    gb_widget_output_string (data, GbName, name);

  /* If widget is a container, save the border width */
  if (GTK_IS_CONTAINER (widget))
    {
      gint border_width;
      gchar buf[128];
      border_width = GTK_CONTAINER (widget)->border_width;
      if (action != GB_SAVING || border_width > 0)
        {
          sprintf (buf, "%s::border_width", class_name);
          gb_widget_output_int (data, buf, border_width);
        }
    }

  get_special_child_properties (widget, data);
  get_position_properties (widget, data);

  visible = wdata->flags & GB_VISIBLE;
  if (action != GB_SAVING || !visible)
    gb_widget_output_bool (data, GbVisible, visible);
  sensitive = wdata->flags & GB_SENSITIVE;
  if (action != GB_SAVING || !sensitive)
    gb_widget_output_bool (data, GbSensitive, sensitive);

  if (action != GB_SAVING || wdata->tooltip)
    gb_widget_output_string (data, GbTooltip, wdata->tooltip);

  can_default = GTK_WIDGET_CAN_DEFAULT (widget);
  if (action != GB_SAVING || can_default)
    gb_widget_output_bool (data, GbCanDefault, can_default);
  has_default = wdata ? wdata->flags & GB_GRAB_DEFAULT : 0;
  if (action != GB_SAVING || has_default)
    gb_widget_output_bool (data, GbHasDefault, has_default);

  can_focus = GTK_WIDGET_CAN_FOCUS (widget);
  if (action != GB_SAVING || can_focus)
    gb_widget_output_bool (data, GbCanFocus, can_focus);
  has_focus = wdata ? wdata->flags & GB_GRAB_FOCUS : 0;
  if (action != GB_SAVING || has_focus)
    gb_widget_output_bool (data, GbHasFocus, has_focus);

  if (!GTK_WIDGET_NO_WINDOW (widget))
    {
      GdkExtensionMode ext_mode = gtk_widget_get_extension_events (widget);
      gint i;

      if (action == GB_SAVING)
        {
          if (wdata && wdata->events)
            {
              gchar buffer[1024];
              gboolean written_first = FALSE;

              buffer[0] = '\0';
              for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
                {
                  if (wdata->events & GbEventMaskValues[i])
                    {
                      if (written_first)
                        strcat (buffer, " | ");
                      strcat (buffer, GbEventMaskSymbols[i]);
                      written_first = TRUE;
                    }
                }

              gb_widget_output_string (data, GbEvents, buffer);
            }
        }
      else
        {
          gb_widget_output_string (data, GbEvents,
                           property_events_value_to_string (wdata->events));
        }

      /* Don't save default extension mode ('None') */
      if (action != GB_SAVING || ext_mode != GDK_EXTENSION_EVENTS_NONE)
        {
          for (i = 0; GbExtensionModeChoices[i]; i++)
            {
              if (GbExtensionModeValues[i] == ext_mode)
                gb_widget_output_choice (data, GbExtEvents, i,
                                         GbExtensionModeSymbols[i]);
            }
        }
      if (action == GB_SHOWING)
        {
          property_set_sensitive (GbEvents, TRUE);
          property_set_sensitive (GbExtEvents, TRUE);
        }
    }
  else if (action == GB_SHOWING)
    {
      gb_widget_output_dialog (data, GbEvents, "N/A", NULL);
      property_set_sensitive (GbEvents, FALSE);
      gb_widget_output_choice (data, GbExtEvents, 0, GbExtensionModeSymbols[0]);
      property_set_sensitive (GbExtEvents, FALSE);
    }

  if (action == GB_SHOWING)
    {
      gb_widget_show_style (widget);
      show_accelerators (widget, data);
      show_signals (widget, data);
    }
  else
    {
      save_style (widget, data);
      save_accelerators (widget, data);
      save_signals (widget, data);
    }
}


static void
get_special_child_properties (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GtkWidget *parent = widget->parent;
  gint expand, fill, padding;
  GtkPackType pack_type;

  if (!parent)
    {
      if (data->action == GB_SHOWING)
        property_hide_special_child_page ();
      return;
    }

  /* If the widget is not the child of a table or a box, hide the special
     child properties (if we're showing the properties) and return. */
  if (!(GTK_IS_TABLE (parent)
        || (GTK_IS_BOX (parent) && !GTK_IS_BUTTON_BOX (parent))
#ifdef GLD_HAVE_GTK_1_1
        || GTK_IS_PACKER (parent)
#endif
      ))
    {
      if (data->action == GB_SHOWING)
        property_hide_special_child_page ();
      return;
    }

  /* If we're saving, start the 'child' section. */
  if (data->action == GB_SAVING)
    {
      save_tag (data, "child");
      data->indent++;
    }

  if (GTK_IS_TABLE (parent))
    {
      GtkTableChild *table_child = gb_table_find_child (GTK_TABLE (parent),
                                                        widget);
      g_return_if_fail (table_child != NULL);

      if (data->action == GB_SAVING)
        {
          /* We use left/right/top/bottom_attach here as they are used in GTK */
          gb_widget_output_int (data, "GtkWidget::left_attach",
                                table_child->left_attach);
          gb_widget_output_int (data, "GtkWidget::right_attach",
                                table_child->right_attach);
          gb_widget_output_int (data, "GtkWidget::top_attach",
                                table_child->top_attach);
          gb_widget_output_int (data, "GtkWidget::bottom_attach",
                                table_child->bottom_attach);
        }
      else
        {
          gb_widget_output_int (data, GbCellX, table_child->left_attach);
          gb_widget_output_int (data, GbCellY, table_child->top_attach);
          gb_widget_output_int (data, GbColSpan, table_child->right_attach
                                - table_child->left_attach);
          gb_widget_output_int (data, GbRowSpan, table_child->bottom_attach
                                - table_child->top_attach);
          property_show_special_child_page (0);
        }
      gb_widget_output_int (data, GbXPad, table_child->xpadding);
      gb_widget_output_int (data, GbYPad, table_child->ypadding);
      gb_widget_output_bool (data, GbXExpand, table_child->xexpand);
      gb_widget_output_bool (data, GbYExpand, table_child->yexpand);
      gb_widget_output_bool (data, GbXShrink, table_child->xshrink);
      gb_widget_output_bool (data, GbYShrink, table_child->yshrink);
      gb_widget_output_bool (data, GbXFill, table_child->xfill);
      gb_widget_output_bool (data, GbYFill, table_child->yfill);

    }
  else if (GTK_IS_BOX (parent) && !GTK_IS_BUTTON_BOX (parent))
    {
      gtk_box_query_child_packing (GTK_BOX (parent), widget,
                                   &expand, &fill, &padding, &pack_type);
      gb_widget_output_int (data, GbPadding, padding);
      gb_widget_output_bool (data, GbExpand, expand);
      gb_widget_output_bool (data, GbFill, fill);

      if (data->action == GB_SAVING)
        {
          /* Save pack type as an enum symbol rather than a bool */
          if (pack_type == GTK_PACK_END)
            {
              gb_widget_output_string (data, GbPack, "GTK_PACK_END");
            }
        }
      else
        {
          gb_widget_output_bool (data, GbPack, pack_type == GTK_PACK_START);
          property_show_special_child_page (1);
        }
    }
#ifdef GLD_HAVE_GTK_1_1
  else if (GTK_IS_PACKER (parent))
    {
      gint i;
      GtkPackerChild *pchild = gb_packer_find_child (GTK_PACKER (parent),
                                                     widget);
      g_return_if_fail (pchild != NULL);

      for (i = 0; GbPackerSideChoices[i]; i++)
        {
          if (GbPackerSideValues[i] == pchild->side)
            gb_widget_output_choice (data, GbPackerSide, i,
                                     GbPackerSideSymbols[i]);
        }
      for (i = 0; GbPackerAnchorChoices[i]; i++)
        {
          if (GbPackerAnchorValues[i] == pchild->anchor)
            gb_widget_output_choice (data, GbPackerAnchor, i,
                                     GbPackerAnchorSymbols[i]);
        }

      gb_widget_output_bool (data, GbPackerExpand,
                             pchild->options & GTK_PACK_EXPAND);
      gb_widget_output_bool (data, GbPackerFillX,
                             pchild->options & GTK_FILL_X);
      gb_widget_output_bool (data, GbPackerFillY,
                             pchild->options & GTK_FILL_Y);
      gb_widget_output_int (data, GbPackerBorder, pchild->border_width);
      gb_widget_output_int (data, GbPackerPadX, pchild->pad_x);
      gb_widget_output_int (data, GbPackerPadY, pchild->pad_y);
      gb_widget_output_int (data, GbPackerIPadX, pchild->i_pad_x);
      gb_widget_output_int (data, GbPackerIPadY, pchild->i_pad_y);

      if (data->action == GB_SAVING)
        {
        }
      else
        {
          property_show_special_child_page (2);
        }
    }
#endif

  if (data->action == GB_SAVING)
    {
      data->indent--;
      save_end_tag (data, "child");
    }
}


static void
get_position_properties (GtkWidget * widget,
                         GbWidgetGetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gboolean sensitive;

  /* Don't bother if widget hasn't been positioned yet. */
  if (data->action == GB_SHOWING && (wdata->flags & GB_INITIAL_EXPOSE))
    {
      if (GTK_IS_WINDOW (widget))
        {
          gb_widget_output_int (data, GbX, wdata->x);
          gb_widget_output_int (data, GbY, wdata->y);
          gb_widget_output_int (data, GbWidth, wdata->width);
          gb_widget_output_int (data, GbHeight, wdata->height);
        }
      return;
    }

  MSG2 ("In get_position_properties Width set:%i Height set:%i",
        wdata->flags & GB_WIDTH_SET, wdata->flags & GB_HEIGHT_SET);

  if (GTK_IS_MENU (widget))
    {
      if (data->action == GB_SHOWING)
        {
          gb_widget_output_int (data, GbX, widget->allocation.x);
          gb_widget_output_int (data, GbY, widget->allocation.y);
          gb_widget_output_int (data, GbWidth, widget->requisition.width);
          gb_widget_output_int (data, GbHeight, widget->requisition.height);
        }
      sensitive = FALSE;
    }
  else if (GTK_IS_WINDOW (widget))
    {
      /* Toplevel window widgets */
      if (data->action == GB_SHOWING)
        {
          gb_widget_output_int (data, GbX, wdata->x);
          gb_widget_output_int (data, GbY, wdata->y);
          gb_widget_output_int (data, GbWidth, wdata->width);
          gb_widget_output_int (data, GbHeight, wdata->height);
        }
      else
        {
          if (wdata->x != -1)
            gb_widget_output_int (data, GbX, wdata->x);
          if (wdata->y != -1)
            gb_widget_output_int (data, GbY, wdata->y);
          if (wdata->flags & GB_WIDTH_SET)
            gb_widget_output_int (data, GbWidth, wdata->width);
          if (wdata->flags & GB_HEIGHT_SET)
            gb_widget_output_int (data, GbHeight, wdata->height);
        }
      sensitive = (GTK_WINDOW (widget)->position == GTK_WIN_POS_NONE);
    }
  else if (widget->parent && GTK_IS_FIXED (widget->parent))
    {
      /* Widgets in fixed containers */
      gb_widget_output_int (data, GbX, widget->allocation.x);
      gb_widget_output_int (data, GbY, widget->allocation.y);
      gb_widget_output_int (data, GbWidth, widget->allocation.width);
      gb_widget_output_int (data, GbHeight, widget->allocation.height);
      sensitive = TRUE;
    }
  else
    {
      /* Widgets in standard containers */
      if (data->action == GB_SHOWING)
        {
          gb_widget_output_int (data, GbX, widget->allocation.x);
          gb_widget_output_int (data, GbY, widget->allocation.y);

          /* If the width or height has been set explicitly we show the size
             set, else we show the current requisition. We always remember
             what we have shown in wdata->width & height so we know if the
             user changes it. */
          if (!(wdata->flags & GB_WIDTH_SET))
            wdata->width = widget->requisition.width;
          gb_widget_output_int (data, GbWidth, wdata->width);

          if (!(wdata->flags & GB_HEIGHT_SET))
            wdata->height = widget->requisition.height;
          gb_widget_output_int (data, GbHeight, wdata->height);

        }
      else
        {
          /* FIXME: only save if user has set it explicitly. */
          if (wdata->flags & GB_WIDTH_SET)
            gb_widget_output_int (data, GbWidth, wdata->width);
          if (wdata->flags & GB_HEIGHT_SET)
            gb_widget_output_int (data, GbHeight, wdata->height);
        }
      sensitive = FALSE;
    }

  /* Only toplevel widgets and children of fixed containers can use x & y */
  if (data->action == GB_SHOWING)
    {
      property_set_sensitive (GbX, sensitive);
      property_set_sensitive (GbY, sensitive);
    }
}


static void
save_style (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GbWidgetData *wdata = data->widget_data;
  g_return_if_fail (wdata != NULL);

  /* Don't save the style if it's the default. */
  if (wdata->gbstyle != gb_widget_default_gb_style)
    {
      /* If the widget is using a named style, just save the name. */
      if (!(wdata->flags & GB_STYLE_IS_UNNAMED))
        {
          save_string (data, GbStyleName, wdata->gbstyle->name);
        }
      else
        {
          gb_widget_save_style (wdata->gbstyle, data, FALSE);
        }
    }
  /* The default propagate value is TRUE. */
  if (!(wdata->flags & GB_STYLE_PROPAGATE))
    save_bool (data, GbStylePropagate, wdata->flags & GB_STYLE_PROPAGATE);
}


void
gb_widget_save_style (GbStyle * gbstyle, GbWidgetGetArgData * data,
                      gboolean save_all)
{
  GtkStyle *style = gbstyle->style;
  GtkStyle *default_style = gb_widget_default_gb_style->style;
  gchar buffer[128];
  gint i;

  save_tag (data, "style");
  data->indent++;

  if (gbstyle->name)
    save_string (data, GbStyleName, gbstyle->name);
  if (save_all
      || strcmp (gbstyle->xlfd_fontname, gb_widget_default_gb_style->xlfd_fontname))
    {
      save_string (data, GbStyleFont, gbstyle->xlfd_fontname);
    }

  /* Colors */
  default_style = gb_widget_default_gb_style->style;
  save_colors (data, save_all, style->fg, default_style->fg, "fg");
  save_colors (data, save_all, style->bg, default_style->bg, "bg");
  save_colors (data, save_all, style->text, default_style->text, "text");
  save_colors (data, save_all, style->base, default_style->base, "base");
  /* FIXME: Delete?
     save_colors(data, save_all, style->light, default_style->light, "light");
     save_colors(data, save_all, style->dark,  default_style->dark,  "dark");
     save_colors(data, save_all, style->mid,   default_style->mid,   "middle");
   */

  /* Background pixmaps */
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      /* Can't use '[' or ']' in XML element names */
      sprintf (buffer, "GtkStyle::%s:%s", GbBgPixmapName, GbStateNames[i]);
      if (gbstyle->bg_pixmap_filenames[i])
        {
          if (save_all
              || gb_widget_default_gb_style->bg_pixmap_filenames[i] == NULL
              || strcmp (gbstyle->bg_pixmap_filenames[i],
                         gb_widget_default_gb_style->bg_pixmap_filenames[i]))
            {
              save_string (data, buffer, gbstyle->bg_pixmap_filenames[i]);
            }
        }
    }

  data->indent--;
  save_tag (data, "/style");
}


static void
save_colors (GbWidgetGetArgData * data, gboolean save_all, GdkColor colors[],
             GdkColor default_colors[], gchar * name)
{
  gint state;
  gchar buf[128];

  for (state = 0; state < GB_NUM_STYLE_STATES; state++)
    {
      if (save_all
          || colors[state].red != default_colors[state].red
          || colors[state].green != default_colors[state].green
          || colors[state].blue != default_colors[state].blue)
        {
          /* Can't use '[' or ']' in XML element names */
          sprintf (buf, "GtkStyle::%s:%s", name, GbStateNames[state]);
          save_color (data, buf, &colors[state]);
        }
    }
}


static void
save_accelerators (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GList *item;
  GbAccelerator *accel;
  gchar key_buffer[128];

  if (data->widget_data == NULL)
    return;
  item = data->widget_data->accelerators;
  while (item)
    {
      accel = (GbAccelerator *) item->data;

      save_tag (data, "Accelerator");
      data->indent++;
      save_string (data, "modifiers",
                   create_modifiers_string (accel->modifiers));

      /* Make sure we don't overflow the buffer. */
      if (strlen (accel->key) < 100)
        {
          strcpy (key_buffer, "GDK_");
          strcat (key_buffer, accel->key);
          save_string (data, "key", key_buffer);
        }

      save_string (data, "signal", accel->signal);
      data->indent--;
      save_tag (data, "/Accelerator");
      item = item->next;
    }
}

/* This creates a string representing the modifiers flags, e.g.
   "GDK_CONTROL_MASK | GDK_SHIFT_MASK". It is used for saving the XML, and
   also for writing the source code. It uses a static buffer. */
static gchar*
create_modifiers_string (guint8 modifier_flags)
{
  static gchar modifiers[128];

  modifiers[0] = '\0';
  if (modifier_flags & GDK_CONTROL_MASK)
    {
      strcat (modifiers, "GDK_CONTROL_MASK");
    }
  if (modifier_flags & GDK_SHIFT_MASK)
    {
      if (strlen (modifiers) > 0)
        strcat (modifiers, " | ");
      strcat (modifiers, "GDK_SHIFT_MASK");
    }
  if (modifier_flags & GDK_MOD1_MASK)
    {
      if (strlen (modifiers) > 0)
        strcat (modifiers, " | ");
      strcat (modifiers, "GDK_MOD1_MASK");
    }
  return modifiers;
}

static void
save_signals (GtkWidget * widget, GbWidgetGetArgData * data)
{
  GList *item;
  GbSignal *signal;

  if (data->widget_data == NULL)
    return;
  item = data->widget_data->signals;
  while (item)
    {
      signal = (GbSignal *) item->data;

      save_tag (data, "Signal");
      data->indent++;
      if (signal->name)
        save_string (data, "name", signal->name);
      if (signal->handler)
        save_string (data, "handler", signal->handler);
      if (signal->object)
        save_string (data, "object", signal->object);
      if (signal->after)
        save_bool (data, "after", signal->after);
      if (signal->data)
        save_string (data, "data", signal->data);
      data->indent--;
      save_tag (data, "/Signal");

      item = item->next;
    }
}



/*************************************************************************
 * Functions for writing C source code
 *************************************************************************/

void
gb_widget_write_source (GtkWidget * widget,
                        GbWidgetWriteSourceData * data)
{
  GtkWidget *parent;
  GbWidget *gbwidget;
  GbWidgetData *widget_data;
  gchar *class_name, *child_name;

  /* This is a temporary kludge so that the code GtkDialogs is OK.
     We stop the source code for the action_area from being written. */
  child_name = gtk_object_get_data (GTK_OBJECT (widget), GB_CHILD_NAME_KEY);
  if (child_name && !strcmp (child_name, "Dialog:action_area")
      && data->create_widget)
    return;

  parent = data->parent;

  class_name = gtk_type_name (GTK_OBJECT_TYPE (widget));

  widget_data = gtk_object_get_data (GTK_OBJECT (widget), GB_WIDGET_DATA_KEY);
  /* If this isn't a gbwidget, skip it. */
  if (widget_data)
    {
      gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table, class_name);
      g_return_if_fail (gbwidget != NULL);

      data->standard_widget
        = (GtkWidget *) g_hash_table_lookup (data->standard_widgets, class_name);
      if (data->standard_widget == NULL)
        {
          data->standard_widget = gtk_type_new (gtk_type_from_name (class_name));
          g_hash_table_insert (data->standard_widgets, class_name,
                               data->standard_widget);
        }

      data->wname = source_create_valid_identifier (gtk_widget_get_name (widget));
      data->widget_data = widget_data;
      data->write_children = TRUE;
      if (gbwidget->gb_widget_write_source)
        (gbwidget->gb_widget_write_source) (widget, data);
      else
        source_add (data, "  /* Skipping %s: unimplemented. */\n", class_name);

      /* Make sure there is a blank line after each widget, for readability. */
      if (data->buffer_pos > 2
          && (data->buffer[data->buffer_pos - 1] != '\n'
              || data->buffer[data->buffer_pos - 2] != '\n'))
        source_add (data, "\n");
      g_free (data->wname);
      data->wname = NULL;
      data->parent = widget;
    }

  /* If widget is a container, recursively write source for children.
     We need to reset the parent after the children have been written. */
  if (GTK_IS_CONTAINER (widget) && data->write_children)
    {
      data->create_widget = TRUE;
      gtk_container_foreach (GTK_CONTAINER (widget),
                             (GtkCallback) gb_widget_write_source, data);

      /* SPECIAL CODE: output code to create notebook tabs. */
      if (GTK_IS_NOTEBOOK (widget))
        notebook_tabs_foreach (GTK_NOTEBOOK (widget),
                               (GtkCallback) gb_widget_write_source, data);

      /* SPECIAL CODE: output menuitem submenus. */
      if (GTK_IS_MENU_ITEM (widget)
          && GTK_MENU_ITEM (widget)->submenu)
        gb_widget_write_source (GTK_MENU_ITEM (widget)->submenu, data);
    }

  /* We need to reset some of the members of the GbWidgetWriteSourceData struct
     so that they work OK for the next sibling. */
  data->parent = parent;
}


/* This is called by each GbWidget's write_source function to write all the
   common source code, including code to add the widget to its parent. */
void
gb_widget_write_standard_source (GtkWidget * widget,
                                 GbWidgetWriteSourceData * data)
{
  GbWidgetData *wdata = data->widget_data;
  gint i, width, height, can_focus, can_default;

  source_add_decl (data, "  GtkWidget *%s;\n", data->wname);

  if (data->set_widget_names)
    source_add (data, "  gtk_widget_set_name (%s, \"%s\");\n", data->wname,
                data->wname);

  if (data->use_widget_hash)
    source_add (data, "  gtk_object_set_data (GTK_OBJECT (%s), \"%s\", %s);\n",
                data->component_name, source_make_string (data->wname),
                data->wname);

  if (widget->parent)
    {
      if (wdata->flags & GB_VISIBLE)
        source_add (data, "  gtk_widget_show (%s);\n", data->wname);

      /* Output code to add widget to parent. */
      gb_widget_write_add_child_source (widget, data);
    }
  /* SPECIAL CODE: menus are attached to the parent menu items. */
  else if (GTK_IS_MENU (widget))
    {
      if (gtk_menu_get_attach_widget (GTK_MENU (widget)))
        {
          MSG1 ("widget %s is a menu with a parent", data->wname);
          gb_widget_write_add_child_source (widget, data);
        }
    }

  if ((!widget->parent || !GTK_IS_FIXED (widget->parent))
      && (wdata->x != -1 || wdata->y != -1))
    {
      source_add (data, "  gtk_widget_set_uposition (%s, %i, %i);\n",
                  data->wname, wdata->x, wdata->y);
    }

  if (widget->parent && GTK_IS_FIXED (widget->parent))
    {
      width = widget->allocation.width;
      height = widget->allocation.height;
    }
  else
    {
      width = wdata->flags & GB_WIDTH_SET ? wdata->width : -1;
      height = wdata->flags & GB_HEIGHT_SET ? wdata->height : -1;
    }
  if (width != -1 || height != -1)
    {
      /* GTK BUG WORKAROUND - a combo should manage the size of its entry. */
      if (GTK_IS_COMBO (widget))
        source_add(data,
                   "  gtk_widget_set_usize (GTK_COMBO (%s)->entry, %i, %i);\n",
                   data->wname, width - 16 < 0 ? -1 : width - 16, height);
      source_add (data, "  gtk_widget_set_usize (%s, %i, %i);\n",
                  data->wname, width, height);
    }


  if (GTK_IS_CONTAINER (widget))
    {
      if (GTK_CONTAINER (widget)->border_width != 0)
        {
          source_add (data,
                 "  gtk_container_border_width (GTK_CONTAINER (%s), %i);\n",
                      data->wname, GTK_CONTAINER (widget)->border_width);
        }
    }


  if (!(wdata->flags & GB_SENSITIVE))
    source_add (data, "  gtk_widget_set_sensitive (%s, FALSE);\n", data->wname);

  can_focus = GTK_WIDGET_CAN_FOCUS (widget);
  if (can_focus != GTK_WIDGET_CAN_FOCUS (data->standard_widget))
    {
      if (can_focus)
        source_add (data, "  GTK_WIDGET_SET_FLAGS (%s, GTK_CAN_FOCUS);\n",
                    data->wname);
      else
        source_add (data, "  GTK_WIDGET_UNSET_FLAGS (%s, GTK_CAN_FOCUS);\n",
                    data->wname);
    }
  can_default = GTK_WIDGET_CAN_DEFAULT (widget);
  if (can_default != GTK_WIDGET_CAN_DEFAULT (data->standard_widget))
    {
      if (can_default)
        source_add (data, "  GTK_WIDGET_SET_FLAGS (%s, GTK_CAN_DEFAULT);\n",
                    data->wname);
      else
        source_add (data, "  GTK_WIDGET_UNSET_FLAGS (%s, GTK_CAN_DEFAULT);\n",
                    data->wname);
    }

  if (wdata->flags & GB_GRAB_FOCUS)
    source_add (data, "  gtk_widget_grab_focus (%s);\n", data->wname);
  if (wdata->flags & GB_GRAB_DEFAULT)
    source_add (data, "  gtk_widget_grab_default (%s);\n", data->wname);

  if (wdata->tooltip)
    {
      data->need_tooltips = TRUE;
      source_add (data, "  gtk_tooltips_set_tip (tooltips, %s, \"%s\", NULL);\n",
                  data->wname, source_make_string (wdata->tooltip));
    }

  if (!GTK_WIDGET_NO_WINDOW (widget))
    {
      GdkExtensionMode ext_mode = gtk_widget_get_extension_events (widget);
      gboolean written_first = FALSE;

      if (wdata->events)
        {
          source_add (data, "  gtk_widget_set_events (%s, ", data->wname);
          for (i = 0; i < GB_EVENT_MASKS_COUNT; i++)
            {
              if (wdata->events & GbEventMaskValues[i])
                {
                  if (!written_first)
                    source_add (data, "%s", GbEventMaskSymbols[i]);
                  else
                    source_add (data, " | %s", GbEventMaskSymbols[i]);
                  written_first = TRUE;
                }
            }
          source_add (data, ");\n");
        }

      if (ext_mode != GDK_EXTENSION_EVENTS_NONE)
        {
          for (i = 0; GbExtensionModeChoices[i]; i++)
            {
              if (GbExtensionModeValues[i] == ext_mode)
                source_add (data, "  gtk_widget_set_extension_events (%s, %s);\n",
                            data->wname, GbExtensionModeSymbols[i]);
            }
        }
    }

  gb_widget_write_signals_source (widget, data);
  gb_widget_write_accelerators_source (widget, data);
}


void
gb_widget_write_add_child_source (GtkWidget * widget,
                                  GbWidgetWriteSourceData * data)
{
  GtkWidget *parent;
  gchar *parent_name;

  /* If the widget is created automatically by its parent, we don't need
     to add source to add it. */
  if (!data->create_widget)
    return;

  /* SPECIAL CODE: to handle menus. */
  if (GTK_IS_MENU (widget))
    parent = gtk_menu_get_attach_widget (GTK_MENU (widget));
  else
    parent = data->parent;

  parent_name = gtk_widget_get_name (parent);
  parent_name = source_create_valid_identifier (parent_name);

  MSG2 ("Adding %s to %s", data->wname, parent_name);

  if (GTK_IS_TABLE (parent))
    {
      gchar xoptions[48], yoptions[48];
      GtkTableChild *tchild = gb_table_find_child (GTK_TABLE (parent), widget);
      g_return_if_fail (tchild != NULL);

      xoptions[0] = yoptions[0] = '\0';
      if (tchild->xexpand)
        strcpy (xoptions, "GTK_EXPAND");
      if (tchild->xshrink)
        {
          if (xoptions[0] != '\0')
            strcat (xoptions, " | ");
          strcat (xoptions, "GTK_SHRINK");
        }
      if (tchild->xfill)
        {
          if (xoptions[0] != '\0')
            strcat (xoptions, " | ");
          strcat (xoptions, "GTK_FILL");
        }
      if (xoptions[0] == '\0')
        strcpy (xoptions, "0");

      if (tchild->yexpand)
        strcpy (yoptions, "GTK_EXPAND");
      if (tchild->yshrink)
        {
          if (yoptions[0] != '\0')
            strcat (yoptions, " | ");
          strcat (yoptions, "GTK_SHRINK");
        }
      if (tchild->yfill)
        {
          if (yoptions[0] != '\0')
            strcat (yoptions, " | ");
          strcat (yoptions, "GTK_FILL");
        }
      if (yoptions[0] == '\0')
        strcpy (yoptions, "0");

      source_add (data,
                  "  gtk_table_attach (GTK_TABLE (%s), %s, %i, %i, %i, %i,\n"
                  "                    %s, %s, %i, %i);\n",
                  parent_name, data->wname,
                  tchild->left_attach, tchild->right_attach,
                  tchild->top_attach, tchild->bottom_attach,
                  xoptions, yoptions, tchild->xpadding, tchild->ypadding);
    }
  else if (GTK_IS_BOX (parent) && !GTK_IS_BUTTON_BOX (parent))
    {
      gint expand, fill, padding;
      GtkPackType pack_type;

      gtk_box_query_child_packing (GTK_BOX (parent), widget,
                                   &expand, &fill, &padding, &pack_type);
      if (pack_type == GTK_PACK_START)
        {
          source_add (data,
                   "  gtk_box_pack_start (GTK_BOX (%s), %s, %s, %s, %i);\n",
                      parent_name, data->wname,
               expand ? "TRUE" : "FALSE", fill ? "TRUE" : "FALSE", padding);
        }
      else
        {
          source_add (data,
                      "  gtk_box_pack_end (GTK_BOX (%s), %s, %s, %s, %i);\n",
                      parent_name, data->wname,
               expand ? "TRUE" : "FALSE", fill ? "TRUE" : "FALSE", padding);
        }
    }
#ifdef GLD_HAVE_GTK_1_1
  else if (GTK_IS_PACKER (parent))
    {
      gchar *side, *anchor, options[48];
      gint i;
      GtkPackerChild *pchild = gb_packer_find_child (GTK_PACKER (parent),
                                                     widget);
      g_return_if_fail (pchild != NULL);

      for (i = 0; GbPackerSideChoices[i]; i++)
        {
          if (GbPackerSideValues[i] == pchild->side)
            side = GbPackerSideSymbols[i];
        }
      for (i = 0; GbPackerAnchorChoices[i]; i++)
        {
          if (GbPackerAnchorValues[i] == pchild->anchor)
            anchor = GbPackerAnchorSymbols[i];
        }
      options[0] = '\0';
      if (pchild->options & GTK_PACK_EXPAND)
        strcpy (options, "GTK_PACK_EXPAND");
      if (pchild->options & GTK_FILL_X)
        {
          if (options[0] != '\0')
            strcat (options, " | ");
          strcat (options, "GTK_FILL_X");
        }
      if (pchild->options & GTK_FILL_Y)
        {
          if (options[0] != '\0')
            strcat (options, " | ");
          strcat (options, "GTK_FILL_Y");
        }
      if (options[0] == '\0')
        strcpy (options, "0");

      source_add (data,
                  "  gtk_packer_add (GTK_PACKER (%s), %s, %s,\n"
                  "                  %s, %s,\n"
                  "                  %i, %i, %i, %i, %i);\n",
                  parent_name, data->wname, side, anchor, options,
                  pchild->border_width, pchild->pad_x, pchild->pad_y,
                  pchild->i_pad_x, pchild->i_pad_y);
    }
#endif
  else if (GTK_IS_FIXED (parent))
    {
      source_add (data, "  gtk_fixed_put (GTK_FIXED (%s), %s, %i, %i);\n",
                  parent_name, data->wname,
                  widget->allocation.x, widget->allocation.y);

    }
  else if (GTK_IS_CLIST (parent))
    {
      /* See if this is a title widget. */
      gchar *child_name = gtk_object_get_data (GTK_OBJECT (widget),
                                               GB_CHILD_NAME_KEY);
      MSG ("Source:parent is clist");
      /* FIXME: This should use CListTitle from gbclist.c */
      if (child_name && !strcmp (child_name, "CList:title"))
        {
          /* We store the last column title written in 'last_child' */
          gint col = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (parent),
                                                           "last_child"));
          source_add (data,
                "  gtk_clist_set_column_widget (GTK_CLIST (%s), %i, %s);\n",
                      parent_name, col + 1, data->wname);

          gtk_object_set_data (GTK_OBJECT (parent), "last_child",
                               GINT_TO_POINTER (col + 1));
        }
      else
        {
          source_add (data, "  gtk_container_add (GTK_CONTAINER (%s), %s);\n",
                      parent_name, data->wname);
        }
    }
  else if (GTK_IS_NOTEBOOK (parent))
    {
      /* See if this is a notebook tab widget. */
      gchar *child_name = gtk_object_get_data (GTK_OBJECT (widget),
                                               GB_CHILD_NAME_KEY);
      MSG ("Source:parent is notebook");
      /* FIXME: This should use NotebookTab from gbnotebook.c */
      if (child_name && !strcmp (child_name, "Notebook:tab"))
        {
          /* We store the last tab written in 'last_child' */
          gint col = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (parent),
                                                           "last_child"));
          /* We use our own function to add the tab to the notebook. */
          source_add (data,
                      "  set_notebook_tab (%s, %i, %s);\n",
                      parent_name, col + 1, data->wname);

          gtk_object_set_data (GTK_OBJECT (parent), "last_child",
                               GINT_TO_POINTER (col + 1));
        }
      else
        {
          source_add (data, "  gtk_container_add (GTK_CONTAINER (%s), %s);\n",
                      parent_name, data->wname);
        }
    }
  else if (GTK_IS_MENU_ITEM (parent) && GTK_IS_MENU (widget))
    {
      MSG ("Source:parent is a menu item and widget is a menu");
      source_add (data,
                  "  gtk_menu_item_set_submenu (GTK_MENU_ITEM (%s), %s);\n",
                  parent_name, data->wname);

    }
  else
    {
      source_add (data, "  gtk_container_add (GTK_CONTAINER (%s), %s);\n",
                  parent_name, data->wname);
    }
  g_free (parent_name);
}


void
gb_widget_write_signals_source (GtkWidget * widget,
                                GbWidgetWriteSourceData * data)
{
  GList *item = data->widget_data->signals;
  GbSignal *signal;
  guint signal_id;
  GtkSignalQuery *query_info;
  gint param, i;
  gchar buffer[1024];
  gchar *ret_type, *pos, *type_name, *arg_name, *object_name, *object_arg;
  gchar *object_arg_start;
  gboolean is_pointer;
  gint param_num, widget_num, event_num, callback_num;
  gint *arg_num;
  gchar *handler, **arg_names;

  while (item)
    {
      signal = (GbSignal *) item->data;
      item = item->next;
      param_num = 1;
      widget_num = event_num = callback_num = 0;

      if (!signal->name)
        {
          /* FIXME: store warnings */
          g_warning ("Signal name not set");
          continue;
        }

      if (!signal->handler)
        {
          /* FIXME: store warnings */
          g_warning ("Signal handler not set");
          continue;

        }

      signal_id = gtk_signal_lookup (signal->name, GTK_OBJECT_TYPE (widget));
      if (signal_id == 0)
        {
          /* FIXME: store warnings */
          g_warning ("Signal unknown");
          continue;
        }

      /* Make sure handler function name is valid. */
      handler = source_create_valid_identifier (signal->handler);

      /* Output code to connect signal. */
      if (signal->object)
        {
          if (signal->after)
            {
              source_add (data,
              "  gtk_signal_connect_object_after (GTK_OBJECT (%s), \"%s\",\n"
                "                                   GTK_SIGNAL_FUNC (%s),\n"
                   "                                   GTK_OBJECT (%s));\n",
                        data->wname, signal->name, handler, signal->object);
            }
          else
            {
              source_add (data,
                   "  gtk_signal_connect_object (GTK_OBJECT (%s), \"%s\",\n"
                      "                             GTK_SIGNAL_FUNC (%s),\n"
                          "                             GTK_OBJECT (%s));\n",
                        data->wname, signal->name, handler, signal->object);
            }
        }
      else
        {
          if (signal->after)
            {
              source_add (data,
                    "  gtk_signal_connect_after (GTK_OBJECT (%s), \"%s\",\n"
                       "                            GTK_SIGNAL_FUNC (%s),\n"
                          "                            %s);\n",
                          data->wname, signal->name, handler,
                          signal->data ? signal->data : "NULL");
            }
          else
            {
              source_add (data,
                          "  gtk_signal_connect (GTK_OBJECT (%s), \"%s\",\n"
                          "                      GTK_SIGNAL_FUNC (%s),\n"
                          "                      %s);\n",
                          data->wname, signal->name, handler,
                          signal->data ? signal->data : "NULL");
            }
        }

      /* Output empty signal handler. */
      query_info = gtk_signal_query (signal_id);
      if (query_info == NULL)
        {
          g_warning ("Couldn't query signal");
          g_free (handler);
          continue;
        }

      /* Output the return type and function name. */
      ret_type = get_type_name (query_info->return_val, &is_pointer);
      pos = buffer;
      sprintf (pos, "%s%s\n%s", ret_type, is_pointer ? "*" : "", handler);
      pos += strlen (pos);

      if (strlen (handler) >= GB_PARAM_INDENT - 1)
        {
          *pos++ = '\n';
          for (i = 0; i < GB_PARAM_INDENT; i++)
            *pos++ = ' ';
        }
      else
        {
          for (i = 0; i < GB_PARAM_INDENT - strlen (handler) - 1; i++)
            *pos++ = ' ';
        }

      /* Output the signal object type and the argument name. We assume the
         type is a pointer - I think that is OK. We remove "Gtk" and convert
         to lower case for the argument name. */
      object_name = gtk_type_name (query_info->object_type);
      sprintf (pos, "(%s ", object_name);
      pos += strlen (pos);
      if (strlen (object_name) + 1 < GB_PARAM_TYPE_WIDTH)
        {
          for (i = 0; i < GB_PARAM_TYPE_WIDTH - strlen (object_name) - 1; i++)
            *pos++ = ' ';
        }
      object_arg = (!strncmp (object_name, "Gtk", 3)) ? object_name + 3
        : object_name;
      object_arg_start = pos;
      sprintf (pos, "*%s", object_arg);
      pos += strlen (pos);
      g_strdown (object_arg_start);
      if (!strcmp (object_arg_start, "widget"))
        widget_num++;
      sprintf (pos, ",\n");
      pos += strlen (pos);

      /* Output the signal parameters. */
      arg_names = lookup_signal_arg_names (object_name, signal->name);

      for (param = 0; param < query_info->nparams; param++)
        {
          for (i = 0; i < GB_PARAM_INDENT; i++)
            *pos++ = ' ';

          if (arg_names)
            {
              sprintf (pos, "%s,\n", arg_names[param]);
              pos += strlen (pos);
            }
          else
            {
              type_name = get_type_name (query_info->params[param], &is_pointer);
              /* Most arguments to the callback are called "arg1", "arg2", etc.
                 GtkWidgets are called "widget", "widget2", ...
                 GdkEvents are called "event", "event2", ...
                 GtkCallbacks are called "callback", "callback2", ... */
              if (!strcmp (type_name, "GtkWidget"))
                {
                  arg_name = "widget";
                  arg_num = &widget_num;
                }
              else if (!strcmp (type_name, "GdkEvent"))
                {
                  type_name = get_gdk_event (signal->name);
                  arg_name = "event";
                  arg_num = &event_num;
                  is_pointer = TRUE;
                }
              else if (!strcmp (type_name, "GtkCallback"))
                {
                  arg_name = "callback";
                  arg_num = &callback_num;
                }
              else
                {
                  arg_name = "arg";
                  arg_num = &param_num;
                }
              sprintf (pos, "%s ", type_name);
              pos += strlen (pos);
              if (strlen (type_name) + 1 < GB_PARAM_TYPE_WIDTH)
                {
                  for (i = 0; i < GB_PARAM_TYPE_WIDTH - strlen (type_name) - 1;
                       i++)
                    *pos++ = ' ';
                }
              if (!arg_num || *arg_num == 0)
                sprintf (pos, "%s%s,\n", is_pointer ? "*" : " ", arg_name);
              else
                sprintf (pos, "%s%s%i,\n", is_pointer ? "*" : " ", arg_name,
                         *arg_num);
              pos += strlen (pos);

              if (arg_num)
                *arg_num += 1;
            }
        }
      for (i = 0; i < GB_PARAM_INDENT; i++)
        *pos++ = ' ';
      sprintf (pos, "gpointer         user_data)");
      pos += strlen (pos);

      /* return FALSE from the handler if the return type is bool. */
      signal_source_add (data, "%s\n{\n\n%s}\n\n", buffer,
               query_info->return_val == GTK_TYPE_BOOL ? "  return FALSE;\n"
               : "");

      signal_header_add (data, "%s;\n", buffer);

      g_free (query_info);
      g_free (handler);
    }
}


static gchar *
get_type_name (GtkType type, gboolean * is_pointer)
{
  static gchar *GbTypeNames[] =
  {
    "char", "gchar",
    "bool", "gboolean",
    "int", "gint",
    "uint", "guint",
    "long", "glong",
    "ulong", "gulong",
    "float", "gfloat",
    "double", "gdouble",
    "string", "gchar",
    "enum", "gint",
    "flags", "gint",
    "boxed", "gpointer",
    "foreign", "gpointer",
    "callback", "GtkCallback",  /* ?? */
    "args", "gpointer",

    "pointer", "gpointer",
    "signal", "gpointer",
    "c_callback", "GtkCallback",        /* ?? */

    NULL
  };

  GtkType parent_type;
  gchar *type_name, *parent_type_name;
  gint i;

  *is_pointer = FALSE;
  type_name = gtk_type_name (type);
  for (i = 0; GbTypeNames[i]; i += 2)
    {
      if (!strcmp (type_name, GbTypeNames[i]))
        {
          if (!strcmp (type_name, "string"))
            *is_pointer = TRUE;
          return GbTypeNames[i + 1];
        }
    }

  for (;;)
    {
      parent_type = gtk_type_parent (type);
      if (parent_type == 0)
        break;
      type = parent_type;
    }
  parent_type_name = gtk_type_name (type);
  if (!strcmp (parent_type_name, "GtkObject")
      || !strcmp (parent_type_name, "boxed"))
    *is_pointer = TRUE;

  return type_name;
}


static gchar *
get_gdk_event (gchar * signal_name)
{
  static gchar *GbGDKEvents[] =
  {
    "button_press_event", "GdkEventButton",
    "button_release_event", "GdkEventButton",
    "motion_notify_event", "GdkEventMotion",
    "delete_event", "GdkEvent",
    "destroy_event", "GdkEvent",
    "expose_event", "GdkEventExpose",
    "key_press_event", "GdkEventKey",
    "key_release_event", "GdkEventKey",
    "enter_notify_event", "GdkEventCrossing",
    "leave_notify_event", "GdkEventCrossing",
    "configure_event", "GdkEventConfigure",
    "focus_in_event", "GdkEventFocus",
    "focus_out_event", "GdkEventFocus",
    "map_event", "GdkEvent",
    "unmap_event", "GdkEvent",
    "property_notify_event", "GdkEventProperty",
    "selection_clear_event", "GdkEventSelection",
    "selection_request_event", "GdkEventSelection",
    "selection_notify_event", "GdkEventSelection",
    "proximity_in_event", "GdkEventProximity",
    "proximity_out_event", "GdkEventProximity",
    "drag_begin_event", "GdkEventDragBegin",
    "drag_request_event", "GdkEventDragRequest",
    "drag_end_event", "GdkEventDragRequest",
    "drop_enter_event", "GdkEventDropEnter",
    "drop_leave_event", "GdkEventDropLeave",
    "drop_data_available_event", "GdkEventDropDataAvailable",
    "other_event", "GdkEventOther",
    "client_event", "GdkEventClient",
    "no_expose_event", "GdkEventNoExpose",
    NULL
  };

  gint i;

  for (i = 0; GbGDKEvents[i]; i += 2)
    {
      if (!strcmp (signal_name, GbGDKEvents[i]))
        return GbGDKEvents[i + 1];
    }
  return "GdkEvent";
}


/* This returns argument names to use for some known GTK signals. */
/* Note: Could possibly return the types to use as well, since several of
   them could be better, e.g. "GtkOrientation" instead of "gint". */
static gchar **
lookup_signal_arg_names (gchar * type, gchar * signal_name)
{
  /* Each arg array starts with the object type name and the signal name,
     and then signal arguments follow. Note that the spacing is hardcoded. */
  static gchar *GbArgTable[][6] =
  {
    {"GtkCList", "select_row",
     "gint             row",
     "gint             column",
     "GdkEventButton  *event"},
    {"GtkCList", "unselect_row",
     "gint             row",
     "gint             column",
     "GdkEventButton  *event"},
    {"GtkCList", "click_column",
     "gint             column"},
    {"GtkNotebook", "switch_page",
     "GtkNotebookPage *page",
     "gint             page_num"},
    {"GtkStatusBar", "text_pushed",
     "guint            context_id",
     "gchar           *text"},
    {"GtkStatusBar", "text_popped",
     "guint            context_id",
     "gchar           *text"},
    {"GtkTipsQuery", "widget_entered",
     "GtkWidget       *widget",
     "gchar           *tip_text",
     "gchar           *tip_private"},
    {"GtkTipsQuery", "widget_selected",
     "GtkWidget       *widget",
     "gchar           *tip_text",
     "gchar           *tip_private",
     "GdkEventButton  *event"},
    {"GtkToolbar", "orientation_changed",
     "GtkOrientation   orientation"},
    {"GtkToolbar", "style_changed",
     "GtkToolbarStyle  style"},
    {"GtkWidget", "draw",
     "GdkRectangle    *area"},
    {"GtkWidget", "size_request",
     "GtkRequisition  *requisition"},
    {"GtkWidget", "size_allocate",
     "GtkAllocation   *allocation"},
    {"GtkWidget", "state_changed",
     "GtkStateType     state"},
    {"GtkWidget", "style_set",
     "GtkStyle        *previous_style"},
  /* FIXME: Update for GTK 1.1 */
    {"GtkWidget", "install_accelerator",
     "gchar           *signal_name",
     "gchar            key",
     "gint             modifiers"},
    {"GtkWidget", "remove_accelerator",
     "gchar           *signal_name"},
    {"GtkWindow", "move_resize",
     "gint            *x",
     "gint            *y",
     "gint             width",
     "gint             height"},
    {"GtkWindow", "set_focus",
     "GtkWidget       *widget"},

    {NULL}
  };

  gint i;

  for (i = 0; GbArgTable[i][0]; i++)
    {
      if (!strcmp (type, GbArgTable[i][0])
          && !strcmp (signal_name, GbArgTable[i][1]))
        return &GbArgTable[i][2];
    }
  return NULL;
}


void
gb_widget_write_accelerators_source (GtkWidget * widget,
                                     GbWidgetWriteSourceData * data)
{
  GList *item = data->widget_data->accelerators;
  GbAccelerator *accel;

  while (item)
    {
      accel = (GbAccelerator *) item->data;
      item = item->next;

      /* If this is the first accelerator, create the accel group/table. */
      if (data->need_accel_group == FALSE)
        {
          data->need_accel_group = TRUE;

#ifdef GLD_HAVE_GTK_1_1
          source_add (data, "  accel_group = gtk_accel_group_new ();\n");
          if (GTK_IS_MENU (data->component))
            source_add (data,
                        "  gtk_menu_set_accel_group (GTK_MENU (%s), accel_group);\n",
                        data->component_name);
          else
            source_add (data,
                        "  gtk_window_add_accel_group (GTK_WINDOW (%s), accel_group);\n",
                        data->component_name);
#else
          source_add (data,
                      "  accelerator_table = gtk_accelerator_table_new ();\n");
          if (GTK_IS_MENU (data->component))
            source_add (data,
                        "  gtk_menu_set_accelerator_table (GTK_MENU (%s), accelerator_table);\n",
                        data->component_name);
          else
            source_add (data,
                        "  gtk_window_add_accelerator_table (GTK_WINDOW (%s), accelerator_table);\n",
                        data->component_name);
#endif
        }

#ifdef GLD_HAVE_GTK_1_1
      source_add (data,
                  "  gtk_widget_add_accelerator (%s, \"%s\", accel_group,\n"
                  "                              GDK_%s, %s, GTK_ACCEL_VISIBLE);\n",
                  data->wname, accel->signal, accel->key,
                  create_modifiers_string (accel->modifiers));
#else
      source_add (data,
                  "  gtk_widget_install_accelerator (%s, accelerator_table, \"%s\",\n"
                  "                                  GDK_%s, %s);\n",
                  data->wname, accel->signal, accel->key,
                  create_modifiers_string (accel->modifiers));
#endif


    }
}



/*************************************************************************
 * Functions for replacing widgets in the user interface
 *************************************************************************/

/*
 * Replace a child with a new child. Used to replace placeholders with
 * a widget when adding widgets, and also to replace widgets with
 * placeholders when deleting.
 * NOTE: gbwidgets do not currently have their own function for this,
 * as there are only a few special cases (I hope!)
 */
void
gb_widget_replace_child (GtkWidget * widget,
                         GtkWidget * current_child,
                         GtkWidget * new_child)
{
  gchar *child_name;

  /* Stop resizes until child is replaced */
  gtk_container_block_resize (GTK_CONTAINER (widget));

  /* We move the child name to the new widget. */
  child_name = gtk_object_get_data (GTK_OBJECT (current_child),
                                    GB_CHILD_NAME_KEY);
  if (child_name)
    gtk_object_set_data (GTK_OBJECT (new_child),
                         GB_CHILD_NAME_KEY, g_strdup (child_name));

  if (GTK_IS_BIN (widget))
    {
      /* For a bin, we just delete the existing child and add the new one. */
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_container_add (GTK_CONTAINER (widget), new_child);

    }
  else if (GTK_IS_BOX (widget))
    {
      /* For a box, we find out the position of the current child, delete it,
         add the new one, and move it into position with reorder_child(). */
      gint expand, fill, padding;
      GtkPackType pack_type;
      gint pos = gb_box_get_pos (GTK_BOX (widget), current_child);
      g_return_if_fail (pos != -1);
      gtk_box_query_child_packing (GTK_BOX (widget), current_child, &expand, &fill,
                                   &padding, &pack_type);
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_container_add (GTK_CONTAINER (widget), new_child);
      gtk_box_set_child_packing (GTK_BOX (widget), new_child, expand, fill,
                                 padding, pack_type);
      gtk_box_reorder_child (GTK_BOX (widget), new_child, pos);

    }
  else if (GTK_IS_LIST (widget))
    {
      /* For a list, we find out the position of the current child, delete it,
         and add the new one at the same position. */
      gint pos = gtk_list_child_position (GTK_LIST (widget), current_child);
      GList glist =
      {NULL, NULL, NULL};
      glist.data = current_child;
      gtk_list_remove_items (GTK_LIST (widget), &glist);
      glist.data = new_child;
      gtk_list_insert_items (GTK_LIST (widget), &glist, pos);

    }
  else if (GTK_IS_NOTEBOOK (widget))
    {
      /* For a notebook, we find out the position of the current child, delete
         it, and add the new one at the same position. If the current_child is
         a notebook tab, just replace it. */
      GtkWidget *tab_label, *notebook_page;
      gint pos;
      GtkNotebookPage *page = find_notebook_page (widget, current_child, &pos);
      g_return_if_fail (page != NULL);
      if (page->child == current_child)
        {
          tab_label = page->tab_label;
          gtk_widget_ref (tab_label);
          gtk_notebook_remove_page (GTK_NOTEBOOK (widget), pos);
          gtk_notebook_insert_page (GTK_NOTEBOOK (widget), new_child, tab_label,
                                    pos);
          gtk_notebook_set_page (GTK_NOTEBOOK (widget), pos);
          gtk_widget_unref (tab_label);
        }
      else
        {
          notebook_page = page->child;
          gtk_widget_ref (notebook_page);
          gtk_notebook_remove_page (GTK_NOTEBOOK (widget), pos);
          gtk_notebook_insert_page (GTK_NOTEBOOK (widget), notebook_page, new_child,
                                    pos);
          gtk_notebook_set_page (GTK_NOTEBOOK (widget), pos);
          gtk_widget_unref (notebook_page);
        }

    }
  else if (GTK_IS_PANED (widget))
    {
      /* For paned, we find out the position of the current child, delete it,
         and add the new one at the same position. */
      gint pos = (GTK_PANED (widget)->child1 == current_child) ? 1 : 2;
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      if (pos == 1)
        gtk_paned_add1 (GTK_PANED (widget), new_child);
      else
        gtk_paned_add2 (GTK_PANED (widget), new_child);

    }
  else if (GTK_IS_TABLE (widget))
    {
      /* For a table, we find out the position & size of the current child,
         delete it, and add the new one in the same place. */
      gint left, right, top, bottom;
      GtkTableChild *tchild = gb_table_find_child (GTK_TABLE (widget),
                                                   current_child);
      g_return_if_fail (tchild != NULL);
      left = tchild->left_attach;
      right = tchild->right_attach;
      top = tchild->top_attach;
      bottom = tchild->bottom_attach;
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_table_attach_defaults (GTK_TABLE (widget), new_child,
                                 left, right, top, bottom);

    }
  else if (GTK_IS_TREE (widget))
    {
      /* For a tree, we find out the position of the current child, delete it,
         and add the new one at the same position. */
      gint pos = gtk_tree_child_position (GTK_TREE (widget), current_child);
      GList glist =
      {NULL, NULL, NULL};
      glist.data = current_child;
      gtk_tree_remove_items (GTK_TREE (widget), &glist);
      gtk_tree_insert (GTK_TREE (widget), new_child, pos);

    }
  else if (GTK_IS_CLIST (widget))
    {
      /* For a clist, we check if the widget is a column title and replace if so.
         If not, we can ?? */
      g_warning ("replacing child of clist - not implemented yet\n");

    }
  else if (GTK_IS_BUTTON (widget))
    {
      /* For a button, we just delete the existing child and add the new one. */
      gtk_container_remove (GTK_CONTAINER (widget), current_child);
      gtk_container_add (GTK_CONTAINER (widget), new_child);

    }
  else if (GTK_IS_CONTAINER (widget))
    {
      /* General code for container - has to remove all children and add back
         NOTE: this may not work for specialised containers.
         NOTE: need to ref widgets? */
      g_warning (_("replacing child of container - not implemented yet\n"));
    }

  /* Now allow a resize */
  gtk_container_unblock_resize (GTK_CONTAINER (widget));
}



/*************************************************************************
 * Functions for showing/hiding tooltips of the widgets
 *************************************************************************/

gboolean
gb_widget_get_show_tooltips ()
{
  return GTK_TOOLTIPS (gb_widget_tooltips)->enabled;
}


void
gb_widget_toggle_show_tooltips (GtkWidget * widget, gpointer data)
{
  if (GTK_CHECK_MENU_ITEM (widget)->active)
    gtk_tooltips_enable (gb_widget_tooltips);
  else
    gtk_tooltips_disable (gb_widget_tooltips);
}


void
gb_widget_reset_tooltips ()
{
  gtk_object_destroy (GTK_OBJECT (gb_widget_tooltips));
  gb_widget_tooltips = gtk_tooltips_new ();
}



/*************************************************************************
 * Functions for creating/copying/destroying GbStyleData structs
 *************************************************************************/

static void
reset_gb_styles_callback (gchar * key, GbStyle * gbstyle, gpointer data)
{
  gb_widget_destroy_gb_style (gbstyle, FALSE);
}

void
gb_widget_reset_gb_styles ()
{
  if (gb_style_hash)
    {
      g_hash_table_foreach (gb_style_hash, (GHFunc) reset_gb_styles_callback,
                            NULL);
      g_hash_table_destroy (gb_style_hash);
    }


  /* Create new GbStyle hash */
  gb_style_hash = g_hash_table_new (g_str_hash, g_str_equal);

  /* Create default GbStyle */
  gb_widget_default_gb_style = gb_widget_new_gb_style ();
  gb_widget_default_gb_style->name = g_strdup (GB_STYLE_DEFAULT);
  gb_widget_default_gb_style->xlfd_fontname = g_strdup (GB_DEFAULT_XLFD_FONTNAME);
  gb_widget_default_gb_style->style = gtk_widget_get_default_style ();
  g_hash_table_insert (gb_style_hash, gb_widget_default_gb_style->name,
                       gb_widget_default_gb_style);
}


GbStyle *
gb_widget_new_gb_style ()
{
  gint i;
  GbStyle *gbstyle = g_new (GbStyle, 1);
  gbstyle->name = NULL;
  gbstyle->xlfd_fontname = NULL;
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      gbstyle->bg_pixmap_filenames[i] = NULL;
    }
  gbstyle->style = NULL;
  gbstyle->ref_count = 0;
  return gbstyle;
}


GbStyle *
gb_widget_copy_gb_style (GbStyle * gbstyle)
{
  gint i;
  GbStyle *gbstyle_copy = g_new (GbStyle, 1);
  gbstyle_copy->name = g_strdup (gbstyle->name);
  gbstyle_copy->xlfd_fontname = g_strdup (gbstyle->xlfd_fontname);
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      gbstyle_copy->bg_pixmap_filenames[i]
        = g_strdup (gbstyle->bg_pixmap_filenames[i]);
    }
  gbstyle_copy->style = gbstyle->style;
  gtk_style_ref (gbstyle_copy->style);
  gbstyle_copy->ref_count = 0;
  return gbstyle_copy;
}


void
gb_widget_ref_gb_style (GbStyle * gbstyle)
{
  gbstyle->ref_count++;
}


void
gb_widget_unref_gb_style (GbStyle * gbstyle)
{
  gbstyle->ref_count--;
  /* Don't destroy named GbStyles - let user do that explicitly. */
  if (gbstyle->ref_count <= 0 && gbstyle->name == NULL)
    {
      gb_widget_destroy_gb_style (gbstyle, TRUE);
    }
}


void
gb_widget_destroy_gb_style (GbStyle * gbstyle, gboolean remove_from_hash)
{
  gint i;

  MSG1 ("Destroying gbstyle:%s", gbstyle->name);

  /* Remove the GbStyle from the hash */
  if (remove_from_hash && gbstyle->name)
    g_hash_table_remove (gb_style_hash, gbstyle->name);

  /* Now free all the memory used */
  g_free (gbstyle->name);
  g_free (gbstyle->xlfd_fontname);
  for (i = 0; i < GB_NUM_STYLE_STATES; i++)
    {
      g_free (gbstyle->bg_pixmap_filenames[i]);
    }
  if (gbstyle->style)
    gtk_style_unref (gbstyle->style);
  g_free (gbstyle);
}



/*************************************************************************
 * Misc. Functions
 *************************************************************************/


/* This is a GTK bug workaround for combo widgets. They should manage the
   size of their GtkEntry, but they don't at present, so we have to set its
   size explicitly (and also in the source code output). */
void
gb_widget_set_usize (GtkWidget *widget,
                     gint w,
                     gint h)
{
  if (GTK_IS_COMBO (widget))
    gtk_widget_set_usize (GTK_COMBO (widget)->entry,
                          w - 16 < 0 ? -1 : w - 16, h);
  gtk_widget_set_usize (widget, w, h);
}

/* This function is used to iterate through the table children in reverse.
   It is needed so we output the XML file in the same order each time. */
static void
table_foreach (GtkTable * table,
               GtkCallback callback,
               gpointer callback_data)
{
  GList *children;
  GtkTableChild *child;

  children = g_list_last (table->children);
  while (children)
    {
      child = children->data;
      children = children->prev;

      (*callback) (child->widget, callback_data);
    }
}


/* This iterates over a notebook's tabs, for saving. */
static void
notebook_tabs_foreach (GtkNotebook *notebook,
                       GtkCallback callback,
                       gpointer callback_data)
{
  GList *children;
  GtkNotebookPage *page;

  children = notebook->children;
  while (children)
    {
      page = children->data;
      (*callback) (page->tab_label, callback_data);
      children = children->next;
    }
}


static GtkNotebookPage *
find_notebook_page (GtkWidget * widget, GtkWidget * current_child, gint * pos)
{
  GList *children;
  GtkNotebookPage *page;
  gint page_num = 0;

  children = GTK_NOTEBOOK (widget)->children;
  while (children)
    {
      page = children->data;
      if (page->child == current_child || page->tab_label == current_child)
        {
          *pos = page_num;
          return page;
        }
      children = children->next;
      page_num++;
    }
  return NULL;
}



/*************************************************************************
 * Common popup menu callbacks
 *************************************************************************/
void
gb_widget_remove_label (GtkWidget * menuitem,
                        GtkWidget * widget)
{
  GtkWidget *child;

  if (GTK_IS_BIN (widget))
    child = GTK_BIN (widget)->child;
  else if (GTK_IS_BUTTON (widget))
    child = GTK_BUTTON (widget)->child;
  else
    return;
  if (child && GTK_IS_LABEL (child))
    editor_delete_widget (child);
}

static void
gb_widget_add_alignment (GtkWidget * menuitem,
                         GtkWidget * widget)
{
  GtkWidget *alignment, *parent;

  parent = widget->parent;
  alignment = gb_widget_new ("GtkAlignment", parent);

  gtk_widget_ref (widget);
  gb_widget_replace_child (parent, widget, alignment);
  if (GTK_BIN (alignment)->child)
    gtk_container_remove (GTK_CONTAINER (alignment),
                          GTK_BIN (alignment)->child);
  gtk_container_add (GTK_CONTAINER (alignment), widget);
  gtk_widget_unref (widget);
  tree_insert_widget_parent (alignment, widget);
}


static void
gb_widget_remove_alignment (GtkWidget * menuitem,
                            GtkWidget * widget)
{
  GtkWidget *alignment, *parent;

  alignment = widget->parent;
  g_return_if_fail (GTK_IS_ALIGNMENT (alignment));
  tree_remove_widget_parent (alignment, widget);
  parent = alignment->parent;

  gtk_widget_ref (widget);
  gtk_container_remove (GTK_CONTAINER (alignment), widget);
  gb_widget_replace_child (parent, alignment, widget);
  gtk_widget_unref (widget);
}


static void
gb_widget_add_event_box (GtkWidget * menuitem,
                         GtkWidget * widget)
{
  GtkWidget *event_box, *parent;

  parent = widget->parent;
  event_box = gb_widget_new ("GtkEventBox", parent);

  gtk_widget_ref (widget);
  gb_widget_replace_child (parent, widget, event_box);
  if (GTK_BIN (event_box)->child)
    gtk_container_remove (GTK_CONTAINER (event_box),
                          GTK_BIN (event_box)->child);
  gtk_container_add (GTK_CONTAINER (event_box), widget);
  gtk_widget_unref (widget);
  tree_insert_widget_parent (event_box, widget);
}

static void
gb_widget_remove_event_box (GtkWidget * menuitem,
                            GtkWidget * widget)
{
  GtkWidget *event_box, *parent;

  event_box = widget->parent;
  g_return_if_fail (GTK_IS_EVENT_BOX (event_box));
  tree_remove_widget_parent (event_box, widget);
  parent = event_box->parent;

  gtk_widget_ref (widget);
  gtk_container_remove (GTK_CONTAINER (event_box), widget);
  gb_widget_replace_child (parent, event_box, widget);
  gtk_widget_unref (widget);
}


/*************************************************************************
 * Common functions used by gbwidgets.
 *************************************************************************/

/* This gets the child label for buttons or menuitems or subclasses.
   This is for showing or saving. */
void
gb_widget_output_child_label (GtkWidget * widget, GbWidgetGetArgData * data,
                              gchar * Label)
{
  GtkWidget *child;

  if (GTK_IS_BUTTON (widget))
    child = GTK_BUTTON (widget)->child;
  else
    child = GTK_BIN (widget)->child;

  /* Note that we don't want to save the child label if it is a GbWidget,
     since it will be saved as a separate widget. */
  if (child && GTK_IS_LABEL (child) && !GB_IS_GB_WIDGET (child))
    {
      gchar *label_text;
      gtk_label_get (GTK_LABEL (child), &label_text);
      gb_widget_output_text (data, Label, label_text);
      if (data->action == GB_SHOWING)
        property_set_sensitive (Label, TRUE);
    }
  else
    {
      if (data->action == GB_SHOWING)
        {
          gb_widget_output_text (data, Label, "");
          property_set_sensitive (Label, FALSE);
        }
    }
}


/* This sets the child label for buttons/items/menuitems or subclasses.
   This is for applying or loading. */
void
gb_widget_input_child_label (GtkWidget * widget, GbWidgetSetArgData * data,
                             gchar * Label)
{
  GtkWidget *child, *label;
  gchar *label_text = gb_widget_input_text (data, Label);

  MSG ("IN input_child_label");

  if (GTK_IS_BUTTON (widget))
    child = GTK_BUTTON (widget)->child;
  else
    child = GTK_BIN (widget)->child;

  if (data->apply)
    {
      if (child && GTK_IS_LABEL (child))
        {
          gtk_label_set (GTK_LABEL (child), label_text);
        }
      else
        {
          if (child != NULL)
            gtk_container_remove (GTK_CONTAINER (widget), child);
          label = gtk_label_new (label_text);
          gtk_widget_show (label);
          gtk_container_add (GTK_CONTAINER (widget), label);

          /* Simple child labels are given different alignments according to
             the parent. GtkButton and GtkToggleButton are centred. All the
             others are aligned left. See the GTK _new_with_label() fns. */
          if (data->action == GB_LOADING)
            {
              if (GTK_IS_CHECK_BUTTON (widget)
                  || GTK_IS_ITEM (widget))
                gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
            }
        }
    }
  /* This isn't very nice. When a text property is got from the property
     editor (i.e. when action is GB_APPLYING) it needs to be freed after. */
  if (data->action == GB_APPLYING)
    g_free (label_text);
}


/* This updates the boxes size to the given value, adding placeholders or
   deleting widgets as necessary. */
void
gb_widget_update_box_size (GtkWidget * widget, gint size)
{
  GtkWidget *new_child;
  gint current_size = g_list_length (GTK_BOX (widget)->children);
  gint i;

  if (current_size < size)
    {
      for (i = 0; i < size - current_size; i++)
        {
          if (GTK_IS_BUTTON_BOX (widget))
            {
              new_child = gb_widget_new ("GtkButton", widget);
            }
          else
            {
              new_child = editor_new_placeholder ();
              /*if (GTK_IS_HBOX(widget)) gtk_widget_set_usize(new_child, 60, 80);
                 else gtk_widget_set_usize(new_child, 80, 60); */
            }
          gtk_box_pack_start (GTK_BOX (widget), new_child, TRUE, TRUE, 0);

          /* This avoids any problems with redrawing the selection. */
          editor_clear_selection (NULL);
        }
    }
  else if (current_size > size)
    {
      gint data[2];
      data[0] = 0;
      data[1] = size;
      gtk_container_foreach (GTK_CONTAINER (widget),
                             (GtkCallback) update_box_size_callback, data);
    }
}


/* This removes widgets from the end of the box. data[0] is the current index,
   data[1] is the desired size of the box. */
static void
update_box_size_callback (GtkWidget * widget, gint data[2])
{
  if (data[0] >= data[1])
    gtk_container_remove (GTK_CONTAINER (widget->parent), widget);
  data[0]++;
}


/*************************************************************************
 * Adjustment convenience functions - handles all 6 values.
 *************************************************************************/
void
gb_widget_output_adjustment (GbWidgetGetArgData * data, gchar * Values[],
                             GtkAdjustment * adjustment)
{
  gb_widget_output_float (data, Values[0], adjustment->value);
  gb_widget_output_float (data, Values[1], adjustment->lower);
  gb_widget_output_float (data, Values[2], adjustment->upper);
  gb_widget_output_float (data, Values[3], adjustment->step_increment);
  gb_widget_output_float (data, Values[4], adjustment->page_increment);
  gb_widget_output_float (data, Values[5], adjustment->page_size);
}


gboolean
gb_widget_input_adjustment (GbWidgetSetArgData * data, gchar * Values[],
                            GtkAdjustment * adjustment)
{
  gfloat value, lower, upper, step_inc, page_inc, page_size;
  gboolean myApply;

  value = gb_widget_input_float (data, Values[0]);
  myApply = data->apply;
  lower = gb_widget_input_float (data, Values[1]);
  myApply |= data->apply;
  upper = gb_widget_input_float (data, Values[2]);
  myApply |= data->apply;
  step_inc = gb_widget_input_float (data, Values[3]);
  myApply |= data->apply;
  page_inc = gb_widget_input_float (data, Values[4]);
  myApply |= data->apply;
  page_size = gb_widget_input_float (data, Values[5]);
  myApply |= data->apply;

  if (myApply)
    {
      adjustment->value = value;
      adjustment->lower = lower;
      adjustment->upper = upper;
      adjustment->step_increment = step_inc;
      adjustment->page_increment = page_inc;
      adjustment->page_size = page_size;
      return TRUE;
    }
  return FALSE;
}
