(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               more_util.ml                               *)
(****************************************************************************)

open Std;;
open Pp;;

(* type option *)


let isSome = function None -> false | _ -> true;;

let option_app f = function
    Some x -> Some (f x)
  | None -> None
;;

let do_option f = function
    Some x -> f x
  | None -> ()
;;

let inSOME x = Some x;;
let outSOME = function
    (Some x) -> x
  | _ -> failwith "outSOME";;


(* Manipulation de caracteres *)

let implode_chars cl =
  let len = List.length cl in
  let dest = String.create len in
  let _ = 
    List.fold_left
      (fun start src -> String.set dest start src; start + 1)
      0 cl
  in
    dest
;;

let explode_chars s =
    let slen = String.length s in
    let rec aux n = if n < slen then (String.get s n)::(aux (n+1)) else [] in
        aux 0
;;

let string_of_char c = String.make 1 c
;;


(* Parsing *)

let in_range (l , h) = function (c:char) ->
    (Char.code l)<=(Char.code c) & (Char.code c)<=(Char.code h)
;;

let islower = in_range('a','z');;

let isupper = in_range('A','Z');;

let isalpha c = islower c or isupper c;;

let isdigit = in_range('0','9');;

let isxdigit c = in_range('0','9') c or in_range('a','f') c or in_range('A','F') c;;

let isalnum c = isdigit c or isalpha c;;

let is_ident_start c = isalpha c or List.mem c ['_';'$'];;

let is_ident_rest c = isalnum c or List.mem c ['_';'$';'\'';'-'];;


(* Constructeurs de parseurs *)

let plist elem = 
  let rec plist_rec = parser
     [< e = elem; l = plist_rec >] -> e::l
   | [< >]                         -> []
  in plist_rec
  
;;

let ne_plist elem = parser
  [< e = elem; l = (plist elem) >] -> (e,l)
;;

let ne_plist_with_sep sep elem = 
 let rec do_rec = parser
  [< e = elem; l = (parser [< () = sep; l = do_rec >] -> l | [< >] -> []) >] -> e::l
 in do_rec
;;

let plist_with_sep sep elem = parser
    [< l = (ne_plist_with_sep sep elem) >] -> l
  | [< >] -> []
;;


let stream_check p= parser [< 'x when p x >] -> x;;

let p_atom = parser
    [< 'a when (is_ident_start a); l= plist(stream_check is_ident_rest)>]
           -> implode_chars(a::l)
;;

(* constructeurs de printers *)

(* quelques separateurs *)
let pr_spc () = [< 'sPC >];;
let pr_fnl () = [< 'fNL >];;
let pr_int n = [< 'iNT n >];;
let pr_str s = [< 'sTR s >];;
let pr_coma () = [< 'sTR","; 'sPC >];;

let rec prlist elem l = match l with 
    []   -> [< >]
  | h::t -> let e = elem h and r = prlist elem t in [< e; r >];;

let rec prlist_with_sep sep elem l = match l with
    []   -> [< >]
  | [h]  -> elem h
  | h::t ->
      let e = elem h and s = sep() and r = prlist_with_sep sep elem t
      in [< e; s; r >];;

let prvect elem v = 
  let rec pr n =
    if n = Array.length v then [< >]
    else let e = elem v.(n) and r = pr (n+1) in [< e; r >]
  in pr 0 
;;

let prvect_with_sep sep elem v =
  let rec pr n =
    if n = 0 then elem v.(0)
    else let r = pr(n-1) and s = sep() and e = elem v.(n) in [< r; s; e >]
  in
    if Array.length v = 0 then [< >]
    else pr (Array.length v - 1);;


type ('a,'b) choice =  Left of 'a | Right of 'b;;

(* $Id: more_util.ml,v 1.11 1999/06/29 07:47:19 loiseleu Exp $ *)



