(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               hashtabl.ml                                *)
(****************************************************************************)
(* Hash tables *)

external hash_param : int -> int -> 'a -> int =  "hash_univ_param"

(* We do dynamic hashing, and we double the size of the table when
   buckets become too long, but without re-hashing the elements. *)


(* Hash parameters *)

let hash_mnodes = 100;;
let hash_nodes = 200;;

let hash x = hash_param hash_mnodes hash_nodes x;;

type ('a, 'b) t =
  { mutable size : int;             (* Total number of elements in the table *)
    mutable data : ('a, 'b) bucketlist array; (* the buckets *)
    mutable mean_len : int}

and ('a, 'b) bucketlist =
    Empty
  | Cons of 'a * 'b * ('a, 'b) bucketlist
;;

type ('a,'b) frozen_t =
    int * ('a,'b) bucketlist array
;;

let freeze t = (t.size, Array.copy t.data);;
let unfreeze (s, v) t = (t.size <- s; t.data <- (Array.copy v); ());;

let create initial_size =
  { size = 0; data = Array.create initial_size Empty; mean_len = 2 }
;;

let clear h =
  for i = 0 to Array.length h.data - 1 do
    h.data.(i) <- Empty
  done;
  h.size <- 0;;

let rec do_on_bucket f = function
 | Empty -> ()
 | Cons (key, info, rest) -> f key info; do_on_bucket f rest;;

let rec rev_do_on_bucket f = function
 | Empty -> ()
 | Cons (key, info, rest) -> do_on_bucket f rest; f key info; ();;

let do_on_buckets f = Array.iter (do_on_bucket f);;
let do_table f h = do_on_buckets f h.data;;

let do_on_buckets_as_added f = Array.iter (rev_do_on_bucket f);;
let do_table_as_added f h = do_on_buckets_as_added f h.data;;

let rec add h key info =
  let i = (hash key) mod (Array.length h.data) in
  h.data.(i) <- Cons(key, info, h.data.(i));
  h.size <- h.size + 1;
  if h.size / Array.length h.data > h.mean_len then resize h

and resize h =
  let data = h.data in
  let n = Array.length h.data in
  h.data <- Array.create (n+n+1) Empty;
  h.size <- 0;
  do_on_buckets_as_added (add h) data;;

let remove h key =
  let i = (hash key) mod (Array.length h.data) in
  let rec remove_bucket = function
      Empty -> Empty
    | Cons(k, i, next) ->
        if k = key
         then begin h.size <- h.size - 1; next end
         else Cons(k, i, remove_bucket next) in
  h.data.(i) <- remove_bucket h.data.(i);;

let find h key =
  match h.data.((hash key) mod (Array.length h.data)) with
    Empty -> 
     raise Not_found
  | Cons(k1, d1, rest1) ->
      if key = k1 then d1 else
      match rest1 with
        Empty ->  raise Not_found
      | Cons(k2, d2, rest2) ->
          if key = k2 then d2 else
          match rest2 with
            Empty ->  raise Not_found
          | Cons(k3, d3, rest3) ->
              if key = k3 then d3 else begin
                let rec find = function
                    Empty ->  
                      raise Not_found
                  | Cons(k, d, rest) ->
                      if key = k then d else find rest
                in find rest3
              end
;;

let find_all h key =
  let rec find_in_bucket = function
    Empty ->
      []
  | Cons(k, d, rest) ->
      if k = key then d :: find_in_bucket rest else find_in_bucket rest in
  find_in_bucket h.data.((hash key) mod (Array.length h.data))
;;

let do_table_rev = do_table_as_added;;

let set_meanlen t n = if n > 0 then t.mean_len <- n;;

(* $Id: hashtabl.ml,v 1.6 1999/06/29 07:47:16 loiseleu Exp $ *)
