(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                coqtop.ml                                 *)
(****************************************************************************)

(* camlp4o ../config/coq_config.cmo ../../tools/camlp4/ocamlpp.cmo *)

(* Afin de rendre Coq plus portable, ce programme Caml remplace le script
   coqtop. 

   Ici, on ne fait que l'analyse des options suivantes:
     -bindir
     -image
     -opt
     -full
     -libdir | -nolib
     -searchisos

   Le reste de la ligne de commande est pass tel quel, et sera analys
   par env/cmd_line.ml

   On essaye au maximum d'utiliser les modules Sys et Filename pour que la
   portabilit soit maximale, mais il reste encore des appels  des fonctions
   du module Unix. Ceux-ci sont prfixs par "Unix."
*)


(* environment *)

let environment = ref (Array.to_list (Unix.environment ()))

let export name value =
  environment := (name^"="^value) :: !environment
;;

let unsetenv name = 
  let rec aux l =
    match l with 
	s::l' -> 
	  let n = String.sub s 0 (String.index s '=') in
	    if n=name then aux l' else s::(aux l')
      | [] -> []
  in 
    environment := aux !environment
;;

export "COQTOP" Coq_config.coqtop;;

export "CAMLP4LIB" Coq_config.camlp4lib;;

unsetenv "COQLIB";;

let bindir = ref Coq_config.bindir

(* La variable d'environnement $COQBIN est prioritaire sur la valeur 
   de Coq_config *)
let _ = 
  try 
    let c = Sys.getenv "COQBIN"
    in if c <> "" then bindir := c
  with Not_found -> () 

let libdir = ref (Some Coq_config.coqlib)

(* idem pour $COQLIB *)
let _ = 
  try
    let c = Sys.getenv "COQLIB"
    in if c <> "" then libdir := Some c
  with Not_found -> ()

let image = ref ""

let native = ref false
let searchonly = ref false
let full = ref false
let profile = ref false

let debugger = ref false

let echo = ref false

let src = ref false

let uninstall = ref false

(* option -where : prints the standard library location and exits *)

let where () =
  print_endline Coq_config.coqlib;
  exit 0
;;

(* option -uninstall : removes the library, the binaries and the man pages *)

let uninstall () = 
  let coqfiles = open_in (Filename.concat Coq_config.coqlib "COQFILES")
  and rm_R = 
    match Sys.os_type with
      "Unix" -> (fun file -> Sys.command ("rm -Rf "^file))
    | "Win32" -> (fun file -> Sys.command ("deltree /y "^file))
    | _ -> assert false
  in
  begin
  print_endline 
    "****** WARNING. The following program will delete all binaries, ";
  print_endline 
    "help files and libraries of the Coq system";
  print_newline ();  
  print_string "Is that really what you want ? ";
  match read_line () with 
    "y" | "Y" | "o" | "O" | "yes" | "oui" -> 
      begin
      print_endline "DELETING COQ FILES...";
      flush stdout;
      try
	while true do
	  let l = (input_line coqfiles) in
	  (if Sys.file_exists l 
	  then Sys.remove l
	  else print_endline ("WARNING : file "^l^" does not exist anymore");
	  print_endline l)
	done
      with End_of_file -> flush stdout
      end
  | _ -> ()
  end;
  exit 0
;;

(* parsing of the command line (for -bindir, -libdir, etc) *)

let usage () =
  Usage.print_usage_coqtop () ;
  flush stderr ;
  exit 1

let version () =
  Printf.printf "The Coq Proof Assistant, version V%s (%s)\n"
    Coq_config.version Coq_config.date;
  Printf.printf "compiled on %s\n" Coq_config.compile_date;
  exit 0

let rec parse = function

    "-image" :: im :: rem -> image := im ; parse rem
  | "-image" :: []        -> usage ()

  | "-bindir" :: d :: rem -> bindir := d ; parse rem
  | "-bindir" :: []       -> usage ()

  | "-libdir" :: d :: rem -> libdir := Some d ; parse rem
  | "-libdir" :: []       -> usage ()

  | "-nolib" :: rem -> libdir := None ; parse rem

  | "-opt" :: rem -> native := true; parse rem

  | "-full" :: rem -> full := true; parse rem

  | "-profile" :: rem -> profile := true; parse rem

  | "-searchisos" :: rem -> searchonly := true; parse rem

  | "-where" :: _ -> where ()

  | "-src" :: rem -> src := true; parse rem

  | "-echo" :: rem -> echo := true ; parse rem

  | ("-?"|"-h"|"-H"|"-help"|"--help") :: _ -> usage ()

  | ("-v"|"--version") :: _ -> version ()

  | ("-db"|"-debugger") :: rem -> debugger := true ; parse rem

  | "-uninstall" :: [] -> uninstall ()
  | "-uninstall" :: _ -> usage()

  | s :: rem -> s :: (parse rem)
  
  | [] -> []
;;

let cl_args = parse (List.tl (Array.to_list Sys.argv));;

if !image="" then
  image :=
    if !full then
      "coq-contrib.out"
    else if !profile then
      "coq-profile.out"
    else if !native then
      if !searchonly then "coq_searchisos_opt.out" else "coqopt.out"
    else
      if !searchonly then "coq_searchisos.out" else "coq.out"
;;


let slash = List.fold_left Filename.concat;;

(* arguments for the debugger *)
let src_args = 
  List.fold_right
    (fun d l -> "-I" :: (slash Coq_config.coqtop d) :: l)
    [ [ "src" ; "config" ] ;
      [ "src" ; "lib" ; "util" ] ;
      [ "src" ; "meta" ] ;
      [ "src" ; "constr" ] ;
      [ "src" ; "proofs" ] ;
      [ "src" ; "parsing" ] ;
      [ "src" ; "env" ] ;
      [ "src" ; "typing" ];
      [ "src" ; "tactics" ]
    ]
    []
;;

let args = 
  (if !src or !debugger then src_args else []) @ cl_args
;;

(* which command to execute. Does the file exist ? *)

let command =
  if (Sys.file_exists !image) then !image
  else if Filename.is_relative !image
    & Sys.file_exists (Filename.concat !bindir !image)
  then Filename.concat !bindir !image
  else
    let (image_name, builtin) =
      match Filename.basename !image with
        | "coq.out" -> ("bytecode image of Coq", true)
        | "coqopt.out" -> ("native code image of Coq", true)
        | "coq_searchisos.out" -> ("bytecode image of Coq_SearchIsos", true)
        | "coq_searchisos_opt.out" ->
            ("native code image of Coq_SearchIsos", true)
        | "coq-contrib.out" ->
            ("native code image of the full Coq system", true)
        | _ -> ("image: " ^ !image, false)
    in
      begin
        prerr_endline ("I can't find the " ^ image_name ^ ".");
        if builtin then prerr_endline " Did you compile it ?";
        flush stderr;
        exit 1
      end;;


(* export COQLIB if !libdir is defined *)
match !libdir with
    None -> 
(* The variable COQLIB is mandatory in the windows version so we put 
	a stupid one *)
	export "COQLIB" ""
  | Some d -> export "COQLIB" d
;;


(* if echo is on then we print the command and exit *)

if !echo then begin
  List.iter (fun s -> print_string s ; print_string " ") (command::args);
  print_newline();
  flush stdout;
  exit 0
end;;


(* execution of [command] with arguments [args]. *)

if !debugger then begin

ifdef Unix then
  Unix.execvpe "ocamldebug"
    (Array.of_list ("ocamldebug" :: src_args @ command :: args))
    (Array.of_list !environment)
else
  ()

end else

ifdef Unix then
  Unix.execve command (Array.of_list (command :: args))
    (Array.of_list !environment)
else
(* Unix.execve is bogus under windows :) *)
 let pid =
    Unix.create_process_env command
       (Array.of_list (command :: args)) 
	(Array.of_list !environment) Unix.stdin Unix.stdout Unix.stderr in
  let status = Unix.waitpid [] pid in
    match status with
        _, Unix.WEXITED c -> exit c
      | _                 -> exit 1
;;

(* $Id: coqtop.ml4,v 1.18 1999/06/29 07:47:11 loiseleu Exp $ *)
