(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                             vernacinterp.ml                              *)
(****************************************************************************)

(* Concrete syntax of the mathematical vernacular MV V2.6 *)


open Std;;
open Names;;
open Pp;;
open Himsg;;
open Proof_trees;;
open CoqAst;;
open Ast;;

exception ProtectedLoop;;
exception Drop;;
exception Quit;;

let disable_drop e =
  if e <> Drop then e
  else UserError("Vernac.disable_drop",[< 'sTR"Drop is forbidden." >])
;;

type vernac_arg = 
    VARG_VARGLIST of vernac_arg list
  | VARG_STRING of string
  | VARG_NUMBER of int
  | VARG_NUMBERLIST of int list
  | VARG_IDENTIFIER of identifier
  | VCALL of string * vernac_arg list
  | VARG_COMMAND of CoqAst.t
  | VARG_COMMANDLIST of CoqAst.t list
  | VARG_TACTIC of CoqAst.t
  | VARG_TACTIC_ARG of tactic_arg
  | VARG_BINDER of identifier list * CoqAst.t
  | VARG_BINDERLIST of (identifier list * CoqAst.t) list
  | VARG_AST of CoqAst.t
  | VARG_ASTLIST of CoqAst.t list
  | VARG_UNIT
  | VARG_DYN of Dyn.t
;;

(* Table of vernac entries *)
let vernac_tab =
  (Mhm.create 51 : (string, vernac_arg list -> unit -> unit) Mhm.t);;

let vinterp_add (s,f) =
    try Mhm.add vernac_tab (s,f)
    with Failure _ ->
        errorlabstrm "vinterp_add"
        [< 'sTR"Cannot add the vernac command " ; 'sTR s ; 'sTR" twice" >]
;;

let overwriting_vinterp_add (s,f) =
  if Mhm.in_dom vernac_tab s then
    Mhm.rmv vernac_tab s;
  Mhm.add vernac_tab (s,f)
;;

let vinterp_map s =
    try Mhm.map vernac_tab s
    with Not_found -> errorlabstrm "Vernac Interpreter"
                      [< 'sTR"Cannot find vernac command " ; 'sTR s >];;

let vinterp_init () = Mhm.empty vernac_tab;;


(* Conversion CoqAst.t -> vernac_arg *)
let rec cvt_varg ast =
  match ast with
      Node(_,"VERNACARGLIST",l) -> VARG_VARGLIST (List.map cvt_varg l)
    | Node(_,"VERNACCALL",(Str (_,na))::l) ->
        VCALL (na,List.map cvt_varg l)
    | Node(_,"VERNACCALL",(Id (_,na))::l) ->
        VCALL (na,List.map cvt_varg l)

    | Nvar(_,s) -> VARG_IDENTIFIER (id_of_string s)
    | Str(_,s) -> VARG_STRING s
    | Num(_,n) -> VARG_NUMBER n
    | Node(_,"NONE",[]) -> VARG_UNIT
    | Node(_,"COMMAND",[c]) -> VARG_COMMAND c
    | Node(_,"COMMANDLIST",l) -> VARG_COMMANDLIST l
    | Node(_,"TACTIC",[c]) -> VARG_TACTIC c
    | Node(_,"BINDER",c::idl) ->
        VARG_BINDER(List.map (comp id_of_string nvar_of_ast) idl, c)
    | Node(_,"BINDERLIST",l) ->
        VARG_BINDERLIST
          (List.map (comp (function (VARG_BINDER (x_0,x_1)) -> (x_0,x_1)
			  | _ -> assert false) cvt_varg) l)
    | Node(_,"NUMBERLIST",ln) ->
        VARG_NUMBERLIST (List.map num_of_ast ln) 

    | Node(_,"AST",[a]) -> VARG_AST a
    | Node(_,"ASTLIST",al) -> VARG_ASTLIST al
    | Node(_,"TACTIC_ARG",[targ]) -> VARG_TACTIC_ARG (cvt_arg targ)
    | Node(_,"VERNACDYN",[Dynamic (_,d)]) -> VARG_DYN d
    | _ -> anomaly_loc (Ast.loc ast, "Vernacinterp.cvt_varg",
                        [< 'sTR "Unrecognizable ast node of vernac arg:";
			  'bRK(1,0); print_ast ast >])
;;


(* Interpretation of a vernac command *)
let call (opn,converted_args) =
  let loc = ref "Looking up command" in
    try
      let callback = vinterp_map opn in
        loc:= "Checking arguments";
        let hunk = callback converted_args in
          loc:= "Executing command";
          hunk()
    with
        Drop -> raise Drop
      | ProtectedLoop -> raise ProtectedLoop
      | e    ->
          if !Errors.debug then
            mSGNL [< 'sTR"Vernac Interpreter " ; 'sTR !loc >];
          raise e
;;

let rec interp = function
(*    Node(_,"Time",l) -> now done in vernac.ml 
      let tstart = System.timestamp() in
        List.iter interp l;
        mSGNL [< 'sTR"Finished transaction in " ;
                 System.fmt_time_difference tstart (System.timestamp()) >]
  | 
*) Node(_,opn,argl) as cmd ->
    let converted_args =
      try List.map cvt_varg argl
      with e ->
        if !Errors.debug then
          mSGNL [< 'sTR"Vernac Interpreter " ; 'sTR"Converting arguments" >];
        raise e
    in call (opn,converted_args)
  | cmd -> errorlabstrm "Vernac Interpreter"
             [< 'sTR"Malformed vernac AST : " ; print_ast cmd >]
;;

let bad_vernac_args s =
  anomalylabstrm s
    [< 'sTR"Vernac "; 'sTR s; 'sTR" called with bad arguments" >]
;;


(* $Id: vernacinterp.ml,v 1.24 1999/11/01 12:41:12 mohring Exp $ *)
