(* camlp4r pa_extend.cmo q_MLast.cmo *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1999 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: pa_lablext.ml,v 1.4 1999/03/04 16:37:26 ddr Exp $ *)

open Pcaml;

value fun_binding = Pa_lo.fun_binding;
value fun_def = Pa_lo.fun_def;
value class_fun_binding = Pa_lo.class_fun_binding;
value label_declaration = Pa_lo.label_declaration;

value mk_ctyp_label loc l t =
  MLast.TyXnd loc "label" (MLast.TyXnd loc l t)
;

value mk_patt_label loc l p =
  MLast.PaXnd loc "label" (MLast.PaXnd loc l p)
;

value mk_patt_variant loc i =
  MLast.PaXnd loc "variant" (MLast.PaUid loc i)
;

value mk_expr_label loc l e =
  MLast.ExXnd loc "label" (MLast.ExXnd loc l e)
;

value mk_expr_variant loc i =
  MLast.ExXnd loc "variant" (MLast.ExUid loc i)
;

value mk_expr_default loc d e =
  match d with
  [ Some d -> MLast.ExXnd loc "default" (MLast.ExTup loc [d; e])
  | None -> e ]
;

value mk_expr_poly loc e t =
  let t =
    match t with
    [ Some t -> t
    | None -> MLast.TyAny loc ]
  in
  MLast.ExXnd loc "poly" (MLast.ExTyc loc e t)
;

value mk_core_variant loc closed max min =
  let closed = MLast.TyQuo loc (if closed then "true" else "false") in
  let max =
    let tl =
      List.map
        (fun (label, t) ->
           let t =
             match t with
             [ Some t -> t
             | None -> MLast.TyQuo loc "" ]
           in
           mk_ctyp_label loc label t)
        max
    in
    MLast.TyTup loc tl
  in
  let min =
    let tl = List.map (fun label -> MLast.TyQuo loc label) min in
    MLast.TyTup loc tl
  in
  MLast.TyXnd loc "variant" (MLast.TyTup loc [closed; max; min])
;

value mk_class_default loc d c =
  match d with
  [ Some d -> MLast.CeXnd loc "default" (MLast.CeApp loc c [d])
  | None -> c ]
;

value mk_class_type_label loc l t =
  MLast.CtXnd loc "label" (MLast.CtXnd loc l t)
;

let label = Grammar.Entry.create gram "label" in
do DELETE_RULE
     class_str_item:
       "method"; "private"; label; fun_binding
   END;
   DELETE_RULE
     class_str_item:
       "method"; label; fun_binding
   END;
return ();

EXTEND
  GLOBAL: sig_item expr patt fun_binding ctyp class_type class_str_item
    fun_binding fun_def class_fun_binding label_declaration;
  sig_item:
    [ [ "val"; i = LABEL; t = ctyp ->
          <:sig_item< value $i$ : $t$ >> ] ]
  ;
  expr: LEVEL "expr1"
    [ [ "fun"; (p, d) = labeled_patt_with_default; e = fun_def ->
          let e = mk_expr_default loc d e in
          <:expr< fun $p$ -> $e$ >>
       | "function"; OPT "|"; l = LIST1 match_labeled_case SEP "|" ->
          <:expr< fun [ $list:l$ ] >> ] ]
  ;
  expr: BEFORE "simple"
    [ "label" NONA
      [ "?"; e = expr -> mk_expr_label loc "?" e
      | "?"; l = ELABEL -> mk_expr_label loc ("?" ^ l) <:expr< $lid:l$ >>
      | l = LABEL; e = expr -> mk_expr_label loc l e
      | l = ELABEL -> mk_expr_label loc l <:expr< $lid:l$ >> ] ]
  ;
  expr: LEVEL "simple"
    [ [ "`"; i = ident -> mk_expr_variant loc i ] ]
  ;
  match_labeled_case:
    [ [ (p, d) = labeled_patt_with_default;
        w = OPT [ "when"; e = expr -> e ]; "->"; e = expr ->
          let e = mk_expr_default loc d e in
          (p, w, e) ] ]
  ;
  fun_def:
    [ [ (p, d) = labeled_patt_with_default; e = fun_def ->
          let e = mk_expr_default loc d e in
          <:expr< fun $p$ -> $e$ >> ] ]
  ;
  fun_binding:
    [ [ (p, d) = labeled_patt_with_default; e = fun_binding ->
          let e = mk_expr_default loc d e in
          <:expr< fun $p$ -> $e$ >> ] ]
  ;
  patt: LEVEL "simple"
    [ [ "`"; i = ident -> mk_patt_variant loc i ] ]
  ;
  labeled_patt_with_default:
    [ [ "?"; (p, d) = patt_with_default ->
          (mk_patt_label loc "?" p, d)
      | "?"; l = LABEL; (p, d) = patt_with_default ->
          (mk_patt_label loc ("?" ^ l) p, d)
      | "?"; l = ELABEL; d = default_arg ->
          (mk_patt_label loc ("?" ^ l) <:patt< $lid:l$ >>, Some d)
      | "?"; l = ELABEL ->
          (mk_patt_label loc ("?" ^ l) <:patt< $lid:l$ >>, None)
      | l = LABEL; (p, d) = patt_with_default -> (mk_patt_label loc l p, d)
      | l = ELABEL; d = default_arg ->
          (mk_patt_label loc l <:patt< $lid:l$ >>, Some d)
      | l = ELABEL -> (mk_patt_label loc l <:patt< $lid:l$ >>, None) ] ]
  ;
  patt_with_default:
    [ [ p = patt LEVEL "simple" -> (p, None)
      | p = patt LEVEL "simple"; d = default_arg -> (p, Some d) ] ]
  ;
  default_arg:
    [ [ "[<"; e = expr; ">]" -> e ] ]
  ;
  ctyp: AFTER "arrow"
    [ NONA
      [ "?"; t = ctyp -> mk_ctyp_label loc "?" t
      | "?"; l = LABEL; t = ctyp -> mk_ctyp_label loc ("?" ^ l) t
      | l = LABEL; t = ctyp -> mk_ctyp_label loc l t ] ]
  ;
  ctyp: LEVEL "simple"
    [ [ t = core_variant -> t ] ]
  ;
  core_variant:
    [ [ "["; l = core_variant_in; "]" ->
          let (closed, min) =
            match snd l with
            [ [] -> (True, List.map fst (fst l))
            | _ -> (False, snd l) ]
          in
          mk_core_variant loc closed (fst l) min ] ]
  ;
  core_variant_in:
    [ [ -> ([], [])
      | l = LIST1 ident_core_type -> (l, [])
      | ">"; l = LIST1 ident_core_type -> (l, List.map fst l) ] ]
  ;
  ident_core_type:
    [ [ i = ident -> (i, None)
      | i = ident; "("; t = ctyp; ")" -> (i, Some t) ] ]
  ;
  ident:
    [ [ i = UIDENT -> i
      | i = LIDENT -> i ] ]
  ;
  class_fun_binding:
    [ [ (p, d) = labeled_patt_with_default; cfb = class_fun_binding ->
          let cfb = mk_class_default loc d cfb in
          <:class_expr< fun $p$ -> $cfb$ >> ] ]
  ;
  class_str_item:
    [ [ "method"; "private"; l = label; fb = strict_binding ->
          let fb = mk_expr_poly loc fb None in
          <:class_str_item< method private $l$ = $fb$ >>
      | "method"; l = label; fb = strict_binding ->
          let fb = mk_expr_poly loc fb None in
          <:class_str_item< method $l$ = $fb$ >> ] ]
  ;
  strict_binding:
    [ [ p = patt LEVEL "simple"; e = fun_binding -> <:expr< fun $p$ -> $e$ >>
      | "="; e = expr -> e
      | (p, d) = labeled_patt_with_default; e = fun_binding ->
          let e = mk_expr_default loc d e in
          <:expr< fun $p$ -> $e$ >> ] ]
  ;
  class_type:
    [ [ l = LABEL; t = ctyp LEVEL "ctyp1" ->
          let ct = Pa_lo.class_type_of_ctyp loc t in
          mk_class_type_label loc l ct ] ]
  ;
  label:
    [ [ i = LIDENT -> i ] ]
  ;
  label_declaration:
    [ [ "mutable"; i = LABEL; t = ctyp -> (i, True, t) ] ]
  ;
END;
