(* camlp4r q_MLast.cmo *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: pr_extend.ml,v 2.0 1998/12/01 08:20:48 ddr Exp $ *)

open Pretty;

(* Utilities *)

value rec list elem el k =
  match el with
  [ [] -> k
  | [x] -> [: `elem x k :]
  | [x :: l] -> [: `elem x [: :]; list elem l k :] ]
;

value rec listws elem sep el k =
  match el with
  [ [] -> k
  | [x] -> [: `elem x k :]
  | [x :: l] -> [: `elem x [: `sep :]; listws elem sep l k :] ]
;

value rec listwbws elem b sep el dg k =
  match el with
  [ [] -> [: b; k :]
  | [x] -> [: `elem b x dg k :]
  | [x :: l] ->
      let sdg =
        match sep with
        [ S _ x -> x
        | _ -> "" ]
      in
      [: `elem b x sdg [: :]; listwbws elem [: `sep :] sep l dg k :] ]
;

(* Extracting *)

value rec get_globals =
  fun
  [ [(<:patt< _ >>, <:expr< ($lid:s$ : Grammar.Entry.e '$_$) >>) :: pel] ->
      [s :: get_globals pel]
  | [] -> []
  | _ -> raise Not_found ]
;

value rec get_locals =
  fun
  [ [(<:patt< $_$ >>,
      <:expr< (grammar_entry_create $_$ : $_$) >>) :: pel] ->
        get_locals pel
  | [] -> ()
  | _ -> raise Not_found ]
;

value unposition =
  fun
  [ <:expr< None >> -> None
  | <:expr< Some Gramext.First >> -> Some Gramext.First
  | <:expr< Some Gramext.Last >> -> Some Gramext.Last
  | <:expr< Some (Gramext.Before $str:s$) >> -> Some (Gramext.Before s)
  | <:expr< Some (Gramext.After $str:s$) >> -> Some (Gramext.After s)
  | <:expr< Some (Gramext.Level $str:s$) >> -> Some (Gramext.Level s)
  | _ -> raise Not_found ]
;

value unlabel =
  fun
  [ <:expr< None >> -> None
  | <:expr< Some $str:s$ >> -> Some s
  | _ -> raise Not_found ]
;

value unassoc =
  fun
  [ <:expr< None >> -> None
  | <:expr< Some Gramext.NonA >> -> Some Gramext.NonA
  | <:expr< Some Gramext.LeftA >> -> Some Gramext.LeftA
  | <:expr< Some Gramext.RightA >> -> Some Gramext.RightA
  | _ -> raise Not_found ]
;

value rec unaction =
  fun
  [ <:expr< fun (loc : (int * int)) -> ($a$ : $_$) >> -> ([], a)
  | <:expr< fun ($p$ : $_$) -> $e$ >> ->
      let (pl, a) = unaction e in ([p :: pl], a)
  | <:expr< fun _ -> $e$ >> ->
      let (pl, a) = unaction e in
      (let loc = (0, 0) in [<:patt< _ >> :: pl], a)
  | _ -> raise Not_found ]
;

value untoken =
  fun
  [ <:expr< ($str:x$, $str:y$) >> -> (x, y)
  | _ -> raise Not_found ]
;

type symbol =
  [ Snterm of MLast.expr
  | Snterml of MLast.expr and string
  | Slist0 of symbol
  | Slist0sep of symbol and symbol
  | Slist1 of symbol
  | Slist1sep of symbol and symbol
  | Sopt of symbol
  | Sself
  | Snext
  | Stoken of Token.pattern
  | Srules of list (list (MLast.patt * symbol) * MLast.expr) ]
;

value rec unsymbol =
  fun
  [ <:expr< Gramext.Snterm (Grammar.Entry.obj ($e$ : $_$)) >> -> Snterm e
  | <:expr< Gramext.Snterml (Grammar.Entry.obj ($e$ : $_$)) $str:s$ >> ->
      Snterml e s
  | <:expr< Gramext.Snterml (Grammar.Entry.obj ($e$ : $_$), $str:s$) >> ->
      Snterml e s
  | <:expr< Gramext.Slist0 $e$ >> -> Slist0 (unsymbol e)
  | <:expr< Gramext.Slist0sep $e1$ $e2$ >> ->
      Slist0sep (unsymbol e1) (unsymbol e2)
  | <:expr< Gramext.Slist0sep ($e1$, $e2$) >> ->
      Slist0sep (unsymbol e1) (unsymbol e2)
  | <:expr< Gramext.Slist1 $e$ >> -> Slist1 (unsymbol e)
  | <:expr< Gramext.Slist1sep $e1$ $e2$ >> ->
      Slist1sep (unsymbol e1) (unsymbol e2)
  | <:expr< Gramext.Slist1sep ($e1$, $e2$) >> ->
      Slist1sep (unsymbol e1) (unsymbol e2)
  | <:expr< Gramext.Sopt $e$ >> -> Sopt (unsymbol e)
  | <:expr< Gramext.Sself >> -> Sself
  | <:expr< Gramext.Snext >> -> Snext
  | <:expr< Gramext.Stoken $e$ >> -> Stoken (untoken e)
  | <:expr< Gramext.srules $e$ >> -> Srules (unrule_list [] e)
  | _ -> raise Not_found ]
and unpsymbol_list pl e =
  match (pl, e) with
  [ ([], <:expr< [] >>) -> []
  | ([p :: pl], <:expr< [$e$ :: $el$] >>) ->
      [(p, unsymbol e) :: unpsymbol_list pl el]
  | _ -> raise Not_found ]
and unrule =
  fun
  [ <:expr< ($e1$, Gramext.action $e2$) >> ->
      let (pl, a) = unaction e2 in
      let sl = unpsymbol_list (List.rev pl) e1 in
      (sl, a)
  | _ -> raise Not_found ]
and unrule_list rl =
  fun
  [ <:expr< [$e$ :: $el$] >> -> unrule_list [unrule e :: rl] el
  | <:expr< [] >> -> rl
  | _ -> raise Not_found ]
;

value unlevel =
  fun
  [ <:expr< ($e1$, $e2$, $e3$) >> ->
      (unlabel e1, unassoc e2, unrule_list [] e3)
  | _ -> raise Not_found ]
;

value rec unlevel_list =
  fun
  [ <:expr< [$e$ :: $el$] >> -> [unlevel e :: unlevel_list el]
  | <:expr< [] >> -> []
  | _ -> raise Not_found ]
;

value unentry =
  fun
  [ <:expr< (Grammar.Entry.obj ($e$ : Grammar.Entry.e '$_$), $pos$, $ll$) >> ->
      (e, unposition pos, unlevel_list ll)
  | _ -> raise Not_found ]
;

value rec unentry_list =
  fun
  [ <:expr< [$e$ :: $el$] >> -> [unentry e :: unentry_list el]
  | <:expr< [] >> -> []
  | _ -> raise Not_found ]
;

value unextend_body e =
  let (globals, e) =
    match e with
    [ <:expr< let $list:pel$ in $e1$ >> ->
        try (get_globals pel, e1) with
        [ Not_found -> ([], e) ]
    | _ -> ([], e) ]
  in
  let e =
    match e with
    [ <:expr<
        let grammar_entry_create s =
          Grammar.Entry.create (Grammar.of_entry $_$) s
        in
        $e$ >> ->
       let e =
         match e with
         [ <:expr< let $list:pel$ in $e1$ >> ->
             try let _ = get_locals pel in e1 with
             [ Not_found -> e ]
         | _ -> e ]
       in
       e
    | _ -> e ]
  in
  let el = unentry_list e in
  (globals, el)
;

(* Printing *)

value ident s k = HVbox [: `S LR s; k :];
value string s k = HVbox [: `S LR ("\"" ^ String.escaped s ^ "\""); k :];

value position =
  fun
  [ None -> [: :]
  | Some Gramext.First -> [: `S LR "FIRST" :]
  | Some Gramext.Last -> [: `S LR "LAST" :]
  | Some (Gramext.Before s) -> [: `S LR "BEFORE"; `string s [: :] :]
  | Some (Gramext.After s) -> [: `S LR "AFTER"; `string s [: :] :]
  | Some (Gramext.Level s) -> [: `S LR "LEVEL"; `string s [: :] :] ]
;

value action expr a dg k =
  expr a dg k
;

value token (con, prm) k =
  if con = "" then string prm k
  else if prm = "" then HVbox [: `S LR con; k :]
  else HVbox [: `S LR con; `string prm k :]
;

value rec symbol (expr, patt) s k =
  match s with
  [ Snterm e -> expr e "" k
  | Snterml e s -> HVbox [: `expr e "" [: :]; `S LR "LEVEL"; `string s k :]
  | Slist0 s -> HVbox [: `S LR "LIST0"; `symbol (expr, patt) s k :]
  | Slist0sep s sep ->
      HVbox
        [: `S LR "LIST0"; `symbol (expr, patt) s [: :]; `S LR "SEP";
           `symbol (expr, patt) sep k :]
  | Slist1 s -> HVbox [: `S LR "LIST1"; `symbol (expr, patt) s k :]
  | Slist1sep s sep ->
      HVbox
        [: `S LR "LIST1"; `symbol (expr, patt) s [: :]; `S LR "SEP";
           `symbol (expr, patt) sep k :]
  | Sopt s -> HVbox [: `S LR "OPT"; `symbol (expr, patt) s k :]
  | Sself -> HVbox [: `S LR "SELF"; k :]
  | Snext -> HVbox [: `S LR "NEXT"; k :]
  | Stoken tok -> token tok k
  | Srules rl -> HVbox [: `HVbox [: :]; rule_list (expr, patt) rl k :] ]
and psymbol (expr, patt) (p, s) k =
  match p with
  [ <:patt< _ >> -> symbol (expr, patt) s k
  | _ -> HVbox [: `patt p "" [: `S LR "=" :]; `symbol (expr, patt) s k :] ]
and psymbol_list expr_patt sl k =
  listws (psymbol expr_patt) (S RO ";") sl k
and rule (expr, patt) b (sl, a) dg k =
  HVbox
    [: b;
       `HOVbox
          [: `HOVbox
               [: `HVbox [: :];
                  psymbol_list (expr, patt) sl [: `S LR "->" :] :];
             `action expr a dg k :] :]
and rule_list expr_patt ll k =
  listwbws (rule expr_patt) [: `S LR "[" :] (S LR "|") ll ""
    [: `S LR "]"; k :]
;

value label =
  fun
  [ Some s -> [: `S LR ("\"" ^ String.escaped s ^ "\"") :]
  | None -> [: :] ]
;

value assoc =
  fun
  [ Some Gramext.NonA -> [: `S LR "NONA" :]
  | Some Gramext.LeftA -> [: `S LR "LEFTA" :]
  | Some Gramext.RightA -> [: `S LR "RIGHTA" :]
  | None -> [: :] ]
;

value level expr_patt b (lab, ass, ll) dg k =
  let s = rule_list expr_patt ll k in
  match (lab, ass) with
  [ (None, None) -> HVbox [: b; s :]
  | _ ->
      Vbox
        [: `HVbox [: b; label lab; assoc ass :];
           `HVbox [: `HVbox [: :]; s :] :] ]
;

value level_list expr_patt ll k =
  Vbox
    [: `HVbox [: :];
       listwbws (level expr_patt) [: `S LR "[" :] (S LR "|") ll ""
         [: `S LR "]"; k :] :]
;

value entry (expr, patt) (e, pos, ll) k =
  BEbox
    [: `HVbox [: `expr e "" [: `S RO ":" :]; position pos :];
       `level_list (expr, patt) ll [: :];
       `HVbox [: `S RO ";"; k :] :]
;

value entry_list expr_patt el k =
  Vbox [: `HVbox [: :]; list (entry expr_patt) el k :]
;

value extend_body expr_patt (globals, e) k =
  let s = entry_list expr_patt e k in
  match globals with
  [ [] -> s
  | sl ->
      HVbox
        [: `HVbox [: :];
           `HOVbox
             [: `S LR "GLOBAL"; `S RO ":"; list ident sl [: `S RO ";" :] :];
           `s :] ]
;

value extend expr patt e dg k =
  match e with
  [ <:expr< Grammar.extend $e$ >> ->
      try
        let ex = unextend_body e in
        BEbox
          [: `S LR "EXTEND"; `extend_body (expr, patt) ex [: :];
             `HVbox [: `S LR "END"; k :] :]
      with
      [ Not_found ->
          HVbox
            [: `S LR "Grammar.extend";
               `HOVbox
                  [: `S LO "(";
                     `expr e "" [: `HVbox [: `S RO ")"; k :] :] :] :] ]
  | _ -> expr e "" k ]
;

Pretty.add_pr_fun "extend" extend;
