/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel3.h"

void ATL_zrefsyrk
(
   const enum ATLAS_UPLO      UPLO,
   const enum ATLAS_TRANS     TRANS,
   const int                  N,
   const int                  K,
   const double               * ALPHA,
   const double               * A,
   const int                  LDA,
   const double               * BETA,
   double                     * C,
   const int                  LDC
)
{
/*
 * Purpose
 * =======
 *
 * ATL_zrefsyrk  performs one of the symmetric rank k operations
 *
 *    C := alpha * A * A' + beta * C,
 *
 * or
 *
 *    C := alpha * A' * A + beta * C,
 *
 * where alpha and beta are scalars, C is an n by n symmetric matrix and
 * A is an  n by k  matrix in the first case and a  k by n matrix in the
 * second case.
 *
 * Arguments
 * =========
 *
 * UPLO    (input)                       const enum ATLAS_UPLO
 *         On entry, UPLO  specifies whether the upper or lower triangu-
 *         lar part of the array C is to be referenced as follows:
 *
 *             UPLO = AtlasUpper   Only the upper triangular part of C
 *                                 is to be referenced.
 *
 *             UPLO = AtlasLower   Only the lower triangular part of C
 *                                 is to be referenced.
 *
 *         Unchanged on exit.
 *
 * TRANS   (input)                       const enum ATLAS_TRANS
 *         On entry,  TRANS  specifies the  operation to be performed as
 *         follows:
 *
 *            TRANS = AtlasNoTrans    C := alpha * A * A' + beta * C,
 *
 *            TRANS = AtlasTrans      C := alpha * A' * A + beta * C.
 *
 *         Unchanged on exit.
 *
 * N       (input)                       const int
 *         On entry, N specifies the order of the matrix C. N must be at
 *         least zero. Unchanged on exit.
 *
 * K       (input)                       const int
 *         On entry, with TRANS = AtlasNoTrans,  K  specifies the number
 *         of columns of the matrix  A,  and otherwise,  K specifies the
 *         number of rows of the  matrix A. K must be at least zero. Un-
 *         changed on exit.
 *
 * ALPHA   (input)                       const double *
 *         On entry, ALPHA specifies the scalar alpha.   When  ALPHA  is
 *         supplied as zero  then  the  entries of the matrix A need not
 *         be set on input. Unchanged on exit.
 *
 * A       (input)                       const double *
 *         On entry,  A  points  to an array of size equal to or greater
 *         than   LDA * ka * sizeof( double[2] ),   where  ka  is k when
 *         TRANS = AtlasNoTrans, and is  n otherwise. Before entry  with
 *         TRANS = AtlasNoTrans, the  leading n by k part of the array A
 *         must contain the matrix A,  otherwise the leading k by n part
 *         of the array A must contain the matrix A. Unchanged  on exit.
 *
 * LDA     (input)                       const int
 *         On entry, LDA  specifies the leading dimension of A as decla-
 *         red  in  the  calling  (sub) program.  LDA  must be  at least
 *         MAX( 1, n ) when TRANS = AtlasNoTrans, and MAX( 1, k ) other-
 *         wise. Unchanged on exit.
 *
 * BETA    (input)                       const double *
 *         On entry,  BETA  specifies the scalar  beta.   When  BETA  is
 *         supplied as zero  then  the  entries of the matrix C need not
 *         be set on input. Unchanged on exit.
 *
 * C       (input/output)                double *
 *         On entry,  C  points  to an array of size equal to or greater
 *         than   LDC * n * sizeof( double[2] ),   Before   entry   with
 *         UPLO = AtlasUpper,  the  leading n by n upper triangular part
 *         of the array C must contain the upper  triangular part of the
 *         symmetric matrix  and the strictly lower triangular part of C
 *         is not referenced.  On exit, the upper triangular part of the
 *         array  C  is  overwritten by the upper triangular part of the
 *         updated  matrix.  Before  entry  with UPLO = AtlasLower,  the
 *         leading n by n lower triangular part of the array C must con-
 *         tain the lower  triangular  part  of the symmetric matrix and
 *         the strictly upper triangular part of C is not referenced. On
 *         exit, the lower triangular part of the array C is overwritten
 *         by the lower triangular part of the updated matrix.
 *
 * LDC     (input)                       const int
 *         On entry, LDC  specifies the leading dimension of A as decla-
 *         red  in  the  calling  (sub) program.  LDC  must be  at least
 *         MAX( 1, n ). Unchanged on exit.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   int                        i, iail, iaj, iajl, iali, ialj, icij, j,
                              jai, jaj, jal, jcj, l, lda2 = ( LDA << 1 ),
                              ldc2 = ( LDC << 1 ), ldcp12 = ( LDC + 1 ) << 1;
   register double            t0_i, t0_r;
/* ..
 * .. Executable Statements ..
 *
 */
   if( ( N == 0 ) ||
       ( ( Mdzero( ALPHA[0], ALPHA[1] ) || ( K == 0 ) ) &&
         Mdone( BETA[0], BETA[1] ) ) ) return;

   if( UPLO == AtlasUpper )
   {
      if( Mdzero( BETA[0], BETA[1] ) )
      {
         for( j = 0, jcj = 0; j < N; j++, jcj += ldc2 )
         {
            for( i = 0, icij = jcj; i <= j; i++, icij += 2 )
            {
               Mset( ATL_dZERO, ATL_dZERO, C[icij], C[icij+1] );
            }
         }
      }
      else if( !Mdone( BETA[0], BETA[1] ) )
      {
         for( j = 0, jcj = 0; j < N; j++, jcj += ldc2 )
         {
            for( i = 0, icij = jcj; i <= j; i++, icij += 2 )
            {
               Mdscl( BETA[0], BETA[1], C[icij], C[icij+1] );
            }
         }
      }
   }
   else
   {
      if( Mdzero( BETA[0], BETA[1] ) )
      {
         for( j = 0, jcj = 0; j < N; j++, jcj += ldcp12 )
         {
            for( i = j, icij = jcj; i < N; i++, icij += 2 )
            {
               Mset( ATL_dZERO, ATL_dZERO, C[icij], C[icij+1] );
            }
         }
      }
      else if( !Mdone( BETA[0], BETA[1] ) )
      {
         for( j = 0, jcj = 0; j < N; j++, jcj += ldcp12 )
         {
            for( i = j, icij = jcj; i < N; i++, icij += 2 )
            {
               Mdscl( BETA[0], BETA[1], C[icij], C[icij+1] );
            }
         }
      }
   }

   if( Mdzero( ALPHA[0], ALPHA[1] ) ) return;

   if( TRANS == AtlasNoTrans )
   {
      if( UPLO == AtlasUpper )
      {
         for( j = 0,      iaj  = 0, jcj  = 0;
              j < N; j++, iaj += 2, jcj += ldc2 )
         {
            for( l = 0,      iajl  = iaj,  jal  = 0;
                 l < K; l++, iajl += lda2, jal += lda2 )
            {
               Mmul( ALPHA[0], ALPHA[1], A[iajl], A[iajl+1], t0_r, t0_i );
               for( i  = 0,      iail  = jal, icij  = jcj;
                    i <= j; i++, iail += 2,   icij += 2 )
               {
                  Mmla( t0_r, t0_i, A[iail], A[iail+1], C[icij], C[icij+1] );
               }
            }
         }
      }
      else
      {
         for( j = 0,      iaj  = 0, jcj  = 0;
              j < N; j++, iaj += 2, jcj += ldc2 )
         {
            for( l = 0,      iajl  = iaj,  jal  = 0;
                 l < K; l++, iajl += lda2, jal += lda2 )
            {
               Mmul( ALPHA[0], ALPHA[1], A[iajl], A[iajl+1], t0_r, t0_i );
               for( i = j,      iail  = (j << 1)+jal, icij  = (j << 1)+jcj;
                    i < N; i++, iail += 2,            icij += 2 )
               {
                  Mmla( t0_r, t0_i, A[iail], A[iail+1], C[icij], C[icij+1] );
               }
            }
         }
      }
   }
   else
   {
      if( UPLO == AtlasUpper )
      {
         for( j = 0,      jaj  = 0,    jcj  = 0;
              j < N; j++, jaj += lda2, jcj += ldc2 )
         {
            for( i  = 0,      jai  = 0,   icij  = jcj;
                 i <= j; i++, jai += lda2, icij += 2 )
            {
               Mset( ATL_dZERO, ATL_dZERO, t0_r, t0_i );
               for( l = 0,      iali  = jai, ialj  = jaj;
                    l < K; l++, iali += 2,   ialj += 2 )
               {
                  Mmla( A[iali], A[iali+1], A[ialj], A[ialj+1], t0_r, t0_i );
               }
               Mmla( ALPHA[0], ALPHA[1], t0_r, t0_i, C[icij], C[icij+1] );
            }
         }
      }
      else
      {
         for( j = 0,      jaj  = 0,    jcj  = 0;
              j < N; j++, jaj += lda2, jcj += ldc2 )
         {
            for( i = j,      jai  = j*lda2, icij  = (j << 1)+jcj;
                 i < N; i++, jai += lda2,   icij += 2 )
            {
               Mset( ATL_dZERO, ATL_dZERO, t0_r, t0_i );
               for( l = 0,      iali  = jai, ialj  = jaj;
                    l < K; l++, iali += 2,   ialj += 2 )
               {
                  Mmla( A[iali], A[iali+1], A[ialj], A[ialj+1], t0_r, t0_i );
               }
               Mmla( ALPHA[0], ALPHA[1], t0_r, t0_i, C[icij], C[icij+1] );
            }
         }
      }
   }
/*
 * End of ATL_zrefsyrk
 */
}
