/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel2.h"

void ATL_zreftrsv
(
   const enum ATLAS_UPLO      UPLO,
   const enum ATLAS_TRANS     TRANS,
   const enum ATLAS_DIAG      DIAG,
   const int                  N,
   const double               * A,
   const int                  LDA,
   double                     * X,
   const int                  INCX
)
{
/*
 * Purpose
 * =======
 *
 * ATL_zreftrsv solves one of the systems of equations
 *
 *    A * x = b,   or   conjg( A  ) * x = b,   or
 *
 *    A'* x = b,   or   conjg( A' ) * x = b,
 *
 * where b and x are n-element vectors and  A is an n by n unit, or non-
 * unit, upper or lower triangular matrix.
 *
 * No test for  singularity  or  near-singularity  is included  in  this
 * routine. Such tests must be performed before calling this routine.
 *
 * Arguments
 * =========
 *
 * UPLO    (input)                       const enum ATLAS_UPLO
 *         On entry, UPLO  specifies whether  the  matrix is an upper or
 *         lower triangular matrix as follows:
 *
 *             UPLO = AtlasUpper   A is an upper triangular matrix.
 *
 *             UPLO = AtlasLower   A is a lower triangular matrix.
 *
 *         Unchanged on exit.
 *
 * TRANS   (input)                       const enum ATLAS_TRANS
 *         On entry,  TRANS specifies the equations to be solved as fol-
 *         lows:
 *
 *            TRANS = AtlasNoTrans     A  * x = b,
 *
 *            TRANS = AtlasConj        conjg( A  ) * x = b,
 *
 *            TRANS = AtlasTrans       A' * x = b,
 *
 *            TRANS = AtlasConjTrans   conjg( A' ) * x = b.
 *
 *         Unchanged on exit.
 *
 * DIAG    (input)                       const enum ATLAS_DIAG
 *         On entry, DIAG specifies whether or not A is unit triangu-
 *         lar as follows:
 *
 *            DIAG = AtlasUnit       A is assumed to be unit triangular,
 *
 *            DIAG = AtlasNonUnit    A is not assumed to be unit trian-
 *                                   gular.
 *
 *         Unchanged on exit.
 *
 * N       (input)                       const int
 *         On entry, N specifies the order of the matrix A. N must be at
 *         least zero. Unchanged on exit.
 *
 * A       (input)                       const double *
 *         On entry,  A  points  to an array of size equal to or greater
 *         than   LDA * n * sizeof( double[2] ).   Before   entry   with
 *         UPLO = AtlasUpper, the leading  n by n  upper triangular part
 *         of the array  A must contain the upper triangular  matrix and
 *         the strictly lower triangular part of  A  is  not referenced.
 *         Before entry with UPLO = AtlasLower, the leading n by n lower
 *         triangular  part of the array A must contain the lower trian-
 *         gular matrix  and the strictly upper triangular part  of A is
 *         not referenced. Unchanged on exit.
 *
 *         Note that when  DIAG = AtlasUnit,  the diagonal elements of A
 *         are not referenced  either,  but are assumed to be unity.
 *
 * LDA     (input)                       const int
 *         On entry, LDA  specifies the leading dimension of A as decla-
 *         red  in  the  calling  (sub) program.  LDA  must be  at least
 *         MAX( 1, n ). Unchanged on exit.
 *
 * X       (input/output)                double *
 *         On entry,  X  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCX ) ) * sizeof( double[2] ),
 *         that contains the vector x. Before entry, the incremented ar-
 *         ray X must contain the n element right-hand side vector b. On
 *         exit, X is overwritten with the solution vector x.
 *
 * INCX    (input)                       const int
 *         On entry, INCX specifies the increment for the elements of X.
 *         INCX must not be zero. Unchanged on exit.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   int                        i, iaij, iajj, incx2 = 2 * INCX, ix, j, jaj,
                              jx, lda2 = (LDA << 1), ldap12 = (LDA+1) << 1;
   register double            t0_i, t0_r;
/* ..
 * .. Executable Statements ..
 *
 */
   if( N == 0 ) return;

   if( TRANS == AtlasNoTrans )
   {
      if( UPLO == AtlasUpper )
      {
         for( j = N-1,     jaj  = (N-1)*lda2, jx  = (N-1)*incx2;
              j >= 0; j--, jaj -= lda2,       jx -= incx2 )
         {
            if( DIAG == AtlasNonUnit )
            {
               iajj = ( j << 1 ) + jaj;
               Mddiv( A[iajj], A[iajj+1], X[jx], X[jx+1] );
            }
            Mset( X[jx], X[jx+1], t0_r, t0_i );
            for( i = 0,      iaij  = jaj, ix  = 0;
                 i < j; i++, iaij += 2,   ix += incx2 )
            {
               Mmls( A[iaij], A[iaij+1], t0_r, t0_i, X[ix], X[ix+1] );
            }
         }
      }
      else
      {
         for( j = 0,      jaj  = 0,      jx  = 0;
              j < N; j++, jaj += ldap12, jx += incx2 )
         {
            if( DIAG == AtlasNonUnit )
               Mddiv( A[jaj], A[jaj+1], X[jx], X[jx+1] );
            Mset( X[jx], X[jx+1], t0_r, t0_i );
            for( i = j+1,    iaij  = jaj+2, ix  = jx+incx2;
                 i < N; i++, iaij += 2,     ix += incx2 )
            {
               Mmls( A[iaij], A[iaij+1], t0_r, t0_i, X[ix], X[ix+1] );
            }
         }
      }
   }
   else if( TRANS == AtlasConj )
   {
      if( UPLO == AtlasUpper )
      {
         for( j = N-1,     jaj  = (N-1)*lda2, jx  = (N-1)*incx2;
              j >= 0; j--, jaj -= lda2,       jx -= incx2 )
         {
            if( DIAG == AtlasNonUnit )
            {
               iajj = ( j << 1 ) + jaj;
               Mddiv( A[iajj], -A[iajj+1], X[jx], X[jx+1] );
            }
            Mset( X[jx], X[jx+1], t0_r, t0_i );
            for( i = 0,      iaij  = jaj, ix  = 0;
                 i < j; i++, iaij += 2,   ix += incx2 )
            {
               Mmls( A[iaij], -A[iaij+1], t0_r, t0_i, X[ix], X[ix+1] );
            }
         }
      }
      else
      {
         for( j = 0,      jaj  = 0,      jx  = 0;
              j < N; j++, jaj += ldap12, jx += incx2 )
         {
            if( DIAG == AtlasNonUnit )
               Mddiv( A[jaj], -A[jaj+1], X[jx], X[jx+1] );
            Mset( X[jx], X[jx+1], t0_r, t0_i );
            for( i = j+1,    iaij  = jaj+2, ix  = jx+incx2;
                 i < N; i++, iaij += 2,     ix += incx2 )
            {
               Mmls( A[iaij], -A[iaij+1], t0_r, t0_i, X[ix], X[ix+1] );
            }
         }
      }
   }
   else if( TRANS == AtlasTrans )
   {
      if( UPLO == AtlasUpper )
      {
         for( j = 0,      jaj  = 0,    jx  = 0;
              j < N; j++, jaj += lda2, jx += incx2 )
         {
            Mset( X[jx], X[jx+1], t0_r, t0_i );
            for( i = 0,      iaij  = jaj, ix  = 0;
                 i < j; i++, iaij += 2,   ix += incx2 )
            {
               Mmls( A[iaij], A[iaij+1], X[ix], X[ix+1], t0_r, t0_i );
            }
            if( DIAG == AtlasNonUnit ) Mddiv( A[iaij], A[iaij+1], t0_r, t0_i );
            Mset( t0_r, t0_i, X[jx], X[jx+1] );
         }
      }
      else
      {
         for( j = N-1,     jaj  = (N-1)*(ldap12), jx  = (N-1)*incx2;
              j >= 0; j--, jaj -= ldap12,         jx -= incx2 )
         {
            Mset( X[jx], X[jx+1], t0_r, t0_i );
            for( i = j+1,    iaij  = 2+jaj, ix  = jx + incx2;
                 i < N; i++, iaij += 2,     ix += incx2 )
            {
               Mmls( A[iaij], A[iaij+1], X[ix], X[ix+1], t0_r, t0_i );
            }
            if( DIAG == AtlasNonUnit ) Mddiv( A[jaj], A[jaj+1], t0_r, t0_i );
            Mset( t0_r, t0_i, X[jx], X[jx+1] );
         }
      }
   }
   else if( TRANS == AtlasConjTrans )
   {
      if( UPLO == AtlasUpper )
      {
         for( j = 0,      jaj  = 0,    jx  = 0;
              j < N; j++, jaj += lda2, jx += incx2 )
         {
            Mset( X[jx], X[jx+1], t0_r, t0_i );
            for( i = 0,      iaij  = jaj, ix  = 0;
                 i < j; i++, iaij += 2,   ix += incx2 )
            {
               Mmls( A[iaij], -A[iaij+1], X[ix], X[ix+1], t0_r, t0_i );
            }
            if( DIAG == AtlasNonUnit )
               Mddiv( A[iaij], -A[iaij+1], t0_r, t0_i );
            Mset( t0_r, t0_i, X[jx], X[jx+1] );
         }
      }
      else
      {
         for( j = N-1,     jaj  = (N-1)*(ldap12), jx  = (N-1)*incx2;
              j >= 0; j--, jaj -= ldap12,         jx -= incx2 )
         {
            Mset( X[jx], X[jx+1], t0_r, t0_i );
            for( i = j+1,    iaij  = 2+jaj, ix  = jx + incx2;
                 i < N; i++, iaij += 2,     ix += incx2 )
            {
               Mmls( A[iaij], -A[iaij+1], X[ix], X[ix+1], t0_r, t0_i );
            }
            if( DIAG == AtlasNonUnit ) Mddiv( A[jaj], -A[jaj+1], t0_r, t0_i );
            Mset( t0_r, t0_i, X[jx], X[jx+1] );
         }
      }
   }
/*
 * End of ATL_zreftrsv
 */
}
