/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel2.h"

void ATL_zrefhpr2
(
   const enum ATLAS_UPLO      UPLO,
   const int                  N,
   const double               * ALPHA,
   const double               * X,
   const int                  INCX,
   const double               * Y,
   const int                  INCY,
   double                     * A
)
{
/*
 * Purpose
 * =======
 *
 * ATL_zrefhpr2 performs the Hermitian rank 2 operation
 *
 *    A := alpha * x * conjg( y' ) + y * conjg( alpha * x' ) + A,
 *
 * where  alpha is a scalar, x and y are n-element vectors and A is an n
 * by n Hermitian matrix, supplied in packed form.
 *
 * Arguments
 * =========
 *
 * UPLO    (input)                       const enum ATLAS_UPLO
 *         On entry, UPLO  specifies whether the upper or lower triangu-
 *         lar part of the matrix A is supplied in the packed array A
 *         as follows:
 *
 *             UPLO = AtlasUpper   The upper triangular part of A is
 *                                 supplied in A.
 *
 *             UPLO = AtlasLower   The lower triangular part of A is
 *                                 supplied in A.
 *
 *         Unchanged on exit.
 *
 * N       (input)                       const int
 *         On entry, N specifies the order of the matrix A. N must be at
 *         least zero. Unchanged on exit.
 *
 * ALPHA   (input)                       const double *
 *         On entry, ALPHA specifies the scalar alpha.   When  ALPHA  is
 *         supplied as  zero then  the arrays X and Y need not be set on
 *         input. Unchanged on exit.
 *
 * X       (input)                       const double *
 *         On entry,  X  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCX ) ) * sizeof( double[2] ),
 *         that contains the vector x. Unchanged on exit.
 *
 * INCX    (input)                       const int
 *         On entry, INCX specifies the increment for the elements of X.
 *         INCX must not be zero. Unchanged on exit.
 *
 * Y       (input)                       const double *
 *         On entry,  Y  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCY ) ) * sizeof( double[2] ),
 *         that contains the vector y. Unchanged on exit.
 *
 * INCY    (input)                       const int
 *         On entry, INCY specifies the increment for the elements of Y.
 *         INCY must not be zero. Unchanged on exit.
 *
 * A       (input/output)                double *
 *         On entry,  A  points  to an array of size equal to or greater
 *         than   (( n*(n+1) ) / 2) * sizeof( double[2] ). Before  entry
 *         with UPLO = AtlasUpper, the array  A  must  contain the upper
 *         triangular part of the Hermitian matrix packed  sequentially,
 *         column  by  column,  so that A[0] contains a(0,0), A[ 1 ] and
 *         A[ 2 ] contain a(0,1) and a(1,1) respectively, and  so on. On
 *         exit, the array A is overwritten by the upper triangular part
 *         of the updated matrix.  Before entry with  UPLO = AtlasLower,
 *         the  array  A  must contain  the lower triangular part of the
 *         Hermitian matrix  packed sequentially, column by  column,  so
 *         that A[ 0 ] contains a(0,0), A[ 1 ] and A[ 2 ] contain a(1,0)
 *         and a(2,0) respectively, and so on. On exit, the array  A  is
 *         overwritten by  the  lower triangular part of the updated ma-
 *         trix.
 *         Note that the imaginary parts of the  diagonal elements  need
 *         not be set, they are assumed to be zero, and on exit they are
 *         set to zero.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   int                        i, iaij, incx2 = 2 * INCX, incy2 = 2 * INCY,
                              ix, iy, j, jx, jy;
   register double            t0_i, t0_r, t1_i, t1_r;
/* ..
 * .. Executable Statements ..
 *
 */
   if( ( N == 0 ) || Mdzero( ALPHA[0], ALPHA[1] ) ) return;

   if( UPLO == AtlasUpper )
   {
      for( j = 0,      iaij = 0, jx  = 0,     jy  = 0;
           j < N; j++,           jx += incx2, jy += incy2 )
      {
         Mmul( ALPHA[0],  ALPHA[1], Y[jy], -Y[jy+1], t0_r, t0_i );
         Mmul( ALPHA[0], -ALPHA[1], X[jx], -X[jx+1], t1_r, t1_i );
         for( i = 0,                 ix  = 0,     iy  = 0;
              i < j; i++, iaij += 2, ix += incx2, iy += incy2 )
         {
            Mmla( X[ix], X[ix+1], t0_r, t0_i, A[iaij], A[iaij+1] );
            Mmla( Y[iy], Y[iy+1], t1_r, t1_i, A[iaij], A[iaij+1] );
         }
         A[iaij] += X[jx] * t0_r - X[jx+1] * t0_i +
                      Y[jy] * t1_r - Y[jy+1] * t1_i;
         A[iaij+1]  = ATL_dZERO;
         iaij      += 2;
      }
   }
   else
   {
      for( j = 0,      iaij = 0, jx  = 0,     jy  = 0;
           j < N; j++,           jx += incx2, jy += incy2 )
      {
         Mmul( ALPHA[0],  ALPHA[1], Y[jy], -Y[jy+1], t0_r, t0_i );
         Mmul( ALPHA[0], -ALPHA[1], X[jx], -X[jx+1], t1_r, t1_i );
         A[iaij] += X[jx] * t0_r - X[jx+1] * t0_i +
                      Y[jy] * t1_r - Y[jy+1] * t1_i;
         A[iaij+1]  = ATL_dZERO;
         iaij      += 2;

         for( i = j+1,               ix  = jx+incx2, iy  = jy+incy2;
              i < N; i++, iaij += 2, ix += incx2,    iy += incy2 )
         {
            Mmla( X[ix], X[ix+1], t0_r, t0_i, A[iaij], A[iaij+1] );
            Mmla( Y[iy], Y[iy+1], t1_r, t1_i, A[iaij], A[iaij+1] );
         }
      }
   }
/*
 * End of ATL_zrefhpr2
 */
}
