/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel2.h"

void ATL_dreftrmv
(
   const enum ATLAS_UPLO      UPLO,
   const enum ATLAS_TRANS     TRANS,
   const enum ATLAS_DIAG      DIAG,
   const int                  N,
   const double               * A,
   const int                  LDA,
   double                     * X,
   const int                  INCX
)
{
/*
 * Purpose
 * =======
 *
 * ATL_dreftrmv performs one of the matrix-vector operations
 *
 *    x := A * x,   or   x := A'*x,
 *
 * where x is an n-element vector and  A is an n by n unit, or non-unit,
 * upper or lower triangular matrix.
 *
 * Arguments
 * =========
 *
 * UPLO    (input)                       const enum ATLAS_UPLO
 *         On entry, UPLO  specifies whether  the  matrix is an upper or
 *         lower triangular matrix as follows:
 *
 *             UPLO = AtlasUpper   A is an upper triangular matrix.
 *
 *             UPLO = AtlasLower   A is a lower triangular matrix.
 *
 *         Unchanged on exit.
 *
 * TRANS   (input)                       const enum ATLAS_TRANS
 *         On entry,  TRANS  specifies the  operation to be performed as
 *         follows:
 *
 *            TRANS = AtlasNoTrans     x := A *x,
 *
 *            TRANS = AtlasConj        x := A *x,
 *
 *            TRANS = AtlasTrans       x := A'*x,
 *
 *            TRANS = AtlasConjTrans   x := A'*x.
 *
 *         Unchanged on exit.
 *
 * DIAG    (input)                       const enum ATLAS_DIAG
 *         On entry, DIAG specifies whether or not A is unit triangu-
 *         lar as follows:
 *
 *            DIAG = AtlasUnit       A is assumed to be unit triangular,
 *
 *            DIAG = AtlasNonUnit    A is not assumed to be unit trian-
 *                                   gular.
 *
 *         Unchanged on exit.
 *
 * N       (input)                       const int
 *         On entry, N specifies the order of the matrix A. N must be at
 *         least zero. Unchanged on exit.
 *
 * A       (input)                       const double *
 *         On entry,  A  points  to an array of size equal to or greater
 *         than   LDA * n * sizeof(   double  ).   Before   entry   with
 *         UPLO = AtlasUpper, the leading  n by n  upper triangular part
 *         of the array  A must contain the upper triangular  matrix and
 *         the strictly lower triangular part of  A  is  not referenced.
 *         Before entry with UPLO = AtlasLower, the leading n by n lower
 *         triangular  part of the array A must contain the lower trian-
 *         gular matrix  and the strictly upper triangular part  of A is
 *         not referenced. Unchanged on exit.
 *
 *         Note that when  DIAG = AtlasUnit,  the diagonal elements of A
 *         are not referenced  either,  but are assumed to be unity.
 *
 * LDA     (input)                       const int
 *         On entry, LDA  specifies the leading dimension of A as decla-
 *         red  in  the  calling  (sub) program.  LDA  must be  at least
 *         MAX( 1, n ). Unchanged on exit.
 *
 * X       (input/output)                double *
 *         On entry,  X  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCX ) ) * sizeof(   double  ),
 *         that contains the vector x.  On exit,  X  is overwritten with
 *         the tranformed vector x.
 *
 * INCX    (input)                       const int
 *         On entry, INCX specifies the increment for the elements of X.
 *         INCX must not be zero. Unchanged on exit.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   int                        i, iaij, ix, j, jaj, jx, ldap1 = LDA + 1;
   register double            t0;
/* ..
 * .. Executable Statements ..
 *
 */
   if( N == 0 ) return;

   if( ( TRANS == AtlasNoTrans ) || ( TRANS == AtlasConj ) )
   {
      if( UPLO == AtlasUpper )
      {
         for( j = 0,      jaj  = 0,   jx  = 0;
              j < N; j++, jaj += LDA, jx += INCX )
         {
            t0 = X[jx];
            for( i = 0,      iaij  = jaj, ix  = 0;
                 i < j; i++, iaij += 1,   ix += INCX )
            {
               X[ix] += t0 * A[iaij];
            }
            if( DIAG == AtlasNonUnit ) X[jx] *= A[iaij];
         }
      }
      else
      {
         for( j = N-1,     jaj  = (N-1)*(ldap1), jx  = (N-1)*INCX;
              j >= 0; j--, jaj -= ldap1,         jx -= INCX )
         {
            t0 = X[jx];
            if( DIAG == AtlasNonUnit ) X[jx] *= A[jaj];
            for( i = j+1,    iaij  = jaj+1, ix  = jx + INCX;
                 i < N; i++, iaij += 1,     ix += INCX )
            {
               X[ix] += A[iaij] * t0;
            }
         }
      }
   }
   else
   {
      if( UPLO == AtlasUpper )
      {
         for( j = N-1,     jaj  = (N-1)*LDA, jx  = (N-1)*INCX;
              j >= 0; j--, jaj -= LDA,       jx -= INCX )
         {
            t0 = ATL_dZERO;
            for( i = 0,      iaij  = jaj, ix  = 0;
                 i < j; i++, iaij += 1,   ix += INCX )
            {
               t0 += A[iaij] * X[ix];
            }
            if( DIAG == AtlasNonUnit ) t0 += A[iaij] * X[jx];
            else                       t0 += X[jx];
            X[jx] = t0;
         }
      }
      else
      {
         for( j = 0,      jaj  = 0,     jx  = 0;
              j < N; j++, jaj += ldap1, jx += INCX )
         {
            t0 = X[jx];
            if( DIAG == AtlasNonUnit ) t0 *= A[jaj];
            for( i = j+1,    iaij  = jaj+1, ix  = jx + INCX;
                 i < N; i++, iaij += 1,     ix += INCX )
            {
               t0 += A[iaij] * X[ix];
            }
            X[jx] = t0;
         }
      }
   }
/*
 * End of ATL_dreftrmv
 */
}
