/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel1.h"

void ATL_zrefscal
(
   const int                  N,
   const double               * ALPHA,
   double                     * X,
   const int                  INCX
)
{
/*
 * Purpose
 * =======
 *
 * ATL_zrefscal performs the following operation:
 *
 *    x := alpha * x,
 *
 * where alpha is a scalar and x is an n-vector.
 *
 * Arguments
 * =========
 *
 * N       (input)                       const int
 *         On entry, N specifies the length of the vector x. N  must  be
 *         at least zero. Unchanged on exit.
 *
 * ALPHA   (input)                       const double *
 *         On entry, ALPHA specifies the scalar alpha.   When  ALPHA  is
 *         supplied as zero, then the entries of the incremented array X
 *         need not be set on input. Unchanged on exit.
 *
 * X       (input/output)                double *
 *         On entry,  X  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCX ) ) * sizeof( double[2] ),
 *         that contains the vector x.  On exit,  the entries of the in-
 *         cremented array X are mutiplied by alpha.
 *
 * INCX    (input)                       const int
 *         On entry, INCX specifies the increment for the elements of X.
 *         INCX must not be zero. Unchanged on exit.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   register int               i;
   int                        incx2 = 2 * INCX,  nu;
   const int                  incX2 = 2 * incx2, incX3 = 3 * incx2,
                              incX4 = 4 * incx2;
   double                     * StX;
   register const double      alpha_r = *ALPHA, alpha_i = ALPHA[1];
   register double            x0_r, x0_i, x1_r, x1_i, x2_r, x2_i, x3_r, x3_i;
/* ..
 * .. Executable Statements ..
 *
 */
   if( ( N > 0 ) && !Mdone( alpha_r, alpha_i ) )
   {
      if( Mdzero( alpha_r, alpha_i ) )
      {
         if( ( nu = ( N >> 2 ) << 2 ) != 0 )
         {
            StX = (double *)X + nu * incx2;

            do
            {
               (*X)       = ATL_dZERO; X[incX2  ] = ATL_dZERO;
               X[1]       = ATL_dZERO; X[incX2+1] = ATL_dZERO;
               X[incx2  ] = ATL_dZERO; X[incX3  ] = ATL_dZERO;
               X[incx2+1] = ATL_dZERO; X[incX3+1] = ATL_dZERO;

               X  += incX4;

            } while( X != StX );
         }

         for( i = N - nu; i != 0; i-- )
         {
            *X   = ATL_dZERO;
            X[1] = ATL_dZERO;
            X   += incx2;
         }
      }
      else
      {
         if( ( nu = ( N >> 2 ) << 2 ) != 0 )
         {
            StX = (double *)X + nu * incx2;

            do
            {
               x0_r  = (*X);       x2_r  = X[incX2  ];
               x0_i  = X[1];       x2_i  = X[incX2+1];
               x1_r  = X[incx2  ]; x3_r  = X[incX3  ];
               x1_i  = X[incx2+1]; x3_i  = X[incX3+1];

               (*X)       = alpha_r * x0_r - alpha_i * x0_i;
               X[incx2  ] = alpha_r * x1_r - alpha_i * x1_i;
               X[1]       = alpha_i * x0_r + alpha_r * x0_i;
               X[incx2+1] = alpha_i * x1_r + alpha_r * x1_i;
               X[incX2  ] = alpha_r * x2_r - alpha_i * x2_i;
               X[incX3  ] = alpha_r * x3_r - alpha_i * x3_i;
               X[incX2+1] = alpha_i * x2_r + alpha_r * x2_i;
               X[incX3+1] = alpha_i * x3_r + alpha_r * x3_i;

               X  += incX4;

            } while( X != StX );
         }

         for( i = N - nu; i != 0; i-- )
         {
            x0_r  = (*X);
            x0_i  = X[1];
            (*X)  = alpha_r * x0_r - alpha_i * x0_i;
            X[1]  = alpha_i * x0_r + alpha_r * x0_i;
            X    += incx2;
         }
      }
   }
/*
 * End of ATL_zrefscal
 */
}
