/*
 * Copyright 1998 by Albert Cahalan; all rights reserved.
 * This file may be used subject to the terms and conditions of the
 * GNU Library General Public License Version 2, or any later version
 * at your option, as published by the Free Software Foundation.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 */
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/utsname.h>
#include "proc/procps.h"
#include "proc/version.h"
#include "proc/sysinfo.h" /* smp_num_cpus */

#define KSYMS_FILENAME "/proc/ksyms"

#if 0
#undef KSYMS_FILENAME
#define KSYMS_FILENAME  "/home/albert/ps/45621/ksyms-2.3.12"
#define SYSMAP_FILENAME "/home/albert/ps/45621/System.map-2.3.12"
#define linux_version_code 131852 /* 2.3.12 */
#define smp_num_cpus 2
#endif

#if 0
#undef KSYMS_FILENAME
#define KSYMS_FILENAME  "/home/albert/ps/45621/ksyms-2.3.18ac8-MODVERS"
#define SYSMAP_FILENAME "/home/albert/ps/45621/System.map-2.3.18ac8-MODVERS"
#define linux_version_code 131858 /* 2.3.18ac8 */
#define smp_num_cpus 2
#endif

#if 0
#undef KSYMS_FILENAME
#define KSYMS_FILENAME  "/home/albert/ps/45621/ksyms-2.3.18ac8-NOMODVERS"
#define SYSMAP_FILENAME "/home/albert/ps/45621/System.map-2.3.18ac8-NOMODVERS"
#define linux_version_code 131858 /* 2.3.18ac8 */
#define smp_num_cpus 2
#endif

/*
 * '?' is a symbol type
 * '.' is part of a name (versioning?)
 * "\t[]" are for the module name in /proc/ksyms
 */
#define LEGAL_SYSMAP_CHARS "0123456789_ ?.\n\t[]" \
                     "abcdefghijklmnopqrstuvwxyz" \
                     "ABCDEFGHIJKLMNOPQRSTUVWXYZ"

/* These are the symbol types, with relative popularity:
 *     4 W  (not for i386)
 *    60 R
 *   100 A
 *   125 r
 *   363 s  (not for i386)
 *   858 B
 *   905 g  (not for i386)
 *   929 G  (not for i386)
 *  1301 b
 *  2750 D
 *  4481 d
 * 11417 ?
 * 13666 t
 * 15442 T
 *
 * For i386, that is: "RArBbDd?tT"
 */

#define SYMBOL_TYPE_CHARS "Tt?dDbBrARGgsW"

typedef struct symb {
  const char *name;
  unsigned long addr;
} symb;

static const symb fail = { "?", 0 };
static const char *dash = "-";

/* These mostly rely on POSIX to make them zero. */

static const symb hashtable[256];

static char       *sysmap_data;
static unsigned    sysmap_room;
static symb       *sysmap_index;
static unsigned    sysmap_count;

static char       *ksyms_data;
static unsigned    ksyms_room     = 4096;
static symb       *ksyms_index;
static unsigned    ksyms_count;
static int         idx_room;

/***********************************/

static const symb *search(unsigned long address, symb *idx, unsigned count){
  unsigned left;
  unsigned mid;
  unsigned right;
  if(!idx) return NULL;   /* maybe not allocated */
  if(address < idx[0].addr) return NULL;
  if(address >= idx[count-1].addr) return idx+count-1;
  left  = 0;
  right = count-1;
  for(;;){
    mid = (left + right) / 2;
    if(address >= idx[mid].addr) left  = mid;
    if(address <= idx[mid].addr) right = mid;
    if(right-left <= 1) break;
  }
  if(address == idx[right].addr) return idx+right;
  return idx+left;
}

/*********************************/

/* allocate if needed, read, and return buffer size */
static void read_file(const char *filename, char **bufp, unsigned *roomp) {
  int fd = 0;
  ssize_t done;
  char *buf;
  ssize_t total = 0;
  unsigned room = *roomp;
  buf = *bufp;
  if(!room) goto hell;     /* failed before */
  if(!buf) buf = malloc(room);
  if(!buf) goto hell;
open_again:
  fd = open(filename, O_RDONLY|O_NOCTTY|O_NONBLOCK);
  if(fd<0){
    switch(errno){
    case EINTR:  goto open_again;
    default:     _exit(101);
    case EACCES:   /* somebody screwing around? */
      /* FIXME: set a flag to disable symbol lookup? */
    case ENOENT:   /* no module support */
    }
    goto hell;
  }
  for(;;){
    done = read(fd, buf+total, room-total-1);
    if(done==0) break;  /* nothing left */
    if(done==-1){
      if(errno==EINTR) continue;  /* try again */
      perror("");
      goto hell;
    }
    if(done==(ssize_t)room-total-1){
      char *tmp;
      total += done;
      /* more to go, but no room in buffer */
      room *= 2;
      tmp = realloc(buf, room);
      if(!tmp) goto hell;
      buf = tmp;
      continue;
    }
    if(done>0 && done<(ssize_t)room-total-1){
      total += done; 
      continue;   /* OK, we read some. Go do more. */
    }
    fprintf(stderr,"%ld can't happen\n", (long)done);
    /* FIXME: memory leak */
    _exit(42);
  }
  *bufp = buf;
  *roomp = room;
  return;
hell:
  if(buf) free(buf);
  *bufp = NULL;
  *roomp = 0;   /* this function will never work again */
  total = 0;
  return;
}

/*********************************/

static int parse_ksyms(void) {
  char *endp;
  if(!ksyms_room || !ksyms_data) goto quiet_goodbye;
  endp = ksyms_data;
  if(idx_room) goto bypass;  /* some space already allocated */
  idx_room = 512;
  for(;;){
    void *vp;
    idx_room *= 2;
    vp = realloc(ksyms_index, sizeof(symb)*idx_room);
    if(!vp) goto bad_alloc;
    ksyms_index = vp;
bypass:
    for(;;){
      char *saved;
      if(!*endp) return 1;
      ksyms_index[ksyms_count].addr = strtoul(endp, &endp, 16);
      if(*endp != ' ') goto bad_parse;
      endp++;
      ksyms_index[ksyms_count].name = endp;
      saved = endp;
      endp = strchr(endp,'\n');
      if(!endp) goto bad_parse;   /* no newline */
      *endp = '\0';
      if(1){ /* The Killing Scope */
        char *cp;
        cp = strchr(endp,'\t');
        if(cp) *cp = '\0';  /* kill trailing module name */
        if(smp_num_cpus==1){
          saved = strrchr(saved, '_');
          if(saved && saved[1]=='R' && saved[2] && !(strlen(saved+2)&7)){
            /* could be crummy version info */
            char *endp2;
            strtoul(saved+2, &endp2, 16);
            if(!*endp2) *saved = '\0';  /* Die version, DIE!!! */
          }
        }else{
          saved = strstr(saved, "_Rsmp_");
          if(saved && !(strlen(saved+6)&7)){
            /* could be crummy version info */
            char *endp2;
            strtoul(saved+6, &endp2, 16);
            if(!*endp2) *saved = '\0';  /* Die version, DIE!!! */
          }
        }
      } /* end killing scope */
      ++endp;
      if(++ksyms_count >= idx_room) break;  /* need more space */
    }
  }

  if(0){
bad_alloc:
    fprintf(stderr, "Warning: not enough memory available\n");
  }
  if(0){
bad_parse:
    fprintf(stderr, "Warning: "KSYMS_FILENAME" not normal\n");
  }
quiet_goodbye:
  idx_room = 0;
  if(ksyms_data) free(ksyms_data) , ksyms_data = NULL;
  ksyms_room = 0;
  if(ksyms_index) free(ksyms_index) , ksyms_index = NULL;
  ksyms_count = 0;
  return 0;
}

/*********************************/

#define VCNT 16

static int sysmap_mmap(const char *filename) {
  struct stat sbuf;
  char *endp;
  int fd;
  char Version[32];
  fd = open(filename, O_RDONLY|O_NOCTTY|O_NONBLOCK);
  if(fd<0) return 0;
  if(fstat(fd, &sbuf) < 0) goto bad_open;
  if(!S_ISREG(sbuf.st_mode)) goto bad_open;
  if(sbuf.st_size < 5000) goto bad_open;  /* if way too small */
  /* Would be shared read-only, but we want '\0' after each name. */
  endp = mmap(0, sbuf.st_size + 1, PROT_READ|PROT_WRITE, MAP_PRIVATE, fd, 0);
  sysmap_data = endp;
  if(sysmap_data == (caddr_t) -1) goto bad_open;
  close(fd);
  fd = -1;
  sprintf(Version, "Version_%d", linux_version_code);
  sysmap_room = 512;
  for(;;){
    void *vp;
    sysmap_room *= 2;
    vp = realloc(sysmap_index, sizeof(symb)*sysmap_room);
    if(!vp) goto bad_alloc;
    sysmap_index = vp;
    for(;;){
      char *vstart;
      if(!*endp){                /* if we reached the end */
        int i = VCNT;            /* check VCNT times to verify this file */
        if(*Version) goto bad_version;
        if(!ksyms_index) return 1; /* if can not verify, assume success */
        while(i--){
#if 1
          const symb *findme;
          const symb *map_symb;
          /* Choose VCNT entries from /proc/ksyms to test */
          findme = ksyms_index + (ksyms_count*i/VCNT);
          /* Search for them in the System.map */
          map_symb = search(findme->addr, sysmap_index, sysmap_count);
          if(map_symb){
            if(map_symb->addr != findme->addr) continue;
            /* backup to first matching address */
            while (map_symb != sysmap_index){
              if (map_symb->addr != (map_symb-1)->addr) break;
              map_symb--;
            }
            /* search for name in symbols with same address */
            while (map_symb != (sysmap_index+sysmap_count)){
              if (map_symb->addr != findme->addr) break;
              if (!strcmp(map_symb->name,findme->name)) goto good_match;
              map_symb++;
            }
            map_symb--; /* backup to last symbol with matching address */
            fprintf(stderr, "{%s} {%s}\n",map_symb->name,findme->name);
            goto bad_match;
          }
good_match:;
#endif
        }
        return 1; /* success */
      }
      sysmap_index[sysmap_count].addr = strtoul(endp, &endp, 16);
      if(*endp != ' ') goto bad_parse;
      endp++;
      if(!strchr(SYMBOL_TYPE_CHARS, *endp)) goto bad_parse;
      endp++;
      if(*endp != ' ') goto bad_parse;
      endp++;
      sysmap_index[sysmap_count].name = endp;
      vstart = endp;
      endp = strchr(endp,'\n');
      if(!endp) goto bad_parse;   /* no newline */
      *endp = '\0';
      ++endp;
      if(*vstart=='V' && *Version && !strcmp(Version,vstart)) *Version='\0';
      if(++sysmap_count >= sysmap_room) break;  /* need more space */
    }
  }

  if(0){
bad_match:
    fprintf(stderr, "Warning: %s does not match kernel data.\n", filename);
  }
  if(0){
bad_version:
    fprintf(stderr, "Warning: %s has an incorrect kernel version.\n", filename);
  }
  if(0){
bad_alloc:
    fprintf(stderr, "Warning: not enough memory available\n");
  }
  if(0){
bad_parse:
    fprintf(stderr, "Warning: %s not parseable as a System.map\n", filename);
  }
  if(0){
bad_open:
    fprintf(stderr, "Warning: %s could not be opened as a System.map\n", filename);
  }

  sysmap_room=0;
  sysmap_count=0;
  if(sysmap_index) free(sysmap_index);
  sysmap_index = NULL;
  if(fd>=0) close(fd);
  if(sysmap_data) munmap(sysmap_data, sbuf.st_size + 1);
  sysmap_data = NULL;
  return 0;
}

/*********************************/

static void read_and_parse(void){
  static time_t stamp;    /* after data gets old, load /proc/ksyms again */
  if(time(NULL) != stamp){
    read_file(KSYMS_FILENAME, &ksyms_data, &ksyms_room);
    parse_ksyms();
    memset((void*)hashtable,0,sizeof(hashtable)); /* invalidate cache */
    stamp = time(NULL);
  }
}

/*********************************/

int open_psdb(const char *override) {
  static const char *sysmap_paths[] = {
    "/boot/System.map-%s",
    "/boot/System.map",
    "/lib/modules/%s/System.map",
    "/usr/src/linux/System.map",
    "/System.map",
    NULL
  };
  struct utsname uts;
  char path[64];
  const char **fmt = sysmap_paths;
  const char *env;
  read_and_parse();
#ifdef SYSMAP_FILENAME    /* debug feature */
  override = SYSMAP_FILENAME;
#endif
  if(override){           /* ought to search some path */
    if(sysmap_mmap(override)) return 0;
    return -1;           /* ought to return "Namelist not found." */
    /* failure is better than ignoring the user & using bad data */
  }
  /* Arrrgh, the old man page and code did not match. */
  if ((env = getenv("PS_SYSMAP"))     && sysmap_mmap(env))    return 0;
  if ((env = getenv("PS_SYSTEM_MAP")) && sysmap_mmap(env))    return 0;
  uname(&uts);
  do{
    snprintf(path, sizeof path, *fmt, uts.release);
    if (sysmap_mmap(path)) return 0;
  }while(*++fmt);
  /* TODO: Without System.map, no need to keep ksyms loaded. */
  return -1;
}

/***************************************/

#define MAX_OFFSET (0x1000*sizeof(long))  /* past this is generally junk */

/* return pointer to temporary static buffer with function name */
const char * wchan(unsigned long address) {
  const symb *mod_symb;
  const symb *map_symb;
  const symb *good_symb;
  const char *ret;
  unsigned hash = (address >> 4) & 0xff;  /* got 56/63 hits & 7/63 misses */
  if(!address) return dash;
  read_and_parse();

  if(hashtable[hash].addr == address) return hashtable[hash].name;
  mod_symb = search(address, ksyms_index,  ksyms_count);
  if(!mod_symb) mod_symb = &fail;
  map_symb = search(address, sysmap_index, sysmap_count);
  if(!map_symb) map_symb = &fail;

  /* which result is closest? */
  good_symb = (mod_symb->addr > map_symb->addr)
            ? mod_symb
            : map_symb
  ;
  if(address > good_symb->addr + MAX_OFFSET) good_symb = &fail;

  /* good_symb->name has the data, but needs to be trimmed */
  ret = good_symb->name;
  switch(*ret){
    case 's': if(!strncmp(ret, "sys_", 4)) ret += 4;   break;
    case 'd': if(!strncmp(ret, "do_",  3)) ret += 3;   break;
    case '_': while(*ret=='_') ret++;                  break;
  }
  /* if(!*ret) ret = fail.name; */  /* not likely (name was "sys_", etc.) */

  /* cache name after abbreviation */
  hashtable[hash].addr = address;
  hashtable[hash].name = ret;

  return ret;
}








