/*
 * A rewrite of the original Debian's start-stop-daemon Perl script
 * in C (faster - it is executed many times during system startup).
 *
 * Written by Marek Michalkiewicz <marekm@i17linuxb.ists.pwr.wroc.pl>,
 * public domain.  Based conceptually on start-stop-daemon.pl, by Ian
 * Jackson <ijackson@gnu.ai.mit.edu>.  May be used and distributed
 * freely for any purpose.  Changes by Christian Schwarz
 * <schwarz@monet.m.isar.de>, to make output conform to the Debian
 * Console Message Standard, also placed in public domain.  Minor
 * changes by Klee Dienes <klee@debian.org>, also placed in the Public
 * Domain. 
 * Modified March 1999 by Ian Main <imain@gnu.org> for use with the
 * HURD.  Changes placed under the GPL */

#define _GNU_SOURCE

#include <hurd.h>
#include <ps.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <getopt.h>
#include <pwd.h>
#include <error.h>
#include <hurd/ihash.h>

#define VERSION "version 0.4.1, 1997-01-29"

static int testmode = 0;
static int quietmode = 0;
static int exitnodo = 1;
static int start = 0;
static int stop = 0;
static int signal_nr = 15;
static const char *signal_str = NULL;
static int user_id = -1;
static const char *userspec = NULL;
static const char *cmdname = NULL;
static char *execname = NULL;
static char *startas = NULL;
static const char *pidfile = NULL;
static const char *progname = "";

static struct stat exec_stat;
static struct ps_context *context;
static struct proc_stat_list *procset;

struct pid_list {
	struct pid_list *next;
	int pid;
};

static struct pid_list *found = NULL;
static struct pid_list *killed = NULL;

static void *xmalloc(int size);
static void push(struct pid_list **list, int pid);
static void do_help(void);
static void parse_options(int argc, char * const *argv);
static int pid_is_user(int pid, int uid);
static int pid_is_cmd(int pid, const char *name);
static void check(int pid);
static void do_pidfile(const char *name);
static void do_psinit(void);
static void do_stop(void);

#ifdef __GNUC__
static void fatal(const char *format, ...)
	__attribute__((noreturn, format(printf, 1, 2)));
static void badusage(const char *msg)
	__attribute__((noreturn));
#else
static void fatal(const char *format, ...);
static void badusage(const char *msg);
#endif

static void
fatal(const char *format, ...)
{
	va_list arglist;

	fprintf(stderr, "%s: ", progname);
	va_start(arglist, format);
	vfprintf(stderr, format, arglist);
	va_end(arglist);
	putc('\n', stderr);
	exit(2);
}


static void *
xmalloc(int size)
{
	void *ptr;

	ptr = malloc(size);
	if (ptr)
		return ptr;
	fatal("malloc(%d) failed", size);
}


static void
push(struct pid_list **list, int pid)
{
	struct pid_list *p;

	p = xmalloc(sizeof(*p));
	p->next = *list;
	p->pid = pid;
	*list = p;
}


static void
do_help(void)
{
	printf("\
start-stop-daemon for Debian Linux - small and fast C version written by\n\
Marek Michalkiewicz <marekm@i17linuxb.ists.pwr.wroc.pl>, public domain.\n"
VERSION "\n\
\n\
Usage:
  start-stop-daemon -S|--start options ... -- arguments ...\n\
  start-stop-daemon -K|--stop options ...\n\
  start-stop-daemon -H|--help\n\
  start-stop-daemon -V|--version\n\
\n\
Options (at least one of --exec|--pidfile|--user is required):
  -x|--exec <executable>        program to start/check if it is running\n\
  -p|--pidfile <pid-file>       pid file to check\n\
  -u|--user <username>|<uid>    stop processes owned by this user\n\
  -n|--name <process-name>      stop processes with this name\n\
  -s|--signal <signal>          signal to send (default TERM)\n\
  -a|--startas <pathname>       program to start (default is <executable>)\n\
  -t|--test                     test mode, don't do anything\n\
  -o|--oknodo                   exit status 0 (not 1) if nothing done\n\
  -q|--quiet                    be more quiet\n\
  -v|--verbose                  be more verbose\n\
\n\
Exit status:  0 = done  1 = nothing done (=> 0 if --oknodo)  2 = trouble\n");
}


static void
badusage(const char *msg)
{
	if (msg)
		fprintf(stderr, "%s: %s\n", progname, msg);
	fprintf(stderr, "Try `%s --help' for more information.\n", progname);
	exit(2);
}

struct sigpair {
	const char *name;
	int signal;
};

const struct sigpair siglist[] = {
	{ "ABRT", SIGABRT },
	{ "ALRM", SIGALRM },
	{ "FPE", SIGFPE },
	{ "HUP", SIGHUP },
	{ "ILL", SIGILL },
	{ "INT", SIGINT },
	{ "KILL", SIGKILL },
	{ "PIPE", SIGPIPE },
	{ "QUIT", SIGQUIT },
	{ "SEGV", SIGSEGV },
	{ "TERM", SIGTERM },
	{ "USR1", SIGUSR1 },
	{ "USR2", SIGUSR2 },
	{ "CHLD", SIGCHLD },
	{ "CONT", SIGCONT },
	{ "STOP", SIGSTOP },
	{ "TSTP", SIGTSTP },
	{ "TTIN", SIGTTIN },
	{ "TTOU", SIGTTOU }
};

static int parse_signal (const char *signal_str, int *signal_nr)
{
	int i;
	for (i = 0; i < sizeof (siglist) / sizeof (siglist[0]); i++) {
		if (strcmp (signal_str, siglist[i].name) == 0) {
			*signal_nr = siglist[i].signal;
			return 0;
		}
	}
	return -1;
}

static void
parse_options(int argc, char * const *argv)
{
	static struct option longopts[] = {
		{ "help",	0, NULL, 'H'},
		{ "stop",	0, NULL, 'K'},
		{ "start",	0, NULL, 'S'},
		{ "version",	0, NULL, 'V'},
		{ "startas",	1, NULL, 'a'},
		{ "name",	1, NULL, 'n'},
		{ "oknodo",	0, NULL, 'o'},
		{ "pidfile",	1, NULL, 'p'},
		{ "quiet",	0, NULL, 'q'},
		{ "signal",	1, NULL, 's'},
		{ "test",	0, NULL, 't'},
		{ "user",	1, NULL, 'u'},
		{ "verbose",	0, NULL, 'v'},
		{ "exec",	1, NULL, 'x'},
		{ NULL,		0, NULL, 0}
	};
	int c;

	for (;;) {
		c = getopt_long(argc, argv, "HKSVa:n:op:qs:tu:vx:",
				longopts, (int *) 0);
		if (c == -1)
			break;
		switch (c) {
		case 'H':  /* --help */
			do_help();
			exit(0);
		case 'K':  /* --stop */
			stop = 1;
			break;
		case 'S':  /* --start */
			start = 1;
			break;
		case 'V':  /* --version */
			printf("start-stop-daemon " VERSION "\n");
			exit(0);
		case 'a':  /* --startas <pathname> */
			startas = optarg;
			break;
		case 'n':  /* --name <process-name> */
			cmdname = optarg;
			break;
		case 'o':  /* --oknodo */
			exitnodo = 0;
			break;
		case 'p':  /* --pidfile <pid-file> */
			pidfile = optarg;
			break;
		case 'q':  /* --quiet */
			quietmode = 1;
			break;
		case 's':  /* --signal <signal> */
			signal_str = optarg;
			break;
		case 't':  /* --test */
			testmode = 1;
			break;
		case 'u':  /* --user <username>|<uid> */
			userspec = optarg;
			break;
		case 'v':  /* --verbose */
			quietmode = -1;
			break;
		case 'x':  /* --exec <executable> */
			execname = optarg;
			break;
		default:
			badusage(NULL);  /* message printed by getopt */
		}
	}

	if (signal_str != NULL) {
		if (sscanf (signal_str, "%d", &signal_nr) != 1) {
			if (parse_signal (signal_str, &signal_nr) != 0) {
				badusage ("--signal takes a numeric argument or name of signal (KILL, INTR, ...)");
			}
		}	
	}

	if (start == stop)
		badusage("need one of --start or --stop");

	if (!execname && !pidfile && !userspec)
		badusage("need at least one of --exec, --pidfile or --user");

	if (!startas)
		startas = execname;

	if (start && !startas)
		badusage("--start needs --exec or --startas");
}


static int
pid_is_user(int pid, int uid)
{
	struct proc_stat *pstat;

	pstat = proc_stat_list_pid_proc_stat (procset, pid);
	if (pstat == NULL)
		fatal ("Error getting process information: NULL proc_stat struct");
	proc_stat_set_flags (pstat, PSTAT_PID | PSTAT_OWNER_UID);
	return (pstat->owner_uid == uid);
}


static int
pid_is_cmd(int pid, const char *name)
{
	struct proc_stat *pstat;

	pstat = proc_stat_list_pid_proc_stat (procset, pid);
	if (pstat == NULL)
		fatal ("Error getting process information: NULL proc_stat struct");
  	proc_stat_set_flags (pstat, PSTAT_PID | PSTAT_ARGS);
	return (!strcmp (name, pstat->args));
}


static void
check(int pid)
{
	/* I will try this to see if it works */
	if (execname && !pid_is_cmd(pid, execname))
		return;
	if (userspec && !pid_is_user(pid, user_id))
		return;
	if (cmdname && !pid_is_cmd(pid, cmdname))
		return;
	push(&found, pid);
}


static void
do_pidfile(const char *name)
{
	FILE *f;
	int pid;

	f = fopen(name, "r");
	if (f) {
		if (fscanf(f, "%d", &pid) == 1)
			check(pid);
		fclose(f);
	}
}

error_t
check_all (void *ptr)
{
	struct proc_stat *pstat = ptr;
	
	check (pstat->pid);
	return (0);
}

static void
do_psinit(void)
{
	error_t err;	

	err = ps_context_create (getproc (), &context);	
	if (err)
		error (1, err, "ps_context_create");
	
	err = proc_stat_list_create (context, &procset);
	if (err)
		error (1, err, "proc_stat_list_create");
	
	err = proc_stat_list_add_all (procset, 0, 0);	
	if (err)
		error (1, err, "proc_stat_list_add_all");

	/* Check all pids */
	ihash_iterate (context->procs, check_all);
}


static void
do_stop(void)
{
	char what[1024];
	struct pid_list *p;

	if (cmdname)
		strcpy(what, cmdname);
	else if (execname)
		strcpy(what, execname);
	else if (pidfile)
		sprintf(what, "process in pidfile `%s'", pidfile);
	else if (userspec)
		sprintf(what, "process(es) owned by `%s'", userspec);
	else
		fatal("internal error, please report");

	if (!found) {
		if (quietmode <= 0)
			printf("No %s found running; none killed.\n", what);
		exit(exitnodo);
	}
	for (p = found; p; p = p->next) {
		if (testmode)
			printf("Would send signal %d to %d.\n",
			       signal_nr, p->pid);
		else if (kill(p->pid, signal_nr) == 0)
			push(&killed, p->pid);
		else
			printf("%s: warning: failed to kill %d: %s\n",
			       progname, p->pid, strerror(errno));
	}
	if (quietmode < 0 && killed) {
		printf("Stopped %s (pid", what);
		for (p = killed; p; p = p->next)
			printf(" %d", p->pid);
		printf(").\n");
	}
}


int
main(int argc, char **argv)
{
	progname = argv[0];

	parse_options(argc, argv);
	argc -= optind;
	argv += optind;

	if (execname && stat(execname, &exec_stat))
		fatal("stat %s: %s", execname, strerror(errno));

	if (userspec && sscanf(userspec, "%d", &user_id) != 1) {
		struct passwd *pw;

		pw = getpwnam(userspec);
		if (!pw)
			fatal("user `%s' not found\n", userspec);

		user_id = pw->pw_uid;
	}

	if (pidfile)
		do_pidfile(pidfile);
	else
		do_psinit();

	if (stop) {
		do_stop();
		exit(0);
	}

	if (found) {
		if (quietmode <= 0)
			printf("%s already running.\n", execname);
		exit(exitnodo);
	}
	if (testmode) {
		printf("Would start %s ", startas);
		while (argc-- > 0)
			printf("%s ", *argv++);
		printf(".\n");
		exit(0);
	}
	if (quietmode < 0)
		printf("Starting %s...\n", startas);
	*--argv = startas;
	execv(startas, argv);
	fatal("Unable to start %s: %s", startas, strerror(errno));
}
