/* fdisk.h -- definitions for fdisk.c
 *
 * Copyright (C) 1995-97 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *               1996-97 Michael Schlueter <schlue00@marvin.informatik.uni-dortmund.de>
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 *
 */
#ifndef _fdisk_h
#define _fdisk_h

/* $Id: fdisk.h,v 1.6 1997/08/22 15:36:26 rnhodek Exp $
 *
 * $Log: fdisk.h,v $
 * Revision 1.6  1997/08/22 15:36:26  rnhodek
 * Implemented basic support for ICD format. Should work, but conversion
 * AHDI<->ICD isn't very clever yet.
 *
 * Revision 1.5  1997/06/21 20:47:43  rnhodek
 * Added RCS keywords
 *
 * Revision 1.4  1997/06/13 12:48:50  rnhodek
 * Include version.h here, so that ALPHA_VERSION is defined everywhere
 * 
 * Revision 1.3  1997/06/12 13:43:38  rnhodek
 * Fix many small bugs here and there. The ones you find when first running a
 * program...
 * 
 * Revision 1.2  1997/06/11 19:49:15  rnhodek
 * Implemented bad sector list handling
 * 
 * Revision 1.1  1997/06/11 14:36:35  rnhodek
 * Initial revision
 * 
 * Revision 1.1.1.1  1997/06/11 14:36:35  rnhodek
 * Started using CVS for atafdisk
 *
 */

#include <setjmp.h>
#include <endian.h>

#include "version.h"

/*
 * Definitions
 */

#define SECTOR_SIZE	512

#define fieldsize(a)	(sizeof(a)/sizeof(*(a)))

#define DEFAULT_DEVICE		"/dev/sda"
#define ALTERNATE_DEVICE	"/dev/hda"
#define ALTERNATE2_DEVICE	"/dev/ada"
#define LINE_LENGTH		80
#define MAXIMUM_PARTS		64
#define SECTOR_SIZE		512

#define	PART_FLAG_VALID		0x01

#define PID_EQ(a,b)	(!memcmp( (a), (b), 3 ))
#define	IS_EXTENDED(p)	((p).rootsec != 0)

/*
 * Structures
 */

enum failure {
    usage,
    unable_to_open,
    unable_to_read,
    unable_to_seek,
    unable_to_write,
    out_of_memory,
    couldnt_save_sectors,
    general
};

enum offset { ignore, lower, deflt, upper };

struct systypes {
    unsigned char index;
    char *name;
};

enum {
    xfmt_unknown,
    xfmt_AHDI,
    xfmt_ICD
} xpart_fmt;			/* which format used for >4 part.s */

#define MAX_PRIMARY_AHDI	4
#define MAX_PRIMARY_ICD		12
#define MAX_PRIMARY		(xpart_fmt == xfmt_ICD ?	\
				 MAX_PRIMARY_ICD :		\
				 MAX_PRIMARY_AHDI)

/* partition format structures on disk */

struct apartition {
    unsigned char	flag;		/* bit 0: valid */
    char		id[3];		/* "GEM", "BGM", "XGM", or other */
    unsigned long	start;		/* start of partition */
    unsigned long	size;		/* length of partition */
} __attribute((packed));

struct rootsector {
    char		unused[0x156];	/* room for boot code */
    struct apartition	icdpart[8];	/* info for ICD-partitions 5..12 */
    char		unused2[0xc];
    unsigned long	hd_size;	/* size of disk in blocks */
    struct apartition	part[4];	/* the four primary partitions */
    unsigned long	bsl_st;		/* start of bad sector list */
    unsigned long	bsl_cnt;	/* length of bad sector list */
    unsigned short	checksum;	/* checksum for bootable disks */
} __attribute((packed));
/* (the packed attribute is necessary to avoid paddings on other systems (with
 * base alignment of 4); not strictly needed for m68k, but it makes sense e.g.
 * with removable disks, to run atari-fdisk on i386 or so...) */

/* in-core representation of a partition */

typedef struct _PARTITION {
    unsigned long	start;		/* abs. start of partition */
    unsigned long	size;		/* size of partition */
    int			flag;		/* boot flags (without VALID) */
    char		id[4];		/* ID, 0-terminated */
    unsigned long	rootsec;	/* sector# where this entry comes from;
					 * if != 0: an extended part.
					 * usually start-1, but weird
					 * partitioning tools could have set
					 * up that differently...
					 */
    int			contents_valid;	/* contains a valid filesystem */
} PARTITION;

struct bootflags {
    unsigned char	flag;
    char		*name;
    char		*short_name;
};

struct partition_ID {
    char id[4];
    char *name;
};

struct option {
    char *name;
    int  *flag;
    char *description;
};


/*
 * Global variables
 */

extern int type_open;		/* normally O_RDWR, -l option gives O_RDONLY */

extern char *disk_device;	/* sda, unless specified */
extern char *line_ptr;		/* interactive input */
extern char line_buffer[LINE_LENGTH];

extern int fd;			/* the disk */
extern int ext_index;		/* the prime extended partition */
extern int listing;		/* no aborts for fdisk -l */
extern int size_flag;
extern int force;		/* force working on disk where partitions are
				 * mounted */
extern int expert_mode;		/* allow certain operations */
extern int partitions;		/* number of existing partitions */

extern char *save_sector_file;
extern char *restore_sector_file;

extern unsigned long hd_size;	/* total size of disk */
extern unsigned long rs_hd_size; /* size of disk noted in root sector */
extern unsigned long bsl_start;	/* start of bad sector list */
extern unsigned long bsl_size;	/* size of bad sector list */
extern unsigned long saved_bsl_start; /* saved versions of above, to detect */
extern unsigned long saved_bsl_size;  /* changes */
extern int bsl_HDX_compat;	/* make HDX compatible bad sec. list */
extern int XGM_flag;		/* boot flags for XGM part. */

/* the in-core partition table */
extern PARTITION part_table[MAXIMUM_PARTS];

extern struct bootflags bootflags[];
extern int n_bootflags;
extern struct partition_ID partition_IDs[];
extern int n_partition_IDs;

#if 0
/* global partition format options */
extern int opt_unordered;
extern int opt_conversions;
extern int opt_overlapping;
extern int opt_separate_aux;
extern int opt_multi_xgm;
extern struct option global_options[];
extern int n_global_options;
#endif

extern jmp_buf listingbuf;


/*
 * Functions for byte swapping
 */

/* Atari partition table is big-endian per definition :-) */
#if BYTE_ORDER == BIG_ENDIAN
#define	swab(__var)
#else
#define	swab(__var)							\
    (sizeof(__var) == 1 ?						\
     	0 :			      					\
     sizeof(__var) == 2 ?						\
     	(__var = (__typeof(__var))swab16((unsigned short)__var))	\
     :									\
     	(__var = (__typeof(__var))swab32((unsigned long)__var)))

extern __inline__ unsigned short swab16( unsigned short val )
{
    return( (val << 8) | (val >> 8) );
}

extern __inline__ unsigned long swab32( unsigned long val )
{
    unsigned short vall = val, valh = val >> 16;
    vall = (vall << 8) | (vall >> 8);
    valh = (valh << 8) | (valh >> 8);
    return( (vall << 16) | valh );
}
#endif


/*
 * Prototypes
 */

#endif  /* _fdisk_h */

/* Local Variables: */
/* tab-width: 8     */
/* End:             */
