/*
   fbgetty : a getty for framebuffer 
   Copyright (C) 1999 Yann DRONEAUD (lch@multimania.com). 

   fbgetty is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   fbgetty is distributed in the hope that it will be useful, but 
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

 */

/*
 * Here is code that test and open devices, update utmp and 
 * signal handling
 *
 */

#include "global.h"

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include <locale.h>
#include <string.h> 

#include <fcntl.h>

#include <utmp.h>

#include <errno.h>

#include <syslog.h>
#include <signal.h>

#include <termios.h>
#include <sys/ioctl.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>


#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
# include <time.h>
#endif

/* Find the major and minor macros (probably defined in sys/type.h */
#if !defined(major) && !defined(minor)
# ifdef MAJOR_IN_SYSMACROS
#  include <sys/sysmacros.h>
# endif
# ifdef MAJOR_IN_MKDEV
#  include <sys/mkdev.h>
# endif
# if defined(__linux__) && !defined(MAJOR_IN_MKDEV) && !defined(MAJOR_IN_SYSMACROS)
#  include <linux/kdev_t.h>
#  define major(rdev) MAJOR(rdev)
#  define minor(rdev) MINOR(rdev)
# endif
#endif

#ifdef __linux__
 #include <linux/major.h>
 #include <linux/tty.h>
 #include <linux/vt.h>
 #ifdef USE_FRAME_BUFFER
  #include <linux/fb.h>
 #endif 
#endif

#include "errors.h"
#include "options.h"

#include "vt.h"
#include "prompt.h"

int init (void);
void update_utmp (void);
RETSIGTYPE hangup (int sig);
RETSIGTYPE sigterm (int sig);
#ifndef HAVE_UPDWTMP
RETSIGTYPE locktimeout (int sig);
#endif

/* 
 * initialize fbgetty 
 * parameters are set by parse_command_line()
 * 
 */
int 
init (void)
{
  struct stat    *st;
  struct vt_stat *vt_stat;

  char *dev_name;
  char *issue_name;
  char *language;

#ifdef USE_FRAME_BUFFER
  struct fb_con2fbmap con2fbmap;
#endif

  /* 
   * I think it's not fatal if fbgetty is not running as root 
   * but bad things could happen 
   */
  if ((getuid() != 0) || (getgid() != 0))
    {
      error("WARNING: %s should be run as on %s",program_invocation_short_name,fgoptions->tty_device);
      error("WARNING: i will try to continue");
    }

  /* 
   * set locale used 
   * This doesn't work for date on my system
   * but error message from the libc are translated
   *
   */
#ifdef FB_GETTY_DEBUG
  error("old locale: %s", setlocale(LC_ALL,""));
  error("test: %m");
#else
  setlocale(LC_ALL,"");
#endif

#ifdef FB_GETTY_DEBUG
  error("installing signal handlers");
#endif

  /* must use sigaction to handle signal ... */
  /* ignore some signal */
  signal(SIGQUIT,SIG_IGN);
  signal(SIGINT,SIG_IGN);
  
  /* quit when hang up */
  signal(SIGHUP,hangup);
 
  /* quit when receive SIGTERM */
  signal(SIGTERM, sigterm);

#ifdef FB_GETTY_DEBUG
  error("testing devices");
#endif

  if ((st = (struct stat *) malloc (sizeof (struct stat))) == NULL)
    fatal_error("can't allocate memory for stat files");
  
  if (strchr(fgoptions->tty_device,'/') == NULL)
    {
      dev_name = fgoptions->tty_device; /* save the device name */
      /* allocate memory for new tty_device 
       * /dev/[tty_device] (about 11 characters long)
       */
      fgoptions->tty_device = (char *) malloc(12*sizeof(char)); 
      sprintf(fgoptions->tty_device,"/dev/%s",dev_name);
    }

#ifdef USE_FRAME_BUFFER
  if (fgoptions->fb_device != NULL)
    {
      if (strchr(fgoptions->fb_device,'/') == NULL)
	{
	  dev_name = fgoptions->fb_device; /* save the device name */
	  /* allocate memory for new fb_device 
	   * /dev/[fb_device] (about 11 characters long)
	   */
	  fgoptions->fb_device = (char *) malloc(12*sizeof(char)); 
	  sprintf(fgoptions->fb_device,"/dev/%s",dev_name);
	}
    }
#endif

  /* locale issue handling */
  language = getenv("LANG");
  if (language != NULL)
    {
      issue_name = fgoptions->issue_file; /* save the issue name */
      /* 
       * allocate memory for new issue file 
       */
      fgoptions->issue_file = (char *) malloc((strlen(issue_name) + strlen(language) + 2) * sizeof(char)); 
      sprintf(fgoptions->issue_file,"%s.%s",issue_name,language);

      if (stat (fgoptions->issue_file, st) == -1)
	{
	  free(fgoptions->issue_file);
          fgoptions->issue_file = issue_name;
	}
    }


  /* 
   * test of tty  device 
   * if not valid, leave the program 
   */
  if (fgoptions->tty_device != NULL)
    {
      if (stat (fgoptions->tty_device, st) == -1) 
	fatal_error("stat %s: %m",fgoptions->tty_device);

      /* test if it's a character device */
      if ((st->st_mode & S_IFMT) != S_IFCHR) 
	fatal_error("%s is not a character device",fgoptions->tty_device);

      if ((major(st->st_rdev) != TTY_MAJOR) || (minor(st->st_rdev) > MAX_NR_CONSOLES))
	fatal_error("%s: invalid tty device",fgoptions->tty_device);

      fgoptions->tty_number = minor(st->st_rdev);

      /* own the tty */
      chmod(fgoptions->tty_device, 0600);
      chown(fgoptions->tty_device, 0, st->st_gid);
    }

#ifdef USE_FRAME_BUFFER
  /* test of frame buffer device */
  /* if not valid ignore it */
  if (fgoptions->fb_device != NULL)
    {
      if ((stat(fgoptions->fb_device,st) == -1) 
	  || ((st->st_mode & S_IFMT) != S_IFCHR) 
	  || (major(st->st_rdev) != FB_MAJOR) 
	  || (minor(st->st_rdev) > FB_NUM_MINORS))
	    {
	      fgoptions->fb_device = NULL;
	      error("invalid framebuffer device file %s",fgoptions->fb_device);
	      error("framebuffer disabled");
	    }

      fgoptions->fb_number = minor(st->st_rdev) >> FB_MODES_SHIFT;

    }
#endif

  /* free memory allocated to the stat buffer (no need to comment !) */
  free(st);

 
  /*************************************************************************
   * And now start the real work 
   *************************************************************************/
    
#ifndef HAVE_LOGIN_TTY
#ifdef FB_GETTY_DEBUG
  error("closing old tty");
#endif
  /* Close old std devices */
  close (STDERR_FILENO);
  close (STDIN_FILENO);
  close (STDOUT_FILENO);
#endif /* HAVE_LOGIN_TTY */
  
#ifdef FB_GETTY_DEBUG
  error("opening tty");
#endif

  /* open the tty device (stdout) */
  if ((fgoptions->tty_fd = open (fgoptions->tty_device, O_RDWR | O_SYNC )) == -1) 
    fatal_error("opening tty device %s: %m",fgoptions->tty_device);
  
#ifdef HAVE_LOGIN_TTY
  /* 
   * login_tty is a facility of glibc 2.0 
   * located in libutils                  
   */
#ifdef FB_GETTY_DEBUG
  error("using login_tty");
#endif
  if (login_tty(fgoptions->tty_fd) == -1)  
    fatal_error("can't open stdout,in,err: %m");

  fgoptions->tty_fd = STDIN_FILENO;
#endif
  
#ifndef HAVE_LOGIN_TTY
  /* reopen stdin and stderr */
  if ((dup (fgoptions->tty_fd) != 1) || (dup (fgoptions->tty_fd) != 2)) 
    fatal_error("can't duplicate stdin and stderr on %s: %m",fgoptions->tty_device);
#endif

  /*
   * Don't buffer... the output,input 
   */
  setvbuf (stdout, NULL,_IONBF,(size_t) 0);
  setvbuf (stdin , NULL,_IONBF,(size_t) 0);
  setvbuf (stderr, NULL,_IONBF,(size_t) 0);

#if 0
  /******** Suspicious code : Must be verified ***********
   * These operation are made by login_tty 
   */
  if (tcsetpgrp(fgoptions->tty_fd,getpid()) == -1)
    error("tcsetpgrp: %m");
  
  if (setpgrp() == -1)
    error("setpgrp: %m");
  
  if (setsid () == -1)
    error ("setsid: %m");
  /*************** End of suspicous code **************/
#endif

#ifdef USE_FRAME_BUFFER
  if (fgoptions->fb_device != NULL)
    {
      /* Opening the framebuffer */
      fgoptions->fb_fd = open(fgoptions->fb_device , O_RDWR | O_SYNC );
      if (fgoptions->fb_fd == -1)
	{
	  error("can't open framebuffer device %s : %m",fgoptions->fb_device);
	  fgoptions->fb_device = NULL;
	}

      /** NEW map the console to other framebuffer (if not fb0) missing in previous release (0.1.2) */
      con2fbmap.console = fgoptions->tty_number;
      con2fbmap.framebuffer = fgoptions->fb_number;

#ifdef FB_GETTY_DEBUG
      error("console: %d",fgoptions->tty_number);
      error("framebuffer: %d",fgoptions->fb_number);
      error("tty fd: %d",fgoptions->tty_fd);
      error("fb fd: %d",fgoptions->fb_fd);
      /* #endif */

      if (ioctl(fgoptions->fb_fd, FBIOPUT_CON2FBMAP, &con2fbmap) == -1 ) 
	{
            fatal_error("ioctl FBIOPUT_CON2FBMAP: %m");
 	}
#endif /* FB_GETTY_DEBUG */

    }
#endif  /* USE_FRAME_BUFFER */

  /* Write we are using tty (or something like that) */
  update_utmp();

  /* install the refresh code */
  install_vt_handler();

  /* flush input and output queues, important for modems */
  ioctl (STDOUT_FILENO, TCFLSH, 2);

  if ((vt_stat = (struct vt_stat *) malloc (sizeof (struct vt_stat))) == NULL)
    fatal_error("Can't allocate memory for state of virtual terminal: %m");
  
  if (ioctl(STDOUT_FILENO,VT_GETSTATE,vt_stat) == -1)
    fatal_error("ioctl VT_GETSTATE failed: %m");

  if (fgoptions->tty_number == vt_stat->v_active)
    refresh_screen(); /* Display issue */

  free(vt_stat);

  return(0);

}



/*
 * SIGHUP -> exit 
 */
RETSIGTYPE 
hangup (int sig)
{
  fatal_error("hangup received, leaving");
}

/* 
 * SIGTERM -> exit 
 */
RETSIGTYPE 
sigterm (int sig)
{
  fatal_error("sigterm received, leaving...");
}

#ifndef HAVE_UPDWTMP
/* Error while locking utmp */
RETSIGTYPE 
locktimeout (int sig)
{
  error("lock failed on wtmp");
}
#endif /* HAVE_UPDWTMP */

/*
 * update_utmp - update our utmp entry
 *
 */
void
update_utmp (void)
{
  struct utmp *ut;
  struct utmp *utp;
  pid_t pid;
  char *tty_number;
#ifndef HAVE_UPDWTMP
  int ut_fd;
#endif 

#ifdef FB_GETTY_DEBUG
  error("Updating utmp");
#endif
  
#ifndef __GLIBC__
  utmpname (_PATH_UTMP); /* This not needed on GNU systems */
#endif
  
  /* find fbgetty in /var/run/utmp (the utmp record is stored by init */
  setutent ();
  pid = getpid();
  while ((utp = getutent ()) != NULL )
    {
      if (utp->ut_pid == pid)
	break;
    }
  
  if ((ut = (struct utmp *) malloc (sizeof(struct utmp))) == NULL)
    fatal_error("can't allocate memory for utmp entry");
  
  /* copy information from the utmp entry found */
  if (utp != NULL )
    {
      
#ifdef FB_GETTY_DEBUG
      error(" ! utmp entry found ... ");
      error(" ! ut_line = %s",ut->ut_line);
      error(" ! ut_id = %s",ut->ut_id);
      error(" ! ut_type = %d",ut->ut_type);
#endif
      memcpy (ut, utp, sizeof (struct utmp));
    }
  else
    {
      
      /* some inits don't initialize utmp... */
      memset (ut, 0, sizeof (struct utmp));
      
#ifdef FB_GETTY_DEBUG
      error(" ! utmp entry not found ... ");
#endif

      if ((tty_number = (char *)  malloc(4 * sizeof(char))) == NULL)
	fatal_error("can't allocate memory for tty number");
      sprintf(tty_number,"%d",fgoptions->tty_number);

      strncpy (ut->ut_id, tty_number , 4 * sizeof(char)); /* ut_id is an array of 4 bytes */

      free(tty_number);
    }

  /** I think this is not needed */
  /* endutent (); */
  strncpy (ut->ut_user, "LOGIN", UT_NAMESIZE);
  strncpy (ut->ut_line, strrchr (fgoptions->tty_device,'/') + 1 , UT_LINESIZE);
    
  time(&ut->ut_time);
 
  ut->ut_type = LOGIN_PROCESS;
  ut->ut_pid = pid;

#ifdef FB_GETTY_DEBUG
  error(" ! values for new utmp/wtmp entries:"); 
  error(" ! ut_line = %s",ut->ut_line);
  error(" ! ut_id = %s",ut->ut_id);
#endif

  pututline (ut);
  endutent ();

#ifdef FB_GETTY_DEBUG
  error(" ! writting wtmp (%s)",_PATH_WTMP);
#endif

#ifdef HAVE_UPDWTMP
  updwtmp(_PATH_WTMP,ut); /* use the facility provided by glibc2 */
#else
  ut_fd = open (_PATH_WTMP, O_APPEND | O_WRONLY);
  if (ut_fd == -1)
    {
      error("can't open wtmp file %s: %m",_PATH_WTMP);
    }
  else
    {
      signal (SIGALRM, locktimeout);
      alarm (3);
      flock (ut_fd, LOCK_EX);
      alarm (0);
      write (ut_fd, ut, sizeof (struct utmp));
      flock (ut_fd, LOCK_UN);
      close (ut_fd);
      signal (SIGALRM, SIG_DFL); /* restore default signal handling */
    }
#endif /* HAVE_UPDWTMP */

  free(ut);

#ifdef FB_GETTY_DEBUG
  error(" ! wtmp update complete");
  error(" ! udpate_utmp finished");
#endif


}




