/*******************************************************************/
/*  bpowerd v1.2 : power daemon for Best Patriot under Linux       */
/*  Author       : Christopher Craig <ccraig@cc.gatech.edu>        */
/*  Copyright    : This program released under the GNU Public      */
/*                 License                                         */
/*******************************************************************/
#include <stdio.h>
#include <strings.h>
#include <ctype.h>
#include <signal.h>
#include <syslog.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <termio.h>
#include "config.h"

#ifdef SysV /* setup initctl defaults for SysV systems */
#include "initreq.h"
#endif

#define LINE_OK    0
#define POWER_FAIL 1 
#define LOW_BATT   2

struct termio ttyparams, init_ttyparams;  /* parameters of port */
char *device;      /* name of device */

/* fd and state are made global so that they can be used by exit_proc */
/*   during a shutdown */
int fd;              /* file descriptor of device (if open) */
int state = LINE_OK; /* state of power  */

void start_daemon(void);
void usage(void);
int check_state();
void change_state();
void exit_proc(int signum);
void killinverter(void);


#ifndef SysV
void init_state(int state);
#endif

int main (int argc, char *argv[])
{
     int i=1;       /* index for argument count */
     int killinv=0; /* boolean to determine if -k was given */

     /* input and parse arguments */
     if (argc<2) {
	  usage();
	  exit(1);
     } else if (!strcmp(argv[i], "-k")) {
	  if (argc!=3) {
	       usage();
	       exit(1);
	  } else {
	       i++;
	       killinv=1;
	  }
     } else if (argc!=2) {
	  usage();
	  exit(1);
     }
     
     device = argv[i]; /* set device */

     /* make port settings */
     if((fd=open(device, O_RDWR))<0) {
	  printf("bpowerd: error opening device (%s), quitting...\n", device);
	  exit(1);
     }
     ioctl(fd, TCFLSH, 2); /* flush I/O queues */
     ioctl(fd, TCSBRK, 1); /* wait for queues */
     ioctl(fd, TCGETA, &init_ttyparams); /* get initial settings */
     signal(SIGTERM, exit_proc); /* set up signal handler for exit */
     signal(SIGINT,  exit_proc); /* set up signal handler for shutdown */
     ioctl(fd, TCGETA, &ttyparams); /* get port settings for program */

     ttyparams.c_cflag = (B1200|CS8|CREAD|HUPCL); /* set control modes */
     ttyparams.c_iflag = 0;     /* set input modes  */
     ttyparams.c_oflag = 0;     /* set output modes */
     ttyparams.c_lflag = 0;     /* set local modes  */
     ttyparams.c_line  = 0;     /* set line discipline */
     ttyparams.c_cc[VMIN] = 1;  /* set min chars to return */
     ttyparams.c_cc[VTIME] = 0; /* no port timer    */
     close(fd);      /* close line */

     /* kill inverter if -k was given */
     if (killinv) killinverter();
          
     /* send start message */
     openlog("bpowerd", LOG_CONS|LOG_PID, LOG_DAEMON); /* open syslog */
     syslog(LOG_INFO, "monitoring UPS on %s\n", device);

#ifndef DEBUG
     start_daemon();  /* spawn daemon only if not debugging */
#endif

     check_state(); /* initialize ups state */

#ifndef SysV     
     init_state();   /* initialize system state */
#endif

     for(;;) {
	  if (check_state()) change_state();
	  sleep(WAIT);
     }
     return(1);
}

/*******************************************************************/
/* start_daemon                                                    */
/*  spawns daemon and kills parent                                 */
/*  No arguments, no return value, no variable changed             */
/*******************************************************************/
void start_daemon(void) 
{
     pid_t pid;
     
     if ((pid=fork())<0) {
	  puts("Error in fork.\n");
	  exit(1);
     } else if (pid!=0) {
	  closelog();
	  exit(0);
     }
     
     setsid();
     chdir("/");
     umask(0);
     return;
}

/*******************************************************************/
/* usage                                                           */
/*  print message stating the correct usage for the program        */
/*  No args, no return, no changes                                 */
/*******************************************************************/
void usage(void)
{
     puts("Usage: bpowerd [-k] device");
     puts(" -k     - sends the kill signal to the inverter and exits, does not run daemon.");
     puts(" device - port device the UPS is connected to. Must be disabled for logins.");
}

/*******************************************************************/
/* check_state                                                     */
/*  checks the state of the power                                  */
/*                                                                 */
/* Side-effects   : state : state of power                         */
/*                     LINE_OK    - Power OK                       */
/*                     POWER_FAIL - Power down, battery OK         */
/*                     LOW_BATT   - battery in alarm state,        */
/*                                  power down                     */
/* returns : 1 if state changed else 0                             */
/*******************************************************************/
int check_state()
{
     int prevstate = state;  /* state at opening */
     int line = TIOCM_RTS;   /* Line for alarm state */
     int flags;              /* flags from port */
     int alarm = 0;          /* Alarms Present? */
     int powerfail = 0;      /* Power failed?   */
     char input = ' ';       /* input from UPS  */

     /* Check for alarm */
     fd = open(device, O_RDWR|O_NDELAY); /* open line */   
     ioctl(fd, TCSETA, &ttyparams);      /* set params */
     ioctl(fd, TIOCMBIS, &line);         /* set line high */
     ioctl(fd, TIOCMGET, &flags);        /* check line */
     ioctl(fd, TCSETA, &init_ttyparams); /* reset line */    
     close(fd);                          /* close line */
     alarm = !( TIOCM_CAR & flags);      /* check for alarm */

#ifdef DEBUG
     printf("Alarm is %s\n", (alarm) ? "present" : "absent");
#endif
          
     if (!alarm) {
	  fd = open(device, O_RDWR|O_NDELAY);
	  ioctl(fd, TCSETA, &ttyparams);
	  write(fd, "t", 1);             /* write output to line */
	  sleep(1);
	  read(fd, &input, 1);           /* read loop */
	  ioctl(fd, TCSETA, &init_ttyparams);
	  close(fd);
	  powerfail = (input != 't');
#ifdef DEBUG
	  printf("Power is %s\n", (powerfail) ? "off" : "on");
#endif
     }

     return ((state = (alarm) ? LOW_BATT :
	      ((powerfail) ? POWER_FAIL : LINE_OK)) != prevstate);
}

#ifndef SysV
/*******************************************************************/
/* init_state                                                      */
/*   reset the files to represent the current state (BSD only)     */
/*                                                                 */
/* Side-effects: state : state of power                            */
/*        LINE_OK : power on                                       */
/*        CABLE   : possible problem with cable                    */
/*******************************************************************/
void init_state()
{
     int pwrstat, upsstat;

     unlink(PWRSTAT);
     unlink(UPSSTAT);
     pwrstat = open(PWRSTAT, O_CREAT|O_WRONLY, 0644);
     upsstat = open(UPSSTAT, O_CREAT|O_WRONLY, 0644);
     if (state) { 
	  write(pwrstat, "FAIL\n", 5);
	  write(upsstat, "CABLE\n", 6);
     } else {
	  write(pwrstat, "OK\n", 3);
	  write(upsstat, "OK\n", 3);
     }
     close(pwrstat);
     close(upsstat);
     if (state) kill(1, SIGPWR); /* signal init if bad cable */
}
#endif

/*******************************************************************/
/* change_state                                                    */
/*  changes the system state and calls init                        */
/*                                                                 */
/*  Uses: state : state of power, see check_state for details      */
/*  returns : nothing                                              */
/*******************************************************************/

void change_state()
{
#ifdef SysV
     /* SysV initctl interface (i.e. everything in this ifdef)
	written by Mitch Blevins <mblevin@debian.org> */
     
     struct init_request req;
     int devfd;

     /* Better safe than sorry */
     memset((void*) &req, '0', sizeof(req));

     req.magic = INIT_MAGIC;

     /* write UPS status to status file */
     switch(state) {
     case LINE_OK: /* Power OK */
	  syslog(LOG_ALERT, "Power returned.\n");
	  req.cmd = INIT_CMD_POWEROK;
	  break;
     case POWER_FAIL: /* Line Fail */
	  syslog(LOG_ALERT, "Power Fail, system on battery power.\n");
	  req.cmd = INIT_CMD_POWERFAIL;
	  break;
     case LOW_BATT: /* Line Fail and Low Batt */
	  syslog(LOG_CRIT, "Battery Low, please log off immediately.\n");
	  req.cmd = INIT_CMD_POWERFAILNOW;
	  break;
     default: 
	  syslog(LOG_CRIT, "Unknown error in UPS program.\n");
	  return;
	  break;
     }                                       
  
     /* Open control pipe for writing */
     if((devfd = open(INIT_FIFO, O_WRONLY)) < 0) {
	  syslog(LOG_CRIT, "Could not open init control pipe.\n");
	  return;
     }

     /* Write request to control pipe */
     if(write(devfd, &req, sizeof(req)) < sizeof(req)) {
	  syslog(LOG_CRIT, "Error writing to init control pipe.\n");
     }

     close(devfd);

#else /* use old BSD style /etc status files */

     int fdstat;

     /* write power status to status file for init */
     unlink(PWRSTAT);
     if ((fdstat = open(PWRSTAT, O_CREAT|O_WRONLY, 0644)) >= 0) {
	  if (state) 
	       write(fdstat, "FAIL\n", 5);
	  else 
	       write(fdstat, "OK\n", 3);
	  close(fdstat);
     }

     /* write UPS status to status file */
     unlink(UPSSTAT);
     if ((fdstat = open(UPSSTAT, O_CREAT|O_WRONLY, 0644)) >= 0) {
	  switch(state) {
	  case LINE_OK: /* Power OK */
	       syslog(LOG_ALERT, "Power returned.\n");
	       write(fdstat, "OK\n", 3);
	       break;
	  case POWER_FAIL: /* Line Fail */
	       syslog(LOG_ALERT, "Power Fail, system on battery power.\n");
	       write(fdstat, "FAIL\n", 5);
	       break;
	  case LOW_BATT: /* Line Fail and Low Batt */
	       syslog(LOG_CRIT, "Battery Low, please log of immediately.\n");
	       write(fdstat, "SCRAM\n", 6);
	       break;
	  default: 
	       syslog(LOG_CRIT, "Error in UPS program.\n");
	       write(fdstat, "ERROR\n", 6);
	  }                                       
	  close(fdstat);
     }
     kill(1, SIGPWR); /* signal init */
#endif
}

/*******************************************************************/
/* exit_proc                                                       */
/*  exits to program cleanly, no arguments does not return         */
/*******************************************************************/
void exit_proc(int signum)
{
     fd=open(device,O_NDELAY | O_RDWR);
     ioctl(fd,TCSETA,&init_ttyparams); /* reset port */
     close(fd);

     if(signum==SIGINT){
	  /* this was called to shutdown instead of just quit */
	  if (state == POWER_FAIL) {
	       killinverter();
	  }
     }
          
     syslog(LOG_INFO, "Process Terminated.\n");
     closelog(); /* close syslog */
     exit(2);
}

/*******************************************************************/
/* killinverter                                                    */
/*  sends shutdown signal to UPS and then quits                    */
/*  No arguments, does not return                                  */
/*******************************************************************/
void killinverter(void)
{
     int line = TIOCM_DTR;

#ifdef DEBUG
     printf("In shutdown, signaling UPS.\n");
#endif
     
     fd=open(device, O_RDWR|O_NDELAY);
     ioctl(fd, TCSETA, &ttyparams);
     ioctl(fd, TIOCMBIS, &line);
     sleep(6);
     ioctl(fd, TCSETA, &init_ttyparams);
     close(fd);
     syslog(LOG_CRIT,
	    "UPS signaled for shutdown, system power off in 2 minutes.\n");
     closelog();
     exit(0);
}


