// vscreen.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Implementation of vscreen stuff.

#include "vscreen.h"
#include <assert.h>

#include <signal.h>
#include <list.h>

vscreen *current_screen=NULL;
int should_exit=0;

bool curses_avail=false;

list<vscreen *> todelete;
// Screens waiting for their turn to be deleted.

void vscreen_handleresize()
{
  resize();
  current_screen->touch();
  current_screen->clearok(true);
  current_screen->repaint();
  current_screen->clearok(false);
}

// The following is black magic >=)

void sigresize(int sig)
{
  resize();
  current_screen->touch();
  current_screen->clearok(true);
  current_screen->repaint();
}

void init_vscreen()
{
  init_curses();
  curses_avail=true;

  if(current_screen)
    vscreen_show(current_screen);
}

vscreen::vscreen():win(NULL), use_nodelay(false), use_leaveok(true)
{
}

void vscreen_mainloop(int synch)
{
  sigset_t signals;

  sigemptyset(&signals);
  sigaddset(&signals, SIGWINCH);

  signal(SIGWINCH, sigresize);

  while(!should_exit)
    {
      sigset_t prevsignals;

      assert(current_screen && current_screen->win);

      chtype ch=current_screen->win.getch();

      assert(ch!=(chtype) ERR);
      assert(ch!=KEY_RESIZE);

      sigprocmask(SIG_BLOCK, &signals, &prevsignals);
      // Oddly enough, the part I want to *protect* from signals is my own code
      // -- the ncurses code above is actually graceful about this situation!

      if(ch=='\f')
	current_screen->repaint();
      else
	{
	  vscreen_widget *focus=current_screen->get_focus();
	  if(!focus || !focus->dispatch_char(ch))
	    current_screen->dispatch_char(ch);
	}

      for(list<vscreen *>::iterator i=todelete.begin(); i!=todelete.end(); i++)
	delete *i;

      todelete.erase(todelete.begin(), todelete.end());

      sigprocmask(SIG_SETMASK, &prevsignals, NULL);
    }
}

void vscreen_preparedelete(vscreen *victim)
{
  todelete.push_back(victim);
}

void vscreen_exitmain()
{
  should_exit=1;
}

vscreen *vscreen_show(vscreen *newscreen)
{
  vscreen *prev=current_screen;
  if(prev)
    prev->set_win(NULL);

  attrset(A_NORMAL);

  if(newscreen)
    {
      if(curses_avail)
	newscreen->set_win(rootwin);
      else
	newscreen->set_win(NULL);
    }
  // If init_curses hasn't run yet, just remember that this is the current
  // screen.
  current_screen=newscreen;

  return prev;
}

void vscreen_suspend()
{
  if(current_screen)
    current_screen->set_win(NULL);
  endwin();
  curses_avail=false;
}

void vscreen_resume()
{
  curses_avail=true;
  refresh();
  if(current_screen)
    current_screen->set_win(rootwin);
}

void vscreen_refresh()
{
  current_screen->clearok(true);
  current_screen->touch();
  current_screen->repaint();
  current_screen->refresh();
  current_screen->clearok(false);
}
