// keybindings.h, -*-c++-*-
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  A global repository for keybindings.

#include "keybindings.h"

#include <apt-pkg/error.h>

keybindings global_bindings;

hash_map<string, chtype> keynames;
hash_map<string, chtype> s_keynames;
// For simplicity, we use the convention that the names are stored in
// lowercase; however, the routines to parse keys take this into account and
// convert the input to lowercase before checking it.
// FIXME: Function keys (F0-Fx) really ought to be handled specially

bool key_tables_initialized=false;

void init_key_tables()
{
  if(key_tables_initialized)
    return;

  key_tables_initialized=true;

  keynames["tab"]='\t';
  keynames["space"]=' ';

  keynames["break"]=KEY_BREAK;
  keynames["down"]=KEY_DOWN;
  keynames["up"]=KEY_UP;
  keynames["left"]=KEY_LEFT;
  keynames["right"]=KEY_RIGHT;
  keynames["home"]=KEY_HOME;
  keynames["backspace"]=KEY_BACKSPACE;
  keynames["f0"]=KEY_F(0);
  keynames["f1"]=KEY_F(1);
  keynames["f2"]=KEY_F(2);
  keynames["f3"]=KEY_F(3);
  keynames["f4"]=KEY_F(4);
  keynames["f5"]=KEY_F(5);
  keynames["f6"]=KEY_F(6);
  keynames["f7"]=KEY_F(7);
  keynames["f8"]=KEY_F(8);
  keynames["f9"]=KEY_F(9);
  keynames["f10"]=KEY_F(10);
  keynames["delete_line"]=KEY_DL;
  keynames["insert_line"]=KEY_IL;
  keynames["delete"]=KEY_DC;
  keynames["insert"]=KEY_IC;
  keynames["insert_exit"]=KEY_EIC;
  keynames["clear"]=KEY_CLEAR;
  keynames["clear_eos"]=KEY_EOS;
  keynames["clear_eol"]=KEY_EOL;
  keynames["scrollf"]=KEY_SF;
  keynames["scrollr"]=KEY_SR;
  keynames["pagedown"]=KEY_NPAGE;
  keynames["pageup"]=KEY_PPAGE;
  keynames["enter"]=KEY_ENTER;
  keynames["return"]=KEY_ENTER;
  keynames["print"]=KEY_PRINT;
  keynames["a1"]=KEY_A1;
  keynames["a3"]=KEY_A3;
  keynames["b2"]=KEY_B2;
  keynames["c1"]=KEY_C1;
  keynames["c3"]=KEY_C3;
  keynames["backtab"]=KEY_BTAB;
  keynames["begin"]=KEY_BEG;
  keynames["cancel"]=KEY_CANCEL;
  keynames["close"]=KEY_CLOSE;
  keynames["command"]=KEY_COMMAND;
  keynames["copy"]=KEY_COPY;
  keynames["create"]=KEY_CREATE;
  keynames["end"]=KEY_END;
  keynames["exit"]=KEY_EXIT;
  keynames["find"]=KEY_FIND;
  keynames["help"]=KEY_HELP;
  keynames["mark"]=KEY_MARK;
  keynames["message"]=KEY_MESSAGE;
  keynames["move"]=KEY_MOVE;
  keynames["next"]=KEY_NEXT;
  keynames["open"]=KEY_OPEN;
  keynames["options"]=KEY_OPTIONS;
  keynames["previous"]=KEY_PREVIOUS;
  keynames["redo"]=KEY_REDO;
  keynames["reference"]=KEY_REFERENCE;
  keynames["refresh"]=KEY_REFRESH;
  keynames["replace"]=KEY_REPLACE;
  keynames["restart"]=KEY_RESTART;
  keynames["resume"]=KEY_RESUME;
  keynames["save"]=KEY_SAVE;
  keynames["select"]=KEY_SELECT;
  keynames["suspend"]=KEY_SUSPEND;
  keynames["undo"]=KEY_UNDO;

  s_keynames["begin"]=KEY_SBEG;
  s_keynames["cancel"]=KEY_SCANCEL;
  s_keynames["command"]=KEY_SCOMMAND;
  s_keynames["copy"]=KEY_SCOPY;
  s_keynames["create"]=KEY_SCREATE;
  s_keynames["delete"]=KEY_SDC;
  s_keynames["delete_line"]=KEY_SDL;
  s_keynames["end"]=KEY_SEND;
  s_keynames["clear_eol"]=KEY_SEOL;
  s_keynames["exit"]=KEY_SEXIT;
  s_keynames["find"]=KEY_SFIND;
  s_keynames["help"]=KEY_SHELP;
  s_keynames["home"]=KEY_SHOME;
  s_keynames["insert"]=KEY_SIC;
  s_keynames["left"]=KEY_SLEFT;
  s_keynames["message"]=KEY_SMESSAGE;
  s_keynames["move"]=KEY_SMOVE;
  s_keynames["next"]=KEY_SNEXT;
  s_keynames["options"]=KEY_SOPTIONS;
  s_keynames["previous"]=KEY_SPREVIOUS;
  s_keynames["print"]=KEY_SPRINT;
  s_keynames["redo"]=KEY_SREDO;
  s_keynames["replace"]=KEY_SREPLACE;
  s_keynames["right"]=KEY_SRIGHT;
  s_keynames["resume"]=KEY_SRSUME;
  s_keynames["save"]=KEY_SSAVE;
  s_keynames["suspend"]=KEY_SSUSPEND;
  s_keynames["undo"]=KEY_SUNDO;
}

// Stolen from pinfo.  I don't like the looks of it, but presumably it works
// (in some circumstances).  This is a FIXME, btw :)
/* adapted from Midnight Commander */
#define KEY_CTRL(x) ((x)&31)	/* macro to get CTRL+key sequence */
#define KEY_ALT(x) (0x200 | (x))	/* macro to get ALT+key sequence */

void keybindings::set(string tag, keybinding strokes)
{
  keymap[tag]=strokes;
}

bool keybindings::key_matches(chtype ch, string tag)
{
  hash_map<string, keybinding>::iterator found=keymap.find(tag);
  if(found==keymap.end())
    return false;
  else
    {
      for(keybinding::iterator i=found->second.begin(); i!=found->second.end(); i++)
	{
	  if(*i==KEY_ENTER)
	    {
	      if(ch==KEY_ENTER || ch=='\r' || ch=='\n')
		return true;
	    }
	  else if(ch==*i)
	    return true;
	}
      return false;
    }
}

chtype parse_key(string keystr)
{
  bool sfound=false,cfound=false,afound=false;
  string tmpstr=keystr;
  chtype rval=(chtype) ERR;

  init_key_tables();

  while(tmpstr.size()>2 && tmpstr[1]=='-')
    {
      switch(tmpstr[0])
	{
	case 's':
	case 'S':
	  sfound=true;
	  break;
	case 'a':
	case 'A':
	case 'm':
	case 'M':
	  afound=true;
	  break;
	case 'c':
	case 'C':
	  cfound=true;
	  break;
	default:
	  _error->Error("Cannot parse key description: %s", keystr.c_str());
	  return (chtype) ERR;
	}
      tmpstr=string(tmpstr,2);
    }

  if(tmpstr.size()==0)
    {
      _error->Error("Invalid null keybinding");
      return (chtype) ERR;
    }

  if(cfound && tmpstr.size()>1)
    {
      _error->Error("Sorry, control modifiers may not be used with unprintable characters");
      return (chtype) ERR;
    }

  if(tmpstr.size()==1)
    {
      rval=sfound?toupper(tmpstr[0]):tmpstr[0];
      if(cfound)
	rval=KEY_CTRL(rval);
      if(afound)
	rval=KEY_ALT(rval);
      return rval;
    }
  else
    {
      for(unsigned int i=0; i<tmpstr.size(); i++)
	tmpstr[i]=tolower(tmpstr[i]);
      hash_map<string,chtype>::iterator found=(sfound?s_keynames:keynames).find(tmpstr);
      if(found==(sfound?s_keynames:keynames).end())
	return (chtype) ERR;
      else
	{
	  rval=found->second;
	  if(afound)
	    rval=KEY_ALT(rval);

	  return rval;
	}
    }
}
