// aptcache.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.

#include "aptcache.h"

#include <apt-pkg/error.h>
#include <apt-pkg/sourcelist.h>
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/tagfile.h>
#include <apt-pkg/fileutl.h>

#include <unistd.h>
#include <stdio.h>
#include <errno.h>

aptitudeDepCache::aptitudeDepCache(MMap &Map, OpProgress &Prog, bool WithLock)
  :pkgDepCache(Map, Prog),package_states(NULL),lock(-1)
{
  if(!_error->PendingError())
    build_selection_list(Prog, WithLock);
}

bool aptitudeDepCache::build_selection_list(OpProgress &Prog, bool WithLock)
{
  string statedir=_config->FindDir("Dir::Aptitude::state", "/var/state/aptitude");
  // Should this not go under Dir:: ?  I'm not sure..
  delete package_states;
  package_states=new aptitude_state[Head().PackageCount];
  for(unsigned int i=0; i<Head().PackageCount; i++)
    {
      package_states[i].new_package=true;
      package_states[i].selection_state=package_states[i].dselect_state=State::Unknown;
      package_states[i].reinstall=false;
    }

  if(WithLock && lock==-1)
    {
      lock=GetLock(statedir+"lock");

      if(_error->PendingError())
	{
	  if(lock!=-1)
	    close(lock);
	  lock=-1;
	  return false;
	}
    }

  // Read in the states that we saved
  FileFd state_file(statedir+"pkgstates", FileFd::ReadOnly);
  if(!state_file.IsOpen())
    {
      _error->Discard();
      if(errno!=ENOENT)
	_error->Warning("Can't open Aptitude extended state file");
    }
  else
    {
      pkgTagFile tagfile(state_file);
      pkgTagSection section;
      bool do_dselect=_config->FindB("Aptitude::Track-Dselect-State", true);
      while(tagfile.Step(section))
	{
	  PkgIterator pkg=FindPkg(section.FindS("Package"));
	  if(!pkg.end()) // Silently ignore unknown packages.
	    {
	      unsigned long tmp=0;
	      aptitude_state &pkg_state=get_ext_state(pkg);

	      section.FindFlag("Unseen", tmp, 1);
	      pkg_state.new_package=(tmp==1);

	      tmp=0;
	      section.FindFlag("ReInstall", tmp, 1);
	      pkg_state.reinstall=(tmp==1);

	      pkg_state.selection_state=(State::PkgSelectedState) section.FindI("State", State::Unknown);
	      pkg_state.dselect_state=(State::PkgSelectedState) section.FindI("Dselect-State", pkg->SelectedState);
	      if(do_dselect && pkg->SelectedState!=pkg_state.dselect_state)
		{
		  MarkFromDselect(pkg);
		  pkg_state.dselect_state=(State::PkgSelectedState) pkg->SelectedState;
		}
	    }
	}
    }

  // Act on them
  for(pkgCache::PkgIterator i=PkgBegin(); !i.end(); i++)
    {
      StateCache &state=(*this)[i];
      aptitude_state &estate=get_ext_state(i);
      switch(estate.selection_state)
	{
	case State::Unknown:
	  if(i.CurrentVer().end())
	    estate.selection_state=State::DeInstall;
	  else
	    {
	      estate.selection_state=State::Install;
	      if(_config->FindB("Aptitude::Auto-Upgrade", true) && state.Status>0)
		MarkInstall(i, _config->FindB("Aptitude::Auto-Install", false));
	    }
	  break;
	case State::Install:
	  if(_config->FindB("Aptitude::Auto-Upgrade", true) && state.Status>0)
	    MarkInstall(i, _config->FindB("Aptitude::Auto-Install", false));
	  SetReInstall(i, estate.reinstall);
	  break;
	case State::Hold:
	  MarkKeep(i, false);
	  break;
	case State::DeInstall:
	  if(!i.CurrentVer().end())
	    MarkDelete(i, false);
	  break;
	case State::Purge:
	  if(!i.CurrentVer().end())
	    MarkDelete(i, true);
	  break;
	}
    }
  return true;
}

bool aptitudeDepCache::save_selection_list(OpProgress &prog)
{
  if(lock==-1)
    return true;
  string statefile=_config->FindDir("Dir::Aptitude::state", "/var/state/aptitude")+"pkgstates";

  FileFd newstate(statefile+".new", FileFd::WriteEmpty);
  if(!newstate.IsOpen())
    _error->Error("Cannot open Aptitude state file");
  else
    {
      for(PkgIterator i=PkgBegin(); !i.end(); i++)
	{
	  aptitude_state &state=get_ext_state(i);
	  char buf[400];
	  int len;

	  len=snprintf(buf,
		       400,
		       "Package: %s\nUnseen: %s\nState: %i\nDselect-State: %i\n%s\n",
		       i.Name(),
		       state.new_package?"yes":"no",
		       state.selection_state,
		       i->SelectedState,
		       state.reinstall?"ReInstall: yes\n":"");
	  if(len>=399)
	    {
	      _error->Error("Internal buffer overflow on package \"%s\" while writing state file", i.Name());
	      newstate.Close();
	      unlink((statefile+".new").c_str());
	      return false;
	    }
	  if(newstate.Failed() || !newstate.Write(buf, len))
	    {
	      _error->Error("Couldn't write state file");
	      newstate.Close();
	      unlink((statefile+".new").c_str());
	      return false;
	    }
	}
      if(newstate.Failed())
	// This is /probably/ redundant, but paranoia never hurts.
	{
	  _error->Error("Error writing state file");
	  newstate.Close();
	  unlink((statefile+".new").c_str());
	  return false;
	}
      newstate.Close();
      // FIXME!  This potentially breaks badly on NFS.. (?) -- actually, it
      //       wouldn't be harmful; you'd just get gratuitous errors..
      if(rename((statefile+".new").c_str(), statefile.c_str())==-1)
	{
	  _error->Errno("save_selection_list", "couldn't replace old state file");
	  unlink((statefile+".new").c_str());
	  return false;
	}
    }

  return true;
}

void aptitudeDepCache::forget_new()
{
  for(unsigned int i=0; i<Head().PackageCount; i++)
    package_states[i].new_package=false;
}

void aptitudeDepCache::MarkInstall(const PkgIterator &Pkg, bool AutoInst)
{
  pkgDepCache::MarkInstall(Pkg, AutoInst);

  get_ext_state(Pkg).selection_state=State::Install;
}

void aptitudeDepCache::MarkDelete(const PkgIterator &Pkg, bool Purge)
{
  pkgDepCache::MarkDelete(Pkg, Purge);
  get_ext_state(Pkg).selection_state=Purge?State::Purge:State::DeInstall;
}

void aptitudeDepCache::MarkKeep(const PkgIterator &Pkg, bool Soft, bool SetHold)
{
  pkgDepCache::MarkKeep(Pkg, Soft);
  if(Pkg.CurrentVer().end())
    {
      if((*this)[Pkg].iFlags&Purge)
	get_ext_state(Pkg).selection_state=State::Purge;
      else
	get_ext_state(Pkg).selection_state=State::DeInstall;
    }
  else if(SetHold)
    get_ext_state(Pkg).selection_state=State::Hold;
  else
    get_ext_state(Pkg).selection_state=State::Install;
}

void aptitudeDepCache::SetReInstall(const PkgIterator &Pkg, bool To)
{
  pkgDepCache::SetReInstall(Pkg, To);
  get_ext_state(Pkg).reinstall=To;
}

void aptitudeDepCache::MarkFromDselect(const PkgIterator &Pkg)
{
  aptitude_state &state=get_ext_state(Pkg);

  if(Pkg->SelectedState!=state.selection_state)
    {
      switch(Pkg->SelectedState)
	{
	case State::Unknown:
	  break;
	case State::Purge:
	  if( (!Pkg.CurrentVer().end()) || !((*this)[Pkg].iFlags&Purge) )
	    MarkDelete(Pkg, true);
	  else
	    MarkKeep(Pkg, false, false);
	  break;
	case State::DeInstall:
	  if(!Pkg.CurrentVer().end())
	    MarkDelete(Pkg, false);
	  else
	    MarkKeep(Pkg, false, false);
	  break;
	case State::Hold:
	  if(!Pkg.CurrentVer().end())
	    MarkKeep(Pkg, false, true);
	  break;
	case State::Install:
	  if(Pkg.CurrentVer().end())
	    MarkInstall(Pkg, false);
	  else
	    MarkKeep(Pkg, false, false);
	  break;
	}
    }
}

aptitudeDepCache::~aptitudeDepCache()
{
  delete[] package_states;
  if(lock!=-1)
    close(lock);
}

aptitudeCacheFile::aptitudeCacheFile():Map(NULL), Cache(NULL), Lock(NULL)
{
}

aptitudeCacheFile::~aptitudeCacheFile()
{
  delete Cache;
  delete Map;
  delete Lock;
}

bool aptitudeCacheFile::Open(OpProgress &Progress, bool WithLock)
{
  if(WithLock)
    Lock=new pkgDpkgLock;

  if(_error->PendingError())
    return false;

  pkgSourceList List;
  if(!List.ReadMainList())
    return _error->Error("The list of sources could not be read.");

  if(WithLock)
    {
      pkgMakeStatusCache(List, Progress);
      Progress.Done();
      if(_error->PendingError())
	return _error->Error("The package lists or status file could not be parsed or opened.");
      if(!_error->empty())
	_error->Warning("You may want to update the package lists to correct these missing files");

      FileFd File(_config->FindFile("Dir::Cache::pkgcache"), FileFd::ReadOnly);
      if(_error->PendingError())
	return false;

      Map=new MMap(File, MMap::Public|MMap::ReadOnly);
      if(_error->PendingError())
	return false;
    }
  else
    {
      Map=pkgMakeStatusCacheMem(List, Progress);
      Progress.Done();
      if(!Map)
	return false;
    }

  Cache=new aptitudeDepCache(*Map, Progress, WithLock);
  Progress.Done();
  if(_error->PendingError())
    return false;

  return true;
}
