/*
 * sid.c - MOS6581 (SID) emulation.
 *
 * Written by
 *   Teemu Rantanen (tvr@cs.hut.fi)
 *
 * AIX support added by
 *   Chris Sharp (sharpc@hursley.ibm.com)
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "vice.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <math.h>

#if defined(HAVE_SYS_IOCTL_H) && !defined(__MSDOS__)
#ifdef __hpux
#define _INCLUDE_HPUX_SOURCE
#endif
#include <sys/ioctl.h>
#endif

#ifndef HAVE_USLEEP
extern int usleep(unsigned long);
#endif

#include "vmachine.h"
#include "mem.h"
#include "ui.h"
#include "interrupt.h"
#include "warn.h"
#include "sid.h"
#include "vsync.h"
#include "resources.h"
#include "utils.h"

#ifdef SOUND

typedef unsigned int u32_t;
typedef int s32_t;
typedef unsigned short u16_t;
typedef short s16_t;

#ifdef SID

#define ATTACK 0
#define DECAY 1
#define SUSTAIN 2
#define RELEASE 3
#define IDLE 4

#define TESTWAVE 0
#define PULSEWAVE 1
#define SAWTOOTHWAVE 2
#define TRIANGLEWAVE 3
#define NOISEWAVE 4
#define NOWAVE 5
#define RINGWAVE 6
#define PULSETRIANGLEWAVE 7
#define PULSESAWTOOTHWAVE 8

/* noise magic */
#define B(v, n, r, x) ((((v)>>((n)-(x)+1))&((1<<(x))-1))<<(r))
#define NSHIFT(v, n) ((((v)<<(n))&0x7fffff)|(B((v),22,0,(n))^B((v),17,0,(n))))
#define NVALUE(v) (B((v),22,7,1)|B((v),20,6,1)|B((v),16,5,1)|B((v),13,4,1)|B((v),11,3,1)|B((v),7,2,1)|B((v),4,1,1)|B((v),2,0,1))
#define NSEED 0x7ffff8

typedef struct voice_s
{
    struct sound_s	*s;
    struct voice_s	*vprev;
    struct voice_s	*vnext;
    int			 nr;

    u32_t		 f;
    u32_t		 fs;
    BYTE		 fm;
    u32_t		 pw;
    
    u32_t		 adsr;
    s32_t		 adsrs;
    u32_t		 adsrz;

    BYTE		 sync;
    BYTE		 filter;
    BYTE		 update;
    BYTE		 gateflip;

    BYTE		 adsrm;
    BYTE		 attack;
    BYTE		 decay;
    BYTE		 sustain;
    BYTE		 release;

    BYTE		*d;

    u32_t		rv;
} voice_t;

typedef struct sound_s
{
    voice_t		 v[4];
    BYTE		 d[64];
    BYTE		 has3;
    BYTE		 vol;
    s16_t		*pbuf;
    s32_t		 bufptr;

    s32_t		 adrs[16];
    u32_t		 sz[16];

    u32_t		 speed1;

    warn_t		*pwarn;

    BYTE		 update;
} sound_t;

static u16_t adrtable[16] =
{
    1, 4, 8, 12, 19, 28, 34, 40, 50, 125, 250, 400, 500, 1500, 2500, 4000
};

/* XXX: check these */
static u32_t exptable[6] =
{
    0x30000000, 0x1c000000, 0x0e000000, 0x08000000, 0x04000000, 0x00000000
};

static u32_t doosc(voice_t *pv)
{
    u32_t		f = pv->f;
    switch (pv->fm)
    {
#ifndef VIC20
    case PULSESAWTOOTHWAVE:
	if (f <= pv->pw)
	    return 0x0000;
    case SAWTOOTHWAVE:
	return f >> 17;
    case RINGWAVE:
	f ^= pv->vprev->f & 0x80000000;
    case TRIANGLEWAVE:
	if (f < 0x80000000)
	    return f >> 16;
	return 0xffff - (f >> 16);
    case PULSETRIANGLEWAVE:
	if (f <= pv->pw)
	    return 0x0000;
	if (f < 0x80000000)
	    return f >> 16;
	return 0xffff - (f >> 16);
#endif
    case NOISEWAVE:
	return NVALUE(NSHIFT(pv->rv, f >> 28)) << 7;
    case PULSEWAVE:
	if (f > pv->pw)
	    return 0x7fff;
    }
    return 0x0000;
}

static void set_adsr(voice_t *pv, BYTE fm)
{
    int				i;
    switch (fm)
    {
    case ATTACK:
	pv->adsrs = pv->s->adrs[pv->attack];
	pv->adsrz = 0;
	break;
    case DECAY:
        /* XXX: fix this */
	if (pv->adsr <= pv->s->sz[pv->sustain])
	{
	    set_adsr(pv, SUSTAIN);
	    return;
	}
	for (i = 0; pv->adsr < exptable[i]; i++);
	pv->adsrs = -pv->s->adrs[pv->decay] >> i;
	pv->adsrz = pv->s->sz[pv->sustain];
	if (exptable[i] > pv->adsrz)
	    pv->adsrz = exptable[i];
	break;
    case SUSTAIN:
	if (pv->adsr > pv->s->sz[pv->sustain])
	{
	    set_adsr(pv, DECAY);
	    return;
	}
	pv->adsrs = 0;
	pv->adsrz = 0;
	break;
    case RELEASE:
	if (!pv->adsr)
	{
	    set_adsr(pv, IDLE);
	    return;
	}
	for (i = 0; pv->adsr < exptable[i]; i++);
	pv->adsrs = -pv->s->adrs[pv->release] >> i;
	pv->adsrz = exptable[i];
	break;
    case IDLE:
	pv->adsrs = 0;
	pv->adsrz = 0;
	break;
    }
    pv->adsrm = fm;
}

static void trigger_adsr(voice_t *pv)
{
    switch (pv->adsrm)
    {
    case ATTACK:
	pv->adsr = 0x7fffffff;
	set_adsr(pv, DECAY);
	break;
    case DECAY:
    case RELEASE:
	if (pv->adsr >= 0x80000000)
	    pv->adsr = 0;
	set_adsr(pv, pv->adsrm);
	break;
    }
}

#ifdef DEBUG
static void print_voice(FILE *fd, voice_t *pv)
{
    char *m = "ADSRI";
    char *w = "TPSTN-R5";
    fprintf(fd, "#SID: V%d: e=%5.1f%%(%c) w=%6.1fHz(%c) f=%5.1f%% p=%5.1f%%\n",
	    pv->nr,
	    (double)pv->adsr*100.0 / (((u32_t)1 << 31) - 1), m[pv->adsrm],
	    (double)pv->fs / (pv->s->speed1*16), w[pv->fm],
	    (double)pv->f*100.0 / ((u32_t)-1),
	    (double)pv->pw*100.0 / ((u32_t)-1));
}

static void print_sid(FILE *fd, sound_t *psid)
{
    int			i;
    fprintf(fd, "#SID: clk=%d v=%d s3=%d\n", clk, psid->vol, psid->has3);
    for (i = 0; i < 3; i++)
	print_voice(fd, &psid->v[i]);
}
#endif

inline static void setup_sid(sound_t *psid)
{
    if (!psid->update)
	return;
    psid->vol = psid->d[0x18] & 0x0f;
    psid->has3 = psid->d[0x18] & 0x80 ? 0 : 1;
    psid->v[0].filter = psid->d[0x17] & 0x01 ? 1 : 0;
    psid->v[1].filter = psid->d[0x17] & 0x02 ? 1 : 0;
    psid->v[2].filter = psid->d[0x17] & 0x04 ? 1 : 0;
    if (psid->d[0x17] & 0x07)
	warn(psid->pwarn, 0, "filters not implemented");
    psid->update = 0;
}

inline static void setup_voice(voice_t *pv)
{
    if (!pv->update)
	return;
    pv->attack = pv->d[5] / 0x10;
    pv->decay = pv->d[5] & 0x0f;
    pv->sustain = pv->d[6] / 0x10;
    pv->release = pv->d[6] & 0x0f;
    pv->pw = (pv->d[2] + (pv->d[3]&0x0f)*0x100) * 0x100100;
    pv->sync = pv->d[4] & 0x02 ? 1 : 0;
    if (pv->sync)
	warn(pv->s->pwarn, 1, "program uses hard sync");
    pv->fs = pv->s->speed1 * (pv->d[0] + pv->d[1]*0x100);
    if (pv->d[4] & 0x08)
    {
	pv->fm = TESTWAVE;
	pv->f = pv->fs = 0;
	pv->rv = NSEED;
    }
    else switch ((pv->d[4] & 0xf0) >> 4)
    {
    case 4:
	pv->fm = PULSEWAVE;
	break;
    case 2:
	pv->fm = SAWTOOTHWAVE;
	break;
    case 1:
	if (pv->d[4] & 0x04)
	{
	    pv->fm = RINGWAVE;
	    warn(pv->s->pwarn, 2, "program uses ring modulation");
	}
	else
	    pv->fm = TRIANGLEWAVE;
	break;
    case 8:
	pv->fm = NOISEWAVE;
	break;
    case 0:
	pv->fm = NOWAVE;
	break;
    case 5:
	pv->fm = PULSETRIANGLEWAVE;
	warn(pv->s->pwarn, 9, "program combines pulse and triangle waveforms");
	break;
    case 6:
	pv->fm = PULSESAWTOOTHWAVE;
	warn(pv->s->pwarn, 10,
	     "program combines pulse and sawtooth waveforms");
	break;
    default:
	pv->fm = NOWAVE;
	warn(pv->s->pwarn, 3, "program combines waveforms");
    }
    switch (pv->adsrm)
    {
    case ATTACK:
    case DECAY:
    case SUSTAIN:
	if (pv->d[4] & 0x01)
	    set_adsr(pv, pv->gateflip ? ATTACK : pv->adsrm);
	else
	    set_adsr(pv, RELEASE);
	break;
    case RELEASE:
    case IDLE:
	if (pv->d[4] & 0x01)
	    set_adsr(pv, ATTACK);
	else
	    set_adsr(pv, pv->adsrm);
	break;
    }
    pv->update = 0;
    pv->gateflip = 0;
}

#ifdef VIC20
inline static void update_sid(sound_t *psid, int nr)
{
    int			i;
    u32_t		o0, o1, o2, o3;
    for (i = 0; i < nr; i++)
    {
	/* addfptrs */
	psid->v[0].f += psid->v[0].fs;
	psid->v[1].f += psid->v[1].fs;
	psid->v[2].f += psid->v[2].fs;
	psid->v[3].f += psid->v[3].fs;
	/* noise */
	if (psid->v[3].f < psid->v[3].fs)
	    psid->v[3].rv = NSHIFT(psid->v[3].rv, 16);
	/* do adsr */
	if ((psid->v[0].adsr += psid->v[0].adsrs) + 0x80000000 <
	    psid->v[0].adsrz + 0x80000000)
	    trigger_adsr(&psid->v[0]);
	if ((psid->v[1].adsr += psid->v[1].adsrs) + 0x80000000 <
	    psid->v[1].adsrz + 0x80000000)
	    trigger_adsr(&psid->v[1]);
	if ((psid->v[2].adsr += psid->v[2].adsrs) + 0x80000000 <
	    psid->v[2].adsrz + 0x80000000)
	    trigger_adsr(&psid->v[2]);
	if ((psid->v[3].adsr += psid->v[3].adsrs) + 0x80000000 <
	    psid->v[3].adsrz + 0x80000000)
	    trigger_adsr(&psid->v[3]);
	/* no sample? */
	if (!psid->pbuf)
	    continue;
	/* volume check */
	if (!psid->vol)
	{
	    psid->pbuf[psid->bufptr++] = 0;
	    continue;
	}
	/* oscillators */
	o0 = psid->v[0].adsr >> 16;
	o1 = psid->v[1].adsr >> 16;
	o2 = psid->v[2].adsr >> 16;
	o3 = psid->v[3].adsr >> 16;
	if (o0)
	    o0 *= doosc(&psid->v[0]);
	if (o1)
	    o1 *= doosc(&psid->v[1]);
	if (o2)
	    o2 *= doosc(&psid->v[2]);
	if (o3)
	    o3 *= doosc(&psid->v[3]);
	/* sample */
	psid->pbuf[psid->bufptr++] = ((s32_t)((o0+o1+o2+o3)>>20)-0x800)*psid->vol;
    }
}
#else
inline static void update_sid(sound_t *psid, int nr)
{
    int			i;
    u32_t		o0, o1, o2;
    for (i = 0; i < nr; i++)
    {
	/* addfptrs */
	psid->v[0].f += psid->v[0].fs;
	psid->v[1].f += psid->v[1].fs;
	psid->v[2].f += psid->v[2].fs;
	/* noise */
	if (psid->v[0].f < psid->v[0].fs)
	    psid->v[0].rv = NSHIFT(psid->v[0].rv, 16);
	if (psid->v[1].f < psid->v[1].fs)
	    psid->v[1].rv = NSHIFT(psid->v[1].rv, 16);
	if (psid->v[2].f < psid->v[2].fs)
	    psid->v[2].rv = NSHIFT(psid->v[2].rv, 16);
	/* hard sync */
	/* XXX: still wrong? */
	if (psid->v[0].sync && psid->v[2].f < psid->v[2].fs)
	    psid->v[0].f = 0;
	if (psid->v[1].sync && psid->v[0].f < psid->v[0].fs)
	    psid->v[1].f = 0;
	if (psid->v[2].sync && psid->v[1].f < psid->v[1].fs)
	    psid->v[2].f = 0;
	/* do adsr */
	if ((psid->v[0].adsr += psid->v[0].adsrs) + 0x80000000 <
	    psid->v[0].adsrz + 0x80000000)
	    trigger_adsr(&psid->v[0]);
	if ((psid->v[1].adsr += psid->v[1].adsrs) + 0x80000000 <
	    psid->v[1].adsrz + 0x80000000)
	    trigger_adsr(&psid->v[1]);
	if ((psid->v[2].adsr += psid->v[2].adsrs) + 0x80000000 <
	    psid->v[2].adsrz + 0x80000000)
	    trigger_adsr(&psid->v[2]);
	/* no sample? */
	if (!psid->pbuf)
	    continue;
	/* volume check */
	if (!psid->vol)
	{
	    psid->pbuf[psid->bufptr++] = 0;
	    continue;
	}
	/* oscillators */
	o0 = psid->v[0].adsr >> 16;
	o1 = psid->v[1].adsr >> 16;
	o2 = psid->v[2].adsr >> 16;
	if (o0)
	    o0 *= doosc(&psid->v[0]);
	if (o1)
	    o1 *= doosc(&psid->v[1]);
	if (psid->has3 && o2)
	    o2 *= doosc(&psid->v[2]);
	else
	    o2 = 0;
	/* sample */
	psid->pbuf[psid->bufptr++] = ((s32_t)((o0+o1+o2)>>20)-0x600)*psid->vol;
    }
}
#endif

static void init_sid(sound_t *psid, s16_t *pbuf, int speed)
{
    u32_t		 i, nr;
    char		*name;

    psid->speed1 = (CYCLES_PER_SEC << 8) / speed;
    for (i = 0; i < 16; i++)
    {
	psid->adrs[i] = 500*8*psid->speed1/adrtable[i];
	psid->sz[i] = 0x8888888*i;
    }
    psid->pbuf = pbuf;
    psid->bufptr = 0;
#ifdef VIC20
    nr = 4;
    name = "SOUND";
#else
    nr = 3;
    name = "SID";
#endif
    if (!psid->pwarn)
	psid->pwarn = warn_init(name, 32);
    else
	warn_reset(psid->pwarn);
    psid->update = 1;
    setup_sid(psid);
    for (i = 0; i < nr; i++)
    {
#ifndef VIC20
	psid->v[i].vprev = &psid->v[(i+2)%3];
	psid->v[i].vnext = &psid->v[(i+1)%3];
#endif
	psid->v[i].nr = i;
	psid->v[i].d = psid->d + i*7;
	psid->v[i].s = psid;
	psid->v[i].rv = NSEED;
	psid->v[i].update = 1;
	setup_voice(&psid->v[i]);
    }
#ifdef VIC20
    psid->v[3].d = psid->d + 57;
#endif
}

#else /* !SID */

/* PET -code */

typedef struct sound_s
{
    warn_t			*pwarn;
    s16_t			*pbuf;
    s32_t			 bufptr;

    int				 on;
    CLOCK			 t;
    BYTE			 sample;

    double			 b;
    double			 bs;

    int				 speed;
} sound_t;

/* XXX: this can be greatly optimized */
static s16_t pet_makesample(double s, double e, BYTE sample)
{
    double				v;
    int					sc, ec, sf, ef, i, nr;
    sc = ceil(s);
    sf = floor(s);
    ec = ceil(e);
    ef = floor(e);
    nr = 0;
    for (i = sc; i < ef; i++)
	if (sample & (1 << (i%8)))
	    nr++;
    v = nr;
    if (sample & (1 << (sf % 8)))
	v += sc - s;
    if (sample & (1 << (ef % 8)))
	v += e - ef;
    return v * 4095.0 / (e-s);
}

static void init_sid(sound_t *psid, s16_t *pbuf, int speed)
{
    if (!psid->pwarn)
	psid->pwarn = warn_init("SOUND", 32);
    else
	warn_reset(psid->pwarn);
    psid->pbuf = pbuf;
    psid->bufptr = 0;
    psid->speed = speed;
    if (!psid->t)
	psid->t = 32;
    psid->b = 0.0;
    psid->bs = (double)CYCLES_PER_SEC/(psid->t*psid->speed);
}

inline static void update_sid(sound_t *psid, int nr)
{
    int				 i;
    u16_t			 v = 0;
    char			*t = "                ";
    if (!psid->pbuf)
	return;
    warn(psid->pwarn, 4,
	 "Sound support for PET is at _very_ experimental stage.\n"
	 "%sIf you think this doesn't sound right, please wait\n"
	 "%sfor the next snapshot or help me get this right.\n"
	 "%s                          //tvr", t, t, t);
    for (i = 0; i < nr; i++)
    {
	if (psid->on)
	    v = pet_makesample(psid->b, psid->b + psid->bs, psid->sample);
	psid->pbuf[psid->bufptr++] = v;
	psid->b += psid->bs;
	while (psid->b >= 8.0)
	    psid->b -= 8.0;
    }
}

#endif /* !SID */


/*
 * devices
 */
typedef struct
{
    char			 *name;
    int				(*init)(sound_t *s, char *device, int *speed,
					int *fragsize, int *fragnr,
					double bufsize);
    int				(*write)(sound_t *s, s16_t *pbuf, int nr);
    int				(*dump)(ADDRESS addr, BYTE byte, CLOCK clks);
    int				(*flush)(sound_t *s);
    int				(*bufferstatus)(sound_t *s, int first);
    void			(*close)(void);
} sid_device_t;

#define FRAGS_PER_SECOND ((int)RFSH_PER_SEC)

/*
 * fs-device
 */
static FILE *fs_fd = NULL;

static int fs_init(sound_t *s, char *param, int *speed,
		   int *fragsize, int *fragnr, double bufsize)
{
    if (!param)
	param = "vicesnd.raw";
    fs_fd = fopen(param, "w");
    if (!fs_fd)
	return 1;
    return 0;
}

static int fs_write(sound_t *s, s16_t *pbuf, int nr)
{
    int			i;
    i = fwrite(pbuf, sizeof(s16_t), nr, fs_fd);
    if (i != nr)
	return 1;
    return 0;
}

static void fs_close(void)
{
    fclose(fs_fd);
    fs_fd = NULL;
}

static sid_device_t fs_device =
{
    "fs",
    fs_init,
    fs_write,
    NULL,
    NULL,
    NULL,
    fs_close
};


/*
 * dummy device to get all the benefits of running sid
 */
static sid_device_t dummy_device =
{
    "dummy",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
};

/*
 * Another dummy device to measure speed (this calculates samples)
 */
static int speed_write(sound_t *s, s16_t *pbuf, int nr)
{
    return 0;
}

static sid_device_t speed_device =
{
    "speed",
    NULL,
    speed_write,
    NULL,
    NULL,
    NULL,
    NULL
};


/*
 * dump device to dump all writes to file for further examination
 */
static FILE *dump_fd = NULL;

static int dump_init(sound_t *s, char *param, int *speed,
		     int *fragsize, int *fragnr, double bufsize)
{
    if (!param)
	param = "vicesnd.sid";
    dump_fd = fopen(param, "w");
    if (!dump_fd)
	return 1;
    return 0;
}

static int dump_dump(ADDRESS addr, BYTE byte, CLOCK clks)
{
    int				i;
    i = fprintf(dump_fd, "%d %d %d\n", (int)clks, addr, byte);
    if (i < 0)
	return 1;
    return 0;
}

static int dump_flush(sound_t *s)
{
    int				i;
#if defined(DEBUG) && defined(CBM64)
    print_sid(dump_fd, s);
#endif
    i = fflush(dump_fd);
    return i;
}

static void dump_close(void)
{
    fclose(dump_fd);
    dump_fd = NULL;
}

static sid_device_t dump_device =
{
    "dump",
    dump_init,
    NULL,
    dump_dump,
    dump_flush,
    NULL,
    dump_close
};


/*
 * timer device to emulate fragmented blocking device behaviour
 */

#ifdef TESTDEVICE

static long test_time_zero = 0;
static long test_time_fragment = 0;
static int test_time_written = 0;
static int test_time_fragsize = 0;
static int test_time_nrfrags = 0;

static long test_now(void)
{
    struct timeval tp;
    gettimeofday(&tp, NULL);
    return tp.tv_sec*1000000 + tp.tv_usec;
}

static int test_bufferstatus(sound_t *s, int first);


static int test_init(sound_t *s, char *param, int *speed,
		     int *fragsize, int *fragnr, double bufsize)
{
    test_time_zero = test_now();
    test_time_fragment = 1000000.0 / ((double)*speed / *fragsize);
    test_time_written = 0;
    test_time_fragsize = *fragsize;
    test_time_nrfrags = *fragnr;
    return 0;
}

static int test_write(sound_t *s, s16_t *pbuf, int nr)
{
    (void)test_bufferstatus(s, 0);
    test_time_written += nr / test_time_fragsize;
    while (test_bufferstatus(s, 0) > test_time_nrfrags*test_time_fragsize)
	usleep(1000000 / (4 * (int)RFSH_PER_SEC));
    return 0;
}

static int test_bufferstatus(sound_t *s, int first)
{
    int			ret;
    long		now = test_now();
    ret = test_time_written - (now - test_time_zero) / test_time_fragment;
    if (ret < 0)
    {
	warn(s->pwarn, -1, "virtual soundbuffer empty");
	test_time_zero = now;
	test_time_written = 0;
	return 0;
    }
    return ret*test_time_fragsize;
}

static sid_device_t test_device =
{
    "test",
    test_init,
    test_write,
    NULL,
    NULL,
    test_bufferstatus,
    NULL
};
#else /* TESTDEVICE */
static sid_device_t test_device;
#endif

/*
 * linux/freebsd -device
 */
#if defined(HAVE_LINUX_SOUNDCARD_H) || defined(HAVE_MACHINE_SOUNDCARD_H)
#if defined(HAVE_LINUX_SOUNDCARD_H)
#include <linux/soundcard.h>
#endif
#if defined(HAVE_MACHINE_SOUNDCARD_H)
#include <machine/soundcard.h>
#endif

static int uss_fd = -1;
static int uss_8bit = 0;
static int uss_bufsize = 0;

static int uss_bufferstatus(sound_t *s, int first);

static int uss_init(sound_t *s, char *param, int *speed,
		    int *fragsize, int *fragnr, double bufsize)
{
    int			 st, tmp, orig;
    if (!param)
	param = "/dev/dsp";
    uss_fd = open(param, O_WRONLY, 0777);
    if (uss_fd < 0)
    {
	warn(s->pwarn, -1, "cannot open '%s' for writing", param);
	return 1;
    }
    /* samplesize 16 bits */
#ifdef WORDS_BIGENDIAN
    orig = tmp = AFMT_S16_BE;
#else
    orig = tmp = AFMT_S16_LE;
#endif    
    st = ioctl(uss_fd, SNDCTL_DSP_SETFMT, &tmp);
    if (st < 0 || orig != tmp || getenv("USS8BIT"))
    {
	/* samplesize 8 bits */
	orig = tmp = AFMT_U8;
	st = ioctl(uss_fd, SNDCTL_DSP_SETFMT, &tmp);
	if (st < 0 || orig != tmp)
	{
	    warn(s->pwarn, -1, "SNDCTL_DSP_SETFMT failed");
	    goto fail;
	}
	warn(s->pwarn, -1, "playing 8bit sample");
	uss_8bit = 1;
    }
    /* no stereo */
    tmp = 0;
    st = ioctl(uss_fd, SNDCTL_DSP_STEREO, &tmp);
    if (st < 0 || tmp != 0)
    {
	warn(s->pwarn, -1, "SNDCTL_DSP_STEREO failed");
	goto fail;
    }
    /* speed */
    tmp = *speed;
    st = ioctl(uss_fd, SNDCTL_DSP_SPEED, &tmp);
    if (st < 0 || tmp <= 0)
    {
	warn(s->pwarn, -1, "SNDCTL_DSP_SPEED failed");
	goto fail;
    }
    *speed = tmp;
    /* fragments */
    for (tmp = 1; 1 << tmp < *fragsize; tmp++);
    orig = tmp = tmp + (*fragnr << 16) + !uss_8bit;
    st = ioctl(uss_fd, SNDCTL_DSP_SETFRAGMENT, &tmp);
    if (st < 0 || (tmp^orig)&0xffff)
    {
	warn(s->pwarn, -1, "SNDCTL_DSP_SETFRAGMENT failed");
	goto fail;
    }
    if (tmp != orig)
    {
	if (tmp >> 16 > *fragnr)
	{
	    warn(s->pwarn, -1, "SNDCTL_DSP_SETFRAGMENT: too many fragments");
	    goto fail;
	}
	*fragnr = tmp >> 16;
	if (*fragnr < 3)
	{
	    warn(s->pwarn, -1, "SNDCTL_DSP_SETFRAGMENT: too few fragments");
	    goto fail;
	}
    }
    uss_bufsize = (*fragsize)*(*fragnr);
    return 0;
fail:
    close(uss_fd);
    uss_fd = -1;
    uss_8bit = 0;
    uss_bufsize = 0;
    return 1;
}

static int uss_write(sound_t *s, s16_t *pbuf, int nr)
{
    int			total, i, now;
    if (uss_8bit)
    {
	/* XXX: ugly to change contents of the buffer */
	for (i = 0; i < nr; i++)
	    ((char *)pbuf)[i] = pbuf[i]/256 + 128;
	total = nr;
    }
    else
	total = nr*sizeof(s16_t);
    for (i = 0; i < total; i += now)
    {
	now = write(uss_fd, (char *)pbuf + i, total - i);
	if (now <= 0)
	{
	    if (now < 0)
		perror("uss_write");
	    return 1;
	}
    }
    return 0;
}

static int uss_bufferstatus(sound_t *s, int first)
{
    audio_buf_info		info;
    int				st, ret;

    st = ioctl(uss_fd, SNDCTL_DSP_GETOSPACE, &info);
    if (st < 0)
    {
	warn(s->pwarn, -1, "SNDCTL_DSP_GETOSPACE failed");
	return -1;
    }
    ret = info.fragments*info.fragsize;
    if (ret != info.bytes)
    {
	warn(s->pwarn, 11, "GETOSPACE: ret(%d)!=bytes(%d)", ret, info.bytes);
	ret = info.bytes;
    }
    if (ret < 0)
    {
        warn(s->pwarn, 12, "GETOSPACE: bytes < 0");
	ret = 0;
    }
    if (!uss_8bit)
	ret /= sizeof(s16_t);
    if (ret > uss_bufsize)
    {
	warn(s->pwarn, 13, "GETOSPACE: bytes > bufsize");
	ret = uss_bufsize;
    }
#if defined(linux)
    /*
     * GETOSPACE before first write returns random value (or actually the
     * value on which the device was when it was closed last time). I hope
     * this has been fixed after 'Sound Driver:3.5-beta2-960210'
     */
    if (first && !ret)
    {
	ret = 1;
	warn(s->pwarn, -1, "SNDCTL_DSP_GETOSPACE not reliable after open()");
    }
#endif
    return ret;
}

static void uss_close(void)
{
    close(uss_fd);
    uss_fd = -1;
    uss_8bit = 0;
    uss_bufsize = 0;
}

static sid_device_t uss_device =
{
    "uss",
    uss_init,
    uss_write,
    NULL,
    NULL,
    uss_bufferstatus,
    uss_close
};

#else
static sid_device_t uss_device;
#endif


/*
 * sgi sound device
 */
#if defined(sgi) && defined(HAVE_DMEDIA_AUDIO_H)
#include <dmedia/audio.h>
#if defined(HAVE_BSTRING_H)
#include <bstring.h>
#endif

static ALconfig    sgi_audioconfig = NULL;
static ALport      sgi_audioport = NULL;

static void sgi_errorhandler(long err, const char *msg, ...)
{
    printf("sgierrorhandler: %d, %s\n", (int)err, msg);
}

static int sgi_init(sound_t *s, char *param, int *speed,
		    int *fragsize, int *fragnr, double bufsize)
{
    long	chpars[] = {AL_OUTPUT_RATE, 0};
    int		st;

    ALseterrorhandler(sgi_errorhandler);
    chpars[1] = *speed;
    st = ALsetparams(AL_DEFAULT_DEVICE, chpars, 2);
    if (st < 0)
	return 1;
    st = ALgetparams(AL_DEFAULT_DEVICE, chpars, 2);
    if (st < 0)
	return 1;
    *speed = chpars[1];

    sgi_audioconfig = ALnewconfig();
    if (!sgi_audioconfig)
	return 1;
    st = ALsetchannels(sgi_audioconfig, AL_MONO);
    if (st < 0)
	goto fail;
    st = ALsetwidth(sgi_audioconfig, AL_SAMPLE_16);
    if (st < 0)
	goto fail;
    st = ALsetqueuesize(sgi_audioconfig, *fragsize * *fragnr);
    if (st < 0)
        goto fail;
    sgi_audioport = ALopenport("outport", "w", sgi_audioconfig);
    if (!sgi_audioport)
	goto fail;
    return 0;
fail:
    ALfreeconfig(sgi_audioconfig);
    sgi_audioconfig = NULL;
    return 1;
}

static int sgi_write(sound_t *s, s16_t *pbuf, int nr)
{
    int				i;
    i = ALwritesamps(sgi_audioport, pbuf, nr);
    if (i < 0)
	return 1;
    return 0;
}

static int sgi_bufferstatus(sound_t *s, int first)
{
    int				i;
    i = ALgetfilled(sgi_audioport);
    return i;
}

static void sgi_close(void)
{
    /* XXX: code missing */
    ALfreeconfig(sgi_audioconfig);
    sgi_audioconfig = NULL;
}

static sid_device_t sgi_device =
{
    "sgi",
    sgi_init,
    sgi_write,
    NULL,
    NULL,
    sgi_bufferstatus,
    sgi_close
};

#else
static sid_device_t sgi_device;
#endif


/*
 * Solaris (untested and unfinished)
 */
#if defined(sun) && defined(HAVE_SYS_AUDIOIO_H)
#include <sys/audioio.h>

static int sun_bufferstatus(sound_t *s, int first);

static int sun_fd = -1;
static int sun_8bit = 0;
static int sun_bufsize = 0;
static int sun_written = 0;

static int toulaw8(s16_t data)
{
    int			v, s, a;

    a = data / 8;

    v = (a < 0 ? -a : a);
    s = (a < 0 ? 0 : 0x80);

    if (v >= 4080)
        a = 0;
    else if (v >= 2032)
        a = 0x0f - (v - 2032) / 128;
    else if (v >= 1008)
        a = 0x1f - (v - 1008) / 64;
    else if (v >= 496)
        a = 0x2f - (v - 496) / 32;
    else if (v >= 240)
        a = 0x3f - (v - 240) / 16;
    else if (v >= 112)
        a = 0x4f - (v - 112) / 8;
    else if (v >= 48)
        a = 0x5f - (v - 48) / 4;
    else if (v >= 16)
        a = 0x6f - (v - 16) / 2;
    else
        a = 0x7f - v;
    
    a |= s;
    
    return a;
}


static int sun_init(sound_t *s, char *param, int *speed,
		    int *fragsize, int *fragnr, double bufsize)
{
    int			st;
    struct audio_info	info;

    if (!param)
	param = "/dev/audio";
    sun_fd = open(param, O_WRONLY, 0777);
    if (sun_fd < 0)
	return 1;
    AUDIO_INITINFO(&info);
    info.play.sample_rate = *speed;
    info.play.channels = 1;
    info.play.precision = 16;
    info.play.encoding = AUDIO_ENCODING_LINEAR;
    st = ioctl(sun_fd, AUDIO_SETINFO, &info);
    if (st < 0)
    {
	AUDIO_INITINFO(&info);
	info.play.sample_rate = 8000;
	info.play.channels = 1;
	info.play.precision = 8;
	info.play.encoding = AUDIO_ENCODING_ULAW;
	st = ioctl(sun_fd, AUDIO_SETINFO, &info);
	if (st < 0)
	    goto fail;
	sun_8bit = 1;
	*speed = 8000;
	warn(s->pwarn, -1, "playing 8 bit ulaw at 8000Hz");
    }
    sun_bufsize = (*fragsize)*(*fragnr);
    sun_written = 0;
    return 0;
fail:
    close(sun_fd);
    sun_fd = -1;
    return 1;
}

static int sun_write(sound_t *s, s16_t *pbuf, int nr)
{
    int			total, i, now;
    if (sun_8bit)
    {
	/* XXX: ugly to change contents of the buffer */
	for (i = 0; i < nr; i++)
	    ((char *)pbuf)[i] = toulaw8(pbuf[i]);
	total = nr;
    }
    else
	total = nr*sizeof(s16_t);
    for (i = 0; i < total; i += now)
    {
	now = write(sun_fd, (char *)pbuf + i, total - i);
	if (now <= 0)
	    return 1;
    }
    sun_written += nr;
    /* XXX: correct? */
    while (sun_bufferstatus(s, 0) > sun_bufsize)
	usleep(1000000 / (4 * (int)RFSH_PER_SEC));
    return 0;
}

static int sun_bufferstatus(sound_t *s, int first)
{
    int			st;
    struct audio_info	info;
    st = ioctl(sun_fd, AUDIO_GETINFO, &info);
    if (st < 0)
	return -1;
    /* XXX: is samples reliable? eof? */
    return sun_written - info.play.samples;
}

static void sun_close(void)
{
    close(sun_fd);
    sun_fd = -1;
    sun_8bit = 0;
    sun_bufsize = 0;
    sun_written = 0;
}


static sid_device_t sun_device =
{
    "sun",
    sun_init,
    sun_write,
    NULL,
    NULL,
    sun_bufferstatus,
    sun_close
};

#else
static sid_device_t sun_device;
#endif


#if defined(HAVE_LIBUMSOBJ) && defined(HAVE_UMS_UMSAUDIODEVICE_H) && defined(HAVE_UMS_UMSBAUDDEVICE_H)

/* AIX -support by Chris Sharp (sharpc@hursley.ibm.com) */

#include <UMS/UMSAudioDevice.h>
#include <UMS/UMSBAUDDevice.h>

/* XXX: AIX: can these be made static and use aix_ -prefix on these? */
UMSAudioDeviceMClass audio_device_class;
UMSAudioDevice_ReturnCode rc;
UMSBAUDDevice audio_device;
Environment *ev;
UMSAudioTypes_Buffer buffer;
UMSAudioDeviceMClass_ErrorCode audio_device_class_error;
char* error_string;
char* audio_formats_alias;
char* audio_inputs_alias;
char* audio_outputs_alias;
char* obyte_order;
long out_rate;
long left_gain, right_gain;


static int aix_init(sound_t *s, char *param, int *speed,
		     int *fragsize, int *fragnr, double bufsize)
{
    int	st, tmp, i;
    /* open device */
    ev = somGetGlobalEnvironment();
    audio_device = UMSBAUDDeviceNew();
    rc = UMSAudioDevice_open(audio_device, ev, "/dev/paud0", "PLAY",
			     UMSAudioDevice_BlockingIO);
    if (audio_device == NULL)
    {
    	fprintf(stderr,"can't create audio device object\nError: %s\n",
		error_string);
	return 1;
    }

    rc = UMSAudioDevice_set_volume(audio_device, ev, 100);
    rc = UMSAudioDevice_set_balance(audio_device, ev, 0);

    rc = UMSAudioDevice_set_time_format(audio_device, ev, UMSAudioTypes_Msecs);

    if (obyte_order)
        free(obyte_order);
    rc = UMSAudioDevice_set_byte_order(audio_device, ev, "LSB");

    /* set 16bit */
    rc = UMSAudioDevice_set_bits_per_sample(audio_device, ev, 16);
    rc = UMSAudioDevice_set_audio_format_type(audio_device, ev, "PCM");
    rc = UMSAudioDevice_set_number_format(audio_device, ev, "TWOS_COMPLEMENT");

    /* set speed */
    rc = UMSAudioDevice_set_sample_rate(audio_device, ev, *speed, &out_rate);

    /* channels */
    rc = UMSAudioDevice_set_number_of_channels(audio_device, ev, 1);

    /* should we use the default? */
    left_gain = right_gain = 100;
    rc = UMSAudioDevice_enable_output(audio_device, ev, "LINE_OUT",
				      &left_gain, &right_gain);

    /* set buffer size */
    tmp = (*fragsize)*(*fragnr)*sizeof(s16_t);
    buffer._maximum = tmp;
    buffer._buffer  = (char *) xmalloc(tmp);
    buffer._length = 0;


    rc = UMSAudioDevice_initialize(audio_device, ev);
    rc = UMSAudioDevice_start(audio_device, ev);

    return 0;
#if 0
    /* XXX: AIX: everything should check rc, this isn't used now */
fail:
    UMSAudioDevice_stop(audio_device, ev);
    UMSAudioDevice_close(audio_device, ev);
    _somFree(audio_device);
    free(buffer._buffer);
    audio_device = NULL;

    return 1;
#endif
}

static int aix_write(sound_t *s, s16_t *pbuf, int nr)
{
    int	total, i, now;
    long samples_written;

    total = nr*sizeof(s16_t);
    buffer._length = total;
    memcpy(buffer._buffer,pbuf,total); 
    rc = UMSAudioDevice_write(audio_device, ev, &buffer, total,
			      &samples_written);
    return 0;
}

static int aix_bufferstatus(sound_t *s, int first)
{
    int i = -1;
    rc = UMSAudioDevice_write_buff_remain(audio_device, ev, &i);
    if (i < 0)
      return -1;
    /* fprintf(stderr,"Audio Buffer remains: %d\n blocks",i); */
    return i/sizeof(s16_t);
}

static void aix_close(void)
{
    rc = UMSAudioDevice_play_remaining_data(audio_device, ev, TRUE);
    UMSAudioDevice_stop(audio_device, ev);
    UMSAudioDevice_close(audio_device, ev);
    _somFree(audio_device);
    free(buffer._buffer);
    audio_device = NULL;
}


static sid_device_t aix_device =
{
    "aix",
    aix_init,
    aix_write,
    NULL,
    NULL,
    aix_bufferstatus,
    aix_close
};

#else
static sid_device_t aix_device;
#endif

#if defined(__hpux) && defined(HAVE_SYS_AUDIO_H)
#include <sys/audio.h>

static int hpux_fd = -1;

static int hpux_init(sound_t *s, char *param, int *speed,
		     int *fragsize, int *fragnr, double bufsize)
{
    int				st, tmp, i;
    if (!param)
	param = "/dev/audio";
    /* open device */
    hpux_fd = open(param, O_WRONLY, 0777);
    if (hpux_fd < 0)
	return 1;
    /* set 16bit */
    st = ioctl(hpux_fd, AUDIO_SET_DATA_FORMAT, AUDIO_FORMAT_LINEAR16BIT);
    if (st < 0)
	goto fail;
    /* set speed */
    st = ioctl(hpux_fd, AUDIO_SET_SAMPLE_RATE, *speed);
    if (st < 0)
	goto fail;
    /* channels */
    st = ioctl(hpux_fd, AUDIO_SET_CHANNELS, 1);
    if (st < 0)
	goto fail;
    /* should we use the default? */
    st = ioctl(hpux_fd, AUDIO_SET_OUTPUT, AUDIO_OUT_SPEAKER);
    if (st < 0)
	goto fail;
    /* set buffer size */
    tmp = (*fragsize)*(*fragnr)*sizeof(s16_t);
    st = ioctl(hpux_fd, AUDIO_SET_TXBUFSIZE, tmp);
    if (st < 0)
    {
	/* XXX: what are valid buffersizes? */
	for (i = 1; i < tmp; i *= 2);
	tmp = i;
	st = ioctl(hpux_fd, AUDIO_SET_TXBUFSIZE, tmp);
	if (st < 0)
	    goto fail;
	*fragnr = tmp / ((*fragsize)*sizeof(s16_t));
    }
    return 0;
fail:
    close(hpux_fd);
    hpux_fd = -1;
    return 1;
}

static int hpux_write(sound_t *s, s16_t *pbuf, int nr)
{
    int			total, i, now;
    total = nr*sizeof(s16_t);
    for (i = 0; i < total; i += now)
    {
	now = write(hpux_fd, (char *)pbuf + i, total - i);
	if (now <= 0)
	    return 1;
    }
    return 0;
}

static int hpux_bufferstatus(sound_t *s, int first)
{
    int				st;
    struct audio_status		ast;
    st = ioctl(hpux_fd, AUDIO_GET_STATUS, &ast);
    if (st < 0)
	return -1;
    return ast.transmit_buffer_count / sizeof(s16_t);
}

static void hpux_close(void)
{
    close(hpux_fd);
    hpux_fd = -1;
}


static sid_device_t hpux_device =
{
    "hpux",
    hpux_init,
    hpux_write,
    NULL,
    NULL,
    hpux_bufferstatus,
    hpux_close
};

#else
static sid_device_t hpux_device;
#endif

#ifdef __MSDOS__

/*
 * MIDAS
 */

#include "vmidas.h"

static int midas_bufferstatus(sound_t *s, int first);

static MIDASstreamHandle midas_stream = NULL;
static int midas_bufsize = -1;
static int midas_maxsize = -1;

static int midas_init(sound_t *s, char *param, int *speed,
		      int *fragsize, int *fragnr, double bufsize)
{
    BOOL		st;

    st = vmidas_startup();
    if (st != TRUE)
	return 1;
    st = MIDASsetOption(MIDAS_OPTION_MIXRATE, *speed);
    if (st != TRUE)
	return 1;
    st = MIDASsetOption(MIDAS_OPTION_MIXING_MODE, MIDAS_MIX_NORMAL_QUALITY);
    if (st != TRUE)
	return 1;
    st = MIDASsetOption(MIDAS_OPTION_OUTPUTMODE, MIDAS_MODE_16BIT_MONO);
    if (st != TRUE)
	return 1;
    st = MIDASsetOption(MIDAS_OPTION_MIXBUFLEN,
			(*fragsize)*(*fragnr)*sizeof(s16_t));
    if (st != TRUE)
	return 1;
    st = MIDASsetOption(MIDAS_OPTION_MIXBUFBLOCKS, *fragnr);
    if (st != TRUE)
	return 1;
#ifdef __MSDOS__
#if 0
    st = MIDASconfig();
    if (st != TRUE)
	return 1;
#endif
#endif    
    st = vmidas_init();
    if (st != TRUE)
	return 1;
    st = MIDASopenChannels(1);
    if (st != TRUE)
    {
	/* st = MIDASclose(); */
	return 1;
    }
    midas_stream = MIDASplayStreamPolling(MIDAS_SAMPLE_16BIT_MONO, *speed,
					  (int)(bufsize*1000));
    if (!midas_stream)
    {
	st = MIDAScloseChannels();
	/* st = MIDASclose(); */
	return 1;
    }
    midas_bufsize = (*fragsize)*(*fragnr);
    midas_maxsize = midas_bufsize / 2;
    return 0;
}

static int midas_write(sound_t *s, s16_t *pbuf, int nr)
{
    BOOL		st = 1;
    unsigned int	ist;

    ist = MIDASfeedStreamData(midas_stream, (unsigned char *)pbuf,
			      nr*sizeof(s16_t), TRUE);
    if (ist != nr*sizeof(s16_t))
	return 1;
#ifndef __MSDOS__
    st = MIDASpoll();
#endif
    return !st;
}

static int midas_bufferstatus(sound_t *s, int first)
{
    int			nr;
    if (first)
	return 0;
    nr = MIDASgetStreamBytesBuffered(midas_stream);
    if (nr < 0)
	nr = 0;
    nr /= sizeof(s16_t);
    if (nr > midas_maxsize)
	midas_maxsize = nr;
    return (int)((double)nr/midas_maxsize*midas_bufsize);
}

static void midas_close(void)
{
    BOOL		st;

    /* XXX: we might come here from `atexit', so MIDAS might have been shut
       down already.  This is a dirty kludge, we should find a cleaner way to
       do it. */
    if (vmidas_available())
    {
	st = MIDASstopStream(midas_stream);
	st = MIDAScloseChannels();
	/* st = MIDASclose(); */
    }
    midas_stream = NULL;
    midas_bufsize = -1;
    midas_maxsize = -1;
}

static sid_device_t midas_device =
{
    "midas",
    midas_init,
    midas_write,
    NULL,
    NULL,
    midas_bufferstatus,
    midas_close
};
#else
static sid_device_t midas_device;
#endif


static sid_device_t *sid_devices[12] =
{
    &uss_device,
    &sgi_device,
    &sun_device,
    &hpux_device,
    &aix_device,
    &midas_device,
    &dummy_device,
    &fs_device,
    &speed_device,
    &dump_device,
    &test_device,
    NULL
};
    
/*
 * and the code itself
 */

#define BUFSIZE 32768
typedef struct
{
    sound_t		 sid;
    double		 clkstep;
    double		 origclkstep;
    double		 clkfactor;
    double		 fclk;
    CLOCK		 wclk;
    s16_t		 buffer[BUFSIZE];
    sid_device_t	*pdev;
    int			 fragsize;
    int			 fragnr;
    int			 bufsize;
    int			 firststatus;
} siddata_t;

static siddata_t siddata;

static int closesid(char *msg)
{
    if (siddata.pdev)
    {
	warn(siddata.sid.pwarn, -1, "closing device '%s'", siddata.pdev->name);
	if (siddata.pdev->close)
	    siddata.pdev->close();
	siddata.pdev = NULL;
    }
    if (msg)
    {
        suspend_speed_eval();
	if (strcmp(msg, ""))
	{
	    UiError(msg);
	    app_resources.sound = 0;
	    UiUpdateMenus();
	}
    }
    return 1;
}

static int disabletime;

static void suspendsid(char *reason)
{
    disabletime = time(0);
    warn(siddata.sid.pwarn, -1, "SUSPEND: disabling sid for %d secs (%s)",
	 app_resources.soundSuspendTime, reason);
    closesid("");
}

static void enablesid(void)
{
    int		now, diff;
    if (!disabletime)
        return;
    now = time(0);
    diff = now - disabletime;
    if (diff < 0 || diff >= app_resources.soundSuspendTime)
    {
        warn(siddata.sid.pwarn, -1, "ENABLE");
        disabletime = 0;
    }
}

static int initsid(void)
{
    int					 i, tmp;
    sid_device_t			*pdev;
    char				*name;
    char				*param;
    int					 speed;
    int					 fragsize;
    int					 fragnr;
    double				 bufsize;
    char				 err[1024];

    if (app_resources.soundSuspendTime > 0 && disabletime)
        return 1;

    name = app_resources.soundDeviceName;
    param = app_resources.soundDeviceArg;
    tmp = app_resources.soundBufferSize;
    if (tmp < 100 || tmp > 1000)
	tmp = SOUND_SAMPLE_BUFFER_SIZE;
    bufsize = tmp / 1000.0;

    speed = app_resources.soundSampleRate;
    if (speed < 8000 || speed > 50000)
	speed = SOUND_SAMPLE_RATE;
    /* calculate optimal fragments */
    fragsize = speed / FRAGS_PER_SECOND;
    for (i = 1; 1 << i < fragsize; i++);
    fragsize = 1 << i;
    fragnr = (speed*bufsize + fragsize - 1) / fragsize;
    if (fragnr < 3)
        fragnr = 3;

    for (i = 0; (pdev = sid_devices[i]); i++)
    {
	if ((name && pdev->name && !strcmp(pdev->name, name)) ||
	    (!name && pdev->name))
	{
	    if (pdev->init)
	    {
		tmp = pdev->init(&siddata.sid, param, &speed,
				 &fragsize, &fragnr, bufsize);
		if (tmp)
		{
		    sprintf(err, "Audio: initialization failed for device `%s'.",
			    pdev->name);
		    return closesid(err);
		}
	    }
	    siddata.pdev = pdev;
	    siddata.fragsize = fragsize;
	    siddata.fragnr = fragnr;
	    siddata.bufsize = fragsize*fragnr;
	    warn(siddata.sid.pwarn, -1,
		 "opened device '%s' speed %dHz fragsize %.3fs bufsize %.3fs",
		 pdev->name, speed, (double)fragsize / speed,
		 (double)siddata.bufsize / speed);
	    app_resources.soundSampleRate = speed;
	    if (pdev->write)
		init_sid(&siddata.sid, siddata.buffer, speed);
	    else
		init_sid(&siddata.sid, NULL, speed);
	    if (pdev->bufferstatus)
		siddata.firststatus = pdev->bufferstatus(&siddata.sid, 1);
	    siddata.clkstep = (double)CYCLES_PER_SEC / speed;
	    siddata.origclkstep = siddata.clkstep;
	    siddata.clkfactor = 1.0;
	    siddata.fclk = clk;
	    siddata.wclk = clk;
	    return 0;
	}
    }
    sprintf(err, "Audio: device `%s' not found or not supported.", name);
    return closesid(err);
}

static int run_sid(void)
{
    int				nr, i;
    if (!app_resources.sound)
	return 1;
    if (app_resources.soundSuspendTime > 0 && disabletime)
        return 1;
    if (!siddata.pdev)
    {
	i = initsid();
	if (i)
	    return i;
    }
    nr = (clk - siddata.fclk) / siddata.clkstep;
    if (!nr)
	return 0;
    if (siddata.sid.bufptr + nr > BUFSIZE)
	return closesid("Audio: sound buffer overflow.");
#ifdef SID
    setup_sid(&siddata.sid);
    setup_voice(&siddata.sid.v[0]);
    setup_voice(&siddata.sid.v[1]);
    setup_voice(&siddata.sid.v[2]);
#ifdef VIC20
    setup_voice(&siddata.sid.v[3]);
#endif
#endif
    update_sid(&siddata.sid, nr);
    siddata.fclk += nr*siddata.clkstep;
    return 0;
}

int flush_sound(void)
{
    int			i, nr, space, used, adjust;
    static int		skipnextframe = 0;
    double		reference;

    if (app_resources.soundSuspendTime > 0)
        enablesid();
    i = run_sid();
    if (i)
	return 0;
    if (siddata.pdev->flush)
    {
	i = siddata.pdev->flush(&siddata.sid);
	if (i)
	{
	    closesid("Audio: cannot flush.");
	    return 0;
	}
    }
    if (siddata.sid.bufptr < siddata.fragsize)
	return skipnextframe;
    nr = siddata.sid.bufptr - siddata.sid.bufptr % siddata.fragsize;
    /* adjust speed */
    if (siddata.pdev->bufferstatus)
    {
	skipnextframe = 1;
	space = siddata.pdev->bufferstatus(&siddata.sid, 0);
	if (!siddata.firststatus)
	    space = siddata.bufsize - space;
	used = siddata.bufsize - space;
	if (space < 0 || used < 0)
	{
	    warn(siddata.sid.pwarn, -1, "fragment problems %d %d %d",
		 space, used, siddata.firststatus);
	    closesid("Audio: fragment problems.");
	    return 0;
	}
	adjust = 0;
	/* buffer a lot too full */
	if (space <= nr/2)
	    adjust = -2;
	/* buffer too full */
	if (space < nr)
	    adjust = -1;
	/* more space than data */
	if (space > nr)
	    adjust = 1;
	/* 1/2 empty */
	if (used <= siddata.bufsize/2)
	    adjust = 2;
	/* buffer really empty */
	if (used <= nr)
	    adjust = 3;
	/* buffer empty */
	if (used <= siddata.fragsize)
	{
	    s16_t		*p;
	    int			 j;
	    static int		 prev;
	    int			 now;
	    if (app_resources.soundSuspendTime > 0)
	    {
	        now = time(0);
		if (now == prev)
		{
		    suspendsid("buffer overruns");
		    return 0;
		}
		prev = now;
	    }
	    j = sizeof(*p)*(siddata.fragsize*siddata.fragnr - nr);
	    if (j > 0)
	    {
	        p = xmalloc(j);
		memset(p, 0, j);
		i = siddata.pdev->write(&siddata.sid, p,
					j / sizeof(*p));
		free(p);
		if (i)
		{
		    closesid("Audio: write to sound device failed.");
		    return 0;
		}
	    }
	    adjust = 4;
	}
	if (!app_resources.soundSpeedAdjustment)
	{
	    siddata.clkfactor = app_resources.speed / 100.0;
	    adjust = 0;
	    siddata.clkfactor *= 0.9 + (used+nr)*0.12/siddata.bufsize;
	}
	/* XXX: perhaps not very good values */
	switch (adjust)
	{
	case -2: siddata.clkfactor *= 1.10; break;
	case -1: siddata.clkfactor *= 1.03; break;
	case  1: siddata.clkfactor /= 1.01; break;
	case  2: siddata.clkfactor /= 1.05; break;
	case  3: siddata.clkfactor /= 1.10; break;
	case  4: siddata.clkfactor /= 1.20; break;
	}
	if (app_resources.speed > 0 && app_resources.soundSpeedAdjustment)
	{
	    reference = app_resources.speed / 100.0;
	    if (siddata.clkfactor >= reference)
		siddata.clkfactor = reference;
	    if (siddata.clkfactor >= 0.99*reference)
		skipnextframe = 0;
	}
	siddata.clkstep = siddata.origclkstep * siddata.clkfactor;
	if (CYCLES_PER_RFSH / siddata.clkstep >= siddata.bufsize)
	{
	    if (app_resources.soundSuspendTime > 0)
	        suspendsid("running too slow");
	    else
	        closesid("Audio: running too slow.");
	    return 0;
	}
	/*
	 * ok, here's the deal. if we aren't running fast enough we might
	 * consider dropping next frame and we will gain more speed.
	 */
	if (adjust < -1)
	    skipnextframe = 0;
    }
    i = siddata.pdev->write(&siddata.sid, siddata.buffer, nr);
    if (i)
    {
	closesid("Audio: write to sounddevice failed.");
	return 0;
    }
    siddata.sid.bufptr -= nr;
    if (siddata.sid.bufptr > 0)
    {
	memcpy(siddata.buffer, siddata.buffer + nr,
	       siddata.sid.bufptr*sizeof(s16_t));
    }
    return skipnextframe;
}

void close_sound(void)
{
    closesid(NULL);
}

int adjusting_sound(void)
{
    if (app_resources.sound && !siddata.pdev)
	initsid();
    if (siddata.pdev && siddata.pdev->bufferstatus &&
	app_resources.soundSpeedAdjustment)
	return 1;
    return 0;
}

void initialize_sound(void)
{
    /* dummy init to get pwarn */
    init_sid(&siddata.sid, NULL, SOUND_SAMPLE_RATE);
}

#ifdef SID
void sid_prevent_clk_overflow(void)
{
    if (!siddata.pdev)
	return;
    siddata.wclk -= PREVENT_CLK_OVERFLOW_SUB;
    siddata.fclk -= PREVENT_CLK_OVERFLOW_SUB;
}

BYTE REGPARM1 read_sid(ADDRESS addr)
{
    BYTE		ret = 0;
    u32_t		ffix;
    run_sid();
    addr &= 0x1f;
    /* XXX: this is not correct, but what can we do without a running sid? */
    if (!siddata.pdev && addr != 0x19 && addr != 0x1a)
    {
	warn(siddata.sid.pwarn, 5, "program reading sid-registers (no sound)");
	if (addr == 0x1b || addr == 0x1c)
	    return rand();
	return siddata.sid.d[addr];
    }
    switch (addr)
    {
    case 0x19:
	/* pot/x */
	break;
    case 0x1a:
	/* pot/y */
	break;
    case 0x1b:
	/* osc3 / random */
	ffix = ((clk - siddata.fclk) / siddata.clkstep)*siddata.sid.v[2].fs;
	siddata.sid.v[2].f += ffix;
	ret = doosc(&siddata.sid.v[2]) >> 7;
	siddata.sid.v[2].f -= ffix;
	warn(siddata.sid.pwarn, 6, "program reading osc3 register");
	/* XXX: not correct, but best we can do now */
	if (siddata.sid.v[2].fm == NOISEWAVE)
	    ret = rand();
	break;
    case 0x1c:
	ret = siddata.sid.v[2].adsr >> 23;
	warn(siddata.sid.pwarn, 7, "program reading env3 register");
	break;
    default:
	ret = siddata.sid.d[addr];
    }
    return ret;
}

void REGPARM2 store_sid(ADDRESS addr, BYTE byte)
{
    int				i;
#ifndef VIC20
    addr &= 0x1f;
#endif
    i = run_sid();
    if (!i && siddata.pdev->dump)
    {
	i = siddata.pdev->dump(addr, byte, clk - siddata.wclk);
	siddata.wclk = clk;
	if (i)
	    closesid("Audio: store to sounddevice failed.");
    }
    switch (addr)
    {
    case 4: 
	if ((siddata.sid.d[addr] ^ byte) & 1)
	    siddata.sid.v[0].gateflip = 1;
    case 0: case 1: case 2: case 3: case 5: case 6:
	siddata.sid.v[0].update = 1;
	break;
    case 11: 
	if ((siddata.sid.d[addr] ^ byte) & 1)
	    siddata.sid.v[1].gateflip = 1;
    case 7: case 8: case 9: case 10: case 12: case 13:
	siddata.sid.v[1].update = 1;
	break;
    case 18: 
	if ((siddata.sid.d[addr] ^ byte) & 1)
	    siddata.sid.v[2].gateflip = 1;
    case 14: case 15: case 16: case 17: case 19: case 20:
	siddata.sid.v[2].update = 1;
	break;
#ifdef VIC20
    case 57: case 58: case 59: case 60: case 61: case 62: case 63:
	siddata.sid.v[3].update = 1;
	break;
#endif
    default:
	siddata.sid.update = 1;
    }
    siddata.sid.d[addr] = byte;
}

#ifdef VIC20
/*
 * XXX: This is _really_ experimental
 */
#define VIC20FREQBASE    65535

void store_vic20sid(ADDRESS addr, BYTE value)
{
    u32_t			freq;
    int				sbase, wval, shift, div;
    addr &= 0x0f;
    if (siddata.pdev)
    {
	char			*t = "                ";
	warn(siddata.sid.pwarn, 8,
	     "Sound support for VIC20 is at _very_ experimental stage.\n"
	     "%sIf you think this doesn't sound right, please wait\n"
	     "%sfor the next snapshot or help me get this right.\n"
	     "%s                          // tvr", t, t, t);
    }
#if 0
    warn(siddata.sid.pwarn, -1, "store_vic20sid(%d, %d)", addr, value);
#endif
    switch (addr)
    {
    case 10:
    case 11:
    case 12:
    case 13:
	sbase = (addr - 10)*7;
	wval = 0x40;
	shift = addr - 10;
	if (addr == 13)
	{
	    sbase = 57;
	    wval = 0x80;
	    shift = 0;
	}
	store_sid(sbase + 2, 0x00);
	store_sid(sbase + 3, 0x08);
	store_sid(sbase + 5, 0x00);
	store_sid(sbase + 6, 0xf0);
	store_sid(sbase + 4, wval+(value>>7));
	div = 255 - value;
	/* XXX: ? */
	if (!div)
	    div = 128;
	freq = VIC20FREQBASE*(1 << shift)/div;
	store_sid(sbase + 0, freq & 0xff);
	store_sid(sbase + 1, (freq / 256) & 0xff);
	break;
    case 14:
	/* volume */
	store_sid(0x18, value & 0x0f);
	break;
    }
}
#endif /* VIC20 */

#endif /* SID */

#ifdef PET
void store_petsnd_onoff(int value)
{
    int				i;
    /* warn(siddata.sid.pwarn, -1, "store_petsnd_onoff(%d)", value); */
    i = run_sid();
    siddata.sid.on = value;
}

void store_petsnd_rate(CLOCK t)
{
    int				i;
    /* warn(siddata.sid.pwarn, -1, "store_petsnd_rate(%d)", t); */
    i = run_sid();
    siddata.sid.t = t;
    /* siddata.sid.b = 0; */
    siddata.sid.bs = (double)CYCLES_PER_SEC/(siddata.sid.t*siddata.sid.speed);
}

void store_petsnd_sample(BYTE sample)
{
    int				i;
    /* warn(siddata.sid.pwarn, -1, "store_petsnd_sample(%d)", sample); */
    i = run_sid();
    siddata.sid.sample = sample;
    while (siddata.sid.b >= 1.0)
	siddata.sid.b -= 1.0;
}
#endif

#endif /* SOUND */

#if !defined(SOUND) && defined(SID)

static BYTE sid[0x20];

void REGPARM2 store_sid(ADDRESS address, BYTE byte)
{
    address &= 0x1f;
    sid[address] = byte;
    return;
}

BYTE REGPARM1 read_sid(ADDRESS address)
{
    address &= 0x1f;
    if (address == 0x1b)
	return rand();
    else
	return sid[address];
}

#endif /* !SOUND && SID */
