/*
 * $Id: lex.c,v 1.26 1995/11/10 07:02:06 coleman Exp coleman $
 *
 * lex.c - lexical analysis
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

#include "zsh.h"

/* lexical state */

static int dbparens, in_brace_param = 0;
int len = 0, bsiz = 256;
char *bptr;

struct lexstack {
    struct lexstack *next;

    int incmdpos;
    int incond;
    int incasepat;
    int dbparens;
    int in_brace_param;
    int alstat;
    int isfirstln;
    int isfirstch;
    int histremmed;
    int histdone;
    int spaceflag;
    int stophist;
    int alstackind;
    int hlinesz;
    char *hline;
    char *hptr;
    char *tokstr;
    char *bptr;
    int bsiz;
    short *chwords;
    int chwordlen;
    int chwordpos;

    unsigned char *cstack;
    int csp;
};

static struct lexstack *lstack = NULL;

/* History word status variables from hist.c */
extern short *chwords;
extern int chwordlen, chwordpos;

/* save the lexical state */

/* is this a hack or what? */

/**/
void
lexsave(void)
{
    struct lexstack *ls;

    ls = (struct lexstack *)malloc(sizeof(struct lexstack));

    ls->incmdpos = incmdpos;
    ls->incond = incond;
    ls->incasepat = incasepat;
    ls->dbparens = dbparens;
    ls->in_brace_param = in_brace_param;
    ls->alstat = alstat;
    ls->alstackind = alstackind;
    ls->isfirstln = isfirstln;
    ls->isfirstch = isfirstch;
    ls->histremmed = histremmed;
    ls->histdone = histdone;
    ls->spaceflag = spaceflag;
    ls->stophist = stophist;
    ls->hline = chline;
    ls->hptr = hptr;
    ls->hlinesz = hlinesz;
    ls->cstack = cmdstack;
    ls->csp = cmdsp;
    cmdstack = (unsigned char *)zalloc(256);
    ls->tokstr = tokstr;
    ls->bptr = bptr;
    ls->bsiz = bsiz;
    ls->chwords = chwords;
    ls->chwordlen = chwordlen;
    ls->chwordpos = chwordpos;
    in_brace_param = 0;
    cmdsp = 0;
    inredir = 0;

    ls->next = lstack;
    lstack = ls;
}

/* restore lexical state */

/**/
void
lexrestore(void)
{
    struct lexstack *ln;

    if (!lstack) {
	zerr("lexrestore without lexsave", NULL, 0);
	return;
    }
    incmdpos = lstack->incmdpos;
    incond = lstack->incond;
    incasepat = lstack->incasepat;
    dbparens = lstack->dbparens;
    in_brace_param = lstack->in_brace_param;
    alstat = lstack->alstat;
    isfirstln = lstack->isfirstln;
    isfirstch = lstack->isfirstch;
    histremmed = lstack->histremmed;
    histdone = lstack->histdone;
    spaceflag = lstack->spaceflag;
    stophist = lstack->stophist;
    chline = lstack->hline;
    hptr = lstack->hptr;
    if (cmdstack)
	free(cmdstack);
    cmdstack = lstack->cstack;
    cmdsp = lstack->csp;
    tokstr = lstack->tokstr;
    bptr = lstack->bptr;
    bsiz = lstack->bsiz;
    chwords = lstack->chwords;
    chwordlen = lstack->chwordlen;
    chwordpos = lstack->chwordpos;
    clearalstack();
    alstackind = lstack->alstackind;
    hlinesz = lstack->hlinesz;
    lexstop = errflag = 0;

    ln = lstack->next;
    free(lstack);
    lstack = ln;
}

/**/
void
yylex(void)
{
    if (tok == LEXERR)
	return;
    do
	tok = gettok();
    while (tok != ENDINPUT && exalias());
    if (tok != NEWLIN)
	isnewlin = 0;
    else
	isnewlin = (inbufct) ? -1 : 1;
    if (tok == SEMI || tok == NEWLIN)
	tok = SEPER;
}

/**/
void
ctxtlex(void)
{
    static int oldpos;

    yylex();
    switch (tok) {
    case SEPER:
    case NEWLIN:
    case SEMI:
    case DSEMI:
    case AMPER:
    case INPAR:
    case INBRACE:
    case DBAR:
    case DAMPER:
    case BAR:
    case BARAMP:
    case INOUTPAR:
    case DO:
    case THEN:
    case ELIF:
    case ELSE:
	incmdpos = 1;
	break;
    case STRING:		/* case ENVSTRING: */
    case ENVARRAY:
    case OUTPAR:
    case CASE:
	incmdpos = 0;
	break;
    }
    if (IS_REDIROP(tok) || tok == FOR || tok == FOREACH || tok == SELECT) {
	inredir = 1;
	oldpos = incmdpos;
	incmdpos = 0;
    } else if (inredir) {
	incmdpos = oldpos;
	inredir = 0;
    }
}

#define LX1_BKSLASH 0
#define LX1_COMMENT 1
#define LX1_NEWLIN 2
#define LX1_SEMI 3
#define LX1_BANG 4
#define LX1_AMPER 5
#define LX1_BAR 6
#define LX1_INPAR 7
#define LX1_OUTPAR 8
#define LX1_INBRACE 9
#define LX1_OUTBRACE 10
#define LX1_INBRACK 11
#define LX1_OUTBRACK 12
#define LX1_INANG 13
#define LX1_OUTANG 14
#define LX1_OTHER 15

#define LX2_BREAK 0
#define LX2_OUTPAR 1
#define LX2_BAR 2
#define LX2_STRING 3
#define LX2_INBRACK 4
#define LX2_OUTBRACK 5
#define LX2_TILDE 6
#define LX2_INPAR 7
#define LX2_INBRACE 8
#define LX2_OUTBRACE 9
#define LX2_OUTANG 10
#define LX2_INANG 11
#define LX2_EQUALS 12
#define LX2_BKSLASH 13
#define LX2_QUOTE 14
#define LX2_DQUOTE 15
#define LX2_BQUOTE 16
#define LX2_COMMA 17
#define LX2_OTHER 18

unsigned char lexact1[256], lexact2[256], lextok2[256];

/**/
void
initlextabs(void)
{
    int t0;
    static char *lx1 = "\\q\n;!&|(){}[]<>xx";
    static char *lx2 = "x)|$[]~({}><=\\\'\"`,x";

    for (t0 = 0; t0 != 256; t0++) {
	lexact1[t0] = LX1_OTHER;
	lexact2[t0] = LX2_OTHER;
	lextok2[t0] = t0;
    }
    for (t0 = 0; lx1[t0]; t0++)
	if (lx1[t0] != 'x')
	    lexact1[(int)lx1[t0]] = t0;
    for (t0 = 0; lx2[t0]; t0++)
	if (lx2[t0] != 'x')
	    lexact2[(int)lx2[t0]] = t0;
    lexact2[';'] = LX2_BREAK;
    lexact2['&'] = LX2_BREAK;
    lextok2['*'] = Star;
    lextok2['?'] = Quest;
    lextok2['{'] = Inbrace;
    lextok2['['] = Inbrack;
    lextok2['$'] = String;
}

/* initialize lexical state */

/**/
void
lexinit(void)
{
    incond = incasepat = nocorrect =
    dbparens = alstat = lexstop = in_brace_param = 0;
    incmdpos = 1;
    tok = ENDINPUT;
    if (isset(EXTENDEDGLOB)) {
	lextok2['#'] = Pound;
	lextok2['^'] = Hat;
    } else {
	lextok2['#'] = '#';
	lextok2['^'] = '^';
    }
}

/* add a char to the string buffer */

/**/
void
add(int c)
{
    *bptr++ = c;
    if (bsiz == ++len) {
	int newbsiz;

	newbsiz = bsiz * 8;
	while (newbsiz < inbufct)
	    newbsiz *= 2;
	bptr = len + (tokstr = (char *)hrealloc(tokstr, bsiz, newbsiz));
	bsiz = newbsiz;
    }
}

static void unadd _((void));

static void
unadd(void)
{
    bptr--;
    len--;
}

/**/
int
gettok(void)
{
    int bct = 0, pct = 0, brct = 0;
    int c, d, intpos = 1;
    int peekfd = -1, peek, ninbracks, intick;

  beginning:
    tokstr = NULL;
    parbegin = -1;
    while (iblank(c = hgetc()) && !lexstop);
    isfirstln = 0;
    wordbeg = inbufct;
    hwbegin(-1);		/* word includes the last character read */
    if (dbparens) {
	hungetc(c);
	c = '\"';		/* " */
    } else if (idigit(c)) {	/* handle 1< foo */
	d = hgetc();
	hungetc(d);
	lexstop = 0;
	if (d == '>' || d == '<') {
	    peekfd = c - '0';
	    c = hgetc();
	}
    }
/* chars in initial position in word */

    if (c == hashchar &&
	(isset(INTERACTIVECOMMENTS) ||
	 (!zleparse && (!interact || unset(SHINSTDIN) || strin)))) {
    /* History is handled here to prevent extra newlines
		 * being inserted into the history.
		 */
	while ((c = ingetc()) != '\n' && !lexstop)
	    hwaddc(c);

	if (errflag)
	    peek = LEXERR;
	else {
	    hwend();
	    hwaddc('\n');
	    peek = NEWLIN;
	}
	return peek;
    }
    if (lexstop)
	return (errflag) ? LEXERR : ENDINPUT;
    switch (lexact1[(unsigned char)c]) {
    case LX1_BKSLASH:
	d = hgetc();
	if (d == '\n')
	    goto beginning;
	hungetc(d);
	break;
    case LX1_NEWLIN:
	return NEWLIN;
    case LX1_SEMI:
	d = hgetc();
	if (d != ';') {
	    hungetc(d);
	    return SEMI;
	}
	return DSEMI;
    case LX1_BANG:
	d = hgetc();
	hungetc(d);
	if (!inblank(d))
	    break;
	if (incmdpos || incond)
	    return BANG;
	break;
    case LX1_AMPER:
	d = hgetc();
	if (d != '&') {
	    hungetc(d);
	    return AMPER;
	}
	return DAMPER;
    case LX1_BAR:
	d = hgetc();
	if (d == '|')
	    return DBAR;
	else if (d == '&')
	    return BARAMP;
	hungetc(d);
	return BAR;
    case LX1_INPAR:
	d = hgetc();
	if (d == '(' && incmdpos) {
	    dbparens = 1;
	    return DINPAR;
	} else if (d == ')')
	    return INOUTPAR;
	hungetc(d);
	if (!(incond == 1 || incmdpos))
	    break;
	return INPAR;
    case LX1_OUTPAR:
	return OUTPAR;
    case LX1_INBRACE:
	if (!incmdpos)
	    break;
	return INBRACE;
    case LX1_OUTBRACE:
	return OUTBRACE;
    case LX1_INBRACK:
	if (!incmdpos)
	    break;
	d = hgetc();
	if (d == '[') {
	    incond = 1;
	    return DINBRACK;
	}
	hungetc(d);
	break;
    case LX1_OUTBRACK:
	if (!incond)
	    break;
	d = hgetc();
	if (d == ']') {
	    incond = 0;
	    return DOUTBRACK;
	}
	hungetc(d);
	break;
    case LX1_INANG:
	d = hgetc();
	if ((!incmdpos && d == '(') || incasepat) {
	    hungetc(d);
	    break;
	} else if (idigit(d) || d == '-' || d == '>') {
	    int tbs = 256, n = 0, nc;
	    char *tbuf, *tbp, *ntb;

	    tbuf = tbp = (char *)zalloc(tbs);
	    hungetc(d);

	    while ((nc = hgetc()) && !lexstop) {
		if (!idigit(nc) && nc != '-')
		    break;
		*tbp++ = (char)nc;
		if (++n == tbs) {
		    ntb = (char *)realloc(tbuf, tbs *= 2);
		    tbp += ntb - tbuf;
		    tbuf = ntb;
		}
	    }
	    if (nc == '>' && !lexstop) {
		lexstop = 0;
		hungetc(nc);
		while (n--)
		    hungetc(*--tbp);
		zfree(tbuf, tbs);
		break;
	    }
	    lexstop = 0;
	    if (nc)
		hungetc(nc);
	    while (n--)
		hungetc(*--tbp);
	    zfree(tbuf, tbs);
	    peek = INANG;
	} else if (d == '<') {
	    int e = hgetc();

	    if (e == '(') {
		hungetc(e);
		hungetc(d);
		peek = INANG;
	    } else if (e == '<')
		peek = TRINANG;
	    else if (e == '-')
		peek = DINANGDASH;
	    else {
		hungetc(e);
		peek = DINANG;
	    }
	} else if (d == '&')
	    peek = INANGAMP;
	else {
	    peek = INANG;
	    hungetc(d);
	}
	tokfd = peekfd;
	return peek;
    case LX1_OUTANG:
	d = hgetc();
	if (d == '(') {
	    hungetc(d);
	    break;
	} else if (d == '&') {
	    d = hgetc();
	    if (d == '!' || d == '|')
		peek = OUTANGAMPBANG;
	    else {
		hungetc(d);
		peek = OUTANGAMP;
	    }
	} else if (d == '!' || d == '|')
	    peek = OUTANGBANG;
	else if (d == '>') {
	    d = hgetc();
	    if (d == '&') {
		d = hgetc();
		if (d == '!' || d == '|')
		    peek = DOUTANGAMPBANG;
		else {
		    hungetc(d);
		    peek = DOUTANGAMP;
		}
	    } else if (d == '!' || d == '|')
		peek = DOUTANGBANG;
	    else if (d == '(') {
		hungetc(d);
		hungetc('>');
		peek = OUTANG;
	    } else {
		hungetc(d);
		peek = DOUTANG;
		if (isset(HISTALLOWCLOBBER))
		    hwaddc('|');
	    }
	} else {
	    hungetc(d);
	    peek = OUTANG;
	    if (!incond && isset(HISTALLOWCLOBBER))
		hwaddc('|');
	}
	tokfd = peekfd;
	return peek;
    }

/* we've started a string, now get the rest of it, performing
		tokenization */

    peek = STRING;
    len = 0;
    bptr = tokstr = (char *)ncalloc(bsiz = 256);
    for (;;) {
	int act;
	int e;
	int endchar;

	if (inblank(c) && !bct && !pct)
	    act = LX2_BREAK;
	else {
	    act = lexact2[(unsigned char)c];
	    c = lextok2[(unsigned char)c];
	}
	switch (act) {
	case LX2_BREAK:
	    if (!in_brace_param)
		goto brk;
	    break;
	case LX2_OUTPAR:
	    if (!pct)
		goto brk;
	    c = Outpar;
	    pct--;
	    break;
	case LX2_BAR:
	    if (!pct && !incasepat)
		goto brk;
	    c = Bar;
	    break;
	case LX2_STRING:
	    e = hgetc();
	    if (e == '[') {
		cmdpush(CS_MATHSUBST);
		add(String);
		add(Inbrack);
		ninbracks = 1;
		while (ninbracks && (c = hgetc()) && !lexstop) {
		    if (c == '[')
			ninbracks++;
		    else if (c == ']')
			ninbracks--;
		    if (ninbracks)
			add(c);
		}
		c = Outbrack;
		cmdpop();
	    } else if (e == '(') {
		add(String);
		if (skipcomm()) {
		    peek = LEXERR;
		    goto brk;
		}
		c = Outpar;
	    } else {
		if (e == '{')
		    in_brace_param = 1;
		hungetc(e);
	    }
	    break;
	case LX2_INBRACK:
	    brct++;
	    break;
	case LX2_OUTBRACK:
	    if (incond && !brct)
		goto brk;
	    brct--;
	    c = Outbrack;
	    break;
	case LX2_TILDE:	/* if (intpos) */
	    c = Tilde;
	    break;
	case LX2_INPAR:
	    e = hgetc();
	    hungetc(e);
	    if (e == ')' ||
		(incmdpos && !brct && peek != ENVSTRING))
		goto brk;
	    pct++;
	    c = Inpar;
	    break;
	case LX2_INBRACE:
	    bct++;
	    break;
	case LX2_OUTBRACE:
	    if (!bct)
		goto brk;
	    if (!--bct && in_brace_param)
		in_brace_param = 0;
	    c = Outbrace;
	    break;
	case LX2_COMMA:
	    if (unset(IGNOREBRACES) && bct > in_brace_param)
		c = Comma;
	    break;
	case LX2_OUTANG:
	    e = hgetc();
	    if (e != '(') {
		hungetc(e);
		goto brk;
	    }
	    add(Outang);
	    if (skipcomm()) {
		peek = LEXERR;
		goto brk;
	    }
	    c = Outpar;
	    break;
	case LX2_INANG:
	    e = hgetc();
	    if (!(idigit(e) || e == '-' || e == '>' || e == '(' || e == ')')) {
		hungetc(e);
		goto brk;
	    }
	    c = Inang;
	    if (e == '(') {
		add(c);
		if (skipcomm()) {
		    peek = LEXERR;
		    goto brk;
		}
		c = Outpar;
	    } else if (e == ')')
		hungetc(e);
	    else {
		add(c);
		c = e;
		while (c != '>' && !lexstop)
		    add(c), c = hgetc();
		c = Outang;
	    }
	    break;
	case LX2_EQUALS:
	    if (intpos) {
		e = hgetc();
		if (e != '(') {
		    hungetc(e);
		    c = Equals;
		} else {
		    add(Equals);
		    if (skipcomm()) {
			peek = LEXERR;
			goto brk;
		    }
		    c = Outpar;
		}
	    } else if (peek != ENVSTRING && incmdpos && !bct) {
		e = hgetc();
		if (e == '(' && incmdpos) {
		    *bptr = '\0';
		    return ENVARRAY;
		}
		hungetc(e);
		peek = ENVSTRING;
		intpos = 2;
	    } else
		c = Equals;
	    break;
	case LX2_BKSLASH:
	    c = hgetc();
	    if (c == '\n') {
		c = hgetc();
		continue;
	    }
	    add(Bnull);
	    add(c);
	    c = hgetc();
	    continue;
	case LX2_QUOTE:
	    add(Snull);

	/* we add the Nularg to prevent this:

				echo $PA'TH'

				from printing the path. */

	    cmdpush(CS_QUOTE);
	    for (;;) {
		int bc = bangchar;

		bangchar = '\0';
		while ((c = hgetc()) != '\'' && !lexstop) {
		    if (isset(CSHJUNKIEQUOTES) && c == '\n') {
			if (bptr[-1] == '\\')
			    unadd();
			else
			    break;
		    }
		    add(c);
		}
		bangchar = bc;
		if (c != '\'') {
		    zerr("unmatched \'", NULL, 0);
		    peek = LEXERR;
		    cmdpop();
		    goto brk;
		}
		e = hgetc();
		if (e != '\'' || unset(RCQUOTES))
		    break;
		add(c);
	    }
	    cmdpop();
	    hungetc(e);
	    c = Snull;
	    break;
	case LX2_DQUOTE:
	    add(Dnull);
	    cmdpush(dbparens ? CS_MATH : CS_DQUOTE);	/* math or dquote */
	    endchar = dbparens ? ')' : '"';
	    intick = 0;
	    while (((c = hgetc()) != endchar || (dbparens && pct > 0) ||
		    intick) && !lexstop)
		if (c == '\\') {
		    c = hgetc();
		    if (c != '\n') {
			add(c == '$' || c == '\\' ||
			    c == '\"' || c == '`' ? Bnull : '\\');
			add(c);
		    }
		} else {
		    if (isset(CSHJUNKIEQUOTES) && c == '\n') {
			if (bptr[-1] == '\\')
			    unadd();
			else
			    break;
		    }
		    if (c == '$' && !intick) {
			e = hgetc();
			if (e == '(') {
			    add(Qstring);
			    if (skipcomm()) {
				peek = LEXERR;
				cmdpop();
				goto brk;
			    }
			    c = Outpar;
			} else if (e == '[') {
			    int qbrct = 1;
			    add(String);
			    add(Inbrack);
			    while (c = hgetc(), !lexstop) {
				if (c == '[')
				    qbrct++;
				if (c == ']' && !--qbrct)
				    break;
				add(c);
				if (c == '\\')
				    if (c = hgetc(), !lexstop)
					add(c);
				    else
					break;
			    }
			    c = Outbrack;
			} else {
			    c = Qstring;
			    hungetc(e);
			}
		    } else if (c == '`') {
			c = Qtick;
			intick = !intick;
		    } else if (dbparens) {
			if (c == '(')
			    pct++;
			else if (c == ')')
			    pct--;
		    }
		    add(c);
		}
	    cmdpop();
	    if (c != endchar || (dbparens && (pct || (c = hgetc()) != ')'))) {
		if (!dbparens)
		    zerr("unmatched \"", NULL, 0);
		peek = LEXERR;
		goto brk;
	    }
	    if (dbparens) {
		dbparens = 0;
		add(Dnull);
		*bptr = '\0';
		return DOUTPAR;
	    }
	    c = Dnull;
	    break;
	case LX2_BQUOTE:
	    add(Tick);
	    cmdpush(CS_BQUOTE);
	    parbegin = inbufct;
	    while ((c = hgetc()) != '`' && !lexstop)
		if (c == '\\') {
		    c = hgetc();
		    if (c != '\n') {
			add(c == '`' || c == '\\' || c == '$' ? Bnull : '\\');
			add(c);
		    }
		} else {
		    if (isset(CSHJUNKIEQUOTES) && c == '\n') {
			if (bptr[-1] == '\\')
			    unadd();
			else
			    break;
		    }
		    add(c);
		}
	    cmdpop();
	    if (c != '`') {
		if (!zleparse)
		    zerr("unmatched `", NULL, 0);
		peek = LEXERR;
		goto brk;
	    }
	    c = Tick;
	    parbegin = -1;
	    break;
	}
	add(c);
	c = hgetc();
	if (intpos)
	    intpos--;
	if (lexstop)
	    break;
    }
  brk:
    hungetc(c);
    *bptr = '\0';
    return peek;
}

/* expand aliases and reserved words */

/**/
int
exalias(void)
{
    Alias an;
    Reswd rw;
    char *s;

    hwend();
    hwget(&yytext);
    s = yytext;
    if (interact && isset(SHINSTDIN) && !strin && !incasepat && tok == STRING &&
	(isset(CORRECTALL) || (isset(CORRECT) && incmdpos)) && !nocorrect)
	spckword(&tokstr, &s, incmdpos, 1);
    if (zleparse && !alstackind) {
	int zp = zleparse;

	gotword(s);
	if (zp && !zleparse) {
	    return 0;
	}
    }

    /* Check for an alias */
    an = noaliases ? NULL : (Alias) aliastab->getnode(aliastab, s);
    if (alstackind != MAXAL && an && !an->inuse &&
	((an->flags & ALIAS_GLOBAL) || incmdpos || alstat == ALSTAT_MORE)) {
	    an->inuse = 1;
	    inpush((alstack[alstackind++] = an)->text, INP_ALIAS);
	    alstat = 0;
	    /* remove from history if it begins with space */
	    if (isset(HISTIGNORESPACE) && an->text[0] == ' ')
		remhist();
	    lexstop = 0;
	    return 1;
	}

    /* Then check for a reserved word (ignored if noaliases is set) */
    rw = (noaliases || !incmdpos) ? NULL : (Reswd) reswdtab->getnode(reswdtab, s);
    if (rw)
	tok = rw->token;
    return 0;
}

/* skip (...) */

/**/
int
skipcomm(void)
{
    int pct = 1, c;

    cmdpush(CS_CMDSUBST);
    parbegin = inbufct;
    c = Inpar;
    do {
	add(c);
	c = hgetc();
	if (itok(c) || lexstop)
	    break;
	else if (c == '(')
	    pct++;
	else if (c == ')')
	    pct--;
	else if (c == '\\') {
	    add(c);
	    c = hgetc();
	} else if (c == '\'') {
	    int bc = bangchar;
	    bangchar = '\0';
	    add(c);
	    while ((c = hgetc()) != '\'' && !lexstop)
		add(c);
	    bangchar = bc;
	} else if (c == '\"') {
	    add(c);
	    while ((c = hgetc()) != '\"' && !lexstop)
		if (c == '\\') {
		    add(c);
		    add(hgetc());
		} else
		    add(c);
	} else if (c == '`') {
	    add(c);
	    while ((c = hgetc()) != '`' && !lexstop)
		if (c == '\\')
		    add(c), add(hgetc());
		else
		    add(c);
	}
    }
    while (pct);
    if (!lexstop)
	parbegin = -1;
    cmdpop();
    return lexstop;
}

