/*
 * $Id: input.c,v 1.8 1995/12/11 04:38:12 coleman Exp coleman $
 *
 * input.c - read and store lines of input
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */


/*
 * This file deals with input buffering, supplying characters to the
 * history expansion code a character at a time.  Input is stored on a
 * stack, which allows insertion of strings into the input, possibly with
 * flags marking the end of alias expansion, with minimal copying of
 * strings.
 * 
 * Input is taken either from zle, if appropriate, or read directly from
 * the input file, or may be supplied by some other part of the shell (such
 * as `eval' or $(...) substitution).  In the last case, it should be
 * supplied by pushing a new level onto the stack, via inpush(input_string,
 * flag); if the current input really needs to be altered, use
 * inputsetline(input_string, flag).  `Flag' can include or's of INP_FREE
 * (if the input string is to be freed when used), INP_CONT (if the input
 * is to continue onto what's already in the input queue), or INP_ALIAS
 * (used as a mark to pop the alias stack).
 * 
 * Note that the input string is itself used as the input buffer: it is not
 * copied, nor is it every written back to, so using a constant string
 * should work.  Consequently, when passing areas of memory from the heap
 * it is necessary that that heap last as long as the operation of reading
 * the string.  After the string is read, the stack should be popped with
 * inpop(), which effectively flushes any unread input as well as restoring
 * the previous input state,
 *
 * PWS 1995/06/30
 */

#include "zsh.h"

/* Input buffer variables:  only inbufct is global, see globals.h. */
static char *inbuf;		/* Current input buffer */
static char *inbufptr;		/* Pointer into input buffer */
static int inbufleft;		/* Characters left in current input
				   stack element */
static int inbufflags;		/* flags: see INP_* in zsh.h  */

static int lastc;		/* used as flag that end of line was reached */


 /* Input must be stacked since the input queue is used by
  * various different parts of the shell.
  */

struct instacks {
    char *buf, *bufptr;
    int bufleft, bufct, flags;
};
static struct instacks *instack, *instacktop;
/*
 * Input stack size.  We need to push the stack for aliases, history
 * expansion, and reading from internal strings: only if these operations
 * are nested do we need more than one extra level.  Thus we shouldn't need
 * too much space as a rule.  Initially, INSTACK_INITIAL is allocated; if
 * more is required, an extra INSTACK_EXPAND is added each time.
 */
#define INSTACK_INITIAL	4
#define INSTACK_EXPAND	4

static int instacksz = INSTACK_INITIAL;

/*
 * Stack to deal with popped aliases.
 * The problem is that we reach the end of the alias text and pop
 * the alias at the point where we read the next character
 * afterwards, which is probably whitespace, *then* go back and
 * look to see if the expanded text needs re-expanding.  This
 * means we need to keep track of the aliases we've popped and
 * maybe add them back in if we unget the whitespace character.
 *
 * What happens is that ingetc(), when it pops an alias, sticks it on the
 * stack, and inungetc() looks at that to see if it has to restore the
 * alias(es); if so, the commands to pop an alias go back in the input
 * queue. If ingetc() is called again, it sets the stack to zero.
 * 
 * We also need to be careful about `lexstop' when reading from a string.
 * 
 * Historical note:  the code used to add a bogus space to the end of
 * aliases, after which the alias would be popped.  With the new
 * history code, which keeps track of spaces correctly, this won't work.
 */
static Alias *inalstack, *inalstacktop;
static int inalstacksz = INSTACK_INITIAL;

/* Get the next character from the input.
 * Will call inputline() to get a new line where necessary.
 */
  
/**/
int
ingetc(void)
{
    inalstacktop = inalstack;
    for (;;) {
	if (inbufleft) {
	    inbufleft--;
	    inbufct--;
	    return lastc = STOUC(*inbufptr++);
	}
	/*
	 * No characters in input buffer.
	 * See if we can pop the alias stack at this point.
	 */
	if ((inbufflags & INP_ALIAS) && alstackind > 0) {
	    /*
	     * Flag that we should pop the alias stack at this point.
	     * Either we have reached the end of an alias expansion,
	     * or the end of a history expansion.
	     */
	    Alias ix;
	    char *t;
	    
	    ix = alstack[--alstackind];
	    if (ix) {
		/* a real alias:  mark it as unused. */
		ix->inuse = 0;
		t = ix->text;
		if (*t && t[strlen(t) - 1] == ' ') {
		    alstat = ALSTAT_MORE;
		    histbackword();
		} else
		    alstat = ALSTAT_JUNK;
		inalpush(ix);
	    }
	}

	/* If the next element down the input stack is a continuation of
	 * this, use it.
	 */ 
	if (inbufflags & (INP_CONT|INP_ALIAS)) {
	    inpoptop();
	    continue;
	}
	/*
	 * Otherwise, see if we have reached the end of input
	 * (due to an error, or to reading from a single string).
	 */
	if (strin || errflag) {
	    lexstop = 1;
	    return lastc = ' ';
	}
	/* As a last resort, get some more input */
	if (inputline())
	    return lastc = ' ';
    }
}

/* Read a line from the current command stream and store it as input */

/**/
int
inputline(void)
{
    unsigned char *ingetcline, *ingetcpmptl = NULL, *ingetcpmptr = NULL;

    /* If reading code interactively, work out the prompts. */
    if (interact && isset(SHINSTDIN))
	if (!isfirstln)
	    ingetcpmptl = (unsigned char *)prompt2;
	else {
	    ingetcpmptl = (unsigned char *)prompt;
	    if (rprompt)
		ingetcpmptr = (unsigned char *)rprompt;
	}
    if (!(interact && isset(SHINSTDIN) && SHTTY != -1 && isset(USEZLE))) {
	/*
	 * If not using zle, read the line straight from the input file.
	 * Possibly we don't get the whole line at once:  in that case,
	 * we get another chunk with the next call to inputline().
	 */
	char *lbuf;

	if (interact && isset(SHINSTDIN)) {
	    /*
	     * We may still be interactive (e.g. running under emacs),
	     * so output a prompt if necessary.  We don't know enough
	     * about the input device to be able to handle an rprompt,
	     * though.
	     */
	    char *pptbuf;
	    int pptlen;
	    pptbuf = putprompt((char *)ingetcpmptl, &pptlen, NULL, 1);
	    write(2, (WRITE_ARG_2_T)pptbuf, pptlen);
	    free(pptbuf);
	}
	lbuf = (char *)zalloc(256);
	do {
	    clearerr(bshin);
	    errno = 0;
	    ingetcline = (unsigned char *)fgets(lbuf, 256, bshin);
	    /* Keep looping in case interrupted by e.g. SIGCHLD. */
	} while (!ingetcline && ferror(bshin) && errno == EINTR);
	if (!ingetcline)
	    zfree(lbuf, 256);
    } else
	ingetcline = zleread((char *)ingetcpmptl, (char *)ingetcpmptr);
    if (!ingetcline) {
	return lexstop = 1;
    }
    if (errflag) {
	free(ingetcline);
	return lexstop = errflag = 1;
    }
    /* Look for a space, to see if this shouldn't be put into history */
    if (isfirstln)
	spaceflag = *ingetcline == ' ';
    if (isset(VERBOSE)) {
	/* Output the whole line read so far. */
	fputs((char *)ingetcline, stderr);
	fflush(stderr);
    }
    if (*ingetcline && ingetcline[strlen((char *)ingetcline) - 1] == '\n') {
	/* We've now read a complete line. */
	lineno++;
	if (interact && isset(SUNKEYBOARDHACK) && isset(SHINSTDIN) &&
	    SHTTY != -1 && *ingetcline && ingetcline[1] &&
	    ingetcline[strlen((char *)ingetcline) - 2] == '`') {
	    /* Junk an unmatched "`" at the end of the line. */
	    int ct;
	    unsigned char *ptr;

	    for (ct = 0, ptr = ingetcline; *ptr; ptr++)
		if (*ptr == '`')
		    ct++;
	    if (ct & 1) {
		ptr[-2] = '\n';
		ptr[-1] = '\0';
	    }
	}
    }
    isfirstch = 1;
    /* Put this into the input channel. */
    inputsetline((char *)ingetcline, INP_FREE);

    return 0;
}

/* Read one line of at most n-1 chars from the input queue */

/**/
char *
ingets(char *buf, int n)
{
    int l;

    for (l = 0; l < n - 1; l++)
	if ((buf[l] = ingetc()) == '\n' || lexstop)
	    break;
    buf[l + (lexstop ? 0 : 1)] = 0;

    return (!lexstop || l) ? buf : NULL;
}

/*
 * Put a string in the input queue:
 * inbuf is only freeable if the flags include INP_FREE.
 */

/**/
void
inputsetline(char *str, int flags)
{
    if ((inbufflags & INP_FREE) && inbuf) {
	free(inbuf);
    }
    inbuf = inbufptr = str;
    inbufleft = strlen(inbuf);

    /*
     * inbufct must reflect the total number of characters left,
     * as it used by other parts of the shell, so we need to take account
     * of whether the input stack continues, and whether there
     * is an extra space to add on at the end.
     */
    if (flags & (INP_ALIAS|INP_CONT))
	inbufct += inbufleft;
    else
	inbufct = inbufleft;
    inbufflags = flags;
}

/*
 * Backup one character of the input.
 * The last character can always be backed up, provided we didn't just
 * expand an alias or a history reference.
 * In fact, the character is ignored and the previous character is used.
 * (If that's wrong, the bug is in the calling code.  Use the #if 0 code
 * to check.) 
 */

/**/
void
inungetc(int c)
{
    if (!lexstop) {
	if (inbufptr != inbuf) {
#if 0
	    /* Just for debugging: enable only if foul play suspected. */
	    if (inbufptr[-1] != c)
		fprintf(stderr, "Warning: backing up wrong character.\n");
#endif
	    /* Just decrement the pointer:  if it's not the same
	     * character being pushed back, we're in trouble anyway.
	     */
	    inbufptr--;
	    inbufct++;
	    inbufleft++;
	}
#if 0
        else {
	    /* Just for debugging */
	    fprintf(stderr, "Attempt to inungetc() at start of input.\n");
	}
#endif
    }
    if (inalstacktop > inalstack)
	inalrestore();
}

/* stuff a whole file into the input queue and print it */

/**/
int
stuff(char *fn)
{
    FILE *in;
    char *buf;
    int len;

    if (!(in = fopen(fn, "r"))) {
	zerr("can't open %s", fn, 0);
	return 1;
    }
    fseek(in, 0, 2);
    len = ftell(in);
    fseek(in, 0, 0);
    buf = (char *)zalloc(len + 1);
    if (!(fread(buf, len, 1, in))) {
	zerr("read error on %s", fn, 0);
	fclose(in);
	zfree(buf, len + 1);
	return 1;
    }
    fclose(in);
    buf[len] = '\0';
    fwrite(buf, len, 1, stdout);
    inputsetline(buf, INP_FREE);
    return 0;
}

/* flush input queue */

/**/
void
inerrflush(void)
{
    /*
     * This always goes character by character, but at present
     * it is only used in the history code, where that is the only
     * completely safe way of discarding input.
     */
    while ((strin || lastc != '\n') && !lexstop)
	ingetc();
}

/* Set some new input onto a new element of the input stack */

/**/
void
inpush(char *str, int flags)
{
    if (!instack) {
	/* Initial stack allocation */
	instack = (struct instacks *)zalloc(instacksz*sizeof(struct instacks));
	instacktop = instack;
    } else if (instacktop == instack + instacksz) {
	/* Expand the stack */
	instack = (struct instacks *)
	    realloc(instack,
		    (instacksz + INSTACK_EXPAND)*sizeof(struct instacks));
	instacktop = instack + instacksz;
	instacksz += INSTACK_EXPAND;
    }
    instacktop->buf = inbuf;
    instacktop->bufptr = inbufptr;
    instacktop->bufleft = inbufleft;
    instacktop->bufct = inbufct;
    instacktop->flags = inbufflags;

    instacktop++;

    inbuf = 0;

    inputsetline(str, flags);
}

/* Remove the top element of the stack */

/**/
void
inpoptop(void)
{
    if (inbuf && (inbufflags & INP_FREE))
	free(inbuf);
	
    instacktop--;

    inbuf = instacktop->buf;
    inbufptr = instacktop->bufptr;
    inbufleft = instacktop->bufleft;
    inbufct = instacktop->bufct;
    inbufflags = instacktop->flags;
}

/* Remove the top element of the stack and all its continuations. */

/**/
void
inpop(void)
{
    int remcont;

    do {
	remcont = inbufflags & (INP_CONT|INP_ALIAS);

	inpoptop();
    } while (remcont);
}

/**/
void
inalpush(Alias al)
{
    if (!inalstack) {
	 inalstack = (Alias *)zalloc(inalstacksz*sizeof(Alias));
	 inalstacktop = inalstack;
    } else if (inalstacktop - inalstack == inalstacksz) {
	inalstack = (Alias *)
	    realloc(inalstack, (inalstacksz + INSTACK_EXPAND)*sizeof(Alias));
	inalstacktop = inalstack + inalstacksz;
	inalstacksz += INSTACK_EXPAND;
    }
    *inalstacktop++ = al;
}

/**/
void
inalrestore(void)
{
    while (inalstacktop > inalstack) {
	Alias al = *--inalstacktop;
	alstack[alstackind++] = al;
	al->inuse = 1;
	inpush("", INP_ALIAS);
    }
}
