/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include "child.h"
#include "file.h"
#include "parse.h"
#include "save.h"
#include "server.h"
#include "util.h"
#include "viewer.h"
#include "widgets.h"
#include "xutil.h"
#include "../Widgets/ArtText.h"
#include "../Widgets/Compat.h"
#include "../Widgets/Dialogue.h"

extern void unref_viewable(VIEWABLE *v)
{
    if (v && --v->ref_count <= 0) {
	XtFree(v->data);
	v->data = NULL;
	v->len = 0;
	XtFree(v->viewer);
	v->viewer = NULL;
	v->ref_count = 0;
	XtFree((char *)v);
    }
}

static void mime_dialogue_callback(Widget w,
				   XtPointer client_data,
				   XtPointer call_data)
{
    DialogueReport	*report = (DialogueReport *)call_data;
    VIEWABLE		*viewable = (VIEWABLE *)client_data;

    switch (report->reply) {
    case DialogueReplyLeft:   /* save */
	if (report->buffer && report->buffer[0] != '\0') {
	    int		fd;
	    char	*data = viewable->data;
	    long	n = viewable->len;

	    fd = open_expand(report->buffer, O_WRONLY|O_CREAT|O_TRUNC,
			     0644, True);
	    if (fd < 0) {
		set_message("Error: Failed to open file!", True);
		return;
	    }

	    while (n > 0) {
		long	i = write(fd, data, n);

		if (i < 0) {
		    perror("knews: write");
		    set_message("Error: Failed to write file!", True);
		    close(fd);
		    return;
		}

		n -= i;
		data += i;
	    }

	    if (close(fd) < 0) {
		perror("knews: close");
		set_message("Error: Failed to write file!", True);
		return;
	    }

	    set_message("Saved OK.", False);
	}
	break;
    case DialogueReplyMiddle: /* pipe */
	if (report->buffer && report->buffer[0] != '\0') {
	    pid_t	pid;
	    char	*cmd;

	    cmd = XtNewString(report->buffer);
	    pid = fork_nicely(cmd, pipe_context_callback,
			      global.stderr_timeout >= 0);

	    if (pid < 0) {
		set_message("Error: fork failed!", True);
		XtFree(cmd);
		return;
	    } else if (pid == 0) { /* child */
		char	*file_name;
		int	fd = create_temp_fd(&file_name);

		if (fd < 0) {
		    perror("knews: open");
		    _exit(127);
		}

		while (viewable->len > 0) {
		    long	i;

		    i = write(fd, viewable->data, viewable->len);
		    if (i < 0) {
			perror("knews: write");
			if (errno == EINTR)
			    continue;
			else
			    _exit(127);
		    }
		    viewable->data += i;
		    viewable->len -= i;
		}

		unlink(file_name);

		if (lseek(fd, SEEK_SET, 0) < 0) {
		    perror("knews: lseek");
		    _exit(127);
		}

		if (fd != STDIN_FILENO) {
		    if (dup2(fd, STDIN_FILENO) < 0) {
			perror("knews: dup2");
			_exit(127);
		    }
		    close(fd);
		}

		execl(BIN_SH, "sh", "-c", report->buffer, (char *)NULL);
		perror("knews: execl " BIN_SH);
		_exit(127);
	    }

	    set_message("Pipe started.", False);
	}
	break;
    case DialogueReplyEnter:  /* do nothing */
	/* maybe we should default to either save or pipe ? */
	return; /* don't fall through */
    case DialogueReplyTab:
	return; /* don't fall through */
    case DialogueReplyRight:
    case DialogueReplyClose:  /* cancel*/
	break;
    }

    unref_viewable(viewable);
    XtDestroyWidget(w);
}

void click_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
    VIEWABLE	*viewable = (VIEWABLE *)client_data;
    int		*clicked = (int *)call_data;

    if (!clicked || !*clicked)
	unref_viewable(viewable);
    else if (viewable->viewer) {
	char	*cmd;
	pid_t	pid;

	cmd = XtNewString(viewable->viewer);
	pid = fork_nicely(cmd, pipe_context_callback, True);

	if (pid < 0) {
	    perror("knews: fork");
	    set_message("Failed to start viewer!", True);
	    return;
	}

	if (pid == 0) { /* child */
	    char	*file_name;
	    char	*viewer;
	    int		fd = create_temp_fd(&file_name);

	    if (fd < 0) {
		perror("knews: open");
		_exit(127);
	    }

	    while (viewable->len > 0) {
		long	i;

		i = write(fd, viewable->data, viewable->len);
		if (i < 0) {
		    perror("knews: write");
		    if (errno == EINTR)
			continue;
		    else
			_exit(127);
		}
		viewable->data += i;
		viewable->len -= i;
	    }

	    viewer = viewable->viewer;
	    if (strstr(viewer, "%s")) {
		int	len, n;
		char	*c;

		n = 0;
		len = 1024;
		c = viewer;
		viewer = malloc(len);
		if (!viewer)
		    _exit(127);

		while (*c != '\0') {
		    if (c[0] == '%' && c[1] == 's') {
			int	i = strlen(file_name);

			if (n + i + 8 > len) {
			    len = n + i + 32;
			    viewer = malloc(len);
			    if (!viewer)
				_exit(127);
			}

			sprintf(viewer + n, "%s", file_name);
			n = strlen(viewer);
			c += 2;
		    } else {
			if (n + 8 > len) {
			    len *= 2;
			    viewer = malloc(len);
			    if (!viewer)
				_exit(127);
			}

			viewer[n++] = *c++;
		    }
		}
		viewer[n] = '\0';
	    } else {
		unlink(file_name);

		if (lseek(fd, SEEK_SET, 0) < 0) {
		    perror("knews: lseek");
		    _exit(127);
		}

		if (fd != STDIN_FILENO) {
		    if (dup2(fd, STDIN_FILENO) < 0) {
			perror("knews: dup2");
			_exit(127);
		    }
		    close(fd);
		}
	    }

	    execl(BIN_SH, "sh", "-c", viewer, (char *)NULL);
	    perror("knews: execl " BIN_SH);
	    _exit(127);
	}

	/* parent */
	set_message("Viewer started.", False);
    } else {
	viewable->ref_count++;
	popup_dialogue("mimedialogue", "Save to file or pipe to shell:",
		       "Save", "Pipe", "Cancel", mime_dialogue_callback,
		       (XtPointer)viewable, XtGrabNone);
    }
}

char *expand_view_command(const char *src, char *type, char *subtype,
			  MimeArg *args, int needs_term, int copious)
{
    char	*dest;
    const char	*p;
    long	len, pos, i, n;

    pos = 0;
    len = 120;
    dest = XtMalloc(len + 3);
    dest[0] = '\0';

    while (*src != '\0')
	if (*src == '%') {
	    char	c = *++src;

	    switch (c) {
	    case '\0':
		continue;
	    case 's':
		src++;
		dest[pos++] = '%';
		dest[pos++] = 's';
		continue;
	    case 't':
		n = strlen(type) + strlen(subtype) + 8;
		if (pos + n + 8 > len) {
		    len += n + 8;
		    dest = XtRealloc(dest, len + 3);
		}
		sprintf(dest + pos, "%s/%s", type, subtype);
		pos += strlen(dest + pos);
		continue;
	    case '{':
		p = strchr(src, '}');
		if (!p)
		    break;
		src++;

		for (i = 0 ; args[i].value ; i++)
		    if (strlen(args[i].name) == p - src &&
			case_strncmp(src, args[i].name, p - src) == 0)
			break;

		src = p + 1;

		if (!args[i].value)
		    continue;

		n = strlen(args[i].value);
		if (pos + n + 8 > len) {
		    len += n + 8;
		    dest = XtRealloc(dest, len + 3);
		}

		strcpy(dest + pos, args[i].value);
		pos += strlen(dest + pos);
		continue;
	    }

	    dest[pos++] = c;
	    src++;
	} else {
	    if (pos + 8 > len) {
		len *= 2;
		dest = XtRealloc(dest, len + 3);
	    }

	    if (*src != '\\')
		dest[pos++] = *src++;
	    else if (*++src != '\0')
		dest[pos++] = *src++;
	}

    dest[pos] = '\0';

    return dest;
}

char *get_server_command(char *res_name)
{
    XrmName	name_list[3];
    XrmClass	class_list[3];
    XrmQuark	rep;
    XrmDatabase	db;
    XrmValue	val;

    while (*res_name == ' ' ||
	   *res_name == '\t')
	res_name++;

    if (*res_name == '\0' ||
	strchr(res_name, '.') ||
	strchr(res_name, '*') ||
	strchr(res_name, ' ') ||
	strchr(res_name, '\t') ||
	strchr(res_name, '/'))
	return NULL;

    class_list[0] = XrmPermStringToQuark("Knews");
    class_list[1] = XrmPermStringToQuark("Server");
    class_list[2] = NULLQUARK;

    name_list[0] = XrmStringToQuark(XtName(main_widgets.shell));
    name_list[1] = XrmStringToQuark(res_name);
    name_list[2] = NULLQUARK;

    db = XtScreenDatabase(XtScreen(main_widgets.shell));

    if (XrmQGetResource(db, name_list, class_list, &rep, &val))
	return (char *)val.addr;
    else
	return NULL;
}

/************************************************************************/

/*
 *  Lots of people enclose URLs in <> or "".  Also, don't allow
 *  quotes for sequrity reasons, since we're passing it to the shell.
 */

#define IS_URL_CHAR(c) (!strchr(" \t<>\"'`\\()[]{}&", (c)))
#define MIN_URL_LEN    4

void text_url_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
    ArtTextUrlReport	*report  = (ArtTextUrlReport *)call_data;
    char		*url_cmd = global.url_command;
    char		*temp;
    long		i;
    pid_t		pid;

    if (!url_cmd) {
	set_message("Error: Resource Knews.urlCommand not set!", True);
	return;
    }

    /*
     *  Very crude parser...
     */

    if (!report->sel_ok) {
	const char	*c;

	c = report->line + report->start;
	if (*c != '\0' && IS_URL_CHAR(*c)) {
	    while (*++c != '\0' && IS_URL_CHAR(*c))
		report->stop++;

	    c = report->line + report->start;
	    while (report->start > 0 && IS_URL_CHAR(*--c))
		report->start--;
	}
    }

    if (report->stop - report->start < MIN_URL_LEN) {
	set_message("Url to short!", True);
	return;
    }

    for (i = report->start ; i <= report->stop ; i++)
	if (!IS_URL_CHAR(report->line[i])) {
	    set_message("Illegal character in URL!", True);
	    return;
	}

    report->sel_ok = True;

    temp = XtNewString("urlCommand");
    pid = fork_nicely(temp, pipe_context_callback,
		      global.stderr_timeout >= 0);

    if (pid < 0) {
	set_message("Error: fork failed!", True);
	XtFree(temp);
	return;
    }

    if (pid == 0) {
	char	*prog, *url;
	long	i, n, url_len;

	url_len = report->stop - report->start + 1;
	url = malloc(url_len + 8);
	n = 1024;
	prog = malloc(1024);
	if (!url || !prog) {
	    fprintf(stderr, "out of memory!\n");
	    _exit(127);
	}

	memcpy(url, report->line + report->start, url_len);
	url[url_len] = '\0';

	for (i = 0 ; *url_cmd != '\0' ; url_cmd++) {
	    if (i + url_len + 8 > n) {
		n += i + url_len + 8;
		prog = realloc(prog, n);
		if (!prog) {
		    fprintf(stderr, "out of memory!\n");
		    _exit(127);
		}
	    }

	    if (*url_cmd == '%' && *++url_cmd == 's') {
		memcpy(prog + i, url, url_len);
		i += url_len;
	    } else {
		prog[i++] = *url_cmd;
	    }
	}
	prog[i] = '\0';

	execl(BIN_SH, "sh", "-c", prog, (char *)0);
	perror("knews: execl " BIN_SH);
	_exit(127);
    }

    set_message("Executing urlCommand.", False);
}

