/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <X11/Shell.h>
#include <sys/wait.h>
#include "cache.h"
#include "child.h"
#include "codes.h"
#include "connect.h"
#include "file.h"
#include "save.h"
#include "server.h"
#include "tag.h"
#include "util.h"
#include "widgets.h"
#include "xutil.h"
#include "../Widgets/ArtText.h"
#include "../Widgets/FileSel.h"
#include "../Widgets/Knapp.h"
#include "../Widgets/Layout.h"
#include "../Widgets/Message.h"
#include "../Widgets/TextField.h"
#include "../Widgets/Toggle.h"
#include "../Widgets/Util.h"

typedef enum {
    SaveScopeArtwin,
    SaveScopeArticle,
    SaveScopeSubject,
    SaveScopeThread,
    SaveScopeSubthread,
    SaveScopeTagged
} SaveScope;

static struct {
    Widget	shell;
    /***/
    Widget	file_message;
    Widget	file_field;
    Widget	shell_message;
    Widget	shell_field;
    Widget	file_chooser;
    Widget	choose_knapp;
    Widget	ok_knapp;
    Widget	close_knapp;
    /***/
    Widget	bogus_from_toggle;
    Widget	bogus_subj_toggle;
    Widget	head_toggle;
    Widget	body_toggle;
    Widget	empty_toggle;
    /***/
    Widget	artwin_toggle;
    Widget	article_toggle;
    Widget	subject_toggle;
    Widget	thread_toggle;
    Widget	subthread_toggle;
    Widget	tagged_toggle;
} save_widgets;

static SaveScope	save_scope = SaveScopeArticle;
static int		is_save = True;

/*************************************************************************/

static void set_save_label(int to_save)
{
    is_save = !to_save;
    KnappSetLabelNo(save_widgets.ok_knapp, !is_save, True);
}

static void get_email(ARTICLE *art, char *buf, int max_len)
{
    char	*c = art->from;
    char	*p1, *p2;
    int		len;

    if ((p1 = strchr(c, '<')) && (p2 = strchr(p1, '>'))) {
	len = p2 - p1 - 1;
	if (len > max_len)
	    len = max_len;
	memcpy(buf, p1 + 1, len);
	buf[len] = '\0';
	return;
    }

    p1 = strchr(c, '(');
    if (p1) {
	do {
	    p1--;
	} while (p1 > c && (*p1 == ' ' || *p1 == '\t'));

	if (p1 > c) {
	    len = p1 - c + 1;
	    if (len > max_len)
		len = max_len;
	    memcpy(buf, c, len);
	    buf[len] = '\0';
	    return;
	}
    }

    len = strlen(c);
    if (len > max_len)
	len = max_len;
    memcpy(buf, c, len);
    buf[len] = '\0';
}

static ARTICLE **get_art_list(SaveScope scope, long *n_arts)
{
    ARTICLE	*art = NULL, *top = NULL;
    ARTICLE	**result = NULL;
    SUBJECT	*subj = NULL;
    int		n, n_alloc;

    switch (scope) {
    case SaveScopeArtwin:
	result = (ARTICLE **)XtMalloc(sizeof(ARTICLE *));
	*result = NULL;
	*n_arts = 0;
	return result;
    case SaveScopeArticle:
	if (global.curr_art) {
	    result = (ARTICLE **)XtMalloc(sizeof result[0]);
	    result[0] = global.curr_art;
	    *n_arts = 1;
	}
	return result;
    case SaveScopeTagged:
	{
	    ARTICLE **tagged = get_tagged_articles();

	    n_alloc = no_tagged_articles();
	    if (n_alloc <= 0)
		return NULL;
	    result = (ARTICLE **)XtMalloc(n_alloc * sizeof(ARTICLE *));
	    for (n = 0 ; n < n_alloc ; n++)
		result[n] = tagged[n];
	    *n_arts = n;
	    return result;
	}
    case SaveScopeSubject:
	subj = global.curr_subj;
	if (!subj)
	    return NULL;
	art = subj->thread;
	break;
    case SaveScopeThread:
	if (!global.curr_subj)
	    return NULL;
	art = global.curr_subj->thread;
	break;
    case SaveScopeSubthread:
	if (!global.curr_art)
	    return NULL;
	top = art = global.curr_art;
	break;
    }

    while (art && (!art->from || (subj && art->subject != subj)))
	if (top)
	    art = next_in_subthread_preorder(art, top);
	else
	    art = next_in_thread_preorder(art);

    if (!art)
	return NULL;

    n = 0;
    n_alloc = 16;
    result = (ARTICLE **)XtMalloc(n_alloc * sizeof(ARTICLE *));

    while (art) {
	if (n + 8 > n_alloc) {
	    n_alloc += 32;
	    result =
		(ARTICLE **)XtRealloc((char *)result,
				      n_alloc * sizeof(ARTICLE *));
	}

	result[n++] = art;

	do {
	    if (top)
		art = next_in_subthread_preorder(art, top);
	    else
		art = next_in_thread_preorder(art);
	} while (art && (!art->from || (subj && art->subject != subj)));
    }
    *n_arts = n;

    return result;
}

static char *save_article(FILE *fp, SERVER *server, ARTICLE *art, int what)
{
    char	*buffer;

    if (what & SAVE_BOGUS_FROM) {
	char	email[128];
	time_t	t = art->date;
	char	*c = ctime(&t);

	get_email(art, email, sizeof email - 1);
	fprintf(fp, "From %s %s", email, c);
    }

    if (what & SAVE_BOGUS_SUBJ)
	fprintf(fp, "Subject: %s\n", art->subject->subject);

    buffer = server_read(server);
    while (buffer && buffer[0] != '\0' && !IS_DOT(buffer)) {
	if (what & SAVE_HEAD)
	    fprintf(fp, "%s\n", buffer);
	buffer = server_read(server);
    }

    if (!buffer || IS_DOT(buffer))
	return buffer;

    if ((what & (SAVE_HEAD | SAVE_BODY)) == (SAVE_HEAD | SAVE_BODY))
	fprintf(fp, "\n");

    buffer = server_read(server);
    while (buffer && !IS_DOT(buffer)) {
	if (what & SAVE_BODY)
	    fprintf(fp, "%s\n", buffer);
	buffer = server_read(server);
    }

    if (what & SAVE_EMPTY)
	fprintf(fp, "\n");

    return buffer;
}

int save_to_file(FILE *fp, char *message, ARTICLE **arts, long n,
		 int what, long *result)
{
    char	*p = message + strlen(message);
    long	i;

    result[1] = 0;

    for (i = 0 ; i < n ; i++) {
	char	*buffer;
	SERVER	*server;

	server = cache_get_server(arts[i]->no, False);
	if (!server) {
	    char	command[32];

	    server = main_server;
	    sprintf(command, "ARTICLE %ld\r\n", arts[i]->no);
	    buffer = server_comm(server, command, True);
	    if (!buffer)
		return -1;
	    if (atoi(buffer) != NNTP_OK_ARTICLE) {
		result[1]++;
		continue;
	    }
	}

	buffer = save_article(fp, server, arts[i], what);
	if (server != main_server)
	    server_free(server);
	else if (!buffer)
	    return -1;

	sprintf(p, "%ld", i + 1);
	set_message(message, False);
    }

    if (fflush(fp) == EOF)
	result[0] = -1;
    else
	result[0] = n - result[1];

    return 0;
}

static int get_what(void)
{
    Boolean	bogus_from = ToggleGet(save_widgets.bogus_from_toggle);
    Boolean	bogus_subj = ToggleGet(save_widgets.bogus_subj_toggle);
    Boolean	head       = ToggleGet(save_widgets.head_toggle);
    Boolean	body       = ToggleGet(save_widgets.body_toggle);
    Boolean	empty      = ToggleGet(save_widgets.empty_toggle);
    int		what = 0;

    if (bogus_from)
	what |= SAVE_BOGUS_FROM;
    if (bogus_subj)
	what |= SAVE_BOGUS_SUBJ;
    if (head)
	what |= SAVE_HEAD;
    if (body)
	what |= SAVE_BODY;
    if (empty)
	what |= SAVE_EMPTY;

    return what;
}

static void do_save(char *file_name, SaveScope scope, int what)
{
    FILE	*file;
    ARTICLE	**art_list = NULL;
    long	result[2], n;
    int		status, fclose_status;
    char	message[128];

    if (global.busy)
	return;

    if (global.mode != NewsModeGroup &&	global.mode != NewsModeThread) {
	set_message("Not in a newsgroup!", True);
	return;
    } else if (!file_name || file_name[0] == '\0') {
	set_message("No file specified!", True);
	return;
    } else if (!(what & (SAVE_HEAD | SAVE_BODY)) ||
	       !(art_list = get_art_list(scope, &n))) {
	set_message("Nothing to save!", True);
	return;
    }

    file = fopen_expand(file_name, "a", True);
    if (!file) {
	set_message("Failed to open file!", True);
	XtFree((char *)art_list);
	return;
    }

    sprintf(message, "Saving...   ");

    if (scope == SaveScopeArtwin) {
	status = ArtTextDumpToFile(main_widgets.text, file);
	result[0] = 1;
	result[1] = 0;
    } else {
	set_busy(True, True);
	status = save_to_file(file, message, art_list, n, what, result);
	unset_busy();
    }

    if ((fclose_status = fclose(file)) < 0)
	perror("knews: fclose");
    XtFree((char *)art_list);

    if (status < 0) {
	set_busy(True, True);
	reconnect_server(True);
	unset_busy();
	return;
    } else if (result[0] < 0 || fclose_status < 0) {
	set_message("File error!", True);
	return;
    } else if (result[1] > 0) {
	sprintf(message, "%ld articles saved, failed to get %ld articles.",
		result[0], result[1]);
	set_message(message, True);
    } else {
	sprintf(message, "%ld articles saved.", result[0]);
	set_message(message, False);
    }

    if (scope == SaveScopeTagged)
	clear_tagged_articles();
}

void pipe_context_callback(void *data, int status, char *stderr_buf)
{
    char	message[128];
    char	*cmd = data;
    int		ok = False;

    if (strlen(cmd) > 100)
	cmd[100] = '\0';

    if (WIFEXITED(status))
	switch (WEXITSTATUS(status)) {
	case 0:
	    sprintf(message, "'%s' exited ok.", cmd);
	    ok = True;
	    break;
	case 127:
	    sprintf(message, "Failed to start '%s'!", cmd);
	    break;
	default:
	    sprintf(message, "'%s' exited abnormally!", cmd);
	    break;
	}
    else if (WIFSIGNALED(status))
	sprintf(message, "'%s' caught %s!", cmd,
		signal_string(WTERMSIG(status)));
    else
	sprintf(message, "Unknown problem with '%s'!", cmd);

    if (!ok)
	stderr_popup(stderr_buf, -1);

    set_message(message, !ok);

    XtFree(cmd);
}

static void do_pipe(char *command, SaveScope scope, int what)
{
    FILE	*file = NULL;
    char	*file_name;
    ARTICLE	**art_list = NULL;
    long	result[2], n;
    int		status, fflush_status;
    char	message[128];
    pid_t	pid;
    char	*temp;

    if (global.busy)
	return;

    if (global.mode != NewsModeGroup &&	global.mode != NewsModeThread) {
	set_message("Not in a newsgroup!", True);
	return;
    } else if (!command || command[0] == '\0') {
	set_message("No command specified!", True);
	return;
    } else if (!(what & (SAVE_HEAD | SAVE_BODY)) ||
	       !(art_list = get_art_list(scope, &n))) {
	set_message("Nothing to pipe!", True);
	return;
    }

    file = create_temp_file(&file_name);
    if (!file) {
	set_message("Failed to create temporary file!", True);
	XtFree((char *)art_list);
	return;
    }
    unlink(file_name);

    sprintf(message, "Saving to temp file...   ");

    if (scope == SaveScopeArtwin) {
	status = ArtTextDumpToFile(main_widgets.text, file);
	result[0] = 1;
	result[1] = 0;
    } else {
	set_busy(True, True);
	status = save_to_file(file, message, art_list, n, what, result);
	unset_busy();
    }

    if ((fflush_status = fflush(file)) < 0)
	perror("knews: fflush");
    XtFree((char *)art_list);

    if (status < 0) {
	fclose(file);
	set_busy(True, True);
	reconnect_server(True);
	unset_busy();
	return;
    } else if (result[0] < 0 || fflush_status < 0) {
	fclose(file);
	set_message("Error with temp file!", True);
	return;
    } else if (result[1] > 0) {
	sprintf(message,
		"%ld articles saved to temp file, failed to get %ld "
		"articles.  Pipe started.",
		result[0], result[1]);
	set_message(message, True);
    } else {
	sprintf(message,
		"%ld articles saved to temp file.  Pipe started.",
		result[0]);
	set_message(message, False);
    }

    if (scope == SaveScopeTagged)
	clear_tagged_articles();

    temp = XtNewString(command);
    pid = fork_nicely(temp, pipe_context_callback,
		      global.stderr_timeout >= 0);

    if (pid < 0) {
	set_message("Fork failed!", True);
	XtFree(temp);
	fclose(file);
	return;
    }

    if (pid == 0) { /* child */
	int	fd;

	fd = fileno(file);
	if (fd != STDIN_FILENO) {
	    fd = dup2(fd, STDIN_FILENO);
	    if (fd < 0) {
		perror("knews: dup2");
		_exit(127);
	    }
	}

	if (lseek(fd, SEEK_SET, 0) < 0) {
	    perror("knews: lseek");
	    _exit(127);
	}

	execl(BIN_SH, "sh", "-c", command, (char *)NULL);
	perror("knews: execl " BIN_SH);
	_exit(127);
    }

    set_message("Pipe started.", False);    
    fclose(file);
}

static void save_callback(Widget w,
			  XtPointer client_data,
			  XtPointer call_data)
{
    Arg		arg;
    char	*file_name = NULL;

    XtSetArg(arg, XtNbuffer, &file_name);
    XtGetValues(save_widgets.file_field, &arg, 1);

    do_save(file_name, save_scope, get_what());
}

static void pipe_callback(Widget w,
			  XtPointer client_data,
			  XtPointer call_data)
{
    Arg		arg;
    char	*command = NULL;

    XtSetArg(arg, XtNbuffer, &command);
    XtGetValues(save_widgets.shell_field, &arg, 1);

    do_pipe(command, save_scope, get_what());
}

static void unset_scope_toggle(SaveScope scope)
{
    Widget	w = NULL;

    switch (scope) {
    case SaveScopeArtwin:
	w = save_widgets.artwin_toggle;
	break;
    case SaveScopeArticle:
	w = save_widgets.article_toggle;
	break;
    case SaveScopeSubject:
	w = save_widgets.subject_toggle;
	break;
    case SaveScopeThread:
	w = save_widgets.thread_toggle;
	break;
    case SaveScopeSubthread:
	w = save_widgets.subthread_toggle;
	break;
    case SaveScopeTagged:
	w = save_widgets.tagged_toggle;
	break;
    }

    ToggleUnset(w);
}

static void set_toggles_sensitive(Boolean sens)
{
    XtSetSensitive(save_widgets.bogus_from_toggle, sens);
    XtSetSensitive(save_widgets.bogus_subj_toggle, sens);
    XtSetSensitive(save_widgets.head_toggle, sens);
    XtSetSensitive(save_widgets.body_toggle, sens);
    XtSetSensitive(save_widgets.empty_toggle, sens);
}

static void file_chooser_callback(Widget gw,
				  XtPointer client_data,
				  XtPointer call_data)
{
    char	*buffer = (char *)call_data;

    if (buffer) {
	Arg	arg;

	XtSetArg(arg, XtNbuffer, buffer);
	XtSetValues(save_widgets.file_field, &arg, 1);
	XtSetKeyboardFocus(save_widgets.shell,
			   save_widgets.file_field);
	set_save_label(True);
    }
}

static void choose_knapp_callback(Widget gw,
				  XtPointer client_data,
				  XtPointer call_data)
{
    if (!save_widgets.file_chooser) {
	Arg	args[4];

	XtSetArg(args[0], XtNcolormap, global.cmap);
	XtSetArg(args[1], XtNvisual, global.visual);
	XtSetArg(args[2], XtNdepth, global.depth);
	save_widgets.file_chooser =
	    XtCreatePopupShell("filechooser", fileSelWidgetClass,
			       main_widgets.shell, args, 3);
	XtAddCallback(save_widgets.file_chooser, XtNchooseCallback,
		      file_chooser_callback, NULL);
    }

    XtPopup(save_widgets.file_chooser, XtGrabExclusive);
}

static void close_knapp_callback(Widget gw,
				 XtPointer client_data,
				 XtPointer call_data)
{
    XtPopdown(save_widgets.shell);
}

static void toggle_callback(Widget gw,
			    XtPointer client_data,
			    XtPointer call_data)
{
    Boolean	*set = (Boolean *)call_data;

    *set = !*set;
}

static void scope_toggle_callback(Widget w,
				  XtPointer client_data,
				  XtPointer call_data)
{
    Boolean	*set = (Boolean *)call_data;

    if (!*set) {
	unset_scope_toggle(save_scope);
	save_scope = (SaveScope)client_data;
	*set = True;
	if (w == save_widgets.artwin_toggle)
	    set_toggles_sensitive(False);
	else
	    set_toggles_sensitive(True);
    }
}

static void knapp_callback(Widget w,
			   XtPointer client_data,
			   XtPointer call_data)
{
    if (is_save)
	save_callback(w, client_data, call_data);
    else
	pipe_callback(w, client_data, call_data);
}

static void focus_callback(Widget w,
			   XtPointer client_data,
			   XtPointer call_data)
{
    if (w == save_widgets.shell_field)
	set_save_label(True);
    else if (w == save_widgets.file_field)
	set_save_label(False);
}

static void tab_callback(Widget w,
			 XtPointer client_data,
			 XtPointer call_data)
{
    if (w == save_widgets.shell_field) {
	XtSetKeyboardFocus(save_widgets.shell, save_widgets.file_field);
	set_save_label(True);
    } else if (w == save_widgets.file_field) {
	XtSetKeyboardFocus(save_widgets.shell, save_widgets.shell_field);
	set_save_label(False);
    }
}

/*************************************************************************/

static void create_save_widgets(void)
{
    Arg		args[8];
    Widget	layout;

    XtSetArg(args[0], XtNallowShellResize, True);
    XtSetArg(args[1], XtNinput, True);
    XtSetArg(args[2], XtNtitle, "knews: save/pipe");
    XtSetArg(args[3], XtNiconName, "save/pipe");
    XtSetArg(args[4], XtNcolormap, global.cmap);
    XtSetArg(args[5], XtNvisual, global.visual);
    XtSetArg(args[6], XtNdepth, global.depth);
    save_widgets.shell =
	XtCreatePopupShell("saveshell", topLevelShellWidgetClass,
			   main_widgets.shell, args, 7);

    save_widgets.file_chooser = NULL;

    layout =
	XtVaCreateManagedWidget("savelayout", layoutWidgetClass,
				save_widgets.shell,
				XtVaTypedArg, XtNlayout, XtRString,
#include "layouts/save.l"
				(int)sizeof(String), NULL);

    XtSetArg(args[0], XtNcenter, True);
    XtCreateManagedWidget("message", messageWidgetClass,
			  layout, args, 1);

    XtSetArg(args[0], XtNcenter, False);
    save_widgets.shell_message =
	XtCreateManagedWidget("shellmessage", messageWidgetClass,
			      layout, args, 1);

    XtSetArg(args[0], XtNfocusRoot, save_widgets.shell);
    save_widgets.shell_field =
	XtCreateManagedWidget("shellfield", textFieldWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.shell_field, XtNcallback,
		  pipe_callback, False);
    XtAddCallback(save_widgets.shell_field, XtNfocusCallback,
		  focus_callback, NULL);
    XtAddCallback(save_widgets.shell_field, XtNtabCallback,
		  tab_callback, NULL);

    XtSetArg(args[0], XtNcenter, False);
    save_widgets.file_message =
	XtCreateManagedWidget("filemessage", messageWidgetClass,
			      layout, args, 1);

    XtSetArg(args[0], XtNfocusRoot, save_widgets.shell);
    save_widgets.file_field =
	XtCreateManagedWidget("filefield", textFieldWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.file_field, XtNcallback,
		  save_callback, NULL);
    XtAddCallback(save_widgets.file_field, XtNfocusCallback,
		  focus_callback, NULL);
    XtAddCallback(save_widgets.file_field, XtNtabCallback,
		  tab_callback, NULL);

    XtSetArg(args[0], XtNresizable, False);
    save_widgets.ok_knapp =
	XtCreateManagedWidget("ok", knappWidgetClass, layout, args, 1);
    XtAddCallback(save_widgets.ok_knapp, XtNcallback,
		  knapp_callback, NULL);

    save_widgets.close_knapp =
	XtCreateManagedWidget("close", knappWidgetClass, layout, NULL, 0);
    XtAddCallback(save_widgets.close_knapp, XtNcallback,
		  close_knapp_callback, NULL);

    save_widgets.choose_knapp =
	XtCreateManagedWidget("choose", knappWidgetClass, layout, NULL, 0);
    XtAddCallback(save_widgets.choose_knapp, XtNcallback,
		  choose_knapp_callback, NULL);

    /***/

    XtSetArg(args[0], XtNset, True);
    save_widgets.bogus_from_toggle =
	XtCreateManagedWidget("bogusfrom", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.bogus_from_toggle, XtNcallback,
		  toggle_callback, NULL);

    XtSetArg(args[0], XtNset, False);
    save_widgets.bogus_subj_toggle =
	XtCreateManagedWidget("bogussubj", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.bogus_subj_toggle, XtNcallback,
		  toggle_callback, NULL);

    XtSetArg(args[0], XtNset, True);
    save_widgets.head_toggle =
	XtCreateManagedWidget("header", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.head_toggle, XtNcallback,
		  toggle_callback, NULL);


    XtSetArg(args[0], XtNset, True);
    save_widgets.body_toggle =
	XtCreateManagedWidget("body", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.body_toggle, XtNcallback,
		  toggle_callback, NULL);

    XtSetArg(args[0], XtNset, True);
    save_widgets.empty_toggle =
	XtCreateManagedWidget("empty", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.empty_toggle, XtNcallback,
		  toggle_callback, NULL);

    /***/

    XtSetArg(args[0], XtNset, False);
    save_widgets.artwin_toggle =
	XtCreateManagedWidget("artwin", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.artwin_toggle, XtNcallback,
		  scope_toggle_callback, (XtPointer)SaveScopeArtwin);

    XtSetArg(args[0], XtNset, True);
    save_widgets.article_toggle =
	XtCreateManagedWidget("article", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.article_toggle, XtNcallback,
		  scope_toggle_callback, (XtPointer)SaveScopeArticle);

    XtSetArg(args[0], XtNset, False);
    save_widgets.subject_toggle =
	XtCreateManagedWidget("subject", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.subject_toggle, XtNcallback,
		  scope_toggle_callback, (XtPointer)SaveScopeSubject);

    XtSetArg(args[0], XtNset, False);
    save_widgets.thread_toggle =
	XtCreateManagedWidget("thread", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.thread_toggle, XtNcallback,
		  scope_toggle_callback, (XtPointer)SaveScopeThread);

    XtSetArg(args[0], XtNset, False);
    save_widgets.subthread_toggle =
	XtCreateManagedWidget("subthread", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.subthread_toggle, XtNcallback,
		  scope_toggle_callback, (XtPointer)SaveScopeSubthread);

    XtSetArg(args[0], XtNset, False);
    save_widgets.tagged_toggle =
	XtCreateManagedWidget("tagged", toggleWidgetClass,
			      layout, args, 1);
    XtAddCallback(save_widgets.tagged_toggle, XtNcallback,
		  scope_toggle_callback, (XtPointer)SaveScopeTagged);

    XtSetKeyboardFocus(save_widgets.shell, save_widgets.file_field);
    XtRealizeWidget(save_widgets.shell);
    XtInstallAllAccelerators(save_widgets.shell, save_widgets.shell);

    add_WM_DELETE_WINDOW_callback(save_widgets.shell,
				  close_knapp_callback, NULL);

    if (global.busy)
	set_busy_save(True);
}

void popup_save(void)
{
    if (!save_widgets.shell)
	create_save_widgets();

    XtPopup(save_widgets.shell, XtGrabNone);

    if (global.busy)
	set_busy_save(True);
}

void popdown_save(void)
{
    if (save_widgets.shell)
	XtPopdown(save_widgets.shell);
}

static void do_action(Boolean save, String *params, Cardinal n)
{
    int		what = 0;
    char	*c;
    SaveScope	scope = SaveScopeArticle;

    if (n == 0) {
	if (save)
	    set_message("No filename specified in save/pipe action!",
			True);
	else
	    set_message("No shell command specified in save/pipe action!",
			True);
	return;
    } else if (n == 1) {
	set_message("To few arguments to save/pipe action!", True);
	return;
    }

    c = params[1];
    while (*c != '\0') {
	switch (*c++) {
	case 'f':
	case 'F':
	    what |= SAVE_BOGUS_FROM;
	    break;
	case 'S':
	case 's':
	    what |= SAVE_BOGUS_SUBJ;
	    break;
	case 'H':
	case 'h':
	    what |= SAVE_HEAD;
	    break;
	case 'B':
	case 'b':
	    what |= SAVE_BODY;
	    break;
	case 'E':
	case 'e':
	    what |= SAVE_EMPTY;
	    break;
	}
    }

    if (!(what & (SAVE_BODY|SAVE_HEAD)) ||
	!params[0] || params[0][0] == '\0') {
	if (save)
	    set_message("Nothing to save!", True);
	else
	    set_message("Nothing to pipe!", True);
	return;
    }

    if (n >= 3) {
	switch (params[2][0]) {
	case 'W':
	case 'w':
	    scope = SaveScopeArtwin;
	    break;
	case 'A':
	case 'a':
	    scope = SaveScopeArticle;
	    break;
	case 'S':
	case 's':
	    if ((params[2][1] == 'U' || params[2][1] == 'u') &&
		(params[2][2] == 'B' || params[2][2] == 'b')) {
		switch (params[2][3]) {
		case 'J':
		case 'j':
		    scope = SaveScopeSubject;
		    break;
		case 'T':
		case 't':
		    scope = SaveScopeSubthread;
		    break;
		}
	    }
	    break;
	case 'T':
	case 't':
	    switch (params[2][1]) {
	    case 'A':
	    case 'a':
		scope = SaveScopeTagged;
		break;
	    case 'H':
	    case 'h':
		scope = SaveScopeThread;
		break;
	    }
	    break;
	}
    }

    if (save)
	do_save(params[0], scope, what);
    else
	do_pipe(params[0], scope, what);
}

void action_save(Widget w, XEvent *string,
		 String *params, Cardinal *no_params)
{
    do_action(True, params, *no_params);
}

void action_pipe(Widget w, XEvent *string,
		 String *params, Cardinal *no_params)
{
    do_action(False, params, *no_params);
}

/*************************************************************************/

void set_busy_save(int busy)
{
    if (!save_widgets.shell)
	return;

    XDefineCursor(XtDisplay(save_widgets.shell), XtWindow(save_widgets.shell),
		  busy ? global.busy_cursor : global.cursor);
    KnappSetActive(save_widgets.ok_knapp, !busy);
}
