/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <X11/Shell.h>
#include "cache.h"
#include "child.h"
#include "codes.h"
#include "connect.h"
#include "file.h"
#include "parse.h"
#include "post.h"
#include "psetup.h"
#include "ppopup.h"
#include "resource.h"
#include "server.h"
#include "util.h"
#include "widgets.h"
#include "xutil.h"
#include "../Widgets/ArtText.h"
#include "../Widgets/Dialogue.h"
#include "../Widgets/Menu.h"
#include "../Widgets/Notice.h"
#include "../Widgets/SeparatorG.h"
#include "../Widgets/StringG.h"
#include "../Widgets/ToggleG.h"
#include "../Widgets/PopdownSh.h"

static Widget		include_toggle;

void free_post_context(PostContext *context)
{
    if (context->file_name) {
	unlink(context->file_name);
	XtFree(context->file_name);
    }
    if (context->fp)
	fclose(context->fp);
    XtFree(context->newsgroups);
    XtFree(context->reply_to);
    context->file_name = NULL;
    context->newsgroups = NULL;
    context->reply_to = NULL;
    XtFree((char *)context);
}

static int generic_post_check(int needs_art, FILE **fp, char **file_name)
{
    if (global.busy)
	XBell(XtDisplay(main_widgets.shell), 0);
    else if (!global.domain_name)
	set_message("Domain name not available, posting will not be allowed!",
		    True);
    else if (!global.user_id)
	set_message("User id not available, posting will not be allowed!",
		    True);
    else if (needs_art &&
	     global.mode != NewsModeGroup &&
	     global.mode != NewsModeThread)
	set_message("Not in a newsgroup!", True);
    else if (needs_art && !global.curr_art)
	set_message("No selected article!", True);
    else if (needs_art && !global.curr_art->from)
	set_message("That's a fake article!", True);
    else if (fp && !(*fp = create_temp_file(file_name)))
	set_message("Failed to create temporary file!", True);
    else
	return True;

    return False;
}

static int insert_extra_headers(FILE *fp)
{
    char	*reply_to      = res_reply_to();
    char	*organization  = res_organization();
    char	*extra_headers = res_extra_headers();
    char	*distr         = res_distribution();
    int		result = 0;

    fprintf(fp, "%s\n", "X-Newsreader: knews " KNEWS_VERSION);
    result++;

    if (distr) {
	fprintf(fp, "Distribution: %s\n", distr);
	result++;
    }

    if (reply_to) {
	fprintf(fp, "Reply-To: %s\n", reply_to);
	result++;
    }

    if (organization) {
	fprintf(fp, "Organization: %s\n", organization);
	result++;
    }

    if (extra_headers && extra_headers[0] != '\0') {
	char	*c;

	fprintf(fp, "%s", extra_headers);
	for (c = extra_headers ; c ; c = strchr(c + 1, '\n'))
	    result++;
	if (extra_headers[strlen(extra_headers) - 1] != '\n') {
	    fputc('\n', fp);
	    result++;
	}
    }

    return result;
}

static void append_signature(PostContext *context)
{
    char	*sig_file_name = res_signature_file();
    FILE	*sig;
    int		c;

    if (!sig_file_name)
	return;
    sig = fopen_expand(sig_file_name, "r", True);
    if (!sig)
	return;

    fprintf(context->fp, "\n-- \n");
    while ((c = getc(sig)) != EOF)
	putc(c, context->fp);
    fclose(sig);
}

#define MAX_REFS	8

/*
 *  Don't generate reference headers deeper than this, with the
 *  exception that the original parent is always included.  This
 *  is in violation of the RFC.  Set MAX_REFS to a big number if
 *  you want maximum compliancy.
 *
 *  Note: knews generates references based on it's internal data
 *  structures, which means that a followup may actually get more
 *  complete references that the article we're following up to.  :-)
 */

static int print_refs(FILE *fp, ARTICLE *art, int *col, int depth)
{
    int	ret = 0;
    int	doit = True;

    if (A_PARENT(art)) {
	ret = print_refs(fp, A_PARENT(art), col, depth + 1);
	if (depth > MAX_REFS)
	    doit = False;
	else if (*col > 0 && *col + art->hash_len > 75) {
	    ret++;
	    fprintf(fp, "\n ");
	    *col = 0;
	}
    }

    if (doit)
	*col += fprintf(fp, " <%s>", art->msgid);

    return ret;
}

static int print_references_header(FILE *fp, ARTICLE *art)
{
    int	col, lines = 1;

    col = fprintf(fp, "References:");
    lines = print_refs(fp, art, &col, 0) + 1;
    fprintf(fp, "\n");

    return lines;
}

static char *skip_line_count(char *c)
{
    while (IS_SPACE(*c))
	c++;
    while (*c >= '0' && *c <= '9')
	c++;
    while (IS_SPACE(*c))
	c++;

    return c;
}

#define INITIAL_SEP_CHARS   " \t-"

static int extract_initials(ARTICLE *art, char *buffer, int maxlen, int cap)
{
    char	*c;
    int		pos = 0;

    maxlen -= 4;
    if (art && art->from && (c = art->tree_data.label)) {
	if (res_show_number_lines())
	    c = skip_line_count(c);

	while (*c != '\0') {
	    while (*c != '\0' && strchr(INITIAL_SEP_CHARS, *c))
		c++;
	    if (*c == '\0' || pos > maxlen)
		break;
	    buffer[pos++] =
		(cap && islower((unsigned char)*c)) ?
		toupper((unsigned char)*c) : *c;
	    while (*c != '\0' && !strchr(INITIAL_SEP_CHARS, *c))
		c++;
	}
    }

    buffer[pos] = '\0';

    return pos;
}

static void print_attribution(FILE *fp, char *attr, ARTICLE *art)
{
    struct tm	*tm = NULL;
    char	*c;

    if (!fp || !attr || attr[0] == '\0' || !art)
	return;

#define FILL_TM(tm, time) ((tm) || ((tm) = gmtime(&(time))))
    while (*attr != '\0')
	if (*attr != '%')
	    fputc(*attr++, fp);
	else {
	    unsigned char	a = *++attr;

	    if (isupper(a))
		a = tolower(a);

	    switch (a) {
	    case '\0':
		continue;
	    case '%': /* literal % */
		fputc('%', fp);
		break;
	    case 'd': /* date: dd mmm */
		if (FILL_TM(tm, art->date) && (unsigned)tm->tm_mon < 12)
		    fprintf(fp, "%2.2d %3.3s", tm->tm_mday,
			    "JanFebMarAprMayJunJulAugSepOctNovDec" +
			    3 * tm->tm_mon);
		break;
	    case 'f': /* from line */
		fprintf(fp, "%s", art->from);
		break;
	    case 'i': /* initials */
		if (art) {
		    char	buffer[32];

		    extract_initials(art, buffer, sizeof buffer, *attr == 'I');
		    fprintf(fp, "%s", buffer);
		}
		break;
	    case 'm': /* message-id */
		fprintf(fp, "<%s>", art->msgid);
		break;
	    case 'n': /* newsgroup */
		if ((global.mode == NewsModeGroup ||
		     global.mode == NewsModeThread) &&
		    global.curr_group)
		    fprintf(fp, "%s", global.curr_group->name);
		break;
	    case 'r': /* real name */
		c = art->tree_data.label;
		if (c && res_show_number_lines())
		    c = skip_line_count(c);
		if (c)
		    fprintf(fp, "%s", c);
		break;
	    case 's': /* subject */
		fprintf(fp, "%s%s", PARENT(art) ? "Re: " : "",
			art->subject->subject);
		break;
	    case 't': /* time */
		if (FILL_TM(tm, art->date))
		    fprintf(fp, "%02d:%02d:%02d",
			    tm->tm_hour, tm->tm_min, tm->tm_sec);
		break;
	    case 'w': /* week day */
		if (FILL_TM(tm, art->date) && (unsigned)tm->tm_wday < 7)
		    fprintf(fp, "%3.3s", month_names + 3 * tm->tm_mon);
		break;
	    case 'y': /* year */
		if (FILL_TM(tm, art->date))
		    fprintf(fp, "%4d", 1900 + tm->tm_year);
		break;
	    default: /* bogus */
		putc('%', fp);
		putc(a, fp);
		break;
	    }
	    attr++;
	}
    fputc('\n', fp);
}
#undef FILL_TM

static void expand_quote_string(ARTICLE *art, char *quote_string,
				char *buffer, int len)
{
    int		pos = 0;

    *buffer = '\0';
    len -= 4;

    if (!quote_string)
	return;

    while (*quote_string != '\0' && pos < len)
	if (*quote_string != '%')
	    buffer[pos++] = *quote_string++;
	else {
	    int		cap = False;

	    switch (*++quote_string) {
	    case 'I':
		cap = True;
		/* fall through */
	    case 'i':
		pos += extract_initials(art, buffer + pos, len, cap);
		break;
	    case '%':
		buffer[pos++] = '%';
		break;
	    default:
		continue;
	    }
	    quote_string++;
	}

    buffer[pos] = '\0';
}

static int get_headers(long art_no,
		       char **newsgroups,
		       char **followup_to,
		       char **reply_to)
{
    SERVER	*server;
    char	*buffer;

    *newsgroups = NULL;
    *followup_to = NULL;
    *reply_to = NULL;

    server = cache_get_server(art_no, False);
    if (!server) {
	char	command[512];

	server = main_server;
	sprintf(command, "HEAD %ld\r\n", art_no);
	buffer = server_comm(server, command, True);
	if (!buffer)
	    return -1;
	if (atoi(buffer) != NNTP_OK_HEAD)
	    return 0;
    }

    buffer = server_read(server);
    while (buffer && buffer[0] != '\0' && !IS_DOT(buffer)) {
	if (!*newsgroups &&
	    case_lstrncmp(buffer, "newsgroups:", 11) == 0) {
	    buffer += 11;
	    while (IS_SPACE(*buffer))
		buffer++;
	    *newsgroups = XtNewString(buffer);
	} else if (!*followup_to &&
		   case_lstrncmp(buffer, "followup-to:", 12) == 0) {
	    buffer += 12;
	    while (IS_SPACE(*buffer))
		buffer++;
	    *followup_to = XtNewString(buffer);
	} else if (!*reply_to &&
		   case_lstrncmp(buffer, "reply-to:", 9) == 0) {
	    buffer += 9;
	    while (IS_SPACE(*buffer))
		buffer++;
	    *reply_to = XtNewString(buffer);
	}

	buffer = server_read(server);
    }

    if (server != main_server)
	server_free(server);
    else {
	while (buffer && !IS_DOT(buffer))
	    buffer = server_read(server);

	if (!buffer) {
	    XtFree(*newsgroups);
	    *newsgroups = NULL;
	    XtFree(*followup_to);
	    *followup_to = NULL;
	    XtFree(*reply_to);
	    *reply_to = NULL;
	    return -1;
	}
    }

    return 0;
}

static int include_quotation(ARTICLE *art, FILE *fp,
			     char *quote_string, int head)
{
    SERVER	*server;
    char	temp[32];
    char	*buffer;


    server = cache_get_server(art->no, False);
    if (!server) {
	server = main_server;
	sprintf(temp, "%s %ld\r\n", "ARTICLE", art->no);
	buffer = server_comm(main_server, temp, True);
	if (!buffer)
	    return -1;
	if (atoi(buffer) != NNTP_OK_ARTICLE)
	    return 0; /* fail silently */
    }

    expand_quote_string(art, quote_string, temp, sizeof temp);

    buffer = server_read(server);
    if (!head) {
	while (buffer && !IS_DOT(buffer) && buffer[0] != '\0')
	    buffer = server_read(server);
	if (buffer && !IS_DOT(buffer))
	    buffer = server_read(server);
    }

    while (buffer && !IS_DOT(buffer)) {
	fprintf(fp, "%s%s\n", temp, buffer);
	buffer = server_read(server);
    }

    if (server == main_server)
	return buffer ? 0 : -1;

    server_free(server);
    return 0;
}

static void setup_file_and_fork_editor(PostContext *context)
{
    FILE	*fp = context->fp;
    int		line = 1;
    char	*full_name = res_full_name();

    line += insert_extra_headers(context->fp);

    if (context->art) { /* a followup */
	line += print_references_header(fp, context->art);

	fprintf(fp, "From: %s@%s (%s)\n",
		global.mail_name, global.domain_name,
		full_name ? full_name : "");
	line++;

	fprintf(fp, "Subject: Re: %s\n", context->art->subject->subject);
	line++;

	if (context->flags & POST) {
	    fprintf(fp, "Newsgroups: %s\n",
		    context->newsgroups ? context->newsgroups : "");
	    line++;
	} else if (context->newsgroups) {
	    fprintf(fp, "X-Original-Newsgroups: %s\n",
		    context->newsgroups ? context->newsgroups : "");
	    line++;
	}

	if (context->flags & MAIL) {
	    if (context->art) {
		fprintf(fp, "In-Reply-To: <%s>\n", context->art->msgid);
		line++;
	    }

	    fprintf(fp, "To: %s\n",
		    context->reply_to ?
		    context->reply_to : context->art->from);
	    line++;		    
	}

	line++;
    } else {
	fprintf(fp, "From: %s@%s (%s)\n",
		global.mail_name, global.domain_name,
		full_name ? full_name : "");
	line++;

	fprintf(fp, "Subject: \n");

	fprintf(fp, "Newsgroups: ");
	if (global.mode == NewsModeGroup || global.mode == NewsModeThread)
	    fprintf(fp, "%s\n", global.curr_group->name);
	else
	    fputc('\n', fp);
    }

    fputc('\n', fp);
    context->line = line;

    if (ToggleGadgetGet(include_toggle) && context->art) {
	char	*attribution  = res_attribution();
	char	*quote_string = res_quote_string();

	print_attribution(context->fp, attribution, context->art);
	set_busy(True, True);
	if (include_quotation(context->art, context->fp,
			      quote_string, False) < 0) {
	    free_post_context(context);
	    reconnect_server(True);
	    unset_busy();
	    return;
	}
	unset_busy();
    }

    append_signature(context);

    fclose(fp);
    context->fp = NULL;

    fork_editor(context);
}

static void forward_dialogue_callback(Widget w,
				      XtPointer client_data,
				      XtPointer call_data)
{
    PostContext		*context = (PostContext *)client_data;
    DialogueReport	*report = (DialogueReport *)call_data;
    char		*full_name = res_full_name();
    int			line = 0;

    if (!report)
	return;

    switch (report->reply) {
    case DialogueReplyClose:
    case DialogueReplyRight:
	XtPopdown(w);
	XtDestroyWidget(w);
	free_post_context(context);
	return;
    case DialogueReplyTab:
	return;
    case DialogueReplyMiddle:
	break;
    case DialogueReplyLeft:
    case DialogueReplyEnter:
	if (!report->buffer)
	    return;
	break;
    }

    XtPopdown(w);
    XtDestroyWidget(w);

    line += insert_extra_headers(context->fp);

    if (full_name)
	fprintf(context->fp, "From: %s@%s (%s)\n",
		global.mail_name, global.domain_name, full_name);
    else
	fprintf(context->fp, "From: %s@%s\n",
		global.mail_name, global.domain_name);
    line++;

    fprintf(context->fp, "Subject: %s%s (fwd)\n",
	    PARENT(context->art) ? "Re: " : "",
	    context->art->subject->subject);
    line++;

    fprintf(context->fp,
	    "To: %s\n\n----- Forwarded Message -----\n",
	    report->buffer);
    line++;

    set_busy(True, True);
    if (include_quotation(context->art, context->fp, NULL, True) < 0) {
	reconnect_server(True);
	unset_busy();
	free_post_context(context);
	return;
    }
    unset_busy();
    context->art = NULL;

    fclose(context->fp);
    context->fp = NULL;

    if (report->reply == DialogueReplyMiddle)
	fork_editor(context);
    else {
	char		message[STDERR_BUFFLEN];
	MailStatus	status;

	set_message("Mailing...", False);
	message[0] = '\0';
	status = mail_article(context, message);

	switch (status) {
	case MailStatusOk:
	    set_message("Mailing done.", False);
	    break;
	case MailStatusFailed:
	    set_message("Mail failed: " MAIL_COMMAND
			" didn't accept it!", True);
	    stderr_popup(message, 0);
	    break;
	case MailStatusStart:
	    set_message("Failed to start " MAIL_COMMAND, True);
	    stderr_popup(message, 0);
	    break;
	case MailStatusOpen:
	    set_message("Failed to open temp file!", True);
	    break;
	case MailStatusFork:
	    set_message("Fork failed!", True);
	    break;

	}

	free_post_context(context);
    }
}

static void followup_to_poster_callback(Widget w,
					XtPointer client_data,
					XtPointer call_data)
{
    PostContext	*context = (PostContext *)client_data;
    NoticeReply	reply = (NoticeReply)call_data;

    switch (reply) {
    case NoticeReplyMiddle:	/* Email */
	context->flags = MAIL | ORIG_MAIL;
	/* fall through */
    case NoticeReplyLeft:	/* Post anyway */
	setup_file_and_fork_editor(context);
	break;
    default:
	free_post_context(context);
	break;
    }

    XtPopdown(w);
    XtDestroyWidget(w);
}

static void post_menu_callback(Widget gw,
			       XtPointer client_data,
			       XtPointer call_data)
{
    PostContext	*context;
    int		flags = (long)client_data;
    FILE	*fp          = NULL;
    char	*file_name   = NULL;
    char	*newsgroups = NULL;
    char	*followup_to = NULL;
    char	*reply_to = NULL;

    if (!generic_post_check(True, &fp, &file_name))
	return;

    file_name = XtNewString(file_name);

    set_busy(True, True);
    if (get_headers(global.curr_art->no, &newsgroups,
		    &followup_to, &reply_to) < 0) {
	fclose(fp);
	unlink(file_name);
	XtFree(file_name);
	reconnect_server(True);
	unset_busy();
	return;
    }
    unset_busy();

    context = (PostContext *)XtMalloc(sizeof(PostContext));
    context->reply_to = reply_to;
    context->art = global.curr_art;
    context->flags = flags;
    context->line = 0;
    context->fp = fp;
    context->file_name = file_name;

    if (!followup_to)
	context->newsgroups = newsgroups;
    else {
	if ((flags & POST) && strcmp(followup_to, "poster") == 0) {
	    XtFree(followup_to);
	    popup_notice("notice",
			 "The original author has requested\n"
			 "that followups be directed to email.\n\n"
			 "Post, email or abort?", "Post", "Email", "Abort", 0,
			 followup_to_poster_callback, (XtPointer)context,
			 XtGrabExclusive);
	    context->newsgroups = newsgroups;
	    XtFree(followup_to);
	    return;
	}
	context->newsgroups = followup_to;
	XtFree(newsgroups);
    }

    if (!(context->flags & POST))
	context->flags |= ORIG_MAIL;

    setup_file_and_fork_editor(context);
}

static void post_new_callback(Widget w,
			      XtPointer client_data,
			      XtPointer call_data)
{
    PostContext *context;
    char        *file_name;
    FILE        *fp;

    if (!generic_post_check(False, &fp, &file_name))
	return;

    context = (PostContext *)XtMalloc(sizeof(PostContext));
    context->file_name = XtNewString(file_name);
    context->fp = fp;
    context->line = 0;
    context->newsgroups = NULL;
    context->reply_to = NULL;
    context->art = NULL;
    context->flags = POST;

    setup_file_and_fork_editor(context);
}

static void forward_menu_callback(Widget gw,
				  XtPointer client_data,
				  XtPointer call_data)
{
    PostContext	*context;
    FILE	*fp;
    char	*file_name;

    if (!generic_post_check(True, &fp, &file_name))
	return;

    context = (PostContext *)XtMalloc(sizeof(PostContext));
    context->file_name = XtNewString(file_name);
    context->fp = fp;
    context->line = 0;
    context->newsgroups = NULL;
    context->reply_to = NULL;
    context->art = global.curr_art;
    context->flags = MAIL|ORIG_MAIL;

    popup_dialogue("forward", "Forward article to:",
		   "Mail", "Edit", "Cancel",
		   forward_dialogue_callback, (XtPointer)context,
		   XtGrabExclusive);
}

static void cancel_post_callback(Widget w,
				 XtPointer client_data,
				 XtPointer call_data)
{
    PostContext	*context;
    FILE	*fp;
    char	*file_name;
    char	*full_name = res_full_name();
    int		line = 1;

    if (!generic_post_check(True, &fp, &file_name))
	return;

    context = (PostContext *)XtMalloc(sizeof(PostContext));
    context->art = global.curr_art;
    context->file_name = XtNewString(file_name);
    context->fp = NULL;
    context->newsgroups = NULL;
    context->reply_to = NULL;
    context->flags = POST;

    line += insert_extra_headers(fp);

    fprintf(fp, "From: %s@%s (%s)\n",
	    global.mail_name, global.domain_name,
	    full_name ? full_name : "");
    line++;

    fprintf(fp, "Subject: cmsg cancel <%s>\n", context->art->msgid);
    line++;

    fprintf(fp, "Control: cancel <%s>\n", context->art->msgid);
    line++;

    fprintf(fp, "Newsgroups: %s\n", global.curr_group->name);
    line++;

    fputc('\n', fp);
    line++;

    fprintf(fp, "Article canceled from within knews.\n");

    fclose(fp);

    context->line = line;
    fork_editor(context);
}

/*************************************************************************/

void create_post_menu(Widget main_shell)
{
    Widget	post_menu, menu, temp;
    Arg		args[4];

    XtSetArg(args[0], XtNcolormap, global.cmap);
    XtSetArg(args[1], XtNvisual, global.visual);
    XtSetArg(args[2], XtNdepth, global.depth);
    post_menu =
	XtCreatePopupShell("postshell", popdownShellWidgetClass,
			   main_shell, args, 3);
    menu = XtCreateManagedWidget("postmenu", menuWidgetClass,
				 post_menu, NULL, 0);

    temp = MenuCreateGadget("followup", stringGadgetClass, menu, NULL, 0);
    XtAddCallback(temp, XtNcallback, post_menu_callback, (XtPointer)POST);

    temp = MenuCreateGadget("mailreply", stringGadgetClass, menu, NULL, 0);
    XtAddCallback(temp, XtNcallback, post_menu_callback, (XtPointer)MAIL);

    temp = MenuCreateGadget("followupreply", stringGadgetClass, menu, NULL, 0);
    XtAddCallback(temp, XtNcallback,
		  post_menu_callback, (XtPointer)(POST | MAIL));

    temp = MenuCreateGadget("postnew", stringGadgetClass, menu, NULL, 0);
    XtAddCallback(temp, XtNcallback, post_new_callback, NULL);

    temp = MenuCreateGadget("cancel", stringGadgetClass, menu, NULL, 0);
    XtAddCallback(temp, XtNcallback, cancel_post_callback, NULL);

    temp = MenuCreateGadget("forward", stringGadgetClass, menu, NULL, 0);
    XtAddCallback(temp, XtNcallback, forward_menu_callback, NULL);

    MenuCreateGadget("separator", separatorGadgetClass, menu, NULL, 0);

    include_toggle =
	MenuCreateGadget("quotetoggle", toggleGadgetClass, menu, NULL, 0);
}

static void do_incl(String *params, Cardinal *no_params)
{
    if (*no_params > 0) {
	if (case_lstrcmp(params[0], "true") == 0 ||
	    case_lstrcmp(params[0], "yes"))
	    ToggleGadgetSet(include_toggle, True);
	else if (case_lstrcmp(params[0], "false") == 0 ||
		 case_lstrcmp(params[0], "no") == 0)
	    ToggleGadgetSet(include_toggle, False);
    }
}

void action_followup(Widget w, XEvent *event,
		     String *params, Cardinal *no_params)
{
    Boolean	old = ToggleGadgetGet(include_toggle);

    do_incl(params, no_params);
    post_menu_callback(w, (XtPointer)POST, NULL);
    ToggleGadgetSet(include_toggle, old);
}

void action_reply(Widget w, XEvent *event,
		  String *params, Cardinal *no_params)
{
    Boolean	old = ToggleGadgetGet(include_toggle);

    do_incl(params, no_params);
    post_menu_callback(w, (XtPointer)MAIL, NULL);
    ToggleGadgetSet(include_toggle, old);
}

void action_followup_and_reply(Widget w, XEvent *event,
			       String *params, Cardinal *no_params)
{
    Boolean	old = ToggleGadgetGet(include_toggle);

    do_incl(params, no_params);
    post_menu_callback(w, (XtPointer)(POST|MAIL), NULL);
    ToggleGadgetSet(include_toggle, old);
}

void action_post_new(Widget w, XEvent *event,
		     String *params, Cardinal *no_params)
{
    post_new_callback(w, NULL, NULL);
}

void action_forward_by_mail(Widget w, XEvent *event,
			    String *params, Cardinal *no_params)
{
    forward_menu_callback(w, NULL, NULL);
}
