/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <X11/Shell.h>
#include <sys/wait.h>
#include "child.h"
#include "pcheck.h"
#include "psetup.h"
#include "post.h"
#include "ppopup.h"
#include "resource.h"
#include "widgets.h"
#include "xutil.h"
#include "../Widgets/Dialogue.h"
#include "../Widgets/Notice.h"

static void post_notice_callback(Widget w,
				 XtPointer client_data,
				 XtPointer call_data)
{
    NoticeReply	reply = (NoticeReply)call_data;
    PostContext	*context = (PostContext *)client_data;
    PostStatus	post_status = PostStatusOk;
    MailStatus	mail_status = MailStatusOk;
    char	message[STDERR_BUFFLEN], *c;
    char	buffer[512];
    char	*agent = res_posting_agent();
    int		do_post, do_mail;

    if (global.busy)
	return;

    do_post = (context->flags & POST) && !(context->flags & POST_DONE);
    do_mail = (context->flags & MAIL) && !(context->flags & MAIL_DONE);

    switch (reply) {
    case NoticeReplyTimeout:
	return;
    case NoticeReplyLeft:
	if (!(context->flags & OK_TO_POST))
	    XBell(XtDisplay(w), 0);
	else if (global.mode == NewsModeDisconnected && do_post && !agent)
	    set_message("Not connected!", True);
	else if (!do_post && !do_mail)
	    XBell(XtDisplay(w), 0);
	else {
	    if (do_post) {
		message[0] = '\0';
		set_message("Posting article...", False);
		if (agent)
		    post_status = post_to_agent(context, agent, message);
		else
		    post_status = post_article(context, message);
	    }

	    if (post_status == PostStatusOk && do_mail) {
		message[0] = '\0';
		set_message("Mailing article...", False);
		mail_status = mail_article(context, message);
	    }

	    if (post_status == PostStatusOk && mail_status == MailStatusOk) {
		if (!do_post)
		    set_message("Mail was successful.", False);
		else if (do_mail)
		    set_message("Post and mail were successful.", False);
		else
		    set_message("Post was successful.", False);
		free_post_context(context);
		break;
	    }

	    if (post_status == PostStatusOk) {
		/* post successful, but not mail */
		NoticeSetLeftLabel(w, "Mail");

		switch (mail_status) {
		case MailStatusOk: /* notreached */
		    break;
		case MailStatusFailed:
		    if (do_post)
			sprintf(buffer,
				"Post was successful, but email failed.\n"
				MAIL_COMMAND " didn't accept it.");
		    else
			sprintf(buffer, "Email failed.\n"
				MAIL_COMMAND " didn't accept it.");
		    stderr_popup(message, 0);
		    break;
		case MailStatusStart:
		    if (do_post)
			sprintf(buffer,
				"Post was successful, but email failed.\n"
				"Failed to start " MAIL_COMMAND);
		    else
			sprintf(buffer, "Email failed.\n"
				"Failed to start " MAIL_COMMAND);
		    stderr_popup(message, 0);
		    break;
		case MailStatusOpen:
		    if (do_post)
			sprintf(buffer,
				"Post was successful, but email failed.\n"
				"Failed to open temp file.");
		    else
			sprintf(buffer, "Email failed.\n"
				"Failed to open temp tile.");
		    break;
		case MailStatusFork:
		    if (do_post)
			sprintf(buffer,
				"Post was successful, but email failed.\n"
				"Fork failed.");
		    else
			sprintf(buffer, "Fork failed.");
		    break;
		}
		NoticeSetMessage(w, buffer);
		c = strchr(buffer, '\n');
		if (c)
		    *c = '\0';
		set_message(buffer, True);
		return;
	    }

	    /* post unsuccessful, mail not tried */
	    switch (post_status) {
	    case PostStatusOk: /* notreached */
		break;
	    case PostStatusBroken:
		sprintf(buffer, "Posting (probably) not completed");
		break;
	    case PostStatusOpen:
		sprintf(buffer, "Failed to open temp file!");
		break;
	    case PostStatusFailed:
		if (agent) {
		    sprintf(buffer, "Posting failed, %s didn't accept it.",
			    agent);
		    stderr_popup(message, 0);
		} else {
		    if (strlen(message) > 128)
			message[128] = '\0';
		    sprintf(buffer, "Posting failed, message "
			    "from server is:\n\n%s", message);
		}
		break;
	    case PostStatusStart:
		if (agent) {
		    sprintf(buffer, "Posting failed: couldn't start %s.",
			    agent);
		    stderr_popup(message, 0);
		} else /* not reached */
		    sprintf(buffer, "Unknown error!");
		break;
	    }
	    NoticeSetMessage(w, buffer);
	    c = strchr(buffer, '\n');
	    if (c)
		*c = '\0';
	    set_message(buffer, True);
	}
	return;
    case NoticeReplyMiddle:
	fork_editor(context);
	break;
    case NoticeReplyRight:
    case NoticeReplyClose:
	free_post_context(context);
	set_standard_message();
	break;
    }

    XtPopdown(w);
    XtDestroyWidget(w);
}

static void post_context_callback(void *client_data,
				  int status,
				  char *stderr_buf)
{
    PostContext	*context = (PostContext *)client_data;
    char	message[1024];
    char	*left_label = "Post";
    int		ok = False;
    Widget	notice;

    if (WIFEXITED(status))
	switch (WEXITSTATUS(status)) {
	case 0:
	    ok = check_article_to_post(context, message, sizeof message);
	    break;
	case 127:
	    strcpy(message, "Failed to start editor!");
	    break;
	default:
	    strcpy(message, "Editor exited abnormally!");
	    break;
	}
    else if (WIFSIGNALED(status))
	sprintf(message, "Editor caught %s!", signal_string(WTERMSIG(status)));
    else
	strcpy(message, "Unknown problem with editor!");

    if (!ok)
	stderr_popup(stderr_buf, 0);

    if (!(context->flags & POST) || (context->flags & POST_DONE))
	left_label = "Mail";
    else if (!(context->flags & MAIL) || (context->flags & MAIL_DONE))
	left_label = "Post";
    else
	left_label = "Post&Mail";

    notice =
	popup_notice("postnotice", message, left_label, "Edit", "Abort",
		     0, post_notice_callback, (XtPointer)context, XtGrabNone);
    if (!ok) {
	Arg	arg;
	Widget	knapp;

	XtSetArg(arg, XtNleftKnapp, &knapp);
	XtGetValues(notice, &arg, 1);
	XtSetSensitive(knapp, False);
	XBell(XtDisplay(main_widgets.shell), 0);
    }
}

void fork_editor(PostContext *context)
{
    pid_t	pid;

    pid = fork_nicely(context, post_context_callback, True);

    if (pid < 0) { /* fork failed */
	Arg	arg;
	Widget	knapp, notice;
	char	*left_label = "Post";

	if ((context->flags & POST) && !(context->flags & POST_DONE))
	    if ((context->flags & MAIL) && !(context->flags & MAIL_DONE))
		left_label = "Post&Mail";
	    else if ((context->flags & MAIL) && !(context->flags & MAIL_DONE))
		left_label = "Mail";

	set_message("Failed to start editor!", True);
	notice = popup_notice("postnotice", "Failed to start editor",
			      left_label, "Edit", "Abort", 0,
			      post_notice_callback, (XtPointer)context,
			      XtGrabNone);

	XtSetArg(arg, XtNleftKnapp, &knapp);
	XtGetValues(notice, &arg, 1);
	XtSetSensitive(knapp, False);
	return;
    }

    if (pid == 0) { /* child */
	char	editcmd[512];
	char	*p1, *p2;

	p1 = strstr(global.edit_command, "%s");
	p2 = strstr(global.edit_command, "%i");

	if (p1)
	    if (p2)
		if (p1 < p2)
		    sprintf(editcmd, global.edit_command,
			    context->file_name, context->line);
		else
		    sprintf(editcmd, global.edit_command,
			    context->line, context->file_name);
	    else
		sprintf(editcmd, global.edit_command, context->file_name);
	else if (p2)
	    sprintf(editcmd, global.edit_command, context->line);
	else
	    sprintf(editcmd, global.edit_command);

	execl(BIN_SH, "sh", "-c", editcmd, (char *)NULL);
	perror("knews: execl " BIN_SH);
	_exit(127);
    }

    /*
     * parent
     */
    if (global.mode == NewsModeDisconnected)
	set_message("Editor started.  Warning: not connected.", False);
    else if (global.posting_allowed || !(context->flags & POST))
	set_message("Editor started.", False);
    else
	set_message("Editor started.  Warning: this server "
		    "doesn't allow posting.", False);
}
