/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include "../Widgets/Compat.h"
#include "color.h"
#include "util.h"

static void set_defaults(Screen *screen)
{
    global.visual = DefaultVisualOfScreen(screen);
    global.cmap   = DefaultColormapOfScreen(screen);
    global.depth  = DefaultDepthOfScreen(screen);
}

static int find_visual(Display *disp, Screen *screen, int class, int depth)
{
    XVisualInfo	info;
    int		screen_no = XScreenNumberOfScreen(screen);

    if (!XMatchVisualInfo(disp, screen_no, depth, class, &info))
	return False;

    global.visual = info.visual;

    return True;
}

static char *get_resource(XrmDatabase db, char *res_name, char *res_class)
{
    XrmValue		val;
    XrmRepresentation	rep;
    XrmQuark		name[3], class[3];

    name[0] = XrmPermStringToQuark("knews");
    name[1] = XrmPermStringToQuark(res_name);
    name[2] = 0;
    class[1] = XrmPermStringToQuark("VisualClass");
    class[0] = XrmPermStringToQuark(res_class);
    class[2] = 0;

    if (XrmQGetResource(db, name, class, &rep, &val))
	return (char *)val.addr;

    return NULL;
}

static int str_to_class(char *class)
{
#define DO(name) if (strcmp(class, #name) == 0) return name;
    DO(StaticGray);
    DO(GrayScale);
    DO(StaticColor);
    DO(PseudoColor);
    DO(TrueColor);
    DO(DirectColor);
#undef DO

    return -1;
}

void color_init(Display *disp)
{
    Screen		*screen = DefaultScreenOfDisplay(disp);
    XrmDatabase		db = XtScreenDatabase(screen);
    char		*visual_class = NULL;
    char		*visual_depth = NULL;
    char		*install      = NULL;
    int			depth, class;

    set_defaults(screen);
    if (!db)
	return;

    visual_class = get_resource(db, "visualClass", "VisualClass");
    visual_depth = get_resource(db, "visualDepth", "VisualDepth");

    if (!visual_depth)
	depth = global.depth;
    else if (sscanf(visual_depth, "%d", &depth) != 1) {
	fprintf(stderr, "knews: visualDepth is not a number: %s\n",
		visual_depth);
	return;
    }

    if (!visual_class)
	class = -1;
    else if ((class = str_to_class(visual_class)) < 0) {
	fprintf(stderr, "knews: no visual type called %s.\n", visual_class);
	return;
    }

    if (visual_class) {
	if (!find_visual(disp, screen, class, depth)) {
	    fprintf(stderr,
		    "knews: couldn't find %d-bit %s visual, using default.\n",
		    depth, visual_class);
	    return;
	}
    } else if (visual_depth) {
	fprintf(stderr,
		"knews: visualDepth given without visualClass, ignoring.\n");
	return;
    } else {
	install = get_resource(db, "installCmap", "InstallCmap");
	if (!install || case_lstrcmp(install, "true") != 0)
	    return;
    }

    global.cmap = XCreateColormap(disp, RootWindowOfScreen(screen),
				  global.visual, AllocNone);
}
