/* `ecc.c' -- A simple application of the rs255-249 and gf256 library.

   Copyright (C) 1993, 1996 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Written by Paul Flaherty (paulf@stanford.edu).
   Speeded up by Fung-Chai Lim (fclim@acm.org).

   This file is part of GNU ecc.  */

#ifdef HAVE_CONFIG_H
#if defined (CONFIG_BROKETS)
/* We use <config.h> instead of "config.h" so that a compilation
   using -I. -I$srcdir will use ./config.h rather than $srcdir/config.h
   (which it would do because it found this file in $srcdir).  */
#include <config.h>
#else
#include "config.h"
#endif
#endif

#include <stdio.h>
#include <fcntl.h>

#if HAVE_UNISTD_H
#include <sys/types.h>
#include <unistd.h>
#endif
#ifndef F_OK
#define F_OK	0
#endif

#include <errno.h>
#ifndef errno
extern int errno;
#endif

#include <rsff-f9.h>

extern void error _PROTO_ ((int exitStatus, int errno, char const *format,
			    ...));
extern void process_options _PROTO_ ((int argc, char *argv[], int *encode,
				      int *force, int *verbose));
extern void usage _PROTO_ ((int status));

static void encode_file _PROTO_ ((int infd, int outfd));
static void decode_file _PROTO_ ((int infd, int outfd, int verbose));

main (argc, argv)
     int argc;
     char *argv[];
{
  int encode;
  int force;
  int verbose;
  char *infname, *outfname;
  int infd, outfd;
  extern int optind;

  process_options (argc, argv, &encode, &force, &verbose);

  infname = outfname = NULL;
  if (optind != argc)
    {
      argc -= optind;
      argv += optind;
      if (argc > 2)
	usage (1);
      if (argc == 2)
	outfname = argv[1];
      if (argc > 0)
	infname = argv[0];
    }

  if (infname == NULL || strEQ (infname, "-"))
    infd = fileno (stdin);
  else if ((infd = open (infname, O_RDONLY)) < 0)
    error (1, errno, "can't open %s for reading", infname);

  if (outfname == NULL || strEQ (outfname, "-"))
    outfd = fileno (stdout);
  else if (!access (outfname, F_OK) && !force)
    {
      error (0, 0, "Output file %s already exists.", outfname);
      close (infd);
      usage (1);
    }
  else if ((outfd = open (outfname, O_WRONLY | O_CREAT | O_TRUNC, PERM_MODE)) < 0)
    {
      close (infd);
      error (1, errno, "can't open %s for writing", outfname);
    }

  if (encode)
    encode_file (infd, outfd);
  else
    decode_file (infd, outfd, verbose);

  close (infd);
  close (outfd);
  exit (0);
}

     static void
encode_file (infd, outfd)
     int infd, outfd;
{
  int n;
  GF256 data[256];

  data[0] = 'G';
  for (n = 1; n < 249; n++)
    data[n] = 0;

  do
    {
      n = read (infd, &data[1], 248);
      data[249] = n;

      rs255_249_encode (&data[1], &data[250]);

      write (outfd, data, 256);
    }
  while (n == 248);
}

     static void
decode_file (infd, outfd, verbose)
     int infd, outfd, verbose;
{
  int n;
  int block;
  int status;
  GF256 data[256];

  block = 0;
  while (1)
    {
      block++;
      n = read (infd, data, 256);

      if (n == 0)
	break;
      if (n != 256 || data[0] != 'G')
	error (1, 0, "sync error in block %d.\n", block);

      status = rs255_249_decode (&data[1]);

      if (status == RS255_249_E4)
	error (0, 0, "unrecoverable error in block %d.", block);
      else if (status != RS255_249_E0 && verbose)
	{
	  n = rs255_249_number_of_errors (status);
	  error (0, 0, "%d byte error in block %d.", n, block);
	}

      n = data[249];
      write (outfd, &data[1], n);
    }
}
