/* `rsff-f9.h' -- The 3-byte error-correcting encoder and decoder based on
   the Reed-Solomon (255, 249) code over the Galois Field (2^8).

   Copyright (C) 1996 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Developed by Paul Flaherty (paulf@stanford.edu).
   Munged for speed by Fung-Chai Lim (fclim@acm.org).

   This file is part of GNU ecc.  */

#ifndef _rs255_249_h
#define _rs255_249_h

#ifdef	__cplusplus
extern "C" {
#endif

/*
#ifdef MAN

rs255-249(3L)	GNU Library Manual Pages		rs255-249(3L)

NAME
   rs255-249 - The 3-byte error-correcting encoder and decoder based on
   the Reed-Solomon (255, 249) code over the Galois Field (2^8).

SYNOPSIS
   #include <gf256.h>
   #include <rsff-f9.h>

   cc ... -lecc

DESCRIPTION
   void rs255_249_encode (data, parity)
   const GF256 *const data;
   GF256 *parity;

   status = rs255_249_decode (received)
   int status;
   GF256 *received;

   int rs255_249_number_of_errors (status)
   int status;

   DATA points to an array of 249 unsigned char, and PARITY points to an
   array of 6 unsigned char.
   RECEIVED points to an array of 255 unsigned char.  See below on the
   relationship between RECEIVED and DATA/PARITY.  The possible values for
   STATUS is the set of symbolic constants defined below with RS255_249_E
   prefix.

   The generating polynomial for the Reed-Solomon (255, 249) code used here
   is
     g(x) = (x - 2) (x - 2^2) ... (x - x^6).

   For each block of 249 bytes passed in as DATA, rs255_249_encode() computes
   the values of the 6 bytes in PARITY such that the polynomial defined as
     c(x) = c254 x^254 + c253 x^253 + ... + c2 x^2 + c1 x + c0
   where
     c254 ... c6 = DATA[248] ... DATA[0]
     c5 ... c0 = PARITY[5] ... PARITY[0]
   is a multiple of the generating polynomial g(x).

   The bytes pointed to by DATA and PARITY may be transmitted in any manner.
   At the receiver, RECEIVED is constructed as follows
     received[0] ... received[248] = d[0] ... d[248]
     received[249] ... received[254] = p[0] ... p[5]
   where D and P are the received versions of DATA and PARITY respectively.

   If 3 or less errors occurred, rs255_249_decode() will correct all the
   errors, and upon return from this function, the first 249 bytes of RECEIVED
   may be safely assumed to be those of DATA.
   Some or all of the errors may occurred in PARITY, ie the bytes from R[249]
   to R[254].  These errors are never corrected.

   If STATUS is RS255_249_E4, some irrecoverable error has occurred.  This
   usually means that there are 4 or more errors in the block of 255 bytes.

   The simplest application is to check STATUS.  If it is RS255_249_E4,
   arrange for the retransmission of the entire block.  Otherwise, ignore
   the last 6 bytes of RECEIVED and return the first 249 bytes as DATA in
   that block.

   For applications that wish to be verbose whenever 1 to 3 errors had
   occurred, the returned value is RS255_249_Ex_Cy, where 1 <= X <= 3 and
   0 <= Y <= X.  X is the number of errors and Y is the number of these
   errors occurring in the DATA portion.  The function
   rs255_249_number_of_errors() returns the value of X when it is passed the
   returned value of rs255_249_decode().

SEE ALSO
   gf256(3L)
#endif MAN
*/

#ifndef GF256
#define GF256 unsigned char
#endif

#ifdef __STDC__
void rs255_249_encode (const GF256 *const data, GF256 *parity);
int rs255_249_decode (GF256 *received);
#else
void rs255_249_encode ();
int rs255_249_decode ();
#endif

/* No error.  */
#define RS255_249_E0		0x00

/* 1-byte error; error corrected.  */
#define RS255_249_E1_C1		0x01

/* 2-byte error; all errors corrected.  */
#define RS255_249_E2_C2		0x02

/* 3-byte error; all errors corrected.  */
#define RS255_249_E3_C3		0x03

/* Irrecoverable error situation.  */
#define RS255_249_E4		0x04

/* 1-byte error in the parity check bytes; no correction needed.  */
#define RS255_249_E1_C0		0x11

/* 2-byte error in the parity check bytes; no correction needed.  */
#define RS255_249_E2_C0		0x22

/* 2-byte error, one in the parity check bytes; the other error was
   corrected.  */
#define RS255_249_E2_C1		0x12

#define RS255_249_E3_C0		0x33
#define RS255_249_E3_C1		0x23
#define RS255_249_E3_C2		0x13

#define rs255_249_number_of_errors(status)	\
  ((status) & 0x0f)

#ifdef	__cplusplus
}
#endif

#endif /* _rs255_249_h */
