/*
   Time-stamp: <95/12/07 00:50:10 yusuf>
*/



/* Tape structure
 * 
 * Taper header (at beginning of each tape)
 * ----------------------------------------
 * long taper_magic number  (identifies taper backups  )
 * long archive ID (unqiue # that IDs archives         ) 
 * int tape number                                     )
 * char archive_title 
 * 
 * 
 * Volume
 * ------
 * long volume_magic
 * long no_in_vol
 * 
 * DATA
 *               : file_info
 *               : file_name
 *               : file (straight or compressed) only if file IS_IFREG
 *               :    S_ISDIR   nothing
 *               :    S_ISLNK   the name of the symbolic link (# chars first)
 * 
 * 
 * File info block:
 * 
 *   long   actual size  with uncompressed files, = size
 *   volume           
 *   position in vol  
 *   rdev             )
 *   uid              )
 *   gid              )
 *   mode             )     Taken from stat call
 *   org mode (mode of original file for links)
 *   size             )
 *   a time           ) 
 *   m time           )
 *   c time           )  
 *   backup time 
 *   length of name+1
 *   compressed or not compressed
 *   checksum
 *
 * 
 * Information file structure
 * 
 * long archive_id                        )
 * int number_tapes                       )   in structure
 * long info_file_size                    )
 * int number_volumes                     )
 * long size_of_volume_headers
 * long no_in_archive                     ) 
 * char archive_title                     )  
 * 
 * For each file:
 *       file_info
 *       file_name
 * 
 * Volume header information
 * 
 * At the end of the file
 *       volume    tape it's on   - for each volume
 * 
 * 
*/


#include "taper.h"
#include "backup.h"

FILE *fdfifo;
char fifo_name[MAXNAMLEN];


_errstat my_gzip(char *file, char *wf)
{
    int ifd, ofd;
    char l[1024];

    sprintf(l, "I:Internal gzipping %s to %s", file, wf);
    if (log_level > 3) write_log(l);
    ifd = open(file, O_RDONLY);
    if (ifd==-1) {
	sprintf(l, "I:%s while opening file", strerror(errno));
	write_log(l);
	return -1;
    }
    ofd = creat(wf, S_IREAD|S_IWRITE);
    if (ofd==-1) {
	sprintf(l, "I:%s while creating file", strerror(errno));
	write_log(l);
	close(ifd); 
	return -1;
    }
    zip(ifd, ofd);
    close(ifd); close(ofd);
    return 0;
}

        
_errstat my_compress(char *file, char *wf)
{
/* Compresses the 'file' to output file 'wf' 
 * 
 * To aid speed, reads the whole file into memory and compresses
 * 
 * Returns 0 if OK, -1 if error, -2 if compressed file would be larger
 * 
 * Returns an error if given a NULL file
 * If the file is > COMPRESS_MAX_ORG (0x700000000), then an error
 *   is returned - I can't ever see a file this long - and if it is,
 *   it's probably best not to compress
 */
    int fd;
    struct stat b;
    ULONG   c_len;
    char    l[1024];

    sprintf(l, "I:Original compressing %s to %s", file, wf);
    if (log_level > 3) write_log(l);
    if ((comp_buffer1 == NULL) || (comp_buffer2 == NULL))
      return -1;
    fd=open(file, O_RDONLY); 
    if (fd == -1) 
	return -1;
    if (fstat(fd, &b) == -1) {close(fd); return -1;} 
    if (b.st_size == 0) {close(fd); return -1;}	 /* not going to compress empty files */
    if (b.st_size > COMPRESS_MAX_COM) {close(fd); return -1;}   /* too large for compress to handle */
    if (b.st_size > COMPRESS2_BUFFER_SIZE) {close(fd); return -1;}   /* too large for compress buffer */
    if (read(fd, comp_buffer1, b.st_size) == -1) {close(fd); return -1;};
    close(fd);
    compress(COMPRESS_ACTION_COMPRESS, cbuf, (UBYTE *) comp_buffer1, b.st_size,
	     (UBYTE *) comp_buffer2, &c_len);
    if (c_len == 0) return -1;			 /* error compressing */
    if (c_len >= b.st_size) return -2;		 /* compressed size is bigger */
    fd=creat(wf, S_IREAD|S_IWRITE); if (fd==-1) return -1;
    if (write(fd, comp_buffer2, c_len)==-1) {close(fd); return -1;}
    close(fd);
    return 0;
}
    

_errstat exclude_compression(char *fn)
{
/* Checks to see if file is one of the files not to compress.
 * 
 * returns 1 if should be excluded (ie. in list)
 * returns 0 if shouldn't be excluded
*/
    return exclude_list(fn, exclude_compress);
}

    
_s8 stop_compressing;				 /* if 1, then child should terminate */
void backup_child_term(int signal)
{
/* Child has received terminate signal */
    stop_compressing=1;
}

void backup_stop_child()
{
/* Stops child from compressing and deletes whatever files
 * are left in temporary buffers */
    char s[MAXNAMLEN];
    
    kill(backup_child, SIGTERM);		 /* tell child to stop compressing */
    waitpid(backup_child, NULL, 0);
    backup_child = 0;
    while (!feof(fdfifo)) {
	if (fgets(s, sizeof(s), fdfifo) == NULL) /* original filename */
	  break;
	if (fgets(s, sizeof(s), fdfifo) == NULL) /* compressed name */
	  break;
	if (s[strlen(s)-1] == '\n')		 /* remove trailing \n */
	  s[strlen(s)-1] = 0;
	unlink(s);				 /* delete it */
    }
    fclose(fdfifo);				 /* close FIFO */
    unlink(fifo_name);				 /* remove file list & FIFO */
}


void do_backup_child(char *cp)
{
/* Loop run in child.
 * 
 * Never returns
*/
    long count;
    char com[MAXNAMLEN], tmpbuf[MAXNAMLEN];    
    int x=0;
    
    close(dv);				 /* child doesn't need tape open */
    if (log_level > 3) write_log("Backup child successfully forked off");
    detach_shared_buffers();		 /* child doesn't need any of these */
    free_non_shared_buffers();
    if (compression == 2) {
	if (malloc_comp_buffers() == -1) {
	    if (log_level > 2) write_log("Unable to make compress 2 buffers - not compressing");
	    goto finbackup;
	}
    }
    signal(SIGTERM, backup_child_term);	 /* for child to stop compressing */
    if ((fdfifo = fopen(fifo_name, "w")) == NULL) /* open fifo */
      goto finbackup;
    
    stop_compressing=0;			 /* install TERM handler */
    cp = info + sizeof(struct info_file_header);   /* loop through */
    for (count=0;count<ifd.no_in_archive;count++) {   /* and look for files to compress */
	if (stop_compressing) 		 /* child has received TERM signal */
	  goto finbackup;
	if (((struct file_info *) cp)->pos_in_archive == 0)
	  if (!exclude_compression(cp+sizeof(struct file_info)))/* check we'd compress this */
	  if (S_ISREG(((struct file_info *) cp)->mode))/* only compress reg files */ {
	      tmpnam(tmpbuf);				 /* get temp filename */
	      sprintf(com, " :Compressing %s to %s", cp+sizeof(struct file_info), tmpbuf);
	      *com = (compression == 1) ? 'E' : 'I';
	      if (log_level > 2) write_log(com);
	      switch(compression) {
	       case 1:		 /* external gzip */
		  sprintf(com, "%s \"%s\" > %s 2>/dev/null", COMPRESS_PROG, 
			  (cp+sizeof(struct file_info)), tmpbuf);/* compress file */
		  change_dollar(com);
		  x = system(com);
		  break;
	       case 2:		 /* internal compress */
		  x = my_compress(cp+sizeof(struct file_info), tmpbuf);
		  break;
	       case 3:		 /* internal gzip */
		  x = my_gzip(cp+sizeof(struct file_info), tmpbuf);
		  break;
		default:
			x = -1; break;
	      }
	      
	      if (x==0) 		 /* OK compression */
		x = fprintf(fdfifo, "%s\n%s\n", cp+sizeof(struct file_info), tmpbuf);
	      else {			 /* problem compressing */
		  (x==-1) ?
		    sprintf(com, "I:couldn't compress %s err=%d", cp+sizeof(struct file_info),
			    errno) :	 /* error copmressing */
		  sprintf(com, "I:didn't compress %s as it grew", cp+sizeof(struct file_info));
		  if (log_level > 2)  write_log(com);   /* compressed was larger */
		  x = fprintf(fdfifo, "%s\n%s\n", cp+sizeof(struct file_info), FIFO_ERR);
	      }
	      fflush(fdfifo);		 /* to ensure received */
	  }
	cp += sizeof(struct file_info);
	while (*cp++);
    }
    finbackup:;
    if (fdfifo != NULL) fclose(fdfifo);
    if (log_level > 3) write_log("Backup child about to finish");
    free_comp_buffers();
    my_free_all();
    detach_shm();
    free_non_shared_buffers();
    exit(0);
}


_errstat backup_dpd(char *full_path, struct stat *b)
{
/* The routine for each file found by process_dir */
    
    struct file_info fi;
    dev_t dev;
    
    dev = get_file_info(full_path, &fi, 0, b);
    if (dev == 0) return 0;			 /* error - ignore */
    if (S_ISDIR(fi.mode)) 			 /* directories have trailing '/' */
	if (full_path[strlen(full_path)-1] != '/') 
          strcat(full_path, "/");
    fi.name_len = strlen(full_path)+1;		 /* recalculate based on above */
    if ( (proc_dev == 0) ||			 /* don't backup if on  */
	 ((proc_dev == 1) && (proc_dev != dev)) ) {/* /proc device */ 
	if (add_one_file_engine(NULL, &fi, full_path) != -1)
	  total_selected += sizeof(struct file_info) + fi.name_len;
    }
    return 0;
}


inline _errstat add_actual_files(WINDOW *mes_box)
{
    long count;
    struct oa_file_entry *se;
    
    for (count=0; count<no_sel; count++) {
	se = find_entry(sel_files, count);
	if ( (se->selected == 1) || /* only if directly selected */
	   (se->selected == 4) )
	process_dir(NULL, 0, name(se), se->incremental, backup_dpd, TRUE);
    }
    return 0;
}


_errstat backup_add_missing_dirs(WINDOW *mes)
{
/* This routines adds any directories that are required. */
    
    cp = info + sizeof(struct info_file_header);
    if (make_cf(cp, ifd.no_in_archive) == -1) return -1; /* find common path */
    return add_missing_dirs(mes, TRUE);
}


_errstat write_nameinfo(struct file_info *fi, char *cp)
{
/* An error writing to the backup device is a fatal error
   Returns 0 if all OK
   Returns -1 if error
*/
    char l[MAXNAMLEN+50];
    
    sprintf(l, "Writing info & filename for %s", cp);
    if (tape_write((char *) fi, sizeof(struct file_info)) == -1) {
	write_fatal_log(l);			 /* write info */
	return -1;
    }
    if (tape_write((char *) cp, strlen(cp)+1) == -1) {/*  write filename */
	write_fatal_log(l);
	return -1;
    }
    return 0;
}


_errstat backup_file(_s32 pos_in_archive)
{
/* Backup the file pointed to by cp and update cp 
 * An error writing to the backup device is a fatal error

   Returns 0 if all OK
   Returns -1 if error
*/
    
    struct file_info *fi;
    char   tmpf[MAXNAMLEN];
    char   *fn, l[MAXNAMLEN+50];
    int    fd;
    _s32   x, totalread;
    struct stat sbuf, csbuf;
    
    fi = (struct file_info *) cp;		 /* fi points to file info */
    fn = cp + sizeof(struct file_info);
    fi->pos_in_archive = pos_in_archive;	 /* set pos in volume */

    if (S_ISLNK(fi->mode)) {			 /* a soft link */
	memset(tmpf, 0, sizeof(tmpf));		 /* name of original file */
	readlink(fn, tmpf, sizeof(tmpf));	 /* is written */
	fi->checksum = 0;
	sprintf(l, "Writing link name for %s [%s]", fn, tmpf);
	if (log_level > 1) write_log(l);
	if (write_nameinfo( fi, fn) == -1) {
	    write_fatal_log(l);
	    return -1;
	}
	x = strlen(tmpf)+1;
	if (tape_write((char *) &x, sizeof(x)) == -1) {
	    write_fatal_log(l);
	    return -1;
	}
	if (tape_write((char *) tmpf, x) == -1) {
	    write_fatal_log(l);
	    return -1;
	}
	return 0;
    }
	
    if (!(S_ISREG(fi->mode))) { 		 /* if file isn't an ordinary */
	fi->checksum = 0;
	sprintf(l, "Storing directory %s", fn);
	if (log_level > 1) write_log(l);
	return write_nameinfo( fi, fn);		 /* file, only write out name */
    }

    *tmpf = 0;					 /* must be a regular file */
    fi->compressed = 0;				 /* not compressed */
    if ((compression) && (!exclude_compression(fn))) {/* compress if want to and not excluded */
	sprintf(l, "Reading filename from FIFO for %s", fn);
	if (log_level > 2) write_log(l);
	if (fgets(tmpf, sizeof(tmpf), fdfifo) == NULL) {   /* gets the original filename */
	    if (tmpf[strlen(tmpf)-1] == '\n')
	      tmpf[strlen(tmpf)-1] = 0;
	    sprintf(l, "Couldn't read filename from FIFO for %s - saving uncompressed", fn);
	    if (log_level > 2) write_log(l);
	    *tmpf = 0;
	    goto non_compress;
	}
	if (tmpf[strlen(tmpf)-1] == '\n')
	  tmpf[strlen(tmpf)-1] = 0;
	if (strcmp(tmpf, fn)) {			 /* checks that FIFO & me agree */
	    sprintf(l, "Mismatch reading in FIFO output for %s [got %s] - saving uncompressed", fn, tmpf);
	    if (log_level > 2) write_log(l); 
	    *tmpf=0;
	    goto non_compress;
	}
	sprintf(l, "Reading compressed filename from FIFO for %s", fn);   /* read name of compressed file */
	if (log_level > 2) write_log(l);
	if (fgets(tmpf, sizeof(tmpf), fdfifo) == NULL) {
	    if (tmpf[strlen(tmpf)-1] == '\n')
	      tmpf[strlen(tmpf)-1] = 0;
	    sprintf(l, "Couldn't read compressed filename from FIFO for %s - saving uncompressed", fn);
	    if (log_level > 2) write_log(l);
	    *tmpf=0;
	    goto non_compress;
	}
	if (tmpf[strlen(tmpf)-1] == '\n')
	  tmpf[strlen(tmpf)-1] = 0;
	if (!strcmp(tmpf, FIFO_ERR)) {
	    sprintf(l, "FIFO didn't compress %s - saving uncompressed", fn);
	    if (log_level > 2) write_log(l);
	    *tmpf=0;
	    goto non_compress;			 /* assume no compression */
	}
	sprintf(l, "Opening compressed file %s", tmpf);
	if (log_level > 2) write_log(l);
	fd = open(tmpf, O_RDONLY);		 /* open the temp file */
	if (fd == -1) {
	    write_error_log(l);
	    unlink(tmpf);			 /* compressed file, delete it */
	    fi->checksum = -1;			 /* mark file as invalid */
	    return write_nameinfo( fi, fn);
	}
	sprintf(l, "Getting compressed file status info for %s", tmpf);
	if (log_level > 2) write_log(l);
	if (fstat(fd, &csbuf) == -1) {		 /* get compressed file size */
	    write_error_log(l);
	    fi->checksum = -1;			 /* mark file as invalid */
	    close(fd);
	    unlink(tmpf);
	    return write_nameinfo( fi, fn);
	}
	
	fi->compressed = compression;		 /* file is compressed */
	fi->act_size = csbuf.st_size;		 /* actual size = size of compressed file */
    }
    else {					 /* no compression needed */
	non_compress:
	sprintf(l, "Opening file %s", fn);
	if (log_level > 2) write_log(l);
	fd = open(fn, O_RDONLY);		 /* open source file */
	if (fd == -1) {
	    write_error_log(l);
	    fi->checksum = -1;			 /* mark file as invalid */
	    return write_nameinfo( fi, fn);
	}
	sprintf(l, "Getting file status info for %s", fn);
	if (log_level > 2) write_log(l);
	if (fstat(fd, &sbuf) == -1) {		 /* get actual file size */
	    write_error_log(l);
	    fi->checksum = -1;			 /* mark file as invalid */
	    close(fd);
	    return write_nameinfo( fi, fn);
	}
	fi->act_size = sbuf.st_size;		 /* update file info */
    }
    
    fi->checksum = calc_checksum(fd);		 /* calculate checksum */

    sprintf(l, "Backing up file %s; actual size %d, on tape size %d.", 
	    fn, fi->size, fi->act_size);
    if (log_level > 1) write_log(l);
    if (write_nameinfo(fi, fn) == -1) {		 /* couldn't */
	write_fatal_log(l);
	close(fd);
	return -1;				 /* write info/name - fatal */
    }
    if (fi->checksum == -1) {			 /* couldn't get checksum - not fatal */
	close(fd);
	return 0;
    }
    totalread = 0;
    while (1) {					 /* copy file accross */
	x = read(fd, tr_buffer, max_tr_size);	 /* to device file   */
	if (!x) break;
	if (x == -1) {
	    write_error_log("Error reading file while transferring data to backup device");
	    while (totalread < fi->act_size) 
	      totalread += tape_write((char *) tr_buffer, min(max_tr_size,/* pad rubbish data to the end so */
				       fi->act_size - totalread));
	    break;				 /* so that the archive integrity is maintained */
	}
	totalread += x;				
	if ((x = tape_write((char *) tr_buffer, x)) == -1) {
	    close(fd);
	    return do_exit(ERROR_WRITING);
	}
    }
    close(fd); 

    if (*tmpf)					 /* remove temporary files */
      unlink(tmpf);
    return 0;
}

_errstat write_volume_header(_s32 no_files)
{
/* Writes the volume header to the tape and appends volume header
 * information to the block in memory.

 *  Returns 0 if OK, -1 if error
*/
    _s32 c;
    struct oa_file_entry *fe;
    int  c1, sz;
    char *cur_pos;
    struct volume_header *vh1, vh;
    char null_string;

    if (log_level > 2) write_log("Writing volume header");
    vh.volume_magic = VOLUME_MAGIC;
    vh.backup_time = time(NULL);		 /* time backed up */
    strcpy(vh.volume_title, volume_title);
    vh.no_sels = no_sel;
    vh.no_in_volume = no_files;
    sz = sizeof(struct volume_header);
    fe=sel_files;				 /* work out how much space */
    for (c=0; c<no_sel; c++) {			 /* needed for volume header */
	sz += sizeof(_s32) + strlen(name(fe)) + 1 +
	  sizeof(_s32)+ 0 + 1;			 /* room for filter */
	advance_ofe(&fe);
    }
    vh.size_header = sz;
    ifd.size_volume_headers += sz;		 /* update size */
    if (tape_write((char *) &vh, sizeof(struct volume_header)) == -1)
      return -1;
    vol_headers = my_realloc(vol_headers, ifd.size_volume_headers);
    if (vol_headers == NULL)
      return do_exit(ERROR_MEMORY);
    cur_pos = vol_headers;
    for (c=0; c<ifd.number_volumes-1; c++) {	 /* skip past existing entries */
	vh1 = (struct volume_header *) cur_pos;	 /* in volume headers */
	cur_pos += sizeof(struct volume_header);/* past header */
	for (c1=0; c1<vh1->no_sels; c1++) {
	    sz = *(_s32 *) cur_pos;		 /* skip selection name */
	    cur_pos += sz + sizeof(_s32);
	    sz = *(_s32 *) cur_pos;		 /* skip filter */
	    cur_pos += sz + sizeof(_s32);
	}
    }
    *((struct volume_header *)  cur_pos) = vh;
    cur_pos += sizeof(struct volume_header);
    fe=sel_files;				 /* write file selections */
    for (c=0; c<no_sel; c++) {
	c1 = strlen(name(fe)) + 1;		 /* file name */
	if (tape_write((char *) &c1, sizeof(c1)) == -1)
	  return -1;
	if (tape_write(name(fe), c1) == -1)
	  return -1;
	*((_s32 *) cur_pos) = c1;		 /* filename length */
	cur_pos += sizeof(_s32);			 /* advance to filename pos */
	strcpy(cur_pos, name(fe));
	cur_pos += c1;				 /* past filename */
	c1 = 0+1;				 /* filter */
	if (tape_write((char *) &c1, sizeof(c1)) == -1)
	  return -1;
	if (tape_write(&null_string, c1) == -1)	 /* filter */
	  return -1;
	*((_s32 *) cur_pos) = c1;
	cur_pos += sizeof(_s32);
	strcpy(cur_pos, "");			 /* filter */
	cur_pos += c1;
	advance_ofe(&fe);
    }
    return 0;					 
}

_errstat write_out_info_file(char *info)
{
    int fd;
    
    if ((fd = open_info_file(FALSE, ifd.archive_id)) == -1) /* error message handled */
	return -1;				 /* by open)info_file */
    if (log_level > 2) write_log("Writing info file information");
    if (write(fd, info, ifd.info_file_size) == -1)
	return do_exit(ERROR_WRITING_INFO);
    if (write(fd, vol_headers, ifd.size_volume_headers) == -1)   /* write out volume header info */
      return do_exit(ERROR_WRITING_INFO);
    if (write(fd, vt_info, ifd.number_volumes*sizeof(struct volume_tape_info)) == -1)
	return do_exit(ERROR_WRITING_INFO);
    if (log_level > 2) write_log("Closing info file");
    close(fd);
    return 0;
}


_errstat  do_backup() {				 
/* For aborting a backup, 
 * 
 * fi.checksum = -2 is written - this means a backup
 *   was aborted. The info file is changed accordingly
*/
    _s32   count=0, no_old_files, bytes_processed, bytes_written, 
           no_written, org_files;
    WINDOW *mes=NULL;
    char   s[100], s2[30], s3[30], tmpbuf[MAXNAMLEN];
    time_t t_start, t_current;
    struct volume_tape_info vti;
    int    x;
    _s8    quitting;
    struct file_info *fi;
    char   sa[9][150];
	
    if (!no_sel)				 /* none selected */
      return 0;
    if (!no_windows)
      mes = status_box(mes, "          Opening backup device...          ", 3, TRUE, 6);

    status_box(mes, "Adding to archive directory...", 3, FALSE, 1);
    if (!append) {
	tdh.magic = TAPER_MAGIC_NUMBER;		 /* identify our directory volume */
	tdh.tape_number = 1;			 /* first volume */
	ifd.archive_id = tdh.archive_id;	 /* setup info file header */
	ifd.info_file_size = sizeof(struct info_file_header);
	ifd.number_tapes = 1;
	ifd.number_volumes = 0;
	ifd.size_volume_headers = 0;
	ifd.no_in_archive = 0;
	strcpy(ifd.archive_title, archive_title);
	strcpy(tdh.archive_title, archive_title);
	no_old_files = 0;
	vt_info = my_realloc(vt_info, sizeof(struct volume_tape_info));
    }
    else {
	ifd = *((struct info_file_header *) info);/* get original ifd */
	no_old_files = ifd.no_in_archive;
	vt_info = my_realloc(vt_info, sizeof(struct volume_tape_info)*(ifd.number_volumes+1));
    }

    org_files = no_old_files;
    if (!append) {
	if (tape_open(O_RDWR) == -1) return -1;
	blocks_passed = 0;
	tape_set_blk_size();
	if (write_tape_header(&tdh) == -1)
	  return -1;
    }
    else {
	if (goto_end_vol(mes, 3, ifd.number_volumes, 1, TRUE) == -1)/* goto end of */
	  return -1;				 /* previous volume */
	blocks_passed = 0;
    }

    ifd.number_volumes++;
    vti.volume = ifd.number_volumes;		 /* update volume/tape info */
    vti.start_tape = ifd.number_tapes;
    cp = info + sizeof(struct info_file_header);
    if (append) 				 /* move past existing entries */
      for (count=0; count < no_old_files; count++) {
	    cp += sizeof(struct file_info);
	    while (*cp++);
      }

    if (add_actual_files(mes) == -1)
      return -1;				 /* add filenames to directory */

    
    memcpy(info, &ifd, sizeof(struct info_file_header));
    status_box(mes, "", 3, FALSE, 1);
    if (backup_add_missing_dirs(mes) == -1)	 /* add directories that are required */
      return -1;
    status_box(mes, "", 1, FALSE, 1);

    if (write_volume_header(ifd.no_in_archive-no_old_files) == -1) return -1;/* write volume header */
/* free superflous memory */    
    sel_files = my_realloc(sel_files, 1);
    archive_files = my_realloc(archive_files, 1);
    len_archive_files = 0; cur_af_size = 0;

    bytes_written=0;
    bytes_processed=0;
    print_kb(s3, total_selected);
    if (compression) {
	tmpnam(fifo_name);
	if ((mknod(fifo_name, S_IFIFO|S_IREAD|S_IWRITE, 0)) == -1) 
	    return do_exit(ERROR_CREATING_FIFO);
	if (backup_child) {				 /* a process still going on */
	    kill(backup_child, SIGKILL);
	    waitpid(backup_child, NULL, 0);
	    backup_child = 0;
	}
	if (log_level > 3) write_log("About to fork of backup child");
	backup_child = fork();
	if (backup_child == 0) {		 /* we are in child process */
	    do_backup_child(cp);		 /* never returns from here */
	}

	if (backup_child == -1) {
	    unlink(fifo_name); 
	    backup_child = 0;
	    return do_exit(ERROR_UNABLE_FORK);
	}
	if ((fdfifo = fopen(fifo_name, "r")) == NULL) {
	    kill(backup_child, SIGKILL);		 /* kill child process  */
	    waitpid(backup_child, NULL, 0);
	    backup_child = 0;
	    return do_exit(ERROR_OPENING_FIFO);
	}
	if (comp_head_start) {			 /* let the compression program */
	    status_box(mes, "Giving compression a head start", 3, FALSE, 1);
	    sleep(60*comp_head_start);		 /* have a head start */
	    t_start = time(NULL);		 /* restart timer */
	}
    }
    cp = info + sizeof(struct info_file_header);
    nodelay(mes, TRUE);
    quitting=0;
    no_written=0;
    t_start = time(NULL);
    for (count=0; count< ifd.no_in_archive; count++) {
	*s = wgetch(mes);
	if (((*s == 'q') || (*s == 'Q')) && (!quitting)) {
	    if (message_box("Confirm abort backup", MB_YESNO)) {
		touchwin(mes); wrefresh(mes);
		backup_stop_child();		 /* stop child from compressing */
		fi = (struct file_info *) cp;
		fi->act_size = -1;
		fi->name_len = 0;
		fi->checksum = -2;
		write_nameinfo(fi, "");
		ifd.no_in_archive = no_old_files;/* correct info file */
		get_vh(vol_headers, ifd.number_volumes)->no_in_volume = no_written; 
		break;
	    }
	}
	if (((struct file_info *) cp)->pos_in_archive == 0) {  /* only need new ones */
	    no_written++;
	    status_box(mes, cp+sizeof(struct file_info), 0, FALSE, 1);
	    sprintf(s, "File %d of %d", count, ifd.no_in_archive);
	    status_box(mes, s, 2, FALSE, 1);
	    sprintf(s, "Processed %s of %s", print_kb(s2, bytes_processed), 
		    print_kb(s3, total_selected));
	    status_box(mes, s, 3, FALSE, 1);
	    if (bytes_written)
	      sprintf(s, "Written %s : Ratio %.2f",
		      print_kb(s2, bytes_written), 
		      (float) bytes_processed/(float) bytes_written);
		else
	      sprintf(s, "Written %s : Ratio 1.0",
		      print_kb(s2, bytes_written));
	    status_box(mes, s, 4, FALSE, 1);
	    sprintf(s, "Total on archive %s", print_kb(s2, bytes_written+total_compressed));
	    status_box(mes, s, 5, FALSE, 1);
	    t_current = time(NULL);
	    x = t_current-t_start;
	    if (total_selected) 
	      sprintf(s, "%.0f%% done, Elapsed %s, Remaining %s",  (float) bytes_processed/(float) total_selected * 100,
		      convtime(s2, t_start, t_current),
		      (bytes_processed == 0) ? "" :
		      convtime(s3, x, x / ((float) bytes_processed/(float) total_selected)));
	    else				 /* if bytes=0, base on # files */
	      sprintf(s, "%d%% done, Elapsed %s, Remaining %s",  100*count/ifd.no_in_archive,
		      convtime(s2, t_start, t_current),
		      (count == 0) ? "" :
		      convtime(s3, x,  x / (count/ifd.no_in_archive)));
	    status_box(mes, s, 7, FALSE, 1);
	    if (x) {
		sprintf(s, "Backup rate %s/min [%s/min]",
			print_mb(tmpbuf, bytes_written/x*60),
			  print_mb(s2, bytes_processed/x*60));
		status_box(mes, s, 8, FALSE, 1);
	    }
	    if (backup_file(++no_old_files) == -1) {
		tape_close(dv);			 /* rewinding device so will auto update headers */
		backup_stop_child();
		return -1;			 /* error message handled by backup file */
	    }
	    bytes_processed += sizeof(struct file_info) + 1 + strlen(cp+sizeof(struct file_info));
	    bytes_written += sizeof(struct file_info) + 1 + strlen(cp+sizeof(struct file_info));
	    if (S_ISREG(((struct file_info *) cp)->mode)) {   /* only add this if regular file */
		bytes_processed += ((struct file_info *) cp)->size;
		bytes_written += ((struct file_info *) cp)->act_size;
	    }
	}
	
	cp += sizeof(struct file_info);
	while (*cp++);
    }
    t_current = time(NULL);
    x = t_current - t_start;
    if (x == 0) x=1;
    status_box(mes, "", 0, FALSE, 1);
    status_box(mes, "", 2, FALSE, 1);
    status_box(mes, "", 4, FALSE, 1);
    status_box(mes, "", 6, FALSE, 1);
    status_box(mes, "Updating header segments", 3, FALSE, 1);
    tape_close(dv);				 /* rewinding device so will auto update headers */
    memcpy(info, &ifd, sizeof(struct info_file_header));
    vti.blocks_on_last_tape = blocks_passed;
    vti.end_tape = ifd.number_tapes;
    memcpy(vt_info+ifd.number_volumes-1, &vti, sizeof(vti));   
    write_out_info_file(info);
    fclose(fdfifo);				 /* close FIFO */
    unlink(fifo_name);				 /* remove file list & FIFO */
    waitpid(backup_child, NULL, 0);
    backup_child = 0;
    close_statusbox(mes);
    touchwin(win_main); wrefresh(win_main);
    sprintf(sa[0], "Backup Finished");
    strcpy(sa[1], "");
    sprintf(sa[2], "Backed up: %d files,  %s  [%s]", ifd.no_in_archive - org_files,
	    print_mb(s2, bytes_written),
	    print_mb(s3, bytes_processed));
    if (log_level > 1) write_log(sa[2]);
    sprintf(sa[3], "Total on archive %s [%s]. Ratio %.2f", 
	    print_mb(s2, bytes_written+total_compressed),
	    print_mb(s3, bytes_processed+total_uncompressed),
	    (total_compressed+bytes_written == 0) ? 1 : 
	       (float) (total_uncompressed+bytes_processed)/
	       (float) (total_compressed+bytes_written));
    if (log_level > 1) write_log(sa[3]);
    strcpy(sa[4], "");
    sprintf(sa[5], "Time elapsed %s.", convtime(s2, t_start, t_current));
    if (log_level > 1) write_log(sa[5]);
    sprintf(sa[6], "Backup rate %s/min [%s/min]", 
	    print_mb(tmpbuf, bytes_written/x*60),
	    print_mb(s3, bytes_processed/x*60));
    if (log_level > 1) write_log(sa[6]);
    strcpy(sa[7], "");
    sprintf(sa[8], "%d warnings, %d errors", log_warnings, log_errors);
    if (log_level > 1) write_log(sa[8]);
    multi_message_box(sa, 9, MB_OK);
    return 0;
}


_errstat read_in_fileset(int argc, char *argv[])
{
/* Reads in the files */
    FILE *f;
    char ln[MAXNAMLEN];
    struct oa_file_entry *ce;
    _s32 c;
    struct direntry entry;
    time_t t;
    
    tmpnam(mailfn);
    mf = open(mailfn, O_CREAT|O_RDWR, S_IREAD|S_IWRITE);
    if (mf==-1) 
      return do_exit(ERROR_OPENING_MAIL);
    
    time(&t);
    sprintf(ln, "To: %s\n", MAIL_TO); write(mf, ln, strlen(ln));
    sprintf(ln, "Date: %s", ctime(&t)), write(mf, ln, strlen(ln));
    strcpy(ln, "Subject: Taper unattended backup\n"); write(mf, ln, strlen(ln));

    c = 0;
    while (c < argc) {
	if ( (!strcmp(argv[c], "-U") ||
	      (!strcmp(argv[c], "-unattended-file"))) ) {
		  c++;
		  if (c == argc) {
		      strcpy(ln, "Illegal file/directory/set name specified\n");
		      write(mf, ln, strlen(ln));
		      strcpy(ln, "BACKUP ABORTED\n");
		      write(mf, ln, strlen(ln));
		      return -1;
		  }
		  if (*argv[c] == '@') {
		      strcpy(dir_cur_dir, taper_info_files);
		      f = backrest_restore_backupset(&argv[c][1]);/* open file set */
		      if (f==NULL) {
			  sprintf(ln, "WARNING: Couldn't open file set %s\n", &argv[c][1]);
			  write(mf, ln, strlen(ln));
			  continue;
		      }
		      sprintf(ln, "Using file set %s\n", &argv[c][1]);
		      write(mf, ln, strlen(ln));
		      while (!feof(f)) {
			  fgets(ln, sizeof(ln), f);
			  if (ln[strlen(ln)-1] == '\n')
			    ln[strlen(ln)-1] = 0;
			  if (*ln) {
			      ce=sel_files;
			      for (c=0; c<no_sel; c++) {		 /* make sure not already in */
				  if (!strcmp(name(ce), ln)) 
				    break;
				  advance_ofe(&ce);
			      }
			      if (c==no_sel)
				if (get_statinfo(ln, &entry.info) != -1) {/* ignore if error */
				    select_entry_1(&entry, ln);
				    write(mf, "  Selected ", strlen("  Selected "));
				    write(mf, ln, strlen(ln));
				    strcpy(ln, " [from fileset]\n");
				    write(mf, ln, strlen(ln));
				}
			      else {
				  sprintf(ln, "WARNING: %s is not found\n", &argv[c][1]);
				  write(mf, ln, strlen(ln));
			      }
			  }
			  fgets(ln, sizeof(ln), f);		 /* ignore filter */
		      }
		      fclose(f);
		  }
		  else {
		      if (get_statinfo(argv[c], &entry.info) != -1) {/* ignore if error */
			  select_entry_1(&entry, argv[c]);
			  sprintf(ln, "Selected %s\n", argv[c]);
			  write(mf, ln, strlen(ln));
		      }
		      else {
			  sprintf(ln, "WARNING: %s is not found\n", &argv[c][1]);
			  write(mf, ln, strlen(ln));
		      }
		  }
	      }
	c++;
    }
    time(&t);
    sprintf(ln, "\nBackup commenced at %s", ctime(&t));
    write(mf, ln, strlen(ln));
    return 1;
}

   
_errstat utils_write_nullheader(WINDOW **mes);
_errstat check_backup() {
/* Make sure backup is a valid device and confirm with use about
   an erase if it has been requested.         */
    int x;
    WINDOW *mes=NULL;
    char info_file[MAXNAMLEN];
    struct stat buf;

    if (stat(tape, &buf) == -1) {		 /* touch file if needed */
	if (tape_type == TAPE_TYPE_FILE) {
	    if (log_level > 2) write_log("Creating backup file");
	    close(creat(tape, S_IREAD|S_IWRITE));
	    x = TAPE_EXIST_EMPTY;
	}
	else
	  return do_exit(ERROR_OPENING_BACKUP);
    }
    else
      x= do_read_vol_dir(-1, tape, O_RDWR, append, FALSE);
    if (x==-1) return -1;
    if (x == BAD_MAGIC) {
	if (!no_windows) {			 
	    if (bmessage_box("Unknown tape data. Overwrite?", MB_YESNO)) append = 0;
	    else return 0;;
	}
	else {					 /* unattended mode */
	    if (!tape_overwrite) {
		write_warning_log("Unknown data on the tape. Tape not overwritten");
		return 0;	 /* need to have authority to overwrite */
	    }
	    append = 0;				 /*   to overwrite tapes in unattended mode */
	}
    }
    
    if ((x == TAPE_EXIST) && (!append)) {
	if (!no_windows) {
	    if (!bmessage_box("Taper data on tape. Confirm erase", MB_YESNO))
	      return 0;
	}
    }
	
    if (x == TAPE_EXIST_EMPTY) {		 /* empty taper tape */
	append = 0;
	return 1;
    }
    
    if ((x == BAD_MAGIC) || (!append)) {	 /* no taper data on tape */
	append = 0;
	if (!no_windows)
	  mes = status_box(mes, "Erasing data on tape", 1, TRUE, 1);
	if (is_regfile(dv)) {
	    unlink(tape);				 /* regular files get deleted */
	    close(creat(tape, S_IREAD|S_IWRITE));/* and re-created to get zero length */
	}
	else 
	    utils_write_nullheader(&mes);
	strcpy(info_file, taper_info_files);	 /* look for information file */
	if (info_file[strlen(info_file)-1] != '/')	 /* associated with this archive */
	  strcat(info_file, "/");
	sprintf(info_file, "%staper_info.%u", info_file, tdh.archive_id);
	unlink(info_file);			 /* attempt to delete info file */
	if (mes) 
	  close_statusbox(mes);
	return 1;
    }
	
    
    if (x == TAPE_NOEXIST) {
	append = 0;
	return 1;
    }
    
    return 1;
   }
    


void taper_backup(int argc, char *argv[])
{
    int      x;
    int      old_append;
    char     s[MAXNAMLEN];

    old_append = append;			 /* save value of append for future use */
    if (open_logfile("Backup") == -1)
      return;
    if (!no_windows) {
	backrest_init_windows() ;			 /* Draw windows */
	backrest_clear_screen();
    }
    if (check_device_names() == -1) goto endbackup;/* check devices & other parms */
    if (backrest_do_mallocs() == -1)		 /* couldn't allocate memory */
      goto endbackup;
    if (check_backup() > 0) {
	if (!append) {				 /* no archive */
	    tdh.archive_id = time(NULL);	 /* set defaults */
	    ifd.archive_id = tdh.archive_id;
	    if (!no_windows)
	      get_string(win_main, archive_title, MAX_ARCHIVETITLE, "Enter archive title");
	    strcpy(tdh.archive_title, archive_title);
	    strcpy(ifd.archive_title, archive_title);
	}
#ifdef MEMORY_TIGHT
	archive_files = my_realloc(archive_files, 1);   /* free this memory */
	len_archive_files = 0; 
#endif	
	print_title_line();
	chdir(cur_dir);
	if (!no_windows) {
	    if (get_string(win_main, volume_title, MAX_ARCHIVETITLE, "Enter volume title"))
	      x = backup_select_files(cur_dir);	 /* select files */
	    else 
	      x = 0;
	}
	else 					 /* unattended mode */
	    x = read_in_fileset(argc, argv);
	
	getcwd(cur_dir, sizeof(cur_dir));
	backrest_kill_windows();
	print_my_name();
	if (x > 0)  do_backup();
    }
    else {
	backrest_kill_windows();
	log_errors = 1;				 /* the fatal error */
    }

    backrest_free_memory();

    if (mf) {					 /* if mail file */
	mail_finish("Backup");
	close(mf);
	sprintf(s, "%s %s < %s", MAIL_PROG, MAIL_TO, mailfn);   /* send mail */
	system(s);
	unlink(mailfn);
    }
    append = old_append;
    
    endbackup:;
    close_logfile("Backup");
}
