//  Copyright (C) 2001 Red Hat, Inc.  All Rights Reserved.
//  Copyright (C) 1999 Silicon Graphics, Inc.  All Rights Reserved.
//
//  This program is free software; you can redistribute it and/or modify it
//  under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  Further, any
//  license provided herein, whether implied or otherwise, is limited to
//  this program in accordance with the express provisions of the GNU
//  General Public License.  Patent licenses, if any, provided herein do not
//  apply to combinations of this program with other product or programs, or
//  any other product whatsoever.  This program is distributed without any
//  warranty that the program is delivered free of the rightful claim of any
//  third person by way of infringement or the like.  See the GNU General
//  Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with this program; if not, write the Free Software Foundation, Inc., 59
//  Temple Place - Suite 330, Boston MA 02111-1307, USA.

#ifndef DNotify_included
#define DNotify_included

#include "config.h"
#include "Monitor.h"
#include <signal.h>

//  DNotify is an object encapsulating the dnotify linux fcntl.
//  It "emulates" the IMon interface.
//  There can only be one instantiation of the DNotify object.
//
//  The user of this object uses express() and revoke() to
//  express/revoke interest in a file.  There is also
//  a callback, the EventHandler.  When an dnotify event comes in,
//  the EventHandler is called.
//
//  The user of the DNotify object is the Interest class.

class DNotify : public Monitor {
public:
    DNotify(EventHandler h);
    ~DNotify();

    static bool is_active();

    virtual Status express(const char *name, struct stat *stat_return);
    virtual Status revoke(const char *name, dev_t dev, ino_t ino);

private:
    struct FileWatch;
    struct DirWatch;
  
    //  Class Variables
    enum { QUEUESIZE = 1024 };
    static int pipe_write_fd;
    static int pipe_read_fd;
    static int change_queue[QUEUESIZE];
    static volatile sig_atomic_t DNotify::queue_overflowed;
    static volatile sig_atomic_t DNotify::queue_changed;
    static volatile int queue_head; // Only modified by read handler
    static volatile int queue_tail; // Only modified by signal handler
    static EventHandler ehandler;
    static void overflow_signal_handler(int sig, siginfo_t *si, void *data);
    static void signal_handler(int sig, siginfo_t *si, void *data);
    static void read_handler(int fd, void *closure);
 
    enum { DIR_HASHSIZE = 257 };
    static DirWatch *dir_hash[DIR_HASHSIZE];
    enum { FILE_HASHSIZE = 257 };
    static FileWatch *file_hash[FILE_HASHSIZE];

    static DirWatch **dir_hashchain(int fd)
			  { return &dir_hash[(unsigned) (fd) % DIR_HASHSIZE]; }
    static FileWatch **file_hashchain(dev_t d, ino_t i)
			  { return &file_hash[(unsigned) (d+i) % FILE_HASHSIZE]; }

    static DirWatch *lookup_dirwatch (int fd);
    static DirWatch *lookup_dirwatch (dev_t dir_dev, ino_t dir_ino);
    static FileWatch *lookup_filewatch (dev_t file_dev, ino_t file_ino);
    static void hash_dirwatch(DirWatch *w);
    static void hash_filewatch(FileWatch *w);
    static void unhash_dirwatch(DirWatch *w);
    static void unhash_filewatch(FileWatch *w);
    static Status watch_dir(const char *notify_dir, dev_t file_dev, ino_t file_ino);

    static void all_watches_changed(void);
    
    DNotify(const DNotify&);			// Do not copy
    DNotify & operator = (const DNotify&);	//  or assign.
};

#endif /* !IMon_included */


