// Copyright (C) 2001 Nils Bokermann <Nils.Bokermann@mediaWays.net>
//
// PURPOSE OF THIS FILE: Implement the ldap_modify and ldap_modrdn
//                       functions
//
// - Automatic Version Information via RCS:
//   $Id: modify.cxx,v 1.2 2001/12/19 05:26:56 nilsb Exp $
//   $Source: /cvsroot/openh323gk/openh323gk/ldap/src/modify.cxx,v $
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "ldapapi.h"
#include <ldap-int.h>

#ifndef lint
// mark object with version info
static const char vcid[] = "@(#) $Id: modify.cxx,v 1.2 2001/12/19 05:26:56 nilsb Exp $";
static const char vcHid[] = __LDAP_OPENH323_H;
#endif

int ldap_modify_ext (LDAP *ld, LDAP_CONST char *dn, LDAPMod **mods, LDAPControl **serverctrls,
		     LDAPControl **clientctrls, int *msgidp ) {

  LDAP_LDAPMessage msg;
  
  *msgidp=++(ld->msg_id);
  msg.m_messageID=*msgidp;
  msg.m_protocolOp=LDAP_LDAPMessage_protocolOp(LDAP_LDAPMessage_protocolOp::e_modifyRequest);
  LDAP_ModifyRequest &modify=msg.m_protocolOp;
  modify.m_object=dn;
  LDAP_ModifyRequest_modifications &ldapmod=modify.m_modifications;

  LDAPMod **mod=mods;

  for(unsigned int number=0;NULL!=*mod;number++) {
    if(ldapmod.GetSize()<=number)
      ldapmod.SetSize(number+1);
    ldapmod[number].m_operation=((*mod)->mod_op & (LDAP_MOD_ADD | LDAP_MOD_DELETE | LDAP_MOD_REPLACE)); // only bit 0 and 1.
    if((*mod)->mod_op & LDAP_MOD_BVALUES) { // mod_bvalue 
      ldapmod[number].m_modification.m_type=(*mod)->mod_type;
      for(unsigned int attribute=0;NULL!=(*mod)->mod_bvalues[attribute];attribute++){
	if(ldapmod[number].m_modification.m_values.GetSize()<=attribute)
	  ldapmod[number].m_modification.m_values.SetSize(attribute+1);
	ldapmod[number].m_modification.m_values[attribute].SetValue(PASN_OctetString((*mod)->mod_bvalues[attribute]->bv_val,
										     (*mod)->mod_bvalues[attribute]->bv_len));
      }
    } else {
      ldapmod[number].m_modification.m_type=(*mod)->mod_type;
      for(unsigned int attribute=0;NULL!=(*mod)->mod_values[attribute];attribute++){
	if(ldapmod[number].m_modification.m_values.GetSize()<=attribute)
	  ldapmod[number].m_modification.m_values.SetSize(attribute+1);
	ldapmod[number].m_modification.m_values[attribute]=(*mod)->mod_values[attribute];
      }
    }
    mod++;
  }


  PBER_Stream encoding_stream;
  msg.Encode(encoding_stream);
  if(encoding_stream.Write(*(ld->socket))) {
    return LDAP_SUCCESS;
  }
  ld->ld_errno=LDAP_UNAVAILABLE;
  return ld->ld_errno;
}

int ldap_modify_ext_s (LDAP *ld, LDAP_CONST char *dn, LDAPMod **mods,
		       LDAPControl **serverctrls, LDAPControl **clientctrls ) {
  int msgid;
  int rv;
  LDAPMessage *result;

  if(LDAP_SUCCESS!=(rv=ldap_modify_ext(ld, dn, mods, serverctrls, clientctrls,&msgid))) {
    ld->ld_errno=rv;
    return ld->ld_errno;
  }
  
  if (ldap_result(ld, msgid, 1, NULL, &result) == -1) {
    return( ld->ld_errno );	/* ldap_result sets ld_errno */
  }
  
  rv = ldap_result2error(ld, result, 0);
  ldap_msgfree(result);
  return rv;
}

int ldap_modify (LDAP *ld, LDAP_CONST char *dn,	LDAPMod **mods) {
  int msgid;
  int rv;
  if(LDAP_SUCCESS!=(rv=ldap_modify_ext(ld, dn, mods, NULL, NULL, &msgid))) {
    ld->ld_errno=rv;
    return -1;
  }
  return msgid;
}

int ldap_modify_s (LDAP *ld, LDAP_CONST char *dn, LDAPMod **mods) {
  int msgid,rv;
  LDAPMessage *result;
  if (-1==(msgid=ldap_modify(ld, dn, mods))){
    return ld->ld_errno;
  }
  if (ldap_result(ld, msgid, 1, NULL, &result) == -1) {
    return( ld->ld_errno );	/* ldap_result sets ld_errno */
  }
  
  rv = ldap_result2error(ld, result, 0);
  ldap_msgfree(result);
  return rv;
}
  

int ldap_modrdn2 (LDAP *ld, LDAP_CONST char *dn, LDAP_CONST char *newrdn, int deleteold) {
  LDAP_LDAPMessage msg;
  msg.m_messageID=++(ld->msg_id);
  msg.m_protocolOp=LDAP_LDAPMessage_protocolOp(LDAP_LDAPMessage_protocolOp::e_modifyRDNRequest);
  LDAP_ModifyRDNRequest &rdn=msg.m_protocolOp;
  rdn.m_entry=dn;
  rdn.m_newrdn=newrdn;
#ifdef V2PLUS
  rdn.m_deleteold=deleteold;
#endif

#ifdef DEBUG
  cerr << msg << endl;
#endif
  PBER_Stream encoding_stream;

  msg.Encode(encoding_stream);

#ifdef V2P_WITH_V2ASN
  // Very dangerous. Adding bool to the stream.

  encoding_stream[1]=encoding_stream[1]+3;
  encoding_stream[encoding_stream.GetPosition()]=1;
  encoding_stream[encoding_stream.GetPosition()+1]=1;
  encoding_stream[encoding_stream.GetPosition()+2]=1;
  encoding_stream.SetPosition(encoding_stream.GetPosition()+3);
#endif
#ifdef BER_DEBUG
  cerr << encoding_stream << endl;
#endif
  if(encoding_stream.Write(*(ld->socket))) {
    return ld->msg_id;
  }
  ld->ld_errno=LDAP_UNAVAILABLE;
  return -1;
}

int ldap_modrdn (LDAP *ld, LDAP_CONST char *dn, LDAP_CONST char *newrdn){
  return ldap_modrdn2(ld, dn, newrdn, 1);
}
int ldap_modrdn_s (LDAP *ld, LDAP_CONST char *dn, LDAP_CONST char *newrdn){
  int msgid,rv;
  LDAPMessage *result;
  if(-1==(msgid=ldap_modrdn(ld, dn, newrdn)))
    return ld->ld_errno;
  
  if (ldap_result(ld, msgid, 1, NULL, &result) == -1) {
    return( ld->ld_errno );	/* ldap_result sets ld_errno */
  }
  
  rv = ldap_result2error(ld, result, 0);
  ldap_msgfree(result);
  return rv;
}
  
int ldap_modrdn2_s (LDAP *ld, LDAP_CONST char *dn, LDAP_CONST char *newrdn, int deleteold){
  int msgid,rv;
  LDAPMessage *result;
  if(-1==(msgid=ldap_modrdn2(ld, dn, newrdn, deleteold)))
    return ld->ld_errno;
  
  if (ldap_result(ld, msgid, 1, NULL, &result) == -1) {
    return( ld->ld_errno );	/* ldap_result sets ld_errno */
  }
  
  rv = ldap_result2error(ld, result, 0);
  ldap_msgfree(result);
  return rv;
}
//
// End of modify.cxx
//
