
/* GnomeMeeting -- A Video-Conferencing application
 * Copyright (C) 2000-2001 Damien Sandras
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *                         videograbber.cpp  -  description
 *                         --------------------------------
 *   begin                : Mon Feb 12 2001
 *   copyright            : (C) 2000-2001 by Damien Sandras
 *   description          : Video4Linux compliant functions to manipulate the 
 *                          webcam device.
 *   email                : dsandras@seconix.com
 *
 */

#include "../config.h"

#include "config.h"
#include "videograbber.h"
#include "misc.h"
#include "gdkvideoio.h"
#include "common.h"
#include "gnomemeeting.h"
#include "misc.h"

#include <sys/time.h>


/* Declarations */
extern GnomeMeeting *MyApp;
extern GtkWidget *gm;


/* The functions */
GMVideoGrabber::GMVideoGrabber (options *o)
  :PThread (1000, NoAutoDeleteThread)
{
  gw = gnomemeeting_get_main_window (gm);
  opts = o;

  /* Internal state */
  is_running = 1;
  has_to_open = 0;
  has_to_close = 0;
  has_to_reset = 0;
  is_grabbing = 0;
  is_opened = 0;

  /* Start the thread */
  this->Resume ();
}


GMVideoGrabber::~GMVideoGrabber ()
{ 
  is_running = 0;
  is_grabbing = 0;

  /* So that we can wait, in the Thread (Main ())
     till the channel and grabber are closed, after the Main method
     has exited */
  quit_mutex.Wait ();

  gnomemeeting_threads_enter ();

  /* Disable the video settings frame */
  gtk_widget_set_sensitive (GTK_WIDGET (gw->video_settings_frame),
			    FALSE);

  /* Enable the video preview button */
  gtk_widget_set_sensitive (GTK_WIDGET (gw->preview_button),
			    TRUE);
  /* Display the logo */
  gnomemeeting_init_main_window_logo ();

  gnomemeeting_threads_leave ();
}


void GMVideoGrabber::Main ()
{
  /* Take the mutex, it will be released at the end of the method
     because at the end of the method, the class can be deleted. */
  quit_mutex.Wait ();

  while (is_running == 1) {

    if (has_to_open == 1)
      VGOpen ();

    if (has_to_close == 1)
      VGClose ();

    if (has_to_reset == 1) {

      if (is_opened) {

	VGClose ();
	VGOpen ();
	is_grabbing = 1;
      }

      has_to_reset = 0;
    }

    if (is_grabbing == 1) {

      grabbing_mutex.Wait ();
      
      channel->Read (video_buffer, height * width * 3);
      channel->Write (video_buffer, height * width * 3);
      
      grabbing_mutex.Signal ();
    }

    Current()->Sleep (100);
  }

  /* Disable the video preview button
     It will be reenabled in the constructeur
     that must be called just after the Main method exits. */
  gnomemeeting_threads_enter ();
  gtk_widget_set_sensitive (GTK_WIDGET (gw->preview_button), FALSE);
  gnomemeeting_threads_leave ();
  
  /* if opened, we close */
  if (is_opened == 1) {

    grabber->Close ();
    channel->Close ();
    Current ()->Sleep (500);

    delete (channel);
  }

  quit_mutex.Signal ();
}


void GMVideoGrabber::Open (int has_to_grab, int synchronous)
{
  /* If the user asks for a synchronous call (typically, from
     OpenVideoChannel), Open synchronously if an async opening
     was not on the road. If an async opening was on the road,
     then we wait till it is opened so that the user can be sure
     that the device is opened when this function returns. */
  if (synchronous == 1) {
    if (has_to_open != 1)
      VGOpen ();
    else
      while (!IsOpened ()) 
	{PThread::Current ()->Sleep (100);}
  }
  else 
    has_to_open = 1;
  
  is_grabbing = has_to_grab;
}


void GMVideoGrabber::Close (void)
{
  has_to_close = 1;
}


void GMVideoGrabber::Reset (void)
{
  has_to_reset = 1;
}


void GMVideoGrabber::Start (void)
{
  MyApp->Endpoint ()->SetCurrentDisplay (0);
  is_grabbing = 1;
}


void GMVideoGrabber::Stop (void)
{
  is_grabbing = 0;
}


int GMVideoGrabber::IsOpened (void)
{
  return is_opened;
}


GDKVideoOutputDevice *GMVideoGrabber::GetEncodingDevice (void)
{
  return encoding_device;
}


PVideoChannel *GMVideoGrabber::GetVideoChannel (void)
{
  return channel;
}


void GMVideoGrabber::SetColour (int colour)
{
  grabber->SetColour (colour);
}


void GMVideoGrabber::SetBrightness (int brightness)
{
  grabber->SetBrightness (brightness);
}


void GMVideoGrabber::SetWhiteness (int whiteness)
{
  grabber->SetWhiteness (whiteness);
}


void GMVideoGrabber::SetContrast (int constrast)
{
  grabber->SetContrast (constrast);
}



void GMVideoGrabber::GetParameters (int *whiteness, int *brightness, 
				  int *colour, int *contrast)
{
  *whiteness = (int) grabber->GetWhiteness () / 256;
  *brightness = (int) grabber->GetBrightness () / 256;
  *colour = (int) grabber->GetColour () / 256;
  *contrast = (int) grabber->GetContrast () / 256;
}


void GMVideoGrabber::VGOpen (void)
{
  gchar *msg = NULL;
  int error_code = -1;  // No error
  int tr_fps;
  


  /* The number of Transmitted FPS must be equal to 0 to disable */
  if (opts->fps == 0)
    tr_fps = 0;
  else
    tr_fps = opts->tr_fps;

  /* Disable the video preview button while opening */
  gnomemeeting_threads_enter ();
  gtk_widget_set_sensitive (GTK_WIDGET (gw->preview_button), FALSE);
  gnome_appbar_push (GNOME_APPBAR (gw->statusbar), _("Opening Video device"));
  gnomemeeting_log_insert (_("Opening Video device"));
  gnomemeeting_threads_leave ();

  channel = new PVideoChannel ();
  encoding_device = new GDKVideoOutputDevice (1, gw);
  grabber = new PVideoInputDevice();

  if (opts->video_size == 0) { 

    height = GM_QCIF_WIDTH; 
    width = GM_QCIF_HEIGHT; 
  }
  else { 

    height = GM_CIF_WIDTH; 
    width = GM_CIF_HEIGHT; 
  }
    
  encoding_device->SetFrameSize (height, width);

  if (!grabber->Open (opts->video_device, FALSE))
    error_code = 0;
  else
  if (!grabber->SetVideoFormat 
      (opts->video_format ? PVideoDevice::NTSC : PVideoDevice::PAL))
    error_code = 1;
  else
  if (!grabber->SetChannel (opts->video_channel))
    error_code = 2;
  else
  if (!grabber->SetColourFormatConverter ("YUV420P"))
    error_code = 3;
  else
  if (!grabber->SetFrameRate (tr_fps))
   error_code = 4;
  else
  if (!grabber->SetFrameSizeConverter (height, width, FALSE))
    error_code = 5;

  /* If no error */
  if (error_code == -1) {

    gnomemeeting_threads_enter ();
    
    msg = g_strdup_printf 
      (_("Successfully opened video device %s, channel %d"), 
       opts->video_device, opts->video_channel);
    gnomemeeting_log_insert (msg);
    gnome_appbar_push (GNOME_APPBAR (gw->statusbar), _("Done"));
    g_free (msg);

    gnomemeeting_threads_leave ();			     
  }
  else {

    gnomemeeting_threads_enter ();
    
    msg = g_strdup_printf 
      (_("Error while opening video device %s, channel %d.\nA test image will be transmitted."), opts->video_device, opts->video_channel);
    
    switch (error_code)	{

    case 0:
      msg = g_strconcat (msg, "\n", _("Error while opening the device."), 
			 NULL);
      break;
      
    case 1:
      msg = g_strconcat (msg, "\n", _("Your video driver doesn't support the requested video format."), NULL);
      break;

    case 2:
      msg = g_strconcat (msg, "\n", _("Could not open the chosen channel."),
			 NULL);
      break;
      
    case 3:
      msg = g_strconcat (msg, "\n", _("Your driver doesn't support the YUV420P format."), NULL);
      break;
      
    case 4:
      msg = g_strconcat (msg, "\n", _("Error with the frame rate."), NULL);
      break;

    case 5:
      msg = g_strconcat (msg, "\n", _("Error with the frame size."), NULL);
      break;
    }

    gnomemeeting_log_insert (msg);
    GtkWidget *msg_box = gnome_message_box_new (msg, GNOME_MESSAGE_BOX_ERROR,
						GNOME_STOCK_BUTTON_OK, NULL);
    g_free (msg);
    gtk_widget_show (msg_box);
    gnome_appbar_push (GNOME_APPBAR (gw->statusbar), _("Failed"));

    gnomemeeting_threads_leave ();			     


    /* delete the failed grabber and open the fake grabber */
    delete grabber;
      
    grabber = new PFakeVideoInputDevice();
    grabber->SetColourFormatConverter ("YUV420P");
    grabber->SetVideoFormat (PVideoDevice::PAL);
    grabber->SetChannel (150);    
    grabber->SetFrameRate (10);
    grabber->SetFrameSize (height, width);
  }

  grabber->Start ();
  
  channel->AttachVideoReader (grabber);
  channel->AttachVideoPlayer (encoding_device);

  has_to_open = 0;
  is_opened = 1;
  
  gnomemeeting_threads_enter ();
  

  /* Setup the video settings */
  GetParameters (&whiteness, &brightness, &colour, &contrast);
 
  gtk_adjustment_set_value (GTK_ADJUSTMENT (gw->adj_brightness),
			    brightness);
  gtk_adjustment_set_value (GTK_ADJUSTMENT (gw->adj_whiteness),
			    whiteness);
  gtk_adjustment_set_value (GTK_ADJUSTMENT (gw->adj_colour),
			    colour);
  gtk_adjustment_set_value (GTK_ADJUSTMENT (gw->adj_contrast),
			    contrast);
 
  /* Enable the video settings frame */
  gtk_widget_set_sensitive (GTK_WIDGET (gw->video_settings_frame),
			    TRUE);
  
  /* Enable the video preview button if not in a call */
  if (MyApp->Endpoint ()->GetCallingState () == 0)
    gtk_widget_set_sensitive (GTK_WIDGET (gw->preview_button),
			      TRUE);

  
  gnomemeeting_threads_leave ();
}


void GMVideoGrabber::VGClose (void)
{
  is_grabbing = 0;

  /* Disable the video preview button while closing */
  gnomemeeting_threads_enter ();
  gtk_widget_set_sensitive (GTK_WIDGET (gw->preview_button), FALSE);
  gnomemeeting_threads_leave ();
  
  grabbing_mutex.Wait ();
  
  grabber->Close ();
  channel->Close ();
  
  grabbing_mutex.Signal ();
  
  delete (channel);
  
  has_to_close = 0;
  is_opened = 0;
  
  /* Enable video preview button */
  gnomemeeting_threads_enter ();
  
  /* Enable the video preview button  */
  gtk_widget_set_sensitive (GTK_WIDGET (gw->preview_button), TRUE);

  /* Display the logo again */
  gnomemeeting_init_main_window_logo ();

  /* Disable the video settings frame */
  gtk_widget_set_sensitive (GTK_WIDGET (gw->video_settings_frame),
			    FALSE);
  
  gnomemeeting_threads_leave ();
}

