/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

/*
 * ContactInterface
 * Author Brandon Lees <brandon@sci.brooklyn.cuny.edu>
 */

#include "ContactInterface.hh"

#include "ContactInfoInterface.hh"
#include "GabberApp.hh"
#include "GabberUtility.hh"
#include "GabberWidgets.hh"

#include <libgnome/gnome-i18n.h>

using namespace jabberoo;
using namespace GabberUtil;

// Init function to register view with MessageManager
void ContactRecvView::init_view(MessageManager& mm)
{
     mm.register_view_type("jabber:x:roster", MessageManager::MessageViewInfo::vfExtension,
                           new_view_msg, new_view_jid, "glade-contact.xpm", "#999966");
}
 
MessageView* ContactRecvView::new_view_msg(const Message& m, ViewMap& vm)
{
     MessageView* mv = manage(new ContactRecvView(m, vm));
     return mv;
}
 
MessageView* ContactRecvView::new_view_jid(const string& jid, ViewMap& vm)
{
     MessageView* mv = manage(new ContactRecvView(jid, vm));
     return mv;
}

ContactRecvView::ContactRecvView(const jabberoo::Message& m, ViewMap& vm)
     : MessageView(m, vm), BaseGabberDialog("ContactRecv_dlg") 
{
     init(m.getFrom());

     render(m);
}

ContactRecvView::ContactRecvView(const string& jid, ViewMap& vm)
     : MessageView(jid, vm), BaseGabberDialog("ContactRecv_dlg")
{
     init(jid);
}

void ContactRecvView::init(string jid)
{
     _ctreePeople = getWidget<Gtk::CTree>("ContactRecv_People_ctree");
     Gtk::ScrolledWindow* tsw = getWidget<Gtk::ScrolledWindow>("ContactRecv_People_scroll");
     _RosterView = new SimpleRosterView(_ctreePeople, tsw, _roster, false); // Set arg 3 to true when groups are supported
     _RosterView->show_jid();

     getButton("ContactRecv_ViewInfo_btn")->clicked.connect(slot(this, &ContactRecvView::on_ViewInfo_clicked));
     getButton("ContactRecv_Remove_btn")->clicked.connect(slot(this, &ContactRecvView::on_Remove_clicked));
     getButton("ContactRecv_Add_btn")->clicked.connect(slot(this, &ContactRecvView::on_Add_clicked));
     getButton("ContactRecv_Cancel_btn")->clicked.connect(slot(this, &ContactRecvView::on_Cancel_clicked));

     PrettyJID* pj = manage(new PrettyJID(_jid, "", PrettyJID::dtNickRes));
     pj->show();
     getWidget<Gtk::HBox>("ContactRecv_JIDInfo_hbox")->pack_start(*pj, true, true, 0);

     _thisWindow->set_title(_("Received Contacts(s) from ") + fromUTF8(pj->get_nickname()) + _(" - Gabber"));
}

ContactRecvView::~ContactRecvView()
{
     delete _RosterView;
}

void ContactRecvView::render(const jabberoo::Message& m)
{
     Tag* x = m.getBaseTag().getTag("x", "jabber:x:roster");
     if (!x)
     {
	  cerr << "Got a Message without a roster extension in ContactRecvView::render?!?!" << endl;
	  return;
     }
     ElementList::const_iterator it = x->getChildren().begin();
     for (; it != x->getChildren().end(); it++)
     {
	  Tag& item = *static_cast<Tag*>(*it);

	  if (item.getName() != "item")
	       continue;
	  string jid = item.getAttrib("jid");
	  string nick = item.getAttrib("name");
	  cerr << "Got roster item " << jid << " " << nick << endl;
	  Roster::Item ritem(jid, nick);
	  ElementList::const_iterator git = item.getChildren().begin();
	  for ( ; git != item.getChildren().end(); git++)
	  {
	       Tag& group = *static_cast<Tag*>(*git);
	       if (group.getName() != "group")
		    continue;

	       ritem.addToGroup(group.getData());
	       cerr << "Got group " << group.getData() << endl;
	  }
	  if (ritem.empty())
	       ritem.addToGroup("Unfiled");
	  _roster.insert(make_pair(jid, ritem));
     }
     _RosterView->refresh();
}

void ContactRecvView::on_Remove_clicked()
{
     if (_ctreePeople->selection().begin() != _ctreePeople->selection().end())
     {
          Gtk::CTree::Row& r = *_ctreePeople->selection().begin();
	  Roster::ItemMap::iterator it = _roster.find(r[_RosterView->_colJID].get_text());
	  if (it != _roster.end())
	  {
	       _roster.erase(it);
	       _RosterView->refresh();
	  }
     }
}

void ContactRecvView::on_ViewInfo_clicked()
{
     if (_ctreePeople->selection().begin() != _ctreePeople->selection().end())
     {
          Gtk::CTree::Row& r = *_ctreePeople->selection().begin();
          // Check the roster for the jid to make sure it isn't a group
          Roster::ItemMap::iterator it = _roster.find(r[_RosterView->_colJID].get_text());
          if (it != _roster.end())
          {
	       Roster::Subscription type;
	       // Try getting the subscription
	       try {
		    type = G_App->getSession().roster()[JID::getUserHost(it->first)].getSubsType();
	       } catch (Roster::XCP_InvalidJID& e) {
		    type = Roster::rsNone;
	       }
               ContactInfoDlg::display(it->first, type);
          }
     }
}

void ContactRecvView::on_Add_clicked()
{
     for (Roster::ItemMap::iterator it = _roster.begin(); it != _roster.end(); it++)
     {
	  // If the JID is not already on the roster, add it
	  if (!G_App->getSession().roster().containsJID(JID::getUserHost(it->first)))
	  {
	       G_App->getSession().roster() << Roster::Item(JID::getUserHost(it->first), it->second.getNickname());
	       G_App->getSession() << Presence(JID::getUserHost(it->first), Presence::ptSubRequest);
	  }
     }
     close();
}

void ContactRecvView::on_Cancel_clicked()
{
     close();
}

///////////////////////////////////////////////////
// Contact Send Dialog
///////////////////////////////////////////////////

ContactSendDlg::ContactSendDlg(const string& jid)
     : BaseGabberDialog("ContactSend_dlg"),
       _jid(jid)
{
     _ctreePeople = getWidget<Gtk::CTree>("ContactSend_People_ctree");
     Gtk::ScrolledWindow* tsw = getWidget<Gtk::ScrolledWindow>("ContactSend_People_scroll");
     _RosterView = new SimpleRosterView(_ctreePeople, tsw, _roster, false); // Set arg 3 to true when groups are supported
     _RosterView->show_jid();
     _entResource = getEntry("ContactSend_Resource_ent");
     _cboResource = getWidget<Gtk::Combo>("ContactSend_Resource_cbo");

     PrettyJID* pj = manage(new PrettyJID(_jid, "", PrettyJID::dtNick));
     pj->show();
     getWidget<Gtk::HBox>("ContactSend_JIDInfo_hbox")->pack_start(*pj, true, true, 0);

     _thisWindow->set_title(_("Send Contacts(s) to ") + fromUTF8(pj->get_nickname()) + _(" - Gabber"));

     getButton("ContactSend_ViewInfo_btn")->clicked.connect(slot(this, &ContactSendDlg::on_ViewInfo_clicked));
     getButton("ContactSend_Remove_btn")->clicked.connect(slot(this, &ContactSendDlg::on_Remove_clicked));
     getButton("ContactSend_Send_btn")->clicked.connect(slot(this, &ContactSendDlg::on_Send_clicked));
     getButton("ContactSend_Cancel_btn")->clicked.connect(slot(this, &ContactSendDlg::on_Cancel_clicked));

     // If it's a group chat id, don't bother with resources
     if (G_App->isGroupChatID(_jid))
     {
	  getWidget<Gtk::HBox>("ContactSend_Resource_hbox")->hide();
     }
     else
     {
	  list<string> resources;
	  try {
	       // Walk the list of resources and add them to the combo box
	       PresenceDB::range r = G_App->getSession().presenceDB().equal_range(JID::getUserHost(_jid));
	       for (PresenceDB::const_iterator it = r.first; it != r.second; it++)
	       {
		    const Presence& p = *it;
		    
		    // If this presence is a NA presence, then skip it
		    if (p.getType() == Presence::ptUnavailable)
			 continue;
		    
		    // Extract the resource
		    const string& res = JID::getResource(p.getFrom());
		    resources.push_back(res);
	  }
	  } catch (PresenceDB::XCP_InvalidJID& e) {
	       // No presences from any resources
	  }
	  if (!resources.empty())
	       _cboResource->set_popdown_strings(resources);
     }
}

ContactSendDlg::~ContactSendDlg()
{
     delete _RosterView;
}

void ContactSendDlg::push_drag_data(GdkDragContext* drag_ctx, gint x, gint y, GtkSelectionData* data, guint info, guint time)
{
     _RosterView->push_drag_data(drag_ctx, x, y, data, info, time);
}

void ContactSendDlg::on_ViewInfo_clicked()
{
     if (_ctreePeople->selection().begin() != _ctreePeople->selection().end())
     {
          Gtk::CTree::Row& r = *_ctreePeople->selection().begin();
	  // Check the roster for the jid to make sure it isn't a group
          Roster::ItemMap::iterator it = _roster.find(r[_RosterView->_colJID].get_text());
          if (it != _roster.end())
          {
	       ContactInfoDlg::display(it->first);
	  }
     }
}

void ContactSendDlg::on_Remove_clicked()
{
     if (_ctreePeople->selection().begin() != _ctreePeople->selection().end())
     {
          Gtk::CTree::Row& r = *_ctreePeople->selection().begin();
          Roster::ItemMap::iterator it = _roster.find(r[_RosterView->_colJID].get_text());
	  if (it != _roster.end())
	  {
	       _roster.erase(it);
	       _RosterView->refresh();
	  }
     }
}

void ContactSendDlg::on_Send_clicked()
{
     string body = _("Jabber Contacts attached:");
     // Walk the list and add the JIDs to the message
     for (Roster::ItemMap::iterator it = _roster.begin(); it != _roster.end(); it++)
     {
	  Roster::Item& ritem = it->second;
	  body += "\n" + ritem.getNickname() + " - " + "jabber:" + ritem.getJID();
     }

     Message m(_jid, body, Message::mtNormal);
     Tag& x = m.addX();
     x.putAttrib("xmlns", "jabber:x:roster");
     // Walk the list and attach the JIDs
     for (Roster::ItemMap::iterator it = _roster.begin(); it != _roster.end(); it++)
     {
	  Roster::Item& ritem = it->second;

	  Tag& item = x.addTag("item");
	  item.putAttrib("jid", ritem.getJID());
	  item.putAttrib("name", ritem.getNickname());
	  for (Roster::Item::iterator git = ritem.begin(); git != ritem.end(); git++)
	       item.addTaggedCDATA("group", *git);
     }
     G_App->getSession() << m;
     close();
}

void ContactSendDlg::on_Cancel_clicked()
{
     close();
}
