/*
 *          Copyright (c) mjh-EDV Beratung, 1996-1999
 *     mjh-EDV Beratung - 63263 Neu-Isenburg - Rosenstrasse 12
 *          Tel +49 6102 328279 - Fax +49 6102 328278
 *                Email info@mjh.teddy-net.com
 *
 *   Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 *   $Id: rnd-pool.c,v 1.2 1999/11/03 23:36:44 jordan Exp $
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common-stuff.h"

#include <fcntl.h>

#ifdef USE_PTHREADS
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif
#endif

#ifdef HAVE_ASSERT_H
#include <assert.h>
#else
#define assert(x)
#endif

#ifndef HAVE_RAND
#define rand() random ()
#endif

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include "cipher.h"
#include "messages.h"
#include "rnd-pool.h"

#define DUMP_MUTEX

/* ------------------------------------------------------------------------- *
 *               private defs and data and structures                        *
 * ------------------------------------------------------------------------- */

#define HASH_TYPE        "ripemd160"
#define COMPRESSION_UNIT 3 /* an integer: 3 units are compressed to 1 */
#define DIM(x)           (sizeof (x) / sizeof (x [0]))

/* #undef  RND_POOL_SIZE */
/* #define RND_POOL_SIZE 500 */

/* ------------------------------------------------------------------------- *
 *                  private variables                                        *
 * ------------------------------------------------------------------------- */

/* the fifo with random data */
static char pool [RND_POOL_SIZE] ;
static int  get_inx = -1, put_inx =  0 ;

/* number of bytes in the pool not compressed, yet */
static int uncompressed = 0;

/* The density gives a means to check whats in the pool. It is
   calculated total 
         
               #incoming-data - #outgoing-data 
      ro   =   -------------------------------
                       #pool-size

   Adding some data #delta-in and probably compresssing results in
   the incremental formula

               #incoming-data + #delta-in - #outgoing-data 
      ro'  =   -------------------------------------------  =
                            #pool-size'

               ro * #pool-size + #delta-in
           =   ---------------------------
                    #pool-size'

   and removing some data #delta-out gives

               #incoming-data - #outgoing-data - #delta-out
      ro''  =   -------------------------------------------  =
                            #pool-size'

               ro * #pool-size - #delta-out
           =   ----------------------------   for #pool-size'' > 0.
                    #pool-size''

*/

static float density = 1.0; 

/* ------------------------------------------------------------------------- *
 *                   private functions                                       *
 * ------------------------------------------------------------------------- */

#define SYSNAME "rnd-pool"
#include "peks-sema.h"

/*
 *  organization of the fifo:
 *  -------------------------
 *
 *
 *   1. standard case
 *   ----------------
 *
 *   output
 *     /\
 *     ||
 *   |    | <- 0, bottom
 *   |    |
 *   | XX | <- get_inx, first busy entry
 *   | XX |
 *   | XX |
 *   | :: |
 *   |    |
 *   | XX |
 *   |    | <- put_inx, first free entry
 *   |    |
 *     /\   <- DIM, 1 beyond largest index
 *     ||
 *    input
 *
 *
 *   2. wrap around
 *   --------------
 *
 *   | XX | <- 0, bottom
 *   | XX |
 *   |    | <- put_inx, first free entry
 *   |    |
 *   |    |
 *   | XX | <- get_inx, first busy entry
 *   | XX |
 *   | :: |
 *   |    |
 *   | XX |
 *   | XX |
 *          <- DIM, 1 beyond largest address
 *
 *
 *   3. fifo full/empty
 *   ------------------
 *
 *   full:     put_inx == get_inx >= 0
 *   not full: put_inx != get_inx
 *   empty:    put_inx  > get_inx == -1
 */

static unsigned
pool_size (void)
{
  int n ;
  
  if ((n = put_inx - get_inx) <= 0)
    return DIM (pool) - n ;

  return get_inx < 0 ? 0 : n ;
}

static unsigned
put_data 
  (const char *p, 
   unsigned    n)
{
  int avail, stored = 0 ;

  if (n == 0)
    /* nothing to do */
    return 0;
  
  /* 1st case, normal mode, fifo may be empty */
  if (get_inx < put_inx) {
    
    /* store the maximum value, possible */
    if ((stored = DIM (pool) - put_inx) > (int)n) 
      stored = n ;
    memcpy (pool + put_inx, p, stored) ;

    /* update indices, wrap around if necessary */
    if ((put_inx += stored) == DIM (pool))
      put_inx = 0;
    if (get_inx < 0)
      get_inx = 0 ;

    /* done, everything stored ? */
    if ((n -= stored) == 0)
      return stored ;

    /* otherwise proceed with the 2nd case */
    p += stored ;

    assert (put_inx == 0);
  }

  /* 2nd case, calculate the space, available */
  if ((avail = get_inx - put_inx) == 0)
    /* fifo full, no more data stored */
    return stored ;

  /* store the maximum value, possible */
  if (avail > (int)n) 
    avail = n ;
  memcpy (pool + put_inx, p, avail) ;
  
  /* update index */
  put_inx += avail ;
  return stored + avail ;
}



static unsigned
get_data 
  (char    *p, 
   unsigned n)
{
  int avail, retrieved = 0 ;

  if (n == 0)
    /* nothing to do */
    return 0;

  /* do the 2nd case, fifo may be full */
  if (get_inx >= put_inx) {

    /* get as much as possible */
    if ((retrieved = DIM (pool) - get_inx) > (int)n)
      retrieved = n ;
    memcpy (p, pool + get_inx, retrieved) ;

    /* update index, fifo might be empty, now */
    if ((get_inx += retrieved) == DIM (pool))
      get_inx = put_inx ? 0 : -1 ;
    
    /* done, everything loaded ? */
    if ((n -= retrieved) == 0)
      return retrieved;

    /* otherwise proceed with the 1st case */
    p += retrieved ;

    assert (get_inx == 0);
  }

  /* 1st case, normal mode, fifo may be empty */
  if (get_inx < 0)
    return retrieved ;

  /* get the maximum value, possible */
  if ((avail = put_inx - get_inx) > (int)n)
    avail = n;
  memcpy (p, pool + get_inx, avail) ;

  /* update index, fifo may become empty */
  if ((get_inx += avail) == put_inx) {
    put_inx = 0 ;
    get_inx = -1;
  }

  return retrieved + avail ;
}


static void
compress_data (void)
{
  static frame_desc *md ;
  char *s ;
  int chunk, len, n ;

  if (md == 0) {
    md = create_frame (find_frame_class (HASH_TYPE), 0);
    assert (md != 0);
  }

  /* buffer size: compression unit + overlapping space */
  chunk = COMPRESSION_UNIT * md->mdlen ;
  s     = ALLOCA (chunk + md->mdlen) ;
  len   = pool_size ();

  /* start: fill the overlapping space */
  len -= get_data (s + chunk,  md->mdlen) ;

  /* loop over the number of remaining data in the pool */
  while (len > 0) {

    /* shift left the overlapping space */
    memcpy (s, s + chunk, md->mdlen);

    /* append next data */
    n = get_data (s + md->mdlen, chunk) ;
    
    /* hash the compression unit + overlapping space */
    XCRCFIRST (md, s, n + md->mdlen) ;

    /* replace the compressed unit */
    put_data (XCRCRESULT0 (md), md->mdlen) ;

    /* update counter */
    len -= n;
  }
  DEALLOCA (s);
}


/* ------------------------------------------------------------------------- *
 *                    public functions                                       *
 * ------------------------------------------------------------------------- */

unsigned
get_random_pool_data
  (char     *buf,
   unsigned  len)
{
  int retrieved, n, available ;
  float old_size ;

  __enter_lock_semaphore () ;

  n         = pool_size () ;
  old_size  = (float)n ;
  available = n - uncompressed ;

  assert (available >= 0);

  /* dont let out any data compressed at least once */
  if (n > available)
    n = available ;
  if (n == 0) {
    __release_lock_semaphore () ;
    return 0;
  }

  retrieved = get_data (buf, len) ;

  if ((n = pool_size ()) == 0)
    density = 1.0 ;
  else {
    density *=  old_size ;
    density -= retrieved ;
    density /=         n ;
  }

  __release_lock_semaphore () ;
  return retrieved ;
}



/* ------------------------------------------------------------ */
#ifdef USE_PTHREADS
static unsigned put_random_pool_data_nosync
  (const char *buf, unsigned len) ;

unsigned
put_random_pool_data
  (const char *buf,
   unsigned    len)
{
  int n ;
  __enter_lock_semaphore () ;
  n = put_random_pool_data_nosync (buf, len);
  __release_lock_semaphore () ;
  return n;
}
static /* function def follows ... */
#else
#define put_random_pool_data_nosync put_random_pool_data
#endif
/* ------------------------------------------------------------ */


unsigned
put_random_pool_data_nosync
  (const char *buf,
   unsigned    len)
{
  int n, processed, left;
  float old_size ;

  old_size  = (float)pool_size () ;
  processed =   0;
  left      = len;
  n         =   0;

  while (left && (n = put_data (buf + processed, left)) < left) {
    processed += n ;
    left      -= n ;
    compress_data () ;
    uncompressed = 0 ;
  }
  uncompressed += n ;
  
  density *=  old_size ;
  density +=       len ;
  density /= pool_size () ;

  return len ;
}


void
point_of_random_time
  (const char *buf,
   unsigned    len)
{
  __enter_lock_semaphore () ;

#ifdef HAVE_GETTIMEOFDAY2
  /* current time in millisecs, or centisecs etc. */
  { struct timeval tv ; gettimeofday2 (&tv, 0);
  put_random_pool_data_nosync ((char*)&tv, sizeof (tv)); }
#else
  { time_t ti = time (0);
  put_random_pool_data_nosync ((char*)&ti, sizeof (ti)); }
#endif

  /* output from the random generator */
  { int r = rand () ; put_random_pool_data_nosync ((char*)&r, sizeof (r)); }

  /* check, whether the user has some more */
  if (buf == 0 || len == 0) {
    __release_lock_semaphore () ;
    return ;
  }
  put_random_pool_data_nosync (buf, len);

#ifdef HAVE_GETTIMEOFDAY2
  /* again, current time in millisecs, or centisecs etc. */
  { struct timeval tv ; gettimeofday2 (&tv, 0);
  put_random_pool_data_nosync ((char*)&tv, sizeof (tv)); }
#else
  { time_t ti = time (0); /* will probably be the same, as above */
  put_random_pool_data_nosync ((char*)&ti, sizeof (ti)); }
#endif
  
  __release_lock_semaphore () ;
}


unsigned
random_pool_density_percentage
  (void)
{
  unsigned n;
  __enter_lock_semaphore () ;
  n = (unsigned)(density * 100) ;
  __release_lock_semaphore () ;
  return n;
}


#ifdef USE_PTHREADS
void rnd_pool_sema_create (void *attr) { __sema_create (attr); }
void rnd_pool_sema_destroy      (void) { __sema_destroy    (); }
#endif
